/* Copyright 2013 Azul Systems, Inc.  All Rights Reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it under 
 * the terms of the GNU General Public License version 2 only, as published by 
 * the Free Software Foundation. 
 *
 * Azul designates this particular file as subject to the "Classpath" exception 
 * as provided by Azul in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT ANY 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 * A PARTICULAR PURPOSE.  See the GNU General Public License version 2 for more
 * details (a copy is included in the LICENSE file that accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version 2 
 * along with this work; if not, write to the Free Software Foundation, Inc., 
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 * 
 * Please contact Azul Systems, Inc., 1173 Borregas Avenue, Sunnyvale, CA 94089
 * USA or visit www.azulsystems.com if you need additional information or have
 * any questions.
 *
 * File: com/azulsystems/misc/VM.java
 */

package com.azulsystems.misc;
import java.lang.ref.Reference;

public class VM {
    private static class UnsafeHolder {
        private static final sun.misc.Unsafe unsafe = sun.misc.Unsafe.getUnsafe();
    }


    private static native void trackReservedDirectMemory0(long reservedMemory);
    private static native long allocateMmapMemory0(long bytes);
    private static native void freeMmapMemory0(long addr, long bytes);
    private static native long getNewThreadID0();
    private static native void releaseThreadID0(long tid);

    /**
     * These actual values for these fields are injected by JVM. The static
     * initialization block is used to set the values here to communicate
     * that these static final fields are not statically foldable, and to
     * avoid any possible circular dependency during vm initialization.
     */
    private static final int threadLocalMapInitialCapacity;
    private static final boolean useMmapForDirectByteBuffers;
    private static final boolean optimizedThreadLookup;
    private static final boolean specialEncodeISOArray;
    private static final int specialEncodeISOArrayMinLength;
    private static final boolean specialStringHashCode;
    private static final int specialStringHashCodeMinLength;
    private static final int javaLangRefReferentOffset;
    static {
      threadLocalMapInitialCapacity = 0;
      useMmapForDirectByteBuffers = false;
      optimizedThreadLookup = false;
      specialEncodeISOArray = false;
      specialEncodeISOArrayMinLength = 0;
      specialStringHashCode = false;
      specialStringHashCodeMinLength = 0;
      javaLangRefReferentOffset = 0;
    }

    public static int getThreadLocalMapInitialCapacity() {
        return threadLocalMapInitialCapacity;
    }

    public static void trackReservedDirectMemory(long reservedMemory) {
        trackReservedDirectMemory0(reservedMemory);
    }

    public static long allocateMmapMemory(long bytes) {
        return allocateMmapMemory0(bytes);
    }

    public static void freeMmapMemory(long addr, long bytes) {
        freeMmapMemory0(addr, bytes);
    }

    public static boolean getUseMmapForDirectByteBuffers() {
        return useMmapForDirectByteBuffers;
    }

    public static long objectFieldOffset(java.lang.reflect.Field f) throws Exception {
        return UnsafeHolder.unsafe.objectFieldOffset(f);
    }

    public static boolean isOptimizedThreadLookupEnabled() {
      return optimizedThreadLookup;
    }

    public static long getNewThreadID() {
      return getNewThreadID0();
    }

    public static void releaseThreadID(long tid) {
      releaseThreadID0(tid);
    }

    public static boolean isSpecialEncodeISOArrayEnabled() {
      return specialEncodeISOArray;
    }

    public static int getSpecialEncodeISOArrayMinLength() {
      return specialEncodeISOArrayMinLength;
    }

    public static boolean isSpecialStringHashCodeEnabled() {
      return specialStringHashCode;
    }

    public static int getSpecialStringHashCodeMinLength() {
      return specialStringHashCodeMinLength;
    }

    public static Object staticGetReferent(Reference<?> ref) {
      return UnsafeHolder.unsafe.getObject(ref, javaLangRefReferentOffset);
    }

    public static void staticSetReferent(Reference<?> ref, Object val) {
      UnsafeHolder.unsafe.putObject(ref, javaLangRefReferentOffset, val);
    }
}
