//===-- VMCallbacks.h -------------------------------------------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2018 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//
//
// Manage a set of callbacks that a running virtual machine (or something
// equivalent) can install into Orca to help it optimize better.
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_SUPPORT_AZUL_VM_CALLBACKS_H
#define LLVM_SUPPORT_AZUL_VM_CALLBACKS_H

#include "llvm/ADT/ArrayRef.h"
#include "llvm/ADT/StringRef.h"
#include <optional>

namespace llvm {
class Function;
}

namespace azul {
namespace orca {

class JavaTypeInfo;
struct FieldInfo;
class FunctionInfo;

/// Specifies a collection of callbacks leveraged by Orca to get information
/// from the VM (or stub implementations when run outside the VM.)  It is
/// *required* that both the arguments and the results of these functions be
/// easily serializable.
struct Callbacks {

  typedef std::optional<bool> (*ReportJavaMethodInliningDecision)(
      uint64_t, 
      llvm::StringRef /* caller_name */,
      llvm::StringRef /* callee_name */,
      llvm::ArrayRef<uint64_t> /* CallerIDs */,
      llvm::ArrayRef<uint64_t> /* CallerBCIs */, 
      llvm::StringRef /* message */,
      bool /* successful */);
  ReportJavaMethodInliningDecision
      _report_java_method_inlining_decision_callback = nullptr;  

  typedef std::optional<bool> (*GetVMInliningAdvice)(
      llvm::StringRef /* caller_name */,
      llvm::StringRef /* callee_name */,
      llvm::ArrayRef<uint64_t> /* CallerIDs */,
      llvm::ArrayRef<uint64_t> /* CallerBCIs */);
  GetVMInliningAdvice
      _get_vm_inlining_advice = nullptr;        
  
  /// (super, sub) -> known relationship
  typedef std::optional<bool> (*IsSubtypeOfType)(int64_t, int64_t);
  IsSubtypeOfType _is_subtype_of_callback = nullptr;

  /// (kid, exact type?) ->
  typedef std::optional<bool> (*RequiresFinalizerType)(int64_t, bool);
  RequiresFinalizerType _requires_finalizer_callback = nullptr;

  /// (name, unused) -> function body
  /// TODO: remove unused argument and rename to get_function_body
  typedef std::optional<llvm::Function *> (*GetInlineCandidate)(
      llvm::StringRef, llvm::StringRef);
  GetInlineCandidate _get_inline_candidate_callback = nullptr;

  /// (known value id, offset, size) -> load result
  typedef std::optional<uint64_t> (*GetLoadResultKnownValue)(uint64_t, int64_t,
                                                             uint32_t);
  GetLoadResultKnownValue _get_load_result_known_value_callback = nullptr;

  typedef std::optional<uint64_t> (*GetLoadResultKlass)(int32_t, int64_t,
                                                        uint32_t);
  GetLoadResultKlass _get_load_result_klass_callback = nullptr;

  typedef std::optional<uint64_t> (*GetKnownValueKey)(uint64_t);
  GetKnownValueKey _get_known_value_key_callback = nullptr;

  typedef std::optional<uint64_t> (*GetKnownValueDependentKey)(uint64_t,
                                                               int64_t);
  GetKnownValueDependentKey _get_known_value_key_dependent_callback = nullptr;

  typedef std::optional<uint64_t> (*GetUniqueSubtype)(uint64_t, bool);
  GetUniqueSubtype _get_unique_subtype_callback = nullptr;

  typedef std::optional<bool> (*IsUnmodifiedByCall)(uint64_t, int64_t, uint32_t,
                                                    llvm::StringRef);
  IsUnmodifiedByCall _is_unmodified_by_call_callback = nullptr;

  typedef std::optional<bool> (*IsUnmodifiedByJavaCall)(uint64_t, int64_t,
                                                        uint32_t, uint64_t,
                                                        uint64_t);
  IsUnmodifiedByJavaCall _is_unmodified_by_java_call_callback = nullptr;

  typedef std::optional<uint64_t> (*GetObjArrayElementKid)(uint64_t);
  GetObjArrayElementKid _get_obj_array_element_kid = nullptr;

  typedef std::optional<llvm::Function *> (*GetInlineCacheCallStub)(uint64_t,
                                                                     uint64_t);
  GetInlineCacheCallStub _get_inline_cache_call_stub_callback = nullptr;

  typedef std::optional<uint64_t> (*GetVMIntegerConstant)(llvm::StringRef);
  GetVMIntegerConstant _get_vm_integer_constant = nullptr;

  typedef std::optional<JavaTypeInfo> (
      *GetJavaTypeInfo)(uint64_t /* klass_id */, uint64_t /* is_exact */,
      std::optional<uint64_t> /* ArrayLen */);
  GetJavaTypeInfo _get_java_type_info_callback = nullptr;
  
  typedef std::optional<FieldInfo> (
      *GetFieldInfoAtOffset)(uint64_t /* klass_id */, uint64_t /* is_exact */,
                             uint64_t /* is_new */, int64_t /* offset */);
  GetFieldInfoAtOffset _get_field_info_at_offset_callback = nullptr;

  typedef std::optional<FieldInfo>(*GetFieldInfoAtOffsetForKnownValue)(
      uint64_t /* KVID */, int64_t /* offset */);
  GetFieldInfoAtOffsetForKnownValue
  _get_field_info_at_offset_for_known_value_callback = nullptr;

  typedef std::optional<uint64_t> (*LeastCommonAncestor)(uint64_t, uint64_t);
  LeastCommonAncestor _least_common_ancestor = nullptr;

  typedef std::optional<uint64_t> (*GetObjectHashcode)(uint64_t);
  GetObjectHashcode _get_object_hashcode_callback = nullptr;

  typedef std::optional<uint64_t> (*GetObjectGeneration)(uint64_t);
  GetObjectGeneration _get_object_generation_callback = nullptr;

  typedef std::optional<uint64_t> (*GetObjectKlassID)(uint64_t);
  GetObjectKlassID _get_object_klass_id_callback = nullptr;

  typedef std::optional<uint64_t> (*GetKlassIDForJavaLangClass)(uint64_t);
  GetKlassIDForJavaLangClass _get_klass_id_for_java_lang_class_callback = nullptr;

  typedef std::optional<llvm::Function *> (*GetMergedAllocFunc)(
      llvm::ArrayRef<uint64_t>);
  GetMergedAllocFunc _get_merged_alloc_func_callback = nullptr;

  // function_name - function to specialize.
  //
  // arguments_info - an array of flags which indicate what we know about the
  //   arguments, see azul::orca::ArgumentInfo for available flags.
  //
  // arguments_kids - an array of klass ids for the arguments. 
  //   If arguments_info[n] has ARGUMENT_INFO_HAS_KNOWN_TYPE flag set 
  //   arguments_kids[n] contains the klass id of the value passed as the n-th 
  //   argument.
  //   If arguments_info[n] has both ARGUMENT_INFO_HAS_KNOWN_TYPE and 
  //   ARGUMENT_INFO_HAS_EXACT_TYPE flags set the type of the value passed as
  //   the n-th argument is known to be exact.
  //   If arguments_info[n] doesn't have ARGUMENT_INFO_HAS_KNOWN_TYPE flag set
  //   arguments_kids[n] content is unspecified.
  //
  // arguments_consts is an array of constant values of the arguments. 
  //   If arguments_info[n] has ARGUMENT_INFO_IS_CONSTANT flag set 
  //   arguments_consts[n] contains:
  //   a) either the primitive constant value passed as the n-th argument, or
  //   b) the known value ID of the pointer passed as the n-th argument. 
  //   If arguments_info[n] doesn't have ARGUMENT_INFO_IS_CONSTANT flag set.
  //   arguments_consts[n] content is unspecified 
  // 
  // caller_ids, caller_bcis contain the calling context information from the 
  //   JVM state of the call or empty if there is no JVM state.
  typedef std::optional<llvm::Function *>(*GetSpecializedImplementation)(
      llvm::StringRef /* function_name */, 
      llvm::ArrayRef<uint64_t> /* arguments_info */, 
      llvm::ArrayRef<uint64_t> /* arguments_kids */,
      llvm::ArrayRef<uint64_t> /* arguments_consts */,
      llvm::StringRef /* caller_name */,
      llvm::ArrayRef<uint64_t> /* caller_ids */,
      llvm::ArrayRef<uint64_t> /* caller_bcis */);
  GetSpecializedImplementation _get_specialized_implementation_callback =
      nullptr;

  // See _get_specialized_implementation_callback for argument description.
  typedef std::optional<uint64_t> (*GetConstantResult)(
      llvm::StringRef /* function_name */,
      llvm::ArrayRef<uint64_t> /* arguments_info */,
      llvm::ArrayRef<uint64_t> /* arguments_kids */,
      llvm::ArrayRef<uint64_t> /* arguments_consts */,
      llvm::ArrayRef<uint64_t> /* caller_ids */,
      llvm::ArrayRef<uint64_t> /* caller_bcis */);
  GetConstantResult _get_constant_result_callback = nullptr;

  typedef std::optional<std::string> (*GetKnownValueGlobalName)(uint64_t);
  GetKnownValueGlobalName _get_known_value_global_name_callback = nullptr;

  typedef std::optional<FunctionInfo> (*GetFunctionInfo)(uint64_t);
  GetFunctionInfo _get_function_info_callback = nullptr;

  typedef std::optional<uint32_t> (*GetSpecializedFunctionID)(llvm::StringRef,
                                                              llvm::StringRef);
  GetSpecializedFunctionID _get_specialized_function_id_callback = nullptr;

  typedef std::optional<uint64_t> (*GetKnownValueID)(llvm::StringRef);
  GetKnownValueID _get_known_value_id_callback = nullptr;

  typedef std::optional<bool> (*IsCachedBoxObject)(uint64_t);
  IsCachedBoxObject _is_cached_box_object_callback = nullptr;

  typedef std::optional<uint64_t> (*GetBoxedValueOffset)(uint64_t);
  GetBoxedValueOffset _get_boxed_value_offset_callback = nullptr;

  typedef std::optional<uint64_t> (*RunTimeToCompileTimeKlassID)(uint64_t);
  RunTimeToCompileTimeKlassID _run_time_to_compile_time_klass_id_callback = nullptr;

  typedef std::optional<uint64_t> (*CompileTimeToRunTimeKlassID)(uint64_t);
  CompileTimeToRunTimeKlassID _compile_time_to_run_time_klass_id_callback = nullptr;

  typedef std::optional<bool> (*HasExceptionHandlerFor)(uint64_t, uint64_t,
                                                        uint64_t);
  HasExceptionHandlerFor _has_exception_handler_for_callback = nullptr;

  typedef std::optional<bool> (*InterruptCompilation)(llvm::StringRef);
  InterruptCompilation _interrupt_compilation_callback = nullptr;

  // This is a temporary callback necessary to run remote compilations within
  // the runtime process.
  bool (*_ignore_callbacks)(void) = nullptr;
};

/// Set the callbacks used by Orca to ask the VM for information.  These are
/// process global and can not vary by thread.  There's no strong reason for
/// this, we simply haven't bothered to store it in the LLVMContext.
///
/// NB! This is **NOT** thread safe.
void setVMCallbacks(const Callbacks &callbacks);

/// Return said VM callbacks.  If the resident callbacks have not yet been
/// initialized then this will return a zeroed struct (i.e. all of the function
/// pointers will be nullptr).
const Callbacks &getVMCallbacks();
}
}

#endif
