///===-- QueryCacheSerialization.h - ----------------------------*- C++ -*-===//
///
///                     The LLVM Compiler Infrastructure
///
/// This file is distributed under the University of Illinois Open Source
/// License. See LICENSE.TXT for details.
///
///===---------------------------------------------------------------------===//
/// Copyright 2013-2018 Azul Systems, Inc.  All Rights Reserved.
/// http://www.azul.com
/// Azul Systems is a contributor to the LLVM Team.
/// Distributed under the same license terms detailed in LICENSE.TXT above.
///===---------------------------------------------------------------------===//
///
/// Utilities to serialize and deserialize query caches.  For an
/// overview of the query cache data structure, see QueryCache.h
///
/// Contract for \p TextualTraits:
///
///     template<> struct QueryTraits<N> {
///       // The result type of the query corresponding to N
///       typedef $TYPE ResultTy;
///
///       // Whether the callbacks backing the queries are expected
///       // to be idempotent or not (now defunct, only used for asserts)
///       typedef $TYPE ResultTy;
///
///       // The string representation of the query type N (e.g.
///       // "is-subtype-of").
///       static llvm::StringRef queryKindString();
///
///       // A function that can serialize \p Value into \p OS in a way that
///       // it can parse back later.
///       static void serializeResult(ResultTy Value, llvm::raw_ostream &OS,
///         ...)
///
///       // A function that can deserialize a value of type \p ResultTy
///       // that was serialized by serializeResult.  On failure, mark \p QC
///       // as failed, and return None.
///       static Optional<ResultTy> deserializeResult(QCLexer &QC, ...)
///     }
///
///===---------------------------------------------------------------------===//

#ifndef LLVM_SUPPORT_AZUL_QUERY_CACHE_SERIALIZATION_H
#define LLVM_SUPPORT_AZUL_QUERY_CACHE_SERIALIZATION_H

#include "llvm/Support/Orca/QueryCache.h"

namespace azul {

namespace qc_utils {
class QCLexer;
}

namespace qc_utils {
/// Utility to print a length prefixed string.  Not in the \c qc_detail
/// namespace since this will be used outside this header.
inline void printLengthPrefixedString(llvm::StringRef Str,
                                      llvm::raw_ostream &OS) {
  OS << "s" << Str.size() + 2 << " |" << Str << "| ";
}

inline bool isWhitespace(char c) { return c == ' ' || c == '\n'; }

/// A helper object used by the deserializer (and hooks it calls to parse result
/// objects) to lex an underlying buffer.  Two design points here:
///
///  - Methods in this class return true to indicate failure.
///  - None of the StringRef's returned by methods in this class are owned by
///    this classes.  They're just shallow pointers into the StringRef the
///    instance was constructed with.
class QCLexer {
  // TODO(sanjoy): currently this parser is fairly strict (it crashes
  // noisily on unexpected input).  At some point in the future, we
  // should make a call on whether a "non-strict mode", where parsing
  // never crashes but continues with malformed input on a best-effort
  // basis, is useful.

  llvm::StringRef Source;
  unsigned Offset = 0;
  bool HasParsingFailed = false;
  unsigned FailedOffset = -1;
  std::string FailureMessage;

  void skipWhitespace();

public:
  /// Create a QCLexer from a \c StringRef, the parsing starting logically at
  /// offset 0.
  explicit QCLexer(llvm::StringRef S) : Source(S) { skipWhitespace(); }

  bool isEOF() const { return Offset == Source.size(); }

  /// Check for failure and potentially check for a failure message.  Once a
  /// QCLexer fails, no further methods can be invoked on it.
  /// @{
  bool hasFailed() const { return HasParsingFailed; }
  llvm::StringRef getFailureMessage() const { return FailureMessage; }
  /// @}

  /// Mark the parser as having "failed", using the printf-like arguments passed
  /// in to construct the failure message.
  void setFailed(const char *fmt, ...)
      __attribute__((__format__(__printf__, 2, 3)));

  void advanceToNextLine(const char *What);

  /// Returns the next whitespace delimited token, where the definition of
  /// "whitespace" is as in \c llvm::qc_utils::isWhitespace.
  llvm::StringRef nextToken(const char *What);

  /// Peek ahead and return the next token (if there is one) without changing
  /// state.
  std::optional<llvm::StringRef> peekNextToken();

  /// Return the next token (i.e. what would've been returned by \c nextToken)
  /// as an integer of type \p T.  If the next token does not parse as an
  /// integer, return an unspecified value and set the parser to the failing
  /// state.
  template <typename T>
  typename std::enable_if<std::is_integral<T>::value, T>::type
  nextTokenAsInt(const char *What);

  /// Return a \c StringRef for the next \p N characters, paying no attention to
  /// whitespace delimition.
  llvm::StringRef nextNCharacters(unsigned N);

  /// Check that the remaining ("unconsumed") token stream contains the list of
  /// tokens in \p ExpectedTokens.  Return true on failure.
  bool consumeExactTokenSeq(const char *What,
                            std::initializer_list<const char *> ExpectedTokens);
};

/// Lex out a string from \p QCP that has the length prefix \p LengthPrefix.
/// Inverse of \c printLengthPrefixedString.
llvm::StringRef lexStringWithLengthPrefix(QCLexer &QCP,
                                          llvm::StringRef LengthPrefix);

/// Lex out a length prefixed string from \p QCP.
llvm::StringRef lexLengthPrefixedString(QCLexer &QCP);
}

/// A utility class that can serialize a \c QueryCache using \p
/// TextualTraits.
template <typename QCTy, template <int> class TextualTraits>
class QueryCacheSerializer {
  template <typename, int, int> friend struct qc_detail::ForEachMapKind;

  template <int Idx, typename... BatonTys>
  static qc_detail::ForEachDirective
  forEachMapKind(QCTy &QC, Query* Q, llvm::raw_ostream &OS, BatonTys &...);

  template <int Idx, typename... BatonTys>
  static void
  serializeResult(const std::optional<typename TextualTraits<Idx>::ResultTy> &,
                  llvm::raw_ostream &OS, BatonTys...);

public:
  /// Entry point into the serialization logic, serializes \p QC into \p OS by
  /// doing a simple walk over the entire query cache and making calls into \c
  /// TextualTraits to serialize result types.
  template <typename... BatonTys>
  static void serialize(QCTy &QC, llvm::raw_ostream &OS, BatonTys &...);
};

/// A utility class that can deserialize a QueryCache using \p TextualTraits.
template <typename QCTy, template <int> class TextualTraits>
class QueryCacheDeserializer {
  template <typename, int, int> friend struct qc_detail::ForEachMapKind;

  template <int Idx, typename... BatonTys>
  static qc_detail::ForEachDirective
  forEachMapKind(QCTy &QC, qc_utils::QCLexer &QCP, const int ActualQueryKind,
                 unsigned NumArgs, BatonTys &...);

  static bool stringToQueryKind(llvm::StringRef Name, int &QueryKind);

public:
  /// Entry point into the deserialization logic.  Deserializes from \p QCP in a
  /// straightforward manner and calls into \p TextualTraits to deserialize
  /// results.
  ///
  /// \p If SkipSection is not nullptr then the query kind names for which it
  /// return true will be dropped during parsing.
  template <typename... BatonTys>
  static bool deserialize(QCTy &QC, qc_utils::QCLexer &QCP,
                          std::function<bool(llvm::StringRef)> SkipSection,
                          BatonTys &...);

  /// Wrapper that deserializes a \p StringRef
  template <typename... BatonTys>
  static bool deserialize(QCTy &QC, llvm::StringRef Source,
                          std::string &ErrorMsg,
                          std::function<bool(llvm::StringRef)> SkipSection,
                          BatonTys &... Batons) {
    typedef QueryCacheDeserializer<QCTy, TextualTraits> SelfTy;

    qc_utils::QCLexer QP(Source);
    if (SelfTy::deserialize(QC, QP, SkipSection, Batons...)) {
      ErrorMsg = std::string(QP.getFailureMessage());
      return true;
    }

    return false;
  }
};

}

#include "QueryCacheSerializationImpl.h"

#endif
