///===-- QueryCache.h - -----------------------------------------*- C++ -*-===//
///
///                     The LLVM Compiler Infrastructure
///
/// This file is distributed under the University of Illinois Open Source
/// License. See LICENSE.TXT for details.
///
///===---------------------------------------------------------------------===//
/// Copyright 2013-2018 Azul Systems, Inc.  All Rights Reserved.
/// http://www.azul.com
/// Azul Systems is a contributor to the LLVM Team.
/// Distributed under the same license terms detailed in LICENSE.TXT above.
///===---------------------------------------------------------------------===//
///
/// This file implements the common "data-structure"y bits of Orca's "record
/// replay" feature (also referred to by an unqualified "query cache").  The
/// "query cache", as implemented in this file, is a "heterogeneous yet strongly
/// typed" "persisted" hashtable, with "some structure" to it.  Let's look at
/// these properties one by one, in reverse order:
///
///  - "some structure": As you'll see, we hard code some concepts like \c
///    QueryArg, \c Query etc. in this file, so this isn't a generic hashtable.
///    This is intentional, and lets us have the following two properties
///    without too much work.
///
///  - "persisted": This is just a fancy way of saying that there are utilities
///    to help with serializing and deserializing (TBD) QueryCache instances to
///    and from raw_ostreams (and hence to and from the disk or in-memory
///    buffers).  The utilities are in QueryCacheSerialization.h
///
///  - "heterogeneous yet strongly typed": This is the reason why this file is
///    so template'y.  The hashtable semantically maps tuples of the form
///    (<query type>, arg0, arg1 ...) to a value whose *type* is a function of
///    <query type>.  That is, query cache knows that a query with <query type>
///    "4" (denoted by some enum, say) always returns a uint64_t, and this fact
///    is encoded in the type system; and it is difficult to get that wrong
///    without the compiler refusing the build your code.
///
///===---------------------------------------------------------------------===//

#ifndef LLVM_SUPPORT_AZUL_QUERY_CACHE_H
#define LLVM_SUPPORT_AZUL_QUERY_CACHE_H

#include "llvm/ADT/ArrayRef.h"
#include "llvm/ADT/DenseMap.h"
#include "llvm/ADT/FoldingSet.h"
#include "llvm/ADT/STLExtras.h"
#include "llvm/ADT/StringRef.h"

#include "llvm/Support/ErrorHandling.h"
#include "llvm/Support/raw_ostream.h"

#include <cassert>
#include <functional>
#include <optional>
#include <string>
#include <type_traits>

namespace azul {

/// Forward declaration for friend directives.
/// @{
template <typename, template <int> class> class QueryCacheSerializer;
template <typename, template <int> class> class QueryCacheDeserializer;

class Query;

namespace qc_utils {
class QCLexer;
}

namespace qc_detail {
class QueryCacheImpl;
void serializeQuery(const Query *Q, llvm::StringRef QueryKindName,
                    llvm::raw_ostream &OS);
Query *deserializeQuery(azul::qc_utils::QCLexer &, QueryCacheImpl &,
                        int, unsigned);
template <typename, int, int> struct ForEachMapKind;

/// Enum for use in \c ForEachDirective
enum class ForEachDirective { Continue, Stop };
}

/// A tagged union of types that can appear as a query argument.  This is a
/// simple immutable data structure.
class QueryArg {
  friend class azul::qc_detail::QueryCacheImpl;

  int Type;

  // Union that commons the storage behind Integer, String, and IntegerArray
  union {
    uint64_t Integer;
    std::string String;
    std::vector<uint64_t> IntegerArray;
    std::optional<uint64_t> OptionalInteger;
  };

public:
  explicit QueryArg(uint64_t Value) : Type(TYPE_UINT64), Integer(Value) {}

  explicit QueryArg(std::string Value)
      : Type(TYPE_STRING), String(std::move(Value)) {}

  explicit QueryArg(llvm::StringRef Value)
      : Type(TYPE_STRING), String(std::move(Value)) {}

  explicit QueryArg(const char* Value)
      : Type(TYPE_STRING), String(Value) {}

  explicit QueryArg(std::vector<uint64_t> Value)
      : Type(TYPE_UINT64_ARRAY), IntegerArray(std::move(Value)) {}

  explicit QueryArg(std::optional<uint64_t> Value)
      : Type(TYPE_OPTIONAL_UINT64), OptionalInteger(std::move(Value)) {}

  QueryArg(const QueryArg &Other) : Type(Other.Type) {
    switch (Type) {
    default:
      llvm_unreachable("unhandled QueryArg type");

    case TYPE_UINT64:
      Integer = Other.getInteger();
      break;

    case TYPE_STRING:
      new (&String) std::string(Other.getString());
      break;

    case TYPE_UINT64_ARRAY:
      new (&IntegerArray) std::vector<uint64_t>(Other.getIntegerArray());
      break;

    case TYPE_OPTIONAL_UINT64:
      new (&OptionalInteger)
          std::optional<uint64_t>(Other.getOptionalInteger());
      break;
    }
  }

  ~QueryArg() {
    if (getType() == TYPE_STRING)
      String.~basic_string<char>();
    else if (getType() == TYPE_UINT64_ARRAY)
      IntegerArray.~vector<uint64_t>();
    else if (getType() == TYPE_OPTIONAL_UINT64)
      OptionalInteger.~optional<uint64_t>();
  }

  enum {
    TYPE_UINT64,
    TYPE_STRING,
    TYPE_UINT64_ARRAY,
    TYPE_OPTIONAL_UINT64,
    TYPE_INVALID,
    TYPE_MAX
  };

  /// @{
  /// Accessor functions.

  llvm::StringRef getString() const {
    assert(getType() == TYPE_STRING);
    return String;
  }

  uint64_t getInteger() const {
    assert(getType() == TYPE_UINT64);
    return Integer;
  }

  llvm::ArrayRef<uint64_t> getIntegerArray() const {
    assert(getType() == TYPE_UINT64_ARRAY);
    return IntegerArray;
  }

  std::optional<uint64_t> getOptionalInteger() const {
    assert(getType() == TYPE_OPTIONAL_UINT64);
    return OptionalInteger;
  }

  int getType() const {
    assert(Type != TYPE_INVALID);
    return Type;
  }

  /// @}

  /// Used only as a helper function by Query -- a QueryArg is intentionally not
  /// a FoldingSetNode
  void Profile(llvm::FoldingSetNodeID &ID) const;

  bool operator==(const QueryArg &That) const {
    if (getType() != That.getType())
      return false;

    switch (getType()) {
    default:
      llvm_unreachable("unhandled QueryArg type");

    case TYPE_UINT64:
      return getInteger() == That.getInteger();

    case TYPE_STRING:
      return getString() == That.getString();

    case TYPE_UINT64_ARRAY:
      return getIntegerArray() == That.getIntegerArray();

    case TYPE_OPTIONAL_UINT64:
      return getOptionalInteger() == That.getOptionalInteger();
    }
  }
};

/// Represents one replayable query.  This is a simple immutable data structure.
class Query : public llvm::FoldingSetNode {
  friend class azul::qc_detail::QueryCacheImpl;

  int Type;
  llvm::SmallVector<QueryArg, 2> Arguments;

  Query(int Type, llvm::ArrayRef<QueryArg> Args)
      : Type(Type), Arguments(Args.begin(), Args.end()) {}

public:
  /// @{
  /// Accessor functions.
  int getKind() const { return Type; }
  llvm::ArrayRef<QueryArg> getArgs() const { return Arguments; }
  /// @}

  bool operator==(const Query &That) const {
    if (getKind() != That.getKind())
      return false;

    return Arguments == That.Arguments;
  }

  /// This is how we interface with FoldingSets.  See llvm::FoldingSetNode.
  void Profile(llvm::FoldingSetNodeID &ID) const {
    ID.AddInteger(Type);
    for (auto &A : Arguments)
      A.Profile(ID);
  }
};

/// Implementation namespace.  Do not use!
namespace qc_detail {

/// Holder for things in \c QueryCache that do not need to depend on the
/// specialization.
class QueryCacheImpl {
  friend Query *deserializeQuery(azul::qc_utils::QCLexer &, QueryCacheImpl &,
                                 int, unsigned);

  llvm::SpecificBumpPtrAllocator<Query> QueryAllocator;
  llvm::FoldingSet<Query> QueryFoldingSet;
  
protected:
  /// Maintains the log of all queries in the order of their appearance
  llvm::SmallVector<Query*, 64> QueryLog;
  
  /// Constructs a unique query instance given a list of QueryArgs, \p Args,
  /// which must be appropriate for the specified query kind \c QueryKind.
  Query *constructUniqueQueryHelper(int QueryKind,
                                    llvm::ArrayRef<QueryArg> Args);

  /// Constructs a unique query instance given a list of QueryArgs, \p Args,
  /// which must be appropriate for the specified query kind \c QueryKind.
  template <typename... ArgsTy>
  Query *constructUniqueQuery(int QueryKind, ArgsTy... Args) {
    QueryArg Params[sizeof...(Args)] = {QueryArg(Args)...}; // "pack expansion"
    return constructUniqueQueryHelper(QueryKind, Params);
  }

public:
  /// @{
  /// Query log iteration
  typedef decltype(QueryLog)::iterator iterator;
  typedef decltype(QueryLog)::const_iterator const_iterator;
  
  iterator begin() { return QueryLog.begin(); }
  iterator end() { return QueryLog.end(); }

  const_iterator begin() const { return QueryLog.begin(); }
  const_iterator end() const { return QueryLog.end(); }
  /// @}
};

/// The \c qc_detail::Storage class is used by the query cache to inject \c N
/// fields of the appropriate type into \c QueryCache.
/// @{
template <int Index, template <int> class QueryTraits>
struct Storage : public Storage<Index - 1, QueryTraits> {
  typedef typename QueryTraits<Index>::ResultTy ResultTy;
  typedef llvm::DenseMap<Query *, std::optional<ResultTy>> MapTy;

  MapTy Map;
};

template <template <int> class QueryTraits> struct Storage<0, QueryTraits> {
  typedef typename QueryTraits<0>::ResultTy ResultTy;
  typedef llvm::DenseMap<Query *, std::optional<ResultTy>> MapTy;

  MapTy Map;
};
/// @}

/// Helper template specialization to "loop" a finite number of times.  Calls a
/// static \c forEachMapKind method on \p QCty \p Remaining number of times,
/// specializing it with 0 the first time, 1 the second time ... (Remaining - 1)
/// the Remaining'th time.  Stops early if \c forEachMapKind returns \c
/// ForEachDirective::Stop.
///
/// @{
template <typename QCTy, int Remaining, int Idx = 0> struct ForEachMapKind {
  template <typename... BatonTys> static void run(BatonTys &... Batons) {
    auto Directive = QCTy::template forEachMapKind<Idx>(Batons...);
    if (Directive == ForEachDirective::Continue)
      ForEachMapKind<QCTy, Remaining - 1, Idx + 1>::run(Batons...);
  }
};

template <typename QCTy, int Idx> struct ForEachMapKind<QCTy, 0, Idx> {
  template <typename... BatonTys> static void run(BatonTys &... Batons) {}
};
/// @}
} // end namespace qc_detail

/// This is a "heterogeneous yet strongly typed" "persisted" hashtable, a
/// described at the top of the file.
///
/// \p QueryKindsMax - A QueryCache instance manages queries of kind 0 to
///     QueryKindsMax - 1 (both inclusive).  \c QueryKindsMax cannot be
///     negative.
///
/// \p QueryTraits - A template that provides traits for each of query kinds
///     from 0 to QueryKindsMax - 1 (both inclusive).  For every N in 0 to
///     QueryKindsMax - 1, QueryTraits<N> needs to contain:
///
///     template<> struct QueryTraits<N> {
///       // The result type of the query corresponding to N
///       typedef $TYPE ResultTy;
///
///       // Whether the callbacks backing the queries are expected
///       // to be idempotent or not (now defunct, only used for asserts)
///       typedef $TYPE ResultTy;
///     }
///
template <int QueryKindsMax, template <int> class QueryTraits>
class QueryCache : public qc_detail::QueryCacheImpl,
                   public qc_detail::Storage<QueryKindsMax - 1, QueryTraits> {

  template <typename, int, int> friend struct qc_detail::ForEachMapKind;

  /// Get the map (always a \p DenseMap mapping a \c Query* to some result type
  /// corresponding to \p QueryKind.
  template <int QueryKind>
  typename qc_detail::Storage<QueryKind, QueryTraits>::MapTy &getMap() {
    // This should be as fast as a field access.
    auto *BaseStorage =
        static_cast<qc_detail::Storage<QueryKind, QueryTraits> *>(this);
    return BaseStorage->Map;
  }

  template <int Idx, typename... ArgTys>
  static qc_detail::ForEachDirective
  forEachMapKind(QueryCache<QueryKindsMax, QueryTraits> &QC, ArgTys... Args) {
    return QC.forEachMapKind<Idx>(Args...);
  }

  template <int Idx, typename ActionTy>
  qc_detail::ForEachDirective forEachMapKind(ActionTy);

  static_assert(QueryKindsMax > 0, "QueryKindsMax cannot be negative!");

  template <int QueryKind>
  static void check(const Query* Q) {
    static_assert(QueryKind < QueryKindsMax, "Incorrectly typed query cache!");
    assert(Q->getKind() == QueryKind && "Mismatched query kind!");
  }

public:
  static constexpr int KindsMax = QueryKindsMax;

  /// Given a set of query arguments passed as function arguments, \p QueryArgs,
  /// and a query kind passed in as a template specialization \p QueryKind,
  /// return a result indicating if the query cache already knows the answer.
  /// The return value should be interpreted as
  ///
  ///  {nullptr, V} -> The query cache thinks the answer to the query is V
  ///                  Note that V can be std::nullopt.
  ///
  ///  {Q(non null), std::nullopt} -> The query cache does not have the answer.
  ///    To insert the answer into the cache, call \c insert with \c Q.
  ///
  ///  {Q(non null), not-std::nullopt} -> Illegal
  template <int QueryKind, typename... QueryArgsTy>
  std::pair<Query *, std::optional<typename QueryTraits<QueryKind>::ResultTy>>
  lookup(QueryArgsTy... QueryArgs) {
    Query *Q = constructUniqueQuery(QueryKind, QueryArgs...);
    check<QueryKind>(Q);

    const auto &Map = getMap<QueryKind>();
    auto ResultIt = Map.find(Q);
    if (ResultIt == Map.end())
      return {Q, std::nullopt};
    return {nullptr, ResultIt->second};
  }

  template <int QueryKind>
  std::optional<typename QueryTraits<QueryKind>::ResultTy>
  lookupByQuery(const Query *Q) {
    check<QueryKind>(Q);

    const auto &Map = getMap<QueryKind>();
    auto ResultIt = Map.find(Q);
    if (ResultIt == Map.end())
      return std::nullopt;
    return ResultIt->second;
  }

  /// Insert the result of a query into this query cache so that subsequent
  /// queries that come in through \p lookup for the tuple represented by \p Q
  /// return the said result.  \p QueryKind must match \c Q->getKind().
  ///
  ///  \p Q - A \c Query* instance obtained via \c lookup.
  ///
  ///  \p Result - The result for the query \c Q.
  template <int QueryKind>
  void
  insert(Query *Q,
      const std::optional<typename QueryTraits<QueryKind>::ResultTy>& Result) {
    check<QueryKind>(Q);

    auto InsertResult = getMap<QueryKind>().insert({Q, Result});
    (void)InsertResult;
    assert(InsertResult.second && "Don't insert twice!");

    QueryLog.push_back(Q);
  }

  /// Update the result of a query so that subsequent queries that come in
  /// through \p lookup for the tuple represented by \p Q return the said result.
  ///  \p QueryKind must match \c Q->getKind().
  ///
  ///  \p Q - A \c Query* instance obtained via \c lookup.
  ///
  ///  \p Result - The result for the query \c Q.
  template <int QueryKind>
  void
  update(const Query *Q,
         const typename QueryTraits<QueryKind>::ResultTy& Result) {
    check<QueryKind>(Q);

    auto &Map = getMap<QueryKind>();
    auto ResultIt = Map.find(Q);
    assert(ResultIt != Map.end() && "Original query not found");
    ResultIt->second = Result;
  }

#ifdef NDEBUG
  template<typename ActionTy>
  void verify(ActionTy) {}
#else
  template<typename ActionTy>
  void verify(ActionTy);
#endif
};
}

#endif // LLVM_SUPPORT_AZUL_QUERY_CACHE_H
