//===-- AzulState.h ---------------------------------------------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2018 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//
//
// This file defines the AzulState class.  AzulState holds some Azul specific
// state inside an LLVMContext.  We don't expose this structure to outside
// the Azul directory.
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_LIB_AZUL_AZUL_STATE_H
#define LLVM_LIB_AZUL_AZUL_STATE_H

#include "llvm/ADT/SmallVector.h"
#include "llvm/ADT/StringRef.h"
#include "llvm/Orca/Orca.h"
#include "llvm/Orca/Utils.h"
#include "llvm/IR/Orca/VMQueryCache.h"
#include "llvm/IR/LLVMContext.h"
#include "llvm/IR/Module.h"
#include "llvm/Pass.h"
#include "llvm/Support/Debug.h"
#include "llvm/Support/raw_ostream.h"

#include <cassert>
#include <cstdint>
#include <functional>

namespace llvm {
  class Function;
}

namespace azul {

class AzulStateKeeper;
class OrcaTimeRecord;
class OrcaInlinerState;

namespace orca {
class OrcaPipeline;
}

namespace internal {

class AzulState {
  std::intptr_t tag;
  VMQueryCache QC;

  /// A scratch module used by InlineCost to emit new functions into. This way
  /// it doesn't modify the module being analyzed.
  std::unique_ptr<llvm::Module> InlineCostScratchModule;

  // there can be a single main instance (installed in LLVM context)
  // as well as some temporary ones created for on-demand query-cache operations
  const bool MainInstance;

  /// QCOut is a non-owning pointer to the stream we serialize the query cache
  /// into.
  llvm::raw_ostream *QCOut = nullptr;
  bool StreamEnabled = false;

  /// Sometimes we need to create a new raw_ostream to store in \c QCOut.  If we
  /// do, we store the owning pointer here, so that we can free it
  /// appropriately.
  std::unique_ptr<llvm::raw_ostream> OwnedStream;

  // Track number of inlined call sites for this compile as it will be used
  // to distinguish query cached callbacks for duplicated call sites.
  uint64_t InlinedCallSites = 0;

  // Allows to interrupt currently running pass pipeline. It works through the
  // new pass manager instrumentation. Interrupt happens in between passes so
  // the currently running pass will need to finish it's work first.
  // Use OptPassGate mechanism to interrupt old pass manager.
  bool IsInterrupted = false;

  /// Keep Orca Pipeline here to provide access to it right from the Context
  /// anywhere during compilation time. Needed mostly to organize
  /// "as-in-orca-opt" pipelines for standalone opt. Not owned by AzulState,
  /// expected to have the same life time.
  orca::OrcaPipeline *Pipeline = nullptr;

  /// A place where the inliner can keep the state it wants to share with
  /// various inliner passes.
  OrcaInlinerState *InlinerState = nullptr;

  /// When this callback is set, it is used for reporting time deltas measured
  /// in various parts of compiler.
  /// The first argument is a string ID, the intention is to accumulate the sum
  /// of all reported measurements for each ID, so if you want to know sum of
  /// some measurements, give them the same ID, use different IDs otherwise.
  /// The second argument should be time difference which you want to report.
  using ReportTimeDiffCallbackType =
      std::function<void(llvm::StringRef, const OrcaTimeRecord &)>;
  ReportTimeDiffCallbackType ReportTimeDiffCallback;

  /// When this callback is set, it is used for updating counters.
  /// The first argument is a string ID. The callback is expected to aggregate
  /// the values reported with the same ID.
  /// The second argument is by how much the counter should be changed. Note
  /// that it's signed, so counters don't necessarily have to only go up.
  using ReportCounterDiffCallbackType =
      std::function<void(llvm::StringRef, int)>;
  ReportCounterDiffCallbackType ReportCounterDiffCallback;

  // Do not invoke callbacks on query cache misses.
  const bool IgnoreCallbacks;

  static const std::intptr_t VALID_TAG = 0x1c8421f47d14da56;
  static const std::intptr_t INVALID_TAG = 0x3e6ab6bee33660fe;

public:
  AzulState(bool MainInstance = false, bool IgnoreCallbacks = false)
      : tag(VALID_TAG), MainInstance(MainInstance), IgnoreCallbacks(IgnoreCallbacks) {}

  ~AzulState() { tag = INVALID_TAG; }

  void streamQueryCacheOut(llvm::raw_ostream *VMQueryCacheOut);
  void streamQueryCacheToFile(llvm::StringRef FName);
  void readQueryCacheFromBuffer(llvm::StringRef Buffer) { QC.read(Buffer); }

  VMQueryCache &getVMQueryCache() { return QC; }
  FunctionCache &getFunctionCache() { return QC.getFunctionCache(); }
  llvm::raw_ostream *getQueryCacheStream() { return QCOut; }
  bool isStreamEnabled() { return StreamEnabled; }
  uint64_t getAndIncInlinedCallSites() { return InlinedCallSites++; }

  /// Returns the InlineCost scratch module. This module is used to emit new
  /// functions into. This way it doesn't modify the module being analyzed.
  llvm::Module *getInlineCostScratchModule(llvm::LLVMContext &C) {
    if (!InlineCostScratchModule)
      InlineCostScratchModule = std::make_unique<llvm::Module>("scratch", C);
    return InlineCostScratchModule.get();
  }

  bool is_interrupted() const { return IsInterrupted; }
  void interrupt() { IsInterrupted = true; }

  orca::OrcaPipeline *getPipeline() const { return Pipeline; }
  void setPipeline(orca::OrcaPipeline *P) { Pipeline = P; }

  bool ignores_callbacks() const { return IgnoreCallbacks; }

  /// install AzulState into LLVM Context.
  void install(llvm::LLVMContext &C);

  /// remove AzulState from LLVM Context and flush the cache
  void finalize(llvm::LLVMContext &C);

  /// dump the whole cache out
  void serializeQueryCache();

  /// dump the whole cache out to given stream
  void writeQueryCache(llvm::raw_ostream &QueryCacheOut);

  /// get previously installed AzulState from LLVM Context
  static AzulState *getStateFromContext(const llvm::LLVMContext &C) {
    return castOrNull(C.getAzulState());
  }

  OrcaInlinerState *getInlinerState() {
    return InlinerState;
  }

  void setInlinerState(OrcaInlinerState *NewInlinerState) {
    InlinerState = NewInlinerState;
  }

  void
  setTimeDiffReportingCallback(ReportTimeDiffCallbackType Callback) {
    ReportTimeDiffCallback = Callback;
  }

  void resetTimeDiffReportingCallback() {
    ReportTimeDiffCallback = ReportTimeDiffCallbackType();
  }

  void
  setCounterDiffReportingCallback(ReportCounterDiffCallbackType Callback) {
    ReportCounterDiffCallback = Callback;
  }

  void resetCounterDiffReportingCallback() {
    ReportCounterDiffCallback = ReportCounterDiffCallbackType();
  }

  static bool tryReportTimeDiff(llvm::LLVMContext &Ctx, llvm::StringRef Id,
                                const OrcaTimeRecord &TR) {
    AzulState *AS = getStateFromContext(Ctx);
    if (!AS || !AS->ReportTimeDiffCallback)
      return false;
    AS->ReportTimeDiffCallback(Id, TR);
    return true;
  }

  static bool tryReportCounterDiff(llvm::LLVMContext &Ctx, llvm::StringRef Id,
                                   int Diff) {
    AzulState *AS = getStateFromContext(Ctx);
    if (!AS || !AS->ReportCounterDiffCallback)
      return false;
    AS->ReportCounterDiffCallback(Id, Diff);
    return true;
  }

  static bool tryIncrementCounter(llvm::LLVMContext &Ctx, llvm::StringRef Id) {
    return tryReportCounterDiff(Ctx, Id, 1);
  }

private:
  static AzulState *castOrNull(void *ptr) {
    if (ptr == nullptr)
      return nullptr;

    AzulState *AS = reinterpret_cast<AzulState *>(ptr);
    assert(AS->tag == VALID_TAG && "invalid object!");
    assert(AS->MainInstance &&
           "should be used from the 'main' AzulState instance only");
    return AS;
  }

  friend class azul::AzulStateKeeper;

  void readQueryCacheFromFile(const llvm::StringRef FileName);
  void verifyQueryCache();
};

} // namespace internal

/// convenience wrapper over main AzulState instance that helps to initialize
/// and dispose it
class AzulStateKeeper {
private:
  internal::AzulState AS;

  /// Keeper performs its operations only if enabled.
  const bool Enabled;

  /// We remember LLVM context when installing to reuse it when disposing later.
  llvm::LLVMContext *Ctx = nullptr;

public:
  AzulStateKeeper(llvm::LLVMContext &C, bool ExplicitlyEnable = false);
  ~AzulStateKeeper() {
    if (Ctx)
      AS.finalize(*Ctx);
  }

  void streamQueryCacheOut(llvm::raw_ostream &VMQueryCacheOut) {
    if (Enabled)
      AS.streamQueryCacheOut(&VMQueryCacheOut);
  }

  void streamQueryCacheToFile(llvm::StringRef FName) {
    if (Enabled)
      AS.streamQueryCacheToFile(FName);
  }

  void readQueryCacheFromBuffer(llvm::StringRef Buffer) {
    if (Enabled)
      AS.QC.read(Buffer);
  }

  /// Track Orca Pipeline for this compilation.
  void setPipeline(orca::OrcaPipeline *P) { AS.setPipeline(P); }

  /// Setup and install managed AzulState into LLVM context.
  void install();
};

/// A simple RAII wrapper over AzulState::tryReportTimeDiff.
/// Use to time a scope:
/// {
///   TimeRegion Timer(V->getContext(), "MyTimer");
///   ...
/// }
struct TimeRegion {
  llvm::LLVMContext &C;
  llvm::StringRef ID;
  OrcaTimeRecord TimerStart;
  TimeRegion(llvm::LLVMContext &C, llvm::StringRef ID);
  ~TimeRegion() {
    internal::AzulState::tryReportTimeDiff(
        C, ID, OrcaTimeRecord::getTimeSince(TimerStart));
  }
};
} // namespace azul

#endif
