//===-- OrcaPipeline.h ------------------------------------------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2018 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//
//
// This header defines the Orca Pipeline API.
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_AZUL_ORCA_PIPELINE_H
#define LLVM_AZUL_ORCA_PIPELINE_H

#include "llvm/ADT/StringRef.h"
#include "llvm/Analysis/CGSCCPassManager.h"
#include "llvm/IR/LegacyPassManager.h"
#include "llvm/IR/PassManager.h"
#include "llvm/Orca/OrcaFeatures.h"
#include "llvm/Orca/OrcaIRStatsInstrumentation.h"
#include "llvm/Orca/OrcaOptimizeAllocations.h"
#include "llvm/Orca/OrcaPassStatsInstrumentation.h"
#include "llvm/Orca/OrcaPeelingPassManager.h"
#include "llvm/Orca/OrcaPipelineInterrupter.h"
#include "llvm/Orca/OrcaYieldInstrumentation.h"
#include "llvm/Passes/StandardInstrumentations.h"
#include "llvm/Transforms/Scalar/LoopPassManager.h"
#include <optional>

namespace llvm {
class Function;
class Module;
class TargetMachine;

namespace json {
  class OStream;
}
}

namespace azul {
namespace internal {
  class AzulState;
}

namespace orca {
using namespace llvm;

enum DropAtomicsPolicy {
  none,
  early,
  late,
};

/// Specify the requested optimization level
enum OptimizationLevel {
  Opt_None,           /* Do no optimization, just run required passes */
  Opt_Full,           /* Normal compile pipeline for Orca */
  Opt_FullWithoutIPO, /* Same as Opt_Full, except that this does not
                         run IPO passes. */
  Opt_Cleanup, /* Run a set of minimum set of cleanup passes.  Don't run any
                * lowering passes */
  Opt_CleanupWithoutIPO, /* Same as Opt_Cleanup, except that it does
                          * not run any IPO passes.  */
  Opt_Last /* NOT a real level */

  // TODO: Consider an Opt_Fast (for a quick compile) and an Opt_Excessive
  // (for reruning passes to find missed optimization/ordering issues)
};

class OrcaPipeline {

  // NB: the order of declaration for these Managers is absolutely
  // critical for the correctness of destruct sequence.
  LoopAnalysisManager LAM;
  CGSCCAnalysisManager CGAM;
  FunctionAnalysisManager FAM;
  ModuleAnalysisManager MAM;
  ModulePassManager MPM;

  // special pass manager for generate-inline-candidates early opt pipeline
  FunctionPassManager InlineCandidateOptFPM;

  // instrumentation callbacks
  PassInstrumentationCallbacks Callbacks;
  OrcaIRStatsInstrumentation OrcaIRStats;
  OrcaPassStatsInstrumentation OrcaPassStats;
  OrcaPipelineInterrupter PipelineInterrupter;
  std::unique_ptr<OrcaYieldInstrumentation> OrcaYield;
  std::unique_ptr<StandardInstrumentations> SI;

  const OrcaFeatures &CompilationFeatures;

  void buildAnalysisManager(TargetMachine *TM,
                            const azul::internal::AzulState *AzState);
  void buildPipeline(OptimizationLevel Level, json::OStream *CompileInfoOut);

  void scheduleFullPasses(ModulePassManager &MPM, OptimizationLevel Level,
                          json::OStream *CompileInfoOut);
  void scheduleCleanupPasses(ModulePassManager &MPM, OptimizationLevel OptLevel);
  void scheduleLateInlineEvent(ModulePassManager &MPM, StringRef Name);
  void scheduleFullUnrollPasses(FunctionPassManager &FPM);
  void scheduleVectorizationPasses(FunctionPassManager &FPM);

public:
  PassInstrumentationCallbacks* getPassInstrumentationCallbacks() {
    return &Callbacks;
  }
  void printPipeline(raw_ostream &OS,
                     function_ref<StringRef(StringRef)> MapClassName2PassName);
  enum InlinerSimplificationPipelineKind { TopLevelPipeline, CandidatePipeline };

private:
  void scheduleInlinerIterationSimplificationPasses(
      ModulePassManager &MPM,
      std::optional<InlinerSimplificationPipelineKind> InlinerPipeline);

  void buildInlineCandidateEarlyOpts();
  void addLoopVectorizeAndSimplify(FunctionPassManager &FPM);
  void addIRCEAndSimplify(FunctionPassManager &FPM);
  void addLoopPassesAndSimplify(FunctionPassManager &FPM, LoopPassManager LPM, bool UseMemorySSA);

public:
  // If Ctx is passed then StandardInstrumentations and YieldInstrumentation
  // will be used. Ctx is not passed in when custom pipelines are created for
  // option parsing.
  OrcaPipeline(const OrcaFeatures &Features, LLVMContext *Ctx = nullptr);

  const OrcaFeatures &getCompilationFeatures() const {
    return CompilationFeatures;
  }

  ModulePassManager getOrcaInlinerSimplificationPipeline(
      InlinerSimplificationPipelineKind P);

  void build(TargetMachine *TM, OptimizationLevel Level,
             const azul::internal::AzulState *AzState = nullptr,
             raw_ostream *TimePassesStream = nullptr,
             json::OStream *CompileInfoOut = nullptr) {
    // Setting the output stream for time-passes handler
    if (SI.get()) {
      if (TimePassesStream)
        SI->getTimePasses().setOutStream(*TimePassesStream);
      // registering instrumentation callbacks
      SI->registerCallbacks(Callbacks);
    }
    if (OrcaYield)
      OrcaYield->registerCallbacks(Callbacks);

    OrcaIRStats.registerCallbacks(Callbacks);

    OrcaPassStats.setOutStream(CompileInfoOut);
    OrcaPassStats.registerCallbacks(Callbacks);

    if (AzState)
      PipelineInterrupter.registerCallbacks(Callbacks, *AzState);

    buildAnalysisManager(TM, AzState);
    buildPipeline(Level, CompileInfoOut);
    buildInlineCandidateEarlyOpts();
  };
  void run(Module &M) {
    MPM.run(M, MAM);
  };

  void optimizeInlineCandidate(Function &F);
  void scheduleCallGraphSCCPasses(ModulePassManager &MPM,
                                  bool enableInliner);
  OrcaPeelingPassManager
  buildPeelingPassManager(OrcaPeelingPassManagerOptions &Opts);

  OrcaOptimizeAllocations
  buildOptimizeAllocationsPipeline(OrcaOptimizeAllocationsOptions Options);
};

} // end namespace orca
} // end namespace azul
#endif
