//===- lib/Orca/OrcaPeelingPassManager.h - ----------------------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2019 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//

#ifndef ORCAPEELINGPASSMANAGER_H
#define ORCAPEELINGPASSMANAGER_H

#include "llvm/Orca/OrcaPassManagers.h"
#include "llvm/Transforms/Scalar/LoopUnrollPass.h"
#include <optional>

namespace azul {
extern const char OrcaPeelingPassManagerName[];
} // namespace azul

namespace llvm {
struct OrcaPeelingPassManagerOptions {
  int PeelOptLevel;
  int MaxUnrollCount;

  OrcaPeelingPassManagerOptions(int PeelOptLevel = 3, int MaxUnrollCount = 0)
    : PeelOptLevel(PeelOptLevel), MaxUnrollCount(MaxUnrollCount) {}

  OrcaPeelingPassManagerOptions &setPeelOptLevel(int POL) {
    PeelOptLevel = POL;
    return *this;
  }

  OrcaPeelingPassManagerOptions &setPeelOptLevel(std::optional<unsigned> POL) {
    if (POL.has_value())
      return setPeelOptLevel(*POL);
    return *this;
  }

  OrcaPeelingPassManagerOptions &setMaxUnrollCount(unsigned MUC) {
    MaxUnrollCount = MUC;
    return *this;
  }

  int getPeelOptLevel() const;

  unsigned getMaxIterations() const;

  unsigned getMaxUnrollCount() const {
    return MaxUnrollCount;
  }
};

// Stores tracking information to identify if the LoopUnroll pass has actually
// done any peeling.
class OrcaPeelingPassManagerChangeTracker {
  DenseMap<Loop *, int> PeeledIterationsPerLoop;
  Function &F;
  FunctionAnalysisManager &FAM;

public:
  OrcaPeelingPassManagerChangeTracker(Function &F,
                                      FunctionAnalysisManager &FAM);

  // Returns true if peeling was made, and though it makes sense to continue
  // iterations.
  bool madeImportantChanges(const PreservedAnalyses &PassPA) const;
};

// OrcaPeelingPassManager is a pass manager which performs peeling and if it
// happened, runs a clean-up passes and iterates the next peeling pass.
typedef OrcaPassWrapper<
    OrcaIterationPass<OrcaConditionalPass<LoopUnrollPass, FunctionPassManager,
                                          OrcaPeelingPassManagerChangeTracker>,
                      OrcaPeelingPassManagerChangeTracker>,
    azul::OrcaPeelingPassManagerName> OrcaPeelingPassManager;

// Should be invoked only from standalone run as uses fake OrcaPipeline.
OrcaPeelingPassManager
createOrcaPeelingPassManager(OrcaPeelingPassManagerOptions &Opts);

} // namespace llvm
#endif /* ORCAPEELINGPASSMANAGER_H */
