//===- lib/Orca/OrcaPassManagers.h - ----------------------------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2020 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//

#ifndef ORCAPASSMANAGERS_H
#define ORCAPASSMANAGERS_H

#include "llvm/IR/Function.h"
#include "llvm/IR/PassManager.h"
#include "llvm/IR/PassInstrumentation.h"
#include "llvm/Support/Debug.h"

namespace azul {
namespace internal {
template <typename PassT, typename IRUnitT,
          typename AnalysisManagerT, typename... ExtraArgTs>
llvm::PreservedAnalyses runOnePass(PassT &Pass, IRUnitT &IRUnit,
                                   AnalysisManagerT &AM, bool updateAM = true,
                                   ExtraArgTs... ExtraArgs) {
  llvm::PassInstrumentation PI =
      llvm::detail::getAnalysisResult<llvm::PassInstrumentationAnalysis>(
          AM, IRUnit, std::tuple<ExtraArgTs...>(ExtraArgs...));

  // Check the PassInstrumentation's BeforePass callbacks before running the
  // pass, skip its execution completely if asked to (callback returns
  // false).
  if (!PI.runBeforePass<IRUnitT>(Pass, IRUnit))
    return llvm::PreservedAnalyses::all();

  llvm::PreservedAnalyses PassPA = Pass.run(IRUnit, AM, ExtraArgs...);

  // Update the analysis manager as each pass runs and potentially
  // invalidates analyses. Important to do it before PassInstrumentation.
  if (updateAM)
    AM.invalidate(IRUnit, PassPA);

  // Call onto PassInstrumentation's AfterPass callbacks immediately after
  // running the pass.
  PI.runAfterPass<IRUnitT>(Pass, IRUnit, PassPA);

  return PassPA;
}
} // end namespace internal
} // end namespace azul
namespace llvm {

// Tracks the fact of any changes made.
class OrcaAnyChangesTracker {
public:
  OrcaAnyChangesTracker(Function &F, FunctionAnalysisManager &FAM) {}

  // Returns true if there are any loops present in the function \p F, and it
  // makes sense to run whole cleanup pipeline.
  bool madeImportantChanges(const PreservedAnalyses &PassPA) const {
    return !PassPA.areAllPreserved();
  }
};

// This is a composition pass which executes MainPass, check the condition
// represented as ConditionTracker and in case condition is true it executes
// the pass ConditionalPass.
template <typename MainPassT, typename ConditionalPassT,
          typename ConditionTrackerT>
class OrcaConditionalPass
    : public PassInfoMixin<OrcaConditionalPass<MainPassT, ConditionalPassT,
                                               ConditionTrackerT> > {
  MainPassT MainPass;
  ConditionalPassT ConditionalPass;
  const bool DebugLogging;

public:
  OrcaConditionalPass(MainPassT &&MainPass, ConditionalPassT &&ConditionalPass,
                      bool DebugLogging = false)
      : MainPass(std::move(MainPass)),
        ConditionalPass(std::move(ConditionalPass)),
        DebugLogging(DebugLogging) {}

  // Runs main pass, asks tracker to continue execution of conditional pass,
  // and run it if answer is yes.
  PreservedAnalyses run(Function &F, FunctionAnalysisManager &FAM) {
    if (DebugLogging)
      dbgs() << "Starting " << this->name() << " conditional pass run.\n";

    // Make preparations before running main pass.
    ConditionTrackerT Tracker(F, FAM);

    if (DebugLogging)
      dbgs() << "Running main pass: " << MainPass.name() << " on "
             << F.getName() << "\n";

    PreservedAnalyses MainPassPA = azul::internal::runOnePass(MainPass, F, FAM);

    // Identify whether the main pass has done changes after which we should
    // continue with conditional pass. Do it after invalidation of analysis
    // because if tracker uses analysis it should not use invalid ones.
    if (Tracker.madeImportantChanges(MainPassPA)) {
      if (DebugLogging)
        dbgs() << "Running conditional pass: " << ConditionalPass.name()
               << " on " << F.getName() << "\n";
      PreservedAnalyses ConditionalPassPA =
          azul::internal::runOnePass(ConditionalPass, F, FAM);

      // Finally, intersect the preserved analyses to compute the aggregate
      // preserved set for this pass manager.
      MainPassPA.intersect(std::move(ConditionalPassPA));
    }

    // We already did all invalidation of analysis, so they are up to date.
    // We can safely report that we preserve all functional analysis.
    MainPassPA.preserveSet<AllAnalysesOn<Function> >();

    if (DebugLogging)
      dbgs() << "Finished " << this->name() << " conditional pass run.\n";
    return MainPassPA;
  }

  void printPipeline(raw_ostream &OS,
                     function_ref<StringRef(StringRef)> MapClassName2PassName) {
    OS << "condpass<";
    OS << getTypeName<ConditionTrackerT>();
    OS << ">(main(";
    MainPass.printPipeline(OS, MapClassName2PassName);
    OS << "),conditional(";
    ConditionalPass.printPipeline(OS, MapClassName2PassName);
    OS << "))";
  }
};

template <typename PassT, typename ConditionTrackerT>
class OrcaIterationPass
    : public PassInfoMixin<OrcaIterationPass<PassT, ConditionTrackerT> > {
  PassT Pass;
  const unsigned MaxIterations;
  const bool DebugLogging;

public:
  OrcaIterationPass(PassT &&Pass, unsigned MaxIterations,
                    bool DebugLogging = false)
      : Pass(std::move(Pass)), MaxIterations(MaxIterations),
        DebugLogging(DebugLogging) {}

  PreservedAnalyses run(Function &F, FunctionAnalysisManager &FAM) {
    if (DebugLogging)
      dbgs() << "Starting " << this->name() << " iteration pass run.\n";

    PreservedAnalyses PA = PreservedAnalyses::all();
    for (unsigned Iter = 0, E = MaxIterations; Iter < E; ++Iter) {
      if (DebugLogging)
        dbgs() << "Running pass: " << Pass.name()
               << " (iteration = " << (Iter + 1) << ") on " << F.getName()
               << "\n";

      // Make preparations before running pass.
      ConditionTrackerT Tracker(F, FAM);

      PreservedAnalyses PassPA = azul::internal::runOnePass(Pass, F, FAM);
      PA.intersect(PassPA);

      // Identify whether the main pass has done changes after which we should
      // continue iterating. Do it after invalidation of analysis because
      // if tracker uses analysis it should not use invalid ones.
      if (!Tracker.madeImportantChanges(PassPA))
        break;
    }

    // We already did all invalidation of analysis, so they are up to date.
    // We can safely report that we preserve all functional analysis.
    PA.preserveSet<AllAnalysesOn<Function> >();

    if (DebugLogging)
      dbgs() << "Finished " << this->name() << " iteration pass run.\n";
    return PA;
  }

  void printPipeline(raw_ostream &OS,
                     function_ref<StringRef(StringRef)> MapClassName2PassName) {
    OS << "iterationpass<";
    OS << getTypeName<ConditionTrackerT>();
    OS << ";";
    OS << MaxIterations;
    OS << ">(";
    Pass.printPipeline(OS, MapClassName2PassName);
    OS << ")";
  }
};

template <typename PassT, const char *PassName> class OrcaPassWrapper {
  PassT Pass;

public:
  OrcaPassWrapper(PassT P) : Pass(std::move(P)) {}
  static StringRef name() { return PassName; }
  PreservedAnalyses run(Function &F, FunctionAnalysisManager &FAM) {
    // No need to update FAM as we execute only one pass, invalidation
    // will be done by caller.
    return azul::internal::runOnePass(Pass, F, FAM, false /*updateFAM*/);
  }
  void printPipeline(raw_ostream &OS,
                     function_ref<StringRef(StringRef)> MapClassName2PassName) {
      Pass.printPipeline(OS, MapClassName2PassName);
  }
};

template <typename PassT> class OrcaPassBuilder {
  PassT Pass;
  bool DebugLogging;

public:
  OrcaPassBuilder(PassT Pass, bool DebugLogging = false)
      : Pass(std::move(Pass)), DebugLogging(DebugLogging) {}

  template <typename ConditionTrackerT, typename ConditionalPassT>
  OrcaPassBuilder<
      OrcaConditionalPass<PassT, ConditionalPassT, ConditionTrackerT> >
  conditional(ConditionalPassT ConditionalPass) {
    return OrcaPassBuilder<
        OrcaConditionalPass<PassT, ConditionalPassT, ConditionTrackerT> >(
        OrcaConditionalPass<PassT, ConditionalPassT, ConditionTrackerT>(
            std::move(Pass), std::move(ConditionalPass), DebugLogging),
        DebugLogging);
  }

  template <typename ConditionTrackerT>
  OrcaPassBuilder<OrcaIterationPass<PassT, ConditionTrackerT> >
  iteration(unsigned MaxIterations) {
    return OrcaPassBuilder<OrcaIterationPass<PassT, ConditionTrackerT> >(
        OrcaIterationPass<PassT, ConditionTrackerT>(
            std::move(Pass), MaxIterations, DebugLogging),
        DebugLogging);
  }

  template <const char *WrapName> OrcaPassWrapper<PassT, WrapName> wrap() {
    return std::move(OrcaPassWrapper<PassT, WrapName>(std::move(Pass)));
  }
};
} // end namespace llvm

#endif // ORCAPASSMANAGERS_H
