//===- llvm/Orca/AzulOptimizeAllocations.h - Definition of the AzulOptimizeAllocations class --------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
// 
//===----------------------------------------------------------------------===//
// Copyright 2013-2018 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//
// \file
// This pass tries to remove redundant allocations.
//===----------------------------------------------------------------------===//

#ifndef ORCAOPTIMIZEALLOCATIONS_H
#define ORCAOPTIMIZEALLOCATIONS_H

#include "llvm/IR/PassManager.h"
#include "llvm/Orca/OrcaPassManagers.h"

namespace azul {
extern const char OrcaOptimizeAllocationsName[];
} // namespace azul

namespace llvm {

class Function;

struct OrcaUnsafeOptimizeAllocationsPass
    : PassInfoMixin<OrcaUnsafeOptimizeAllocationsPass> {
  OrcaUnsafeOptimizeAllocationsPass() {}

  PreservedAnalyses run(Function &F, FunctionAnalysisManager &FAM);

  bool dropAllocationsFromDeopt(Function &F, FunctionAnalysisManager &FAM) {
    return !run(F, FAM).areAllPreserved();
  }
};

struct OrcaOptimizeAllocationsPass
    : PassInfoMixin<OrcaOptimizeAllocationsPass> {
  OrcaOptimizeAllocationsPass() = default;

  PreservedAnalyses run(Function &F, FunctionAnalysisManager &FAM);

  bool rewriteAllocationUses(Function &F, FunctionAnalysisManager &FAM) {
    return !run(F, FAM).areAllPreserved();
  }
};

struct OrcaRemoveAllocationsPass : PassInfoMixin<OrcaRemoveAllocationsPass> {
  OrcaRemoveAllocationsPass() = default;

  PreservedAnalyses run(Function &F, FunctionAnalysisManager &FAM);

  bool removeAllocations(Function &F, FunctionAnalysisManager &FAM) {
    return !run(F, FAM).areAllPreserved();
  }
};

struct OrcaOptimizeAllocationsOptions {
  bool Unroll;
  bool Cleanup;
  bool RewriteUsingFlowSensitiveEA;
  bool RewriteUsingSymbolicOptimizer;
  bool RewriteAllocationsUses;
  bool UnsafeDropAllocationsFromDeoptStateOptAlloc;
  unsigned FullUnrollCount;
  unsigned MaxIterations;
  OrcaOptimizeAllocationsOptions();
  OrcaOptimizeAllocationsOptions &setRewriteUsingFlowSensitiveEA(bool V);
  OrcaOptimizeAllocationsOptions &setUnroll(bool V) {
    Unroll = V;
    return *this;
  }
};

// Tracks the fact of any changes made.
class OrcaAnyAllocationTracker {
  Function &F;

public:
  OrcaAnyAllocationTracker(Function &F, FunctionAnalysisManager &) : F(F) {}

  // Returns true if there are any allocations present in the function \p F.
  bool madeImportantChanges(const PreservedAnalyses &) const;
};

typedef OrcaPassWrapper<
    OrcaIterationPass<
        OrcaConditionalPass<
            OrcaConditionalPass<OrcaRemoveAllocationsPass, FunctionPassManager,
                                OrcaAnyAllocationTracker>,
            FunctionPassManager, OrcaAnyChangesTracker>,
        OrcaAnyChangesTracker>,
    azul::OrcaOptimizeAllocationsName> OrcaOptimizeAllocations;

OrcaOptimizeAllocations
createOrcaOptimizeAllocations(OrcaOptimizeAllocationsOptions Opts);
} // end namespace llvm
#endif // ORCAOPTIMIZEALLOCATIONS_H
