//===- llvm/Orca/OptimizeSafepointPolls.h - Definition of the OptimizeSafepointPolls class --------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
// 
//===----------------------------------------------------------------------===//
// Copyright 2013-2018 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//
// \file
// This file provides the interface for Azul's own OptimizeSafepointPolls pass.
// This pass removes redundant safepoint polls.
///
//===----------------------------------------------------------------------===//

#ifndef ORCAOPTIMIZESAFEPOINTPOLLS_H
#define ORCAOPTIMIZESAFEPOINTPOLLS_H

#include "llvm/IR/PassManager.h"
#include <optional>

namespace azul {
namespace orca {

class OrcaFeatures;

}
}

namespace llvm {

class OptimizeSafepointPollsOpts {
  bool RemoveSafepointsFromAllLoops;
  bool UseCodeLatencyEstimator;
  bool CanonicalizingMode;

  template <typename OptType>
  void setOpt(OptType &Opt, std::optional<OptType> NewOptValue) {
    if (NewOptValue)
      Opt = *NewOptValue;
  }

public:
  OptimizeSafepointPollsOpts();
  OptimizeSafepointPollsOpts(const azul::orca::OrcaFeatures &Features);

  OptimizeSafepointPollsOpts &
  setRemoveSafepointsFromAllLoops(std::optional<bool> Opt) {
    setOpt(RemoveSafepointsFromAllLoops, Opt);
    return *this;
  }

  OptimizeSafepointPollsOpts &
  setCanonicalizingMode(std::optional<bool> Opt) {
    setOpt(CanonicalizingMode, Opt);
    return *this;
  }

  OptimizeSafepointPollsOpts &setUseCodeLatencyEstimator(std::optional<bool> Opt) {
    setOpt(UseCodeLatencyEstimator, Opt);
    return *this;
  }

  bool getRemoveSafepointsFromAllLoops() const {
    return RemoveSafepointsFromAllLoops;
  }
  bool getCanonicalizingMode() const { return CanonicalizingMode; }
  bool shouldUseCodeLatencyEstimator() const { return UseCodeLatencyEstimator; }
};

class OptimizeSafepointPollsPass
    : public PassInfoMixin<OptimizeSafepointPollsPass> {
  OptimizeSafepointPollsOpts Opts;

public:
  OptimizeSafepointPollsPass() = default;
  OptimizeSafepointPollsPass(const OptimizeSafepointPollsOpts &Opts)
      : Opts(Opts) {}

  PreservedAnalyses run(Function &F, FunctionAnalysisManager &AM);
};

} // end namespace llvm

#endif /* ORCAOPTIMIZESAFEPOINTPOLLS_H */
