//===-- llvm/Orca/LoopDataAligning.h --------------------------------------===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2023 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//

#ifndef LLVM_ORCA_LOOP_DATA_ALIGNING
#define LLVM_ORCA_LOOP_DATA_ALIGNING

#include "llvm/IR/Instructions.h"
#include "llvm/IR/PassManager.h"
#include "llvm/IR/ValueHandle.h"

namespace llvm {

class DominatorTree;
class Function;
class Loop;
class LoopInfo;
class LoopStructure;
class ScalarEvolution;
class SCEV;

struct LoopDataAligningPass : public PassInfoMixin<LoopDataAligningPass> {
  PreservedAnalyses run(Function &F, FunctionAnalysisManager &AM);

  bool runImpl(Function &F, ScalarEvolution &SE, LoopInfo &LI,
               DominatorTree &DT);

private:
  // Check that the loop contains only one Load/Store operation.
  Value *findSingleLoadStoreMemAccess(Loop *L);

  // Build preloop for N iterations to start the main loop cache aligned.
  bool buildPreloop(ScalarEvolution &SE, LoopInfo &LI, DominatorTree &DT,
                    Loop *L, LoopStructure &LS, Value *Ptr);

  // Build a dummy "get_padding" call that will be replaced with an actual
  // alignment gap calculation in the 'post-vectorizer-pass' according to the
  // vectorizing results. The call takes 4 arguments:
  // - pointer at the first loop iteration;
  // - loop exit value;
  // - pointer increment step;
  // - IV increment step;
  Value *buildGetPaddingCall(ScalarEvolution &SE, Loop *L,
                             const SCEV *PtrStartSCEV, Value *LoopExit,
                             Value *PtrStep, Value *IVStep);
};

struct LoopPostDataAligningPass
    : public PassInfoMixin<LoopPostDataAligningPass> {
  bool ExpandGetPaddingCalls;

  LoopPostDataAligningPass(bool ExpandGetPaddingCalls = false)
      : ExpandGetPaddingCalls(ExpandGetPaddingCalls){};

  PreservedAnalyses run(Function &F, FunctionAnalysisManager &AM);

  bool runImpl(Function &F, ScalarEvolution &SE, LoopInfo &LI);

  // Do not skip pass by PassInstrumentation
  static bool isRequired() { return true; }

private:
  // Find a get_padding() call associated with the vectorized loop.
  CallInst *findGetPadding(ScalarEvolution &SE, const Loop *L, Value *Ptr,
                           SmallVector<AssertingVH<CallInst>> &PaddingCalls);

  // Find SIMD Load/Store in the loop.
  Instruction *findVectorizedMemoryOp(const Loop *L);

  void replaceCallWithZero(CallInst *Call,
                           SmallPtrSetImpl<BasicBlock *> &BlocksToSimplify);

  void replaceCallWithPaddingCalculation(
      ScalarEvolution &SE, CallInst *Call, unsigned Alignment,
      SmallPtrSetImpl<BasicBlock *> &BlocksToSimplify);
};

} // end namespace llvm

#endif // LLVM_ORCA_LOOP_DATA_ALIGNING
