//===-- InlinerUtils.h ------------------------------------------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2021 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//
//
// This header contains utility functions facilitating OrcaInliner operations,
// but intended to be used outside of the inliner. For example, by module passes
// that we want to use in inliner's simplification pipelines.
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_ORCA_INLINER_UTILS
#define LLVM_ORCA_INLINER_UTILS

#include "llvm/IR/PassManager.h"
#include "llvm/Orca/OrcaState.h"

namespace llvm {

Function *getTopOfInlinerStack(LLVMContext &Ctx);

/// Use this function if you have a module pass which can run on one function
/// and you want to use this pass in inliner's simplification pipelines. It'll
/// run `runOnOneFunction` method of your pass on the function that is on top of
/// the inliner stack, if there's any. If the stack doesn't exist or it's empty,
/// it'll run on all functions from the module.
/// This function is supposed to be called form `run` methods of module passes.
/// The fact that you call it from that method allows you to do any extra setup
/// necessary if `runOnOneFunction` needs anything more than just a function
/// on which to run.
/// The returned PreservedAnalyses object can be returned from the `run` method
/// as-is.
template <typename PassT, typename... ExtraArgTs>
PreservedAnalyses
runWithInlinerStackSupport(PassT &ThePass, Module &M,
                           ModuleAnalysisManager &MAM,
                           ExtraArgTs &&... ExtraArgs) {
  PassInstrumentation &PI = MAM.getResult<PassInstrumentationAnalysis>(M);
  PreservedAnalyses PA = PreservedAnalyses::all();

  Function *CurrentFunction = getTopOfInlinerStack(M.getContext());
  for (Function &F : M) {
    if (CurrentFunction && CurrentFunction != &F)
      continue;
    // Since we check the inliner stack explicitly, we don't have to call PI's
    // callbacks, but they provide nice debug prints when -debug-pass-manager is
    // set.
    if (!PI.runBeforePass(ThePass, F))
      continue;
    PreservedAnalyses PassPA =
        ThePass.runOnOneFunction(F, std::forward<ExtraArgTs>(ExtraArgs)...);
    MAM.invalidate(M, PassPA);
    PI.runAfterPass(ThePass, F, PassPA);
    PA.intersect(std::move(PassPA));
  }
  return PA;
}

} // namespace llvm

#endif
