//===- llvm/Orca/FlowSensitiveEATransform.h ---------------------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
// Copyright 2021 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//
// \file
// Flow-sensitive escape analysis transformation.
//
//===----------------------------------------------------------------------===//

#ifndef FLOWSENSITIVEEATRANSFORM_H
#define FLOWSENSITIVEEATRANSFORM_H

#include "llvm/ADT/BitmaskEnum.h"
#include "llvm/IR/ValueHandle.h"

using namespace llvm;

namespace azul::FlowSensitiveEA {

/// Transform is a base class for all transformations that are generated
/// based on Flow Sensitive Escape Analysis (FSEA).
/// Transformation handling is split in two main stages:
///  - construction of transformations, while traversing over non-phi
///    instructions of a basic block in direct order. There should be no
///    modification of IR that leads to invalidation of FSEA results here.
///  - actual application of transformations, modifying IR and and thus
///    invalidating FSEA results.
/// This allows FSEA results to avoid keeping track of IR transformation
/// changes which would make the analysis complicated.
/// The second transformation stage needs to remember what it is going to
/// generate. All references it needs must be kept as tracking values but
/// outside of the IR. This implies an overhead both memory and computational.
/// That is why transformations should try to apply all non-affecting IR
/// changes at the first stage to minimize complexity of the second stage.
struct Transform {
  enum IRChange {
    None = 0,
    NonCFG = 1,
    CFG = 3,
    LLVM_MARK_AS_BITMASK_ENUM(/*LargestValue=*/CFG)
  };

  virtual IRChange apply() = 0;
  virtual ~Transform(){};
#if !defined(NDEBUG) || defined(LLVM_ENABLE_DUMP)
  LLVM_DUMP_METHOD
  virtual void dump() const {};
#endif
};

struct RAUW : public Transform {
  TrackingVH<Instruction> ToReplace;
  TrackingVH<Value> ReplaceWith;

  RAUW(Instruction *I, Value *V) : ToReplace(I), ReplaceWith(V) {
    assert(I->getType() == V->getType());
  }
  ~RAUW() override {}

  IRChange apply() override {
    auto *I = ToReplace.getValPtr();
    // Do not call azul::Utils::replaceAllUsesAndErase() here because
    // ReplaceWith should not get its name changed. It might be old and have
    // some users that do not expect its name changed.
    I->replaceAllUsesWith(ReplaceWith.getValPtr());
    I->eraseFromParent();
    return IRChange::NonCFG;
  }
};

struct Erase : public Transform {
  TrackingVH<Instruction> ToErase;

  Erase(Instruction *I) : ToErase(I) {}
  ~Erase() override {}

  IRChange apply() override {
    auto *I = ToErase.getValPtr();
    I->eraseFromParent();
    return IRChange::NonCFG;
  }
};

} // namespace azul::FlowSensitiveEA

#endif /* FLOWSENSITIVEEATRANSFORM_H */
