//===- llvm/Orca/AddGCBarriers.h - Definition of the AddGCBarriers class --------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
// 
//===----------------------------------------------------------------------===//
// Copyright 2013-2018 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//
// \file
// This file provides the interface for Azul's own Add GC Barriers pass.
// This pass modifies load and store instructions to add garbage collection
// read barriers and write barriers.
///
//===----------------------------------------------------------------------===//

#include "llvm/IR/IRBuilder.h"
#include "llvm/IR/PassManager.h"

namespace llvm {

class Module;
class Function;
class Instruction;
class LoopInfo;
class LoadInst;
class PointerType;
class Value;

struct AddGCBarriers : PassInfoMixin<AddGCBarriers> {
  Module *M = nullptr; // Module that the pass was initialized for

public:
  AddGCBarriers() = default;

  static bool isRequired() { return true; }

  PreservedAnalyses run(Function &F, FunctionAnalysisManager &);

  /// Do one-time initialization (for performance)
  void init(Module &);

  /// Helper used by public run method and by the legacy pass
  PreservedAnalyses runImpl(Function &F, LoopInfo &);

private:
  // Returns true if we can generate a vector barrier (LVB/SVB) for the given
  // vector type.
  bool canGenerateVectorBarrierFor(FixedVectorType *VT);

  // Helper function used by get[LVB|SVB|Poison]Function
  Function *getFunctionForVF(unsigned VF,
                             const StringRef* FuncNames);

  // Get a LVB function from the module corresponding to load of type \p ValueTy
  // from address space \p AS.
  Function *getLVBFunction(Type *ValueTy, unsigned AS);

  // Get a SVB function from the module corresponding to a store of a \p
  // ValueTy.
  Function *getSVBFunction(Type *ValueTy);

  // Get an azul.poison.oop function from the module.
  //  VF -- "vectorization factor"; fetch an poisoning function that
  //        will poison a vector of oops of width VF. Must be a power
  //        of 2
  Function *getPoisonFunction(unsigned VF);

  /// Returns the new load instruction
  Instruction *insertReadBarrier(LoadInst *loadInst, MaybeAlign Alignment);

  /// Generate the readBarrier and replace the original load with that barrier.
  /// Returns iterator pointing to the new instruction, as any iterators
  /// pointing at original load are invalidated.
  BasicBlock::iterator replaceLoadWithReadBarrier(LoadInst *loadInst,
                                                  MaybeAlign Alignment);

  /// Returns the new value to be stored by the store instruction.
  Value *insertStoreBarrier(Value *value, Value *addr, IRBuilder<> &Builder);

  /// Replaces the store value with the store barrier generated through
  /// insertStoreBarrier.
  void replaceStoreValueWithStoreBarrier(StoreInst *SI);
};

} // end namespace llvm
