//===-- NewJVMStateBundle.h ---------------------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2018 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//
//
// This header describes an in-memory representation for a JVM State.
//
//===----------------------------------------------------------------------===//

#ifndef NEW_JVM_STATE_BUNDLE_H
#define NEW_JVM_STATE_BUNDLE_H
#include "llvm/IR/Constants.h"
#include "llvm/IR/LLVMContext.h"
#include "llvm/ADT/SmallVector.h"
#include "llvm/IR/Value.h"
#include <cstdint>
#include <optional>

namespace azul {
namespace jvmstate {

struct StateValue {
  enum ValueType : uint16_t {
    FirstType = 0,
    OopValue = FirstType,
    FirstPrimitiveType,
    Primitive8 = FirstPrimitiveType,
    Primitive16,
    Primitive32,
    Primitive64,
    // Below we have the equivalents for oop and primitive types when they are
    // indexed into a separate values section
    FirstIndexType,
    OopIndex = FirstIndexType,
    Primitive8Index,
    Primitive16Index,
    Primitive32Index,
    Primitive64Index,
    LastIndexType = Primitive64Index,
    Address,
    // Materialized values are eagerly computed by the compiled code. They are
    // available as LLVM values. Non-materialized values are not available in
    // the compiled code, but materialized lazily during deoptimization.
    LastMaterializedType = Address,
    LazyObject,
    Dead,
    KnownValue,
    NumValueTypes,
  };
  ValueType Ty;
  uint32_t Val;

  bool isIndexType() const {
    return Ty >= ValueType::FirstIndexType && Ty <= ValueType::LastIndexType;
  }
};

struct InitializedStateSlot {
  StateValue SV;
  uint16_t SlotIndex;
};

struct InitCommand {
  enum Type : int8_t {
    FIELD = -1,
    MEMCPY = -2
  };
  Type Ty;

  llvm::SmallVector<StateValue, 4> Operands;
  // Currently used only for MEMCPY commands.
  // TODO: move element size into the operands instead
  uint8_t MemcpyElementSize;
};

struct LazyObject {
  enum LazyObjectFlags : uint8_t {
    LAZY_OBJECT_FLAG_NONE = 0,
    LAZY_OBJECT_FLAG_ARRAY = 1,
    LAZY_OBJECT_FLAG_LAZY_BOX = 2
  };
  unsigned ID;
  StateValue KlassID;
  LazyObjectFlags Flags;
  StateValue ArrayLength;
  unsigned LockCount;
  llvm::SmallVector<InitCommand, 16> InitCommands;

  void setFlag(LazyObjectFlags NewFlags) {
    Flags = (LazyObjectFlags)(Flags | NewFlags);
  }

  bool isFlagSet(LazyObjectFlags F) const { return (Flags & F) == F; }
};

struct AbstractFrame {
  /// Specifies possible flags for Abstract frame.
  enum Flags : uint8_t {
    FLAG_NONE = 0,

    // Frame state corresponds to bytecode before execution if 0 or after if 1.
    FLAG_REEXECUTABLE = 1 << 0,

    // The flag indicates that if exception happens then we should deoptimize.
    FLAG_DEOPT_ON_THROW = 1 << 1,

    // Frame state contains extra lock. We should release it if we deoptimize.
    FLAG_EXTRA_LOCK = 1 << 2,

    // There is an OOM handler for this location in the current frame that
    // can access local references.
    FLAG_OOM_HANDLER_INSPECTS_LOCALS = 1 << 3,

    // Frame state contains an eliminated nested lock (which is pushed as the
    // last lock on
    // the stack through PushMonitor, similar to as done for FLAG_EXTRA_LOCK).
    // If we deoptimize we should lock it and drop it from the monitor stack.
    FLAG_ELIMINATED_LOCK = 1 << 4,
  };

  // Describes the Kind of InitializedStateSlot.
  enum ValueKind : unsigned {
    FirstValueKind = 0,
    StackValue = FirstValueKind,
    LocalValue,
    Monitor,
    NumValueKinds
  };

  Flags Flag;
  uint16_t CallerID;
  uint16_t BCI;
  uint16_t NumStack;
  uint16_t NumLocals;
  uint16_t NumMonitors;

  llvm::SmallVector<LazyObject, 4> LazyObjects;
  // TODO: Only the Stack can contain StateValue::ValueType::Address values. We
  // should create wrapper functions for inserting into the below vectors to
  // impose this constraint.
  llvm::SmallVector<InitializedStateSlot, 8> Stack;
  llvm::SmallVector<InitializedStateSlot, 8> Locals;
  llvm::SmallVector<InitializedStateSlot, 1> Monitors;
  llvm::SmallVector<llvm::Value*, 16> Values;
};

struct AbstractJVMState {
  llvm::SmallVector<AbstractFrame, 8> Frames;
};

} // namespace jvmstate
} // namespace azul

#endif // NEW_JVM_STATE_BUNDLE_H
