//===- JVMStateBundle.h - Helpers for deopt bundles for the JVM -*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2018 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//

#ifndef LLVM_IR_JVM_STATE_BUNDLE_H
#define LLVM_IR_JVM_STATE_BUNDLE_H

#include "llvm/ADT/ArrayRef.h"
#include "llvm/ADT/BitmaskEnum.h"
#include "llvm/ADT/DenseMap.h"

#include "llvm/IR/Constants.h"
#include "llvm/IR/InstrTypes.h"
#include "llvm/IR/ValueHandle.h"
#include "llvm/IR/ValueMap.h"

#include "llvm/Support/CommandLine.h"

#include <map>
#include <optional>

extern llvm::cl::opt<bool> UseNewDeoptFormatAtSerialization;

namespace azul {
// ConstantDataArraySlice has been copied from llvm/Analysis/ValueTracking.h
struct ConstantDataArraySlice {
  /// ConstantDataArray pointer. nullptr indicates a zeroinitializer (a valid
  /// initializer, it just doesn't fit the ConstantDataArray interface).
  const llvm::ConstantDataArray *Array;

  /// Slice starts at this Offset.
  uint64_t Offset;

  /// Length of the slice.
  uint64_t Length;

  /// Moves the Offset and adjusts Length accordingly.
  void move(uint64_t Delta) {
    assert(Delta < Length);
    Offset += Delta;
    Length -= Delta;
  }

  /// Convenience accessor for elements in the slice.
  uint64_t operator[](unsigned I) const {
    return Array == nullptr ? 0 : Array->getElementAsInteger(I + Offset);
  }
};
} // namespace azul

namespace llvm {
bool isKnownDeoptBundleFormat(const CallBase *CB);
void writeBriefDeoptBundleAnnotation(const Module *M, OperandBundleUse BU,
                                     raw_ostream &Out, StringRef Prefix = "");

class AbstractFrameBase {
  friend class LazyObjectSection;

public:
  enum ValueKind : unsigned {
    FirstValueKind = 0,
    StackValue = FirstValueKind,
    LocalValue,
    Monitor,
    NumValueKinds
  };

  /// Specifies possible flags for Abstract frame.
  enum Flags : unsigned {
    FLAG_NONE = 0,

    // Frame state corresponds to bytecode before execution if 0 or after if 1.
    FLAG_REEXECUTABLE = 1 << 0,

    // The flag indicates that if exception happens then we should deoptimize.
    FLAG_DEOPT_ON_THROW = 1 << 1,

    // Frame state contains extra lock. We should release it if we deoptimize.
    FLAG_EXTRA_LOCK = 1 << 2,

    // There is an OOM handler for this location in the current frame that
    // can access local references.
    FLAG_OOM_HANDLER_INSPECTS_LOCALS = 1 << 3,

    // Frame state contains an eliminated nested lock (which is pushed as the last lock on
    // the stack through PushMonitor, similar to as done for FLAG_EXTRA_LOCK).
    // If we deoptimize we should lock it and drop it from the monitor stack.
    FLAG_ELIMINATED_LOCK = 1 << 4,

    // By default, if we deoptimize upon return from a non-void non-reexecutable
    // call, the deopt handler will take the returned value and put it onto
    // expression stack.
    // This flag indicates that the result of the call, the state after which we
    // describe, is already on the expression stack, so the deopt handler
    // doesn't need to do anything in this regard. This flag allows us to
    // transfer deopt states from non-reexecutable non-void calls, since by
    // explicitly communicating the result of the original call, we can fully
    // describe the correct deoptimization state.
    FLAG_EXPLICIT_RET_VAL = 1 << 5,

    LLVM_MARK_AS_BITMASK_ENUM(
        /* LargestValue = */ FLAG_EXPLICIT_RET_VAL)
  };

  enum class ValueType : uint16_t {
    FirstType = 0,
    OopValue = FirstType,
    FirstPrimitiveType,
    Primitive8 = FirstPrimitiveType,
    Primitive16,
    Primitive32,
    Primitive64,
    // Below we have the equivalents for oop and primitive types when they are
    // indexed into a separate values section
    FirstIndexType,
    OopIndex = FirstIndexType,
    Primitive8Index,
    Primitive16Index,
    Primitive32Index,
    Primitive64Index,
    LastIndexType = Primitive64Index,
    // TODO: Remove FirstNonPrimitiveType
    FirstNonPrimitiveType,
    Address = FirstNonPrimitiveType,
    // Materialized values are eagerly computed by the compiled code. They are
    // available as LLVM values. Non-materialized values are not available in
    // the compiled code, but materialized lazily during deoptimization.
    LastMaterializedType = Address,
    LazyObject,
    Dead,
    KnownValue,
    NumValueTypes,
  };

  struct StateValue {
    ValueType Ty;
    Value *Val;

    StateValue() = default;
    StateValue(ValueType Ty, Value *Val) : Ty(Ty), Val(Val) {};
    
    static StateValue fromValues(Value *Ty, Value *Val);
    static StateValue fromValues(uint32_t Ty, Value *Val);
    static bool isStateValue(Value *Ty, Value *Val);
    static bool isStateValue(uint32_t Ty, Value *Val);
    
    static bool canEncode(Type *Ty) {
      return getValueType(Ty).has_value();
    }
    static bool canEncode(Value *V) {
      return canEncode(V->getType());
    }

    static bool isPrimitiveType(ValueType VT) {
      return VT == ValueType::Primitive8 || VT == ValueType::Primitive16 ||
             VT == ValueType::Primitive32 || VT == ValueType::Primitive64;
    }

    bool isDead() const {
      return Ty == ValueType::Dead;
    }

    bool isNullValue() const {
      if (!isPrimitiveType(Ty) && Ty != ValueType::OopValue)
        return false;
      if (auto ConstValue = dyn_cast<llvm::Constant>(Val))
        return ConstValue->isNullValue();
      return false;
    }

    // This function returns true if the StateValue can be omitted during
    // serialization.
    bool canSkip() const {
      if (isDead())
        return true;
      if (Ty == ValueType::Primitive64)
        return false; // Do not skip in this case.
      if (isNullValue())
        return true;
      return false;
    }
    
    std::optional<Value *> asValue() const {
      if (Ty == ValueType::Dead || 
          Ty == ValueType::LazyObject ||
          Ty == ValueType::KnownValue)
        return std::nullopt;
      return Val;
    }

    bool operator==(const StateValue &SV) const {
      return Ty == SV.Ty && Val == SV.Val;
    };

    std::optional<uint32_t> asLazyObjectID() const {
      if (Ty != ValueType::LazyObject)
        return std::nullopt;
      return cast<ConstantInt>(Val)->getZExtValue();
    }

    std::optional<uint32_t> asKnownValueID() const {
      if (Ty != ValueType::KnownValue)
        return std::nullopt;
      return cast<ConstantInt>(Val)->getZExtValue();
    }

    static std::optional<StateValue> fromValue(Value *V) {
      if (auto VTY = getValueType(V->getType()))
        return StateValue(*VTY, V);
      return std::nullopt;
    }

    static StateValue fromKnownValue(LLVMContext &C, uint32_t KVID);
    static StateValue fromLazyObjectID(LLVMContext &C, uint32_t LOID);

    static std::optional<ValueType> getValueType(Type *Ty);

    static bool isMaterializedValueType(ValueType VT) {
      return VT <= ValueType::LastMaterializedType;
    }

    static bool isIndexType(ValueType VT) {
      return VT >= ValueType::FirstIndexType && VT <= ValueType::LastIndexType;
    }

    // A function to check that the type specified by ExpectedVT matches with
    // the type encoded in V for materialized types.
    static bool checkIfValueTypesMatch(ValueType ExpectedVT, Value *V);

    StateValue makeIndexType(Value *NewVal) {
      assert(Ty >= ValueType::FirstType && Ty < ValueType::FirstIndexType &&
             "Can make only non-index materialized values to index type");
      return {(ValueType)((uint16_t)Ty + (uint16_t)ValueType::FirstIndexType),
              NewVal};
    }

    StateValue makeNonIndexType(Value *NewVal) {
      assert(Ty >= ValueType::FirstIndexType &&
             Ty <= ValueType::LastIndexType &&
             "Can make only indexed materializable values to non-index type");
      return {(ValueType)((uint16_t)Ty - (uint16_t)ValueType::FirstIndexType),
              NewVal};
    }

    // Returns true if the StateValue must be placed in the Values section
    // once serialized.
    bool serializedInValuesSection() const {
      // We serialize i64 constants through an indirection in Values section
      // so as to keep the size of constants i32.
      return !isa<llvm::ConstantInt>(Val) || Ty == ValueType::Primitive64;
    }

    // Returns Non-Index Type StateValue
    static std::optional<StateValue>
    fromValuesNonIndexed(LLVMContext &C, unsigned Ty, unsigned Val,
                         ArrayRef<Use> ValuesSection);
  };

  // InitializedStateSlot encodes an initialized slot (stack, local or monitor)
  // of an abstract frame. It's essentially a pair of a StateValue and the  
  // position of the slot, densely packed for compact representation in deopt 
  // bundles. 
  //
  // In deopt bundles InitializedStateSlots are represented as a pair of
  // values: value tag and payload value. The value tag is a 32-bit integer
  // whose most significant 16 bits contain the slot index and the lower 16 bits
  // contain the ValueType of the StateValue. The payload value is the value
  // component of the StateValue.

  struct InitializedStateSlot {
    StateValue SV;
    uint16_t SlotIndex;

    InitializedStateSlot(StateValue SV, uint16_t SlotIndex = 0)
        : SV(SV), SlotIndex(SlotIndex){};

    static InitializedStateSlot fromValues(Value *ValueTag, Value *Val);
    static bool isInitializedStateSlot(Value *ValueTag, Value *Val);

    uint32_t getValueTagAsInt() const {
      uint32_t VTL = SlotIndex;
      VTL = (VTL << 16) | (((uint32_t)SV.Ty & 0xFFFF));
      return VTL;
    }

    /// Replaces the stored StateValue with the Dead one
    void kill() {
      SV = {ValueType::Dead, Constant::getNullValue(SV.Val->getType())};
    }

    // Returns ISS with Non-Index Type StateValue
    static std::optional<InitializedStateSlot>
    fromValuesNonIndexed(LLVMContext &C, unsigned Ty, unsigned Val,
                         ArrayRef<Use> ValuesSection);
  };

  // Currently, an AbstractJVMState contains a single LazyObjectSection
  // before the list of AbstractFrames. We are in transition to the scheme
  // were any of the AbstractFrames can have a separate LazyObjectSection.
  // Once complete, we should be able to remove the custom composition 
  // logic for inlining (ComposeDeoptBundles) and fall back to the upstream
  // concatenation of the deopt bundles.   
  //
  // When LazyObjectSection belongs to an AbstractFrame StateValues can refer
  // to LazyObjects from ancestor frames. LazyObjectRef encodes a reference to
  // a LazyObject (possibly cross-frame reference) in a LazyObjectSection.
  // RelativeFrameOffset field contains the relative offset of the ancestor
  // frame where the lazy materialized object is defined. These relative offsets
  // are stable during inlining. LazyObjectID is the ID of the lazy object in
  // the corresponding LazyObjectSection.
  //
  // During the transition we implicitly assume that the single LazyObjectSection
  // belongs to the root frame (even though technically it's not yet a part of
  // an AbstractFrame).
  // A Copy of LazyObjectRef exists in DeoptStateSerialization.h
  struct LazyObjectRef {
    unsigned RelativeFrameOffset;
    unsigned LazyObjectID;

    LazyObjectRef(unsigned LazyObjectID, unsigned RelativeFrameOffset)
        : RelativeFrameOffset(RelativeFrameOffset), LazyObjectID(LazyObjectID) {
      unsigned MaxAllowedLazyObjectID = 1 << 16;
      unsigned MaxAllowedRelativeFrameOffset = 1 << 16;
      assert(RelativeFrameOffset < MaxAllowedRelativeFrameOffset &&
             "cannot reference objects that are more than 65536 frames away");
      assert(LazyObjectID < MaxAllowedLazyObjectID &&
             "can have only 65536 LazyObjects in a frame");
    }

    LazyObjectRef(unsigned LazyObjectRefAsInt) {
      RelativeFrameOffset =  LazyObjectRefAsInt >> 16;
      LazyObjectID = LazyObjectRefAsInt & 0xFFFF;
    }

    unsigned getAsInt() {
      return (RelativeFrameOffset << 16) | (((uint32_t)LazyObjectID & 0xFFFF));
    }
  };

protected:
  enum : unsigned {
    FlagIndex = 0,
    CallerIDIndex,
    BCIIndex,
    NumInitializedStackElementsIndex,
    NumStackElementsIndex,
    NumInitializedLocalsIndex,
    NumLocalsIndex,
    NumInitializedMonitorIndex,
    NumMonitorIndex,
    OldHeaderSize,
    NumValuesIndex = OldHeaderSize,
    HeaderSize
  };
};

/// Length, ID, StateValue(KlassID), HasArrayLength, StateValue(ArrayLength),
/// LockCount
constexpr unsigned LAZY_OBJECT_HEADER_LENGTH = 8;
constexpr unsigned LAZY_OBJECT_FIELD_LENGTH = 4;

enum LazyObjectFlags {
  LAZY_OBJECT_FLAG_NONE = 0,
  LAZY_OBJECT_FLAG_ARRAY = 1,
  LAZY_OBJECT_FLAG_LAZY_BOX = 2
};

struct InitCommand {
  using StateValue = AbstractFrameBase::StateValue;
  using ValueType = AbstractFrameBase::ValueType;
  enum {
    // Command IDs.
    // Use small negative numbers to easily find them in IR dumps.
    FIELD = -1,  // StateValue(Offset), StateValue(Value)
    MEMCPY = -2, // StateValue(DstOffset), StateValue(Src), StateValue(SrcOffset), StateValue(Length), ElemSize

    // Number of arguments.
    NUM_OF_ARGS_FIELD = 4,
    NUM_OF_ARGS_MEMCPY = 9,
  };

  // In the old format LazyObjectSection is built from a sequence of operands
  // FullSection.
  // In the new format LazyObjectSection is built from ConstantsSection and
  // ValuesSection.
  bool IsNewFormat;

  // The underlying sequence of operands for the old format.
  ArrayRef<Use> Args; // Includes command type as Args[0].

  // The underlying constant data and values section for the new format.
  azul::ConstantDataArraySlice ConstantArgs;
  ArrayRef<Use> ValuesSection;

  // Construct an InitCommand in the old format from a sequence of operands
  InitCommand(ArrayRef<Use> Args) : Args(Args) {
    IsNewFormat = false;
    assert(getLength() == 1 + getNumOfArgs(getCmd()));
  }

  // Construct an InitCommand in the new format from a ConstantDataArray
  InitCommand(azul::ConstantDataArraySlice ConstantArgs,
              ArrayRef<Use> ValuesSection)
      : ConstantArgs(ConstantArgs), ValuesSection(ValuesSection) {
    IsNewFormat = true;
    assert(getLength() == 1 + getNumOfArgs(getCmd()));
  }

  int32_t getCmd() const;
  unsigned getLength() const {
    return IsNewFormat ? ConstantArgs.Length : Args.size();
  }

  static size_t isInitCommand(int InitCommand) {
    switch (InitCommand) {
    case FIELD:
    case MEMCPY:
      return true;
    }
    return false;
  }

  static size_t getNumOfArgs(int InitCommand) {
    switch (InitCommand) {
    case FIELD:
      return NUM_OF_ARGS_FIELD;
    case MEMCPY:
      return NUM_OF_ARGS_MEMCPY;
    }
    llvm_unreachable("Unknown init command");
  }

  template <typename VisitFunc>
  void visitStateValues(const VisitFunc &Visit) const;

  AbstractFrameBase::StateValue getMemcpyDstOffset(bool NonIndexed =
                                                       true) const;
  AbstractFrameBase::StateValue getMemcpySrc(bool NonIndexed = true) const;
  AbstractFrameBase::StateValue getMemcpySrcOffset(bool NonIndexed =
                                                       true) const;
  AbstractFrameBase::StateValue getMemcpyLength(bool NonIndexed = true) const;
  Value *getMemcpyElemSize() const;

  AbstractFrameBase::StateValue getFieldOffset(bool NonIndexed = true) const;
  AbstractFrameBase::StateValue getFieldValue(bool NonIndexed = true) const;

  LLVMContext &getContext() const {
    return IsNewFormat ? ConstantArgs.Array->getContext()
                       : Args[0]->getContext();
  }

private:
  std::optional<AbstractFrameBase::StateValue>
  getStateValueAtOffset(unsigned Offset, bool NonIndexed) const;

  AbstractFrameBase::StateValue getStateValueOperand(int32_t ExpectedCmd,
                                                     unsigned Index,
                                                     bool NonIndexed) const;
};

class LazyObject {
  using StateValue = AbstractFrameBase::StateValue;
  using ValueType = AbstractFrameBase::ValueType;

  uint32_t ID;
  // We store two versions of KlassID and ArrayLength -
  // 1. Version that does not use Index ValueTypes
  // 2. Version that may use Index ValueTypes, name prefixed with the word
  // "Indexed". If this version stores an Indexed Type, then the corresponding
  // Value* will be a ConstantInt pointing to the real Value* in ValuesSection
  // of the Parent Frame of the LazyObject.
  StateValue KlassID;
  StateValue IndexedKlassID;
  std::optional<StateValue> ArrayLength;
  std::optional<StateValue> IndexedArrayLength;
  unsigned LockCount = 0;
  /// Indicates that the lazy object is a lazily materialized box object.
  /// A box object is a wrapper over a primitive value produced as a
  /// result of a call to the corresponding BoxType::valueOf method.
  /// Some box types have a cache of small values which is required by the JLS.
  /// If the boxed value is within the cached range valueOf returns the cached 
  /// box, otherwise a new instance is allocated. Lazy materialization of boxes
  /// implements the same behavior.
  bool LazyBox = false;
  SmallVector<InitCommand, 8> InitCommands;

private:
  LazyObject() = default;

public:
  // In the old format LazyObjectSection is built from a sequence of operands
  // FullSection.
  // In the new format LazyObjectSection is built from ConstantsSection and
  // ValuesSection.
  // Build a LazyObject in the old format from a sequence of operands
  static std::optional<LazyObject> TryBuild(ArrayRef<Use> Uses);
  static LazyObject Build(ArrayRef<Use> Uses) { return *TryBuild(Uses); }
  // Build a LazyObject in the new format from a ConstantDataArray
  static std::optional<LazyObject>
  TryBuild(azul::ConstantDataArraySlice Constants, ArrayRef<Use> ValuesSection);
  static LazyObject Build(azul::ConstantDataArraySlice Constants,
                          ArrayRef<Use> ValuesSection) {
    return *TryBuild(Constants, ValuesSection);
  }

  uint32_t getID() const { return ID; }
  StateValue getKlassID(bool NonIndexed = true) const {
    return NonIndexed ? KlassID : IndexedKlassID;
  }
  std::optional<StateValue> getArrayLength(bool NonIndexed = true) const {
    return NonIndexed ? ArrayLength : IndexedArrayLength;
  }
  unsigned getLockCount() const { return LockCount; }
  bool isLazyBox() const { return LazyBox; }

  unsigned getLength() const {
    unsigned Result = LAZY_OBJECT_HEADER_LENGTH;
    for (auto &IC : InitCommands)
      Result += IC.getLength();
    return Result;
  }

  const SmallVectorImpl<InitCommand> &getInitCommands() const {
    return InitCommands;
  }

  unsigned getNumInitCommands() const { return InitCommands.size(); }

  const InitCommand &getInitCommandAt(unsigned Index) const {
    return InitCommands[Index];
  }
};

class LazyObjectSection {
  // In the old format LazyObjectSection is built from a sequence of operands
  // FullSection.
  // In the new format LazyObjectSection is built from ConstantsSection and
  // ValuesSection.
  bool IsNewFormat;

  // The underlying sequence of operands for the old format.
  ArrayRef<Use> FullSection;

  // The underlying constant data and values section for the new format.
  azul::ConstantDataArraySlice ConstantsSection;
  ArrayRef<Use> ValuesSection;

  SmallVector<LazyObject, 8> LazyObjects;

private:
  LazyObjectSection() = default;

public:
  static std::optional<LazyObjectSection> TryBuild(ArrayRef<Use> Uses);
  static LazyObjectSection Build(ArrayRef<Use> Uses) { return *TryBuild(Uses); }
  static LazyObjectSection BuildEmpty() { return LazyObjectSection(); }

  unsigned getLength() const {
    return IsNewFormat ? ConstantsSection.Length : FullSection.size();
  }
  unsigned getNumObjects() const { return LazyObjects.size(); }

  const LazyObject &getObjectAt(unsigned Index) const {
    return LazyObjects[Index];
  }

  void serialize(std::vector<Value *> &Out) const {
    Out.insert(Out.end(), FullSection.begin(), FullSection.end());
  }
};

/// \brief This class specifies layout of elements of a single abstract frame.
///
/// To use this class you have to derive your class from it and pass your class
/// as a template parameter. Your class should implement following methods:
/// getNumStackElements, getNumLocals and getNumMonitors.
template <class T>
class AbstractFrameLayout : public AbstractFrameBase {
protected:
  /// Returns an offset at which the first stack element should be.
  unsigned getStackElementsBeginOffset() const {
    auto ThisT = static_cast<const T *>(this);
    return ThisT->getHeaderSize() + ThisT->getLazyObjectSection().getLength();
  }
  /// Returns an offset which is the next to the end of the last stack element.
  unsigned getStackElementsEndOffset() const {
    auto ThisT = static_cast<const T *>(this);
    return getStackElementsBeginOffset() +
           ThisT->getNumInitializedStackElements() * 2;
  }

  /// Returns an offset at which the first local value should be.
  unsigned getLocalsBeginOffset() const { return getStackElementsEndOffset(); }
  /// Returns an offset which is the next to the end of the last local value.
  unsigned getLocalsEndOffset() const {
    auto ThisT = static_cast<const T *>(this);
    return getLocalsBeginOffset() + ThisT->getNumInitializedLocals() * 2;
  }

  /// Returns an offset at which the first monitor should be.
  unsigned getMonitorsBeginOffset() const { return getLocalsEndOffset(); }
  /// Returns an offset which is the next to the end of the last monitor.
  unsigned getMonitorsEndOffset() const {
    auto ThisT = static_cast<const T *>(this);
    return getMonitorsBeginOffset() + ThisT->getNumInitializedMonitors() * 2;
  }

  unsigned getValuesBeginOffset(ValueKind Kind) const {
    switch (Kind) {
    case StackValue:
      return getStackElementsBeginOffset();
    case LocalValue:
      return getLocalsBeginOffset();
    case Monitor:
      return getMonitorsBeginOffset();
    default:
      llvm_unreachable("Invalid value kind!");
    }
  }

  unsigned getValuesEndOffset(ValueKind Kind) const {
    switch (Kind) {
    case StackValue:
      return getStackElementsEndOffset();
    case LocalValue:
      return getLocalsEndOffset();
    case Monitor:
      return getMonitorsEndOffset();
    default:
      llvm_unreachable("Invalid value kind!");
    }
  }

public:
  // Returns the number of stack/locals/monitors in this frame
  unsigned getNumValues(ValueKind Kind) const {
    auto ThisT = static_cast<const T *>(this);
    switch (Kind) {
    case StackValue:
      return ThisT->getNumStackElements();
    case LocalValue:
      return ThisT->getNumLocals();
    case Monitor:
      return ThisT->getNumMonitors();
    default:
      llvm_unreachable("Invalid value kind!");
    }
  }
};

/// \brief Represents a single abstract inlined frame in a full deoptimization
/// state. AbstractFrame is meant to be a view into the OperandBundle that
/// actually stores the deopt state.
class AbstractFrame : public AbstractFrameLayout<AbstractFrame> {
private:
  Flags Flag;
  uint32_t CallerID;
  uint32_t BCI;
  uint32_t NumInitializedStackElements;
  uint32_t NumStackElements;
  uint32_t NumInitializedLocals;
  uint32_t NumLocals;
  uint32_t NumInitializedMonitors;
  uint32_t NumMonitors;
  uint32_t NumValues;

  ArrayRef<Use> AbstractState;
  ArrayRef<Use> ValuesSection;

  // In NewFormat, ConstantsInFrame is an Array that contains:
  // 1. The Header of a Frame, which  contains Flags, Caller, BCI, etc.
  // 2. All InitializedStateSlots in the frame. Non-constant and i64 constants
  // in InitializedStateSlots are stored in the ValuesSection
  // ConstantsInFrame is an array that contains elements of type i32. The array
  // can be of type ConstantDataArray or ConstantAggregateZero, hence we mark it
  // to be of type Constant.
  azul::ConstantDataArraySlice ConstantsInFrame;
  LazyObjectSection LazyObjects;
  AbstractFrame() : LazyObjects(LazyObjectSection::BuildEmpty()){};

public:
  /// \brief Construct an AbstractFrame instance from a sequence of operands.
  ///
  /// This parses out a single abstract frame from the *prefix* of \p FullState.
  /// The exact number of elements of \p FullState that are part of this
  /// abstract frame is reported by getLength.
  static std::optional<AbstractFrame> TryBuild(ArrayRef<Use> FullState);
  static AbstractFrame Build(ArrayRef<Use> FullState);

  bool hasAnyFlags() const { return Flag; }
  bool isReexecutable() const { return Flag & FLAG_REEXECUTABLE; }
  bool isDeoptOnThrow() const { return Flag & FLAG_DEOPT_ON_THROW; }
  bool hasExtraLock() const { return Flag & FLAG_EXTRA_LOCK; }
  bool hasEliminatedLock() const { return Flag & FLAG_ELIMINATED_LOCK; }
  bool canOOMHandlerInspectLocals() const {
    return Flag & FLAG_OOM_HANDLER_INSPECTS_LOCALS;
  }
  bool hasExplicitRetVal() const {
    return Flag & FLAG_EXPLICIT_RET_VAL;
  }
  Flags getFlags() const { return Flag; }
  uint32_t getCallerID() const { return CallerID; }
  uint32_t getBCI() const { return BCI; }
  unsigned getNumInitializedStackElements() const {
    return NumInitializedStackElements;
  }
  unsigned getNumStackElements() const { return NumStackElements; }
  unsigned getNumInitializedLocals() const { return NumInitializedLocals; }
  unsigned getNumLocals() const { return NumLocals; }
  unsigned getNumInitializedMonitors() const { return NumInitializedMonitors; }
  unsigned getNumMonitors() const { return NumMonitors; }
  unsigned getNumValues() const { return NumValues; }
  // In the new format, getHeaderSize demarcates the header in ConstantsInFrame,
  // while in the old format, it demarcates the header in the llvm::Use Array.
  unsigned getHeaderSize() const {
    return IsNewFormat ? HeaderSize : OldHeaderSize;
  }

  unsigned getConstantArrayLength() const {
    return LazyObjects.getLength() + getHeaderSize() +
           getNumInitializedStackElements() * 2 +
           getNumInitializedLocals() * 2 + getNumInitializedMonitors() * 2;
  }

  /// Calculates length of an abstract frame. Length here is number of Values
  /// required to encode state of a frame.
  unsigned getLength() const {
    // getConstantArrayLength gives the length of the frame in old format
    return IsNewFormat ? 1 + getNumValues() : getConstantArrayLength();
  }

  // Returns number of initalized state slots of the specified kind in this
  // frame.
  unsigned getNumInitializedValues(ValueKind Kind) const;

  InitializedStateSlot getInitializedStateSlotAt(ValueKind Kind, unsigned Index,
                                                 bool NonIndexed = true) const;

  LLVMContext &getContext() const {
    return AbstractState.front()->getContext();
  }

  const LazyObjectSection &getLazyObjectSection() const { return LazyObjects; }

  /// Returns AbstractState converted into vector of Value*.
  /// MutableAbstractJVMState constructs MutableAbstractFrames lazily, so this
  /// method is required to avoid constructing missing mutable frames when
  /// MutableAbstractJVMState::serialize is called.
  void serialize(std::vector<Value *> &Out) const;

  static StringRef ValueKindAsString(ValueKind Kind);

  // We are in the process of transitioning to a new deopt format, where in the
  // new format, we have -
  // 1. All constant operands are encoded as a ConstantArray.
  // 2. All non-constant values are stored in a separate section called
  // ValuesSection.
  bool IsNewFormat = false;

  Value *getValueAt(unsigned Index) const {
    assert(IsNewFormat);
    assert(Index < NumValues);
    return ValuesSection[Index].get();
  }

private:
  Value *getOperandAt(unsigned Index) const { return AbstractState[Index]; }
  InitializedStateSlot getInitializedStateSlotAtOffset(unsigned Offset,
                                                       bool NonIndexed) const;
};

/// \brief Represents a "full" deoptimization state, consisting of a sequence of
/// abstract frames.
///
/// Note: while this a wrapper class, constructing this isn't as cheap as
/// constructing a CallSite or an ArrayRef.
class AbstractJVMState {
  ArrayRef<Use> FullState;
  SmallVector<AbstractFrame, 8> Frames;

private:
  AbstractJVMState() = default;

public:
  AbstractJVMState(const AbstractJVMState &JVMS) {
    // Cannot delete this method because it is needed by
    // Optional<AbstractJVMState>.
    llvm_unreachable("Avoid copying AbstractJVMState");
  };
  AbstractJVMState &operator=(const AbstractJVMState &JVMS) {
    // Cannot delete this method because it is needed by
    // Optional<AbstractJVMState>.
    llvm_unreachable("Avoid copying AbstractJVMState");
  };

  AbstractJVMState(AbstractJVMState &&JVMS) = default;
  AbstractJVMState &operator=(AbstractJVMState &&JVMS) = default;

  static std::optional<AbstractJVMState> TryBuild(OperandBundleUse OBU);
  static AbstractJVMState Build(OperandBundleUse OBU);
  static AbstractJVMState BuildEmpty() { return AbstractJVMState(); }

  unsigned getNumFrames() const { return Frames.size(); }
  const AbstractFrame &getFrameAt(unsigned Index) const {
    return Frames[Index];
  }
  const AbstractFrame &getYoungestFrame() const { return Frames.back(); }
  // Iterates through all StateValues and pass them to the specified function
  // in an unspecified order. For Frames, it iterates through all StateValues
  // inside InitializedStateSlots.
  void visitStateValues(
      std::function<void(const AbstractFrame::StateValue &SV)> Visit) const;
};

using ChangeStateValuesCallbackTy =
    std::function<bool(AbstractFrame::StateValue &SV, unsigned FrameIdx)>;

struct MutableInitCommand {
  using StateValue = AbstractFrameBase::StateValue;
  using ValueType = AbstractFrameBase::ValueType;
  int32_t Cmd;
  // Maximum of 9 arguments (4 from Field, 9 from MemCpy).
  SmallVector<Value *, 9> Args;

  MutableInitCommand(int32_t Cmd) : Cmd(Cmd) {}
  MutableInitCommand(const InitCommand &IC);

  unsigned getLength() const { return 1 + InitCommand::getNumOfArgs(Cmd); }

  template <typename ChangeFunc>
  bool changeStateValues(const ChangeFunc &Change, unsigned FrameIdx);
};

class MutableLazyObject {
  using StateValue = AbstractFrameBase::StateValue;
  uint32_t ID;
  StateValue KlassID;
  std::optional<StateValue> ArrayLength;
  unsigned LockCount = 0;
  bool LazyBox = false;
  SmallVector<MutableInitCommand, 8> InitCommands;

public:
  explicit MutableLazyObject(const LazyObject &L)
      : ID(L.getID()), KlassID(L.getKlassID()), ArrayLength(L.getArrayLength()),
        LockCount(L.getLockCount()), LazyBox(L.isLazyBox()),
        InitCommands(parse(L.getInitCommands())) {}

  explicit MutableLazyObject(uint32_t ID, StateValue KlassID,
                             std::optional<StateValue> ArrayLength,
                             unsigned LockCount, bool LazyBox)
      : ID(ID), KlassID(KlassID), ArrayLength(ArrayLength),
        LockCount(LockCount), LazyBox(LazyBox) {
    assert((!LazyBox || LockCount == 0) &&
           "LazyBoxes can't have non-zero LockCount");
  }

  static SmallVector<MutableInitCommand, 8>
  parse(const SmallVectorImpl<InitCommand> &InitCommands);

  uint32_t getID() const { return ID; }
  StateValue getKlassID() const { return KlassID; }
  std::optional<StateValue> getArrayLength() const { return ArrayLength; }
  unsigned getLockCount() const { return LockCount; }
  bool isLazyBox() const { return LazyBox; }

  void setKlassID(StateValue KlassID) { this->KlassID = KlassID; }
  void setArrayLength(std::optional<StateValue> ArrayLength) {
    this->ArrayLength = ArrayLength;
  }
  void setLockCount(unsigned LockCount) {
    this->LockCount = LockCount;
  }
  void setLazyBox() { LazyBox = true; }

  SmallVectorImpl<MutableInitCommand> &initCommands() { return InitCommands; }
  const SmallVectorImpl<MutableInitCommand> &initCommands() const {
    return InitCommands;
  }

  unsigned getLength() const {
    unsigned Length = LAZY_OBJECT_HEADER_LENGTH;
    for (auto &IC : InitCommands)
      Length += IC.getLength();
    return Length;
  }

  void serialize(std::vector<Value *> &Out,
                 LLVMContext &Context) const; // old format
  void serialize(SmallVectorImpl<unsigned> &Constants,
                 DenseMap<Value *, unsigned> &Values,
                 LLVMContext &C) const; // new format

  // Iterates through all StateValues and pass them to the specified function
  // in an unspecified order. The function may change the values and if so
  // must return true.
  bool changeStateValues(ChangeStateValuesCallbackTy Change, unsigned FrameIdx);
};

class MutableLazyObjectSection {
  using StateValue = AbstractFrameBase::StateValue;
  // TODO: Make MutableLazyObjectSection lazy initialized with the immutable
  // LazyObjectSection and mutable creating objects if a change is made. That is
  // similar to the mutable frames in MutableAbstractJVMState.
  // std::map is used here to get the natural order from the iterator, so the
  // lazy objects get into the lazy section in order of their numbers.
  using LazyObjectMap = std::map<uint32_t, MutableLazyObject>;
  LazyObjectMap LazyObjects;
  unsigned MaxID;

public:
  explicit MutableLazyObjectSection(const LazyObjectSection &L) : MaxID(0) {
    for (unsigned Index = 0; Index < L.getNumObjects(); ++Index) {
      auto Object = L.getObjectAt(Index);
      LazyObjects.emplace(Object.getID(), MutableLazyObject(Object));
      MaxID = std::max(MaxID, Object.getID());
    }
  }

  void serialize(std::vector<Value *> &Out, LLVMContext &C) const; // old format
  void serialize(SmallVectorImpl<unsigned> &Constants,
                 DenseMap<Value *, unsigned> &Values,
                 LLVMContext &C) const; // new format

  MutableLazyObject &addLazyObject(StateValue KlassID,
                                   std::optional<StateValue> ArrayLength,
                                   unsigned LockCount, bool LazyBox) {
    auto NewID = ++MaxID;
    auto O = LazyObjects.emplace(NewID,
      MutableLazyObject(NewID, KlassID, ArrayLength, LockCount, LazyBox));
    return O.first->second;
  }

  // Returns cumulative size of all LazyObjects
  unsigned getLength() const {
    unsigned Length = 0;
    for (auto &LazyObject : LazyObjects)
      Length += LazyObject.second.getLength();
    return Length;
  }

  LazyObjectMap::iterator begin() { return LazyObjects.begin(); }
  LazyObjectMap::iterator end() { return LazyObjects.end(); }
};

/// \brief Represents a single abstract inlined frame in a full deoptimization
/// state. This class enables you to alter abstract frame contents and generate
/// a new state that contains your modifications.
///
/// Note: it doesn't affect exsisting state, it just remembers all
/// modifications you want to perform and generates NEW state that you can use
/// when you generate new deopt bundles.
class MutableAbstractFrame : public AbstractFrameLayout<MutableAbstractFrame> {
private:
  using StateValue = AbstractFrameBase::StateValue;
  Flags Flag;
  uint32_t CallerID;
  uint32_t BCI;

  unsigned NumStackElements = 0;
  unsigned NumLocals = 0;
  unsigned NumMonitors = 0;

  SmallVector<InitializedStateSlot, 8> State[ValueKind::NumValueKinds];
  MutableLazyObjectSection Objects;

public:
  // This constructor only reserves space in the stack/locals/monitors state
  // vectors with the sizes provided as arguments. It does not initialize 
  // NumStackElements/NumLocals/NumMonitors fields. These fields shoud be
  // initialized implicitly by push API for the corresponding sections.
  MutableAbstractFrame(int function_id, int bci, Flags flags,
                       unsigned InitializedStackSize,
                       unsigned InitializedLocalsSize,
                       unsigned InitializedMonitorsSize)
      : Flag(flags), CallerID(function_id), BCI(bci),
        Objects(LazyObjectSection::BuildEmpty()) {
    reserve(StackValue, InitializedStackSize);
    reserve(LocalValue, InitializedLocalsSize);
    reserve(Monitor, InitializedMonitorsSize);
  }

  explicit MutableAbstractFrame(int function_id, int bci, Flags flags)
      : Flag(flags), CallerID(function_id), BCI(bci),
        Objects(LazyObjectSection::BuildEmpty()) {}

  explicit MutableAbstractFrame(const AbstractFrame &Frame);

  bool hasAnyFlags() const { return Flag; }
  bool isReexecutable() const { return Flag & Flags::FLAG_REEXECUTABLE; }
  bool isDeoptOnThrow() const { return Flag & Flags::FLAG_DEOPT_ON_THROW; }
  bool hasExtraLock() const { return Flag & Flags::FLAG_EXTRA_LOCK; }
  bool hasEliminatedLock() const { return Flag & Flags::FLAG_ELIMINATED_LOCK; }
  bool hasExplicitRV() const { return Flag & Flags::FLAG_EXPLICIT_RET_VAL; }
  bool canOOMHandlerInspectLocals() const {
    return Flag & Flags::FLAG_OOM_HANDLER_INSPECTS_LOCALS;
  }
  bool hasExplicitRetVal() const {
    return Flag & Flags::FLAG_EXPLICIT_RET_VAL;
  }
  Flags getFlags() const { return Flag; }
  uint32_t getCallerID() const { return CallerID; }
  uint32_t getBCI() const { return BCI; }
  unsigned getNumInitializedStackElements() const {
    return getNumInitializedValues(StackValue);
  }
  unsigned getNumStackElements() const { return NumStackElements; }
  unsigned getNumInitializedLocals() const {
    return getNumInitializedValues(LocalValue);
  }
  unsigned getNumLocals() const { return NumLocals; }
  unsigned getNumInitializedMonitors() const {
    return getNumInitializedValues(Monitor);
  }
  unsigned getNumMonitors() const { return NumMonitors; }
  void reserveStackElements(unsigned Size) { reserve(StackValue, Size); }
  void reserveLocals(unsigned Size) { reserve(LocalValue, Size); }
  void reserveMonitors(unsigned Size) { reserve(Monitor, Size); }

  MutableLazyObjectSection &getLazyObjectSection() { return Objects; }

  void setFlags(Flags NewValue) { Flag = NewValue; }
  void addFlags(Flags NewValue) { Flag |= NewValue; }
  void setCallerID(uint32_t NewValue) { CallerID = NewValue; }
  void setBCI(uint32_t NewValue) { BCI = NewValue; }

  void pushInitializedStateSlot(ValueKind Kind, InitializedStateSlot NewValue);

  void pushMonitor(InitializedStateSlot NewMonitor) {
    pushInitializedStateSlot(ValueKind::Monitor, NewMonitor);
    NumMonitors++;
  }

  void pushStackValue(InitializedStateSlot NewStackValue) {
    pushInitializedStateSlot(ValueKind::StackValue, NewStackValue);
    NumStackElements++;
  }

  void pushLocalValue(InitializedStateSlot NewLocalValue) {
    pushInitializedStateSlot(ValueKind::LocalValue, NewLocalValue);
    NumLocals++;
  }

  // Returns number of Initialized States of the specified kind in this frame.
  unsigned getNumInitializedValues(ValueKind Kind) const;

  // Reserves space for given amount of values of the specified kind.
  void reserve(ValueKind Kind, unsigned Size);

  // Note that Index here is the index in the array of InitializedStateSlots of
  // the specified kind and not the slot index in the abstract state.
  InitializedStateSlot &valueAt(ValueKind Kind, unsigned Index);

  // Refer to comment above AbstractFrame.IsNewFormat for definition of
  // NewFormat.
  void serialize(std::vector<Value *> &Out, LLVMContext &C,
                 bool NewFormat) const;

  // Iterates through all StateValues and pass them to the specified function
  // in an unspecified order. The function may change the values and if so
  // must return true.
  bool changeStateValues(ChangeStateValuesCallbackTy Change, unsigned FrameIdx);

  MutableLazyObject &addLazyObject(StateValue KlassID,
                                   std::optional<StateValue> ArrayLength,
                                   unsigned LockCount, bool LazyBox) {
    return Objects.addLazyObject(KlassID, ArrayLength, LockCount, LazyBox);
  }
};

/// \brief Represents a "full" deoptimization state, consisting of a sequence of
/// abstract frames. You can modify contents of abstract frames and generate new
/// deoptimization state that contains your modifications.
///
/// Note: it doesn't affect any existing "deopt" bundle! To use your new state
/// you have to generate a new instruction with this state in its deopt bundle.
class MutableAbstractJVMState {
public:
  using LazyObjectRef = AbstractFrameBase::LazyObjectRef;
  using StateValue = AbstractFrameBase::StateValue;

private:
  AbstractJVMState JVMS;
  DenseMap<unsigned, MutableAbstractFrame> Frames;
  int NumFrames;

  // This map contains references to allocation instructions added by
  // addDematerializableAllocation(Instruction*). All these allocations are
  // prepared for being dematerialized by dematerializeLazyObjects().
  // All New LazyObjects are added in the LazyObjectSection of the first frame.
  DenseMap<Value *, unsigned> NewLazyObjects;

public:
  explicit MutableAbstractJVMState(AbstractJVMState &&JVMS)
      : JVMS(std::move(JVMS)), NumFrames(this->JVMS.getNumFrames()) {}
  // Constructs AbstractJVMState and then uses it to construct
  // MutableAbstractJVMState. Fails only when AbstractJVMState::TryBuild fails.
  static std::optional<MutableAbstractJVMState> TryBuild(OperandBundleUse OBU);
  static MutableAbstractJVMState Build(OperandBundleUse OBU);

  static MutableAbstractJVMState BuildEmpty() {
    return MutableAbstractJVMState{AbstractJVMState::BuildEmpty()};
  }

  unsigned getNumFrames() const { return NumFrames; }
  // Returns a reference to mutable abstract frame that you can modify. Use
  // method `serialize` to get abstract state that contains your modifications.
  MutableAbstractFrame &getFrameAt(unsigned Index);
  MutableAbstractFrame &getYoungestFrame() {
    return getFrameAt(getNumFrames() - 1);
  }

  MutableAbstractFrame &appendFrame(const AbstractFrame &F);
  MutableAbstractFrame &appendFrame(MutableAbstractFrame &&F);

  // Iterates through all StateValues and pass them to the specified function
  // in an unspecified order. The function may change the values and if so
  // must return true.
  bool changeStateValues(ChangeStateValuesCallbackTy Change);

  // Adds a new lazy object for the specified Allocation in the First Frame of
  // the MutableAbstractJVMState. References to all allocations must be
  // dematerialized by method dematerializeLazyObjects(LLVMContext&) before this
  // MutableAbstractJVMState is serialized.
  MutableLazyObject &addDematerializableObject(Instruction *Alloc);
  MutableLazyObject &
  addDematerializableObject(StateValue KlassID,
                            std::optional<StateValue> ArrayLength,
                            unsigned LockCount, bool LazyBox,
                            const SmallVectorImpl<Value *> &Aliases);
  MutableLazyObject &
  addDematerializableObject(StateValue KlassID,
                            std::optional<StateValue> ArrayLength,
                            unsigned LockCount, bool LazyBox, Value *V) {
    return addDematerializableObject(KlassID, ArrayLength, LockCount,
                                     LazyBox, SmallVector<Value *, 1>(1, V));
  }

  // Once both adding lazy objects and adding their fields are finished
  // this method should be called before serialization can be done.
  // This method changes all references StateValue(OopType, Value*) to the
  // lazy object references StateValue(LazyObject, LazyObjectID).
  // Returns true if at least one change has been made.
  void dematerializeLazyObjects(LLVMContext &C);

  // Refer to comment above AbstractFrame.IsNewFormat for definition of
  // NewFormat. Returns new abstract deoptimization state that contains all the
  // modifications performed on states returned by getFrameAt/getYoungestFrame.
  std::vector<Value *>
  serialize(LLVMContext &C,
            bool IsNewFormat = UseNewDeoptFormatAtSerialization);

private:
  void serializeFrameAt(std::vector<Value *> &Out, LLVMContext &C,
                        unsigned FrameIndex, bool IsNewFormat);
};

struct LazyObjectBuilder {
  struct InitCommandVH {
    int32_t Cmd;
    // Maximum of 9 arguments (4 from Field, 9 from MemCpy).
    SmallVector<TrackingVH<llvm::Value>, 9> Args;

    InitCommandVH(int32_t Cmd) : Cmd(Cmd) {}
    static std::optional<InitCommandVH> GetField(Value *Offset, Value *Value);
    static std::optional<InitCommandVH> GetMemcpy(Value *DstOffset, Value *Src,
                                                  Value *SrcOffset,
                                                  Value *Length,
                                                  Value *ElemSize);
  };

  LazyObjectBuilder() = default;
  LazyObjectBuilder(Instruction *Alloc);
  LazyObjectBuilder(Value *KlassID, std::optional<Value *> ArrayLength);

  Value *KlassID;
  std::optional<TrackingVH<llvm::Value>> ArrayLength;
  unsigned LockCount = 0;
  bool LazyBox = false;

  SmallVector<InitCommandVH, 4> InitCommands;

  std::optional<llvm::Value *> getArrayLength() const {
    if (ArrayLength.has_value())
      return ArrayLength->getValPtr();
    return std::nullopt;
  }
};

/// Collects all values in the given deopt state which can't be replaces with
/// lazy objects.
void collectNonReplaceableValues(OperandBundleUse OBU,
                                 SmallPtrSetImpl<Value *> &Result);
/// Checks whether it's legal to replace new allocation values in the deopt 
/// state with symbolic description of lazy materialized object.
/// 
/// The call site must have a deopt state with known deopt format.
bool canReplaceWithLazyObject(OperandBundleUse OBU, const Value *New);
bool canReplaceWithLazyObject(const CallBase *CB, const Value *New);

/// The caller is responsible to check whether the replacement is legal
///
/// Objects is a list of lazy objects.
/// Mapping is a map between values to replace and corresponding lazy objects.
/// Mapping contains the index of the lazy object description in Objects list.
/// There might be more than one value pointing to one lazy object.
bool ReplaceWithLazyObjects(OperandBundleUse OBU, 
                            SmallVectorImpl<LazyObjectBuilder> &Objects,
                            ValueMap<Value*, unsigned> &Mapping,
                            std::vector<Value*> &NewBundle);
/// A convenience wrapper for ReplaceWithLazyObjects which takes one lazy object
/// and one value pointing to this object.
bool ReplaceWithLazyObject(OperandBundleUse OBU, Value *V,
                           LazyObjectBuilder &Object,
                           std::vector<Value*> &NewBundle);

/// Convert JVM state sitting on a call-site \p CB from after-call state
/// to before-call state.
/// This involves setting reexecutable flag and filling corresponding
/// abstract frame with appropriate (JVM) stack values.
///
///
/// Say, when inlining the call
///
///   call void @call_stub(i32 -1) [
///     ; F0 { Deopt-On-Throw, CallerID=1, BCI=2 }
///     ; F1 { Deopt-On-Throw, CallerID=2, BCI=3, L0=<Dead> }
///     ... ]
///
/// And for the call-site inside the call_stub:
///   call void @llvm.experimental.deoptimize() [ "deopt"() ]
///
/// we should get the following deopt bundle for inlined call-site:
///   call void @llvm.experimental.deoptimize() [
///     ; F0 { Deopt-On-Throw, CallerID=1, BCI=2 }
///     ; F1 { Re-Executable Deopt-On-Throw, CallerID=2, BCI=3, S0=<i32 -1>, L0=<Dead> }
///     ... ]
///
void convertAfterCallToBeforeCall(MutableAbstractFrame &CallState,
                                  CallBase *CB);

}

#endif
