//===-- DeoptStateSerialization.h -------------------------------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2018 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//
//
// This header defines API support for all the Serialization formats that we
// support for the New Deopt State.
// At the moment we have:
//   1. Yaml Parser that Parses/Emits Yaml Description of the Deopt State.
//
//===----------------------------------------------------------------------===//

#ifndef DEOPT_STATE_SERIALIZATION_H
#define DEOPT_STATE_SERIALIZATION_H
#include "llvm/ADT/SmallVector.h"
#include "llvm/IR/LLVMContext.h"
#include "llvm/IR/IRBuilder.h"
#include "llvm/IR/InstrTypes.h"
#include "llvm/IR/Orca/NewJVMStateBundle.h"
#include "llvm/IR/Orca/JVMStateBundle.h"
#include "llvm/Support/YAMLTraits.h"

// Helper structs to store data;
LLVM_YAML_DECLARE_ENUM_TRAITS(azul::jvmstate::StateValue::ValueType)
LLVM_YAML_DECLARE_ENUM_TRAITS(azul::jvmstate::InitCommand::Type)
LLVM_YAML_DECLARE_BITSET_TRAITS(azul::jvmstate::LazyObject::LazyObjectFlags)
LLVM_YAML_DECLARE_BITSET_TRAITS(azul::jvmstate::AbstractFrame::Flags)
LLVM_YAML_IS_SEQUENCE_VECTOR(azul::jvmstate::StateValue)
LLVM_YAML_IS_SEQUENCE_VECTOR(azul::jvmstate::InitializedStateSlot)
LLVM_YAML_IS_SEQUENCE_VECTOR(azul::jvmstate::InitCommand)
LLVM_YAML_IS_SEQUENCE_VECTOR(azul::jvmstate::LazyObject)
LLVM_YAML_IS_SEQUENCE_VECTOR(azul::jvmstate::AbstractFrame)
LLVM_YAML_DECLARE_MAPPING_TRAITS(azul::jvmstate::InitCommand)
LLVM_YAML_DECLARE_MAPPING_TRAITS(azul::jvmstate::LazyObject)
LLVM_YAML_DECLARE_MAPPING_TRAITS(azul::jvmstate::AbstractFrame)
LLVM_YAML_DECLARE_MAPPING_TRAITS(azul::jvmstate::AbstractJVMState)

namespace llvm {
namespace yaml {

// Manual declaration of Mapping traits to set flow as true
template <> struct MappingTraits<azul::jvmstate::StateValue> {
  static void mapping(IO &io, azul::jvmstate::StateValue &SV);
  static const bool flow = true;
};

template <> struct MappingTraits<azul::jvmstate::InitializedStateSlot> {
  static void mapping(IO &io, azul::jvmstate::InitializedStateSlot &ISS);
  static const bool flow = true;
};

} // namespace yaml
} // namespace llvm

namespace azul {
namespace jvmstate {

class DeoptStateBinarySerializer {
public:
  static std::vector<llvm::Value *> serialize(llvm::LLVMContext &C,
                                              const AbstractJVMState &JVMS);

private:
  static void serializeFrame(llvm::LLVMContext &C,
                             std::vector<llvm::Value *> &Out,
                             const AbstractFrame &AF);

  static unsigned serializeLazyObject(std::vector<unsigned> &Out,
                                      const LazyObject &LO);

  static unsigned serializeInitCommand(std::vector<unsigned> &Out,
                                       const InitCommand &IC);

  static void serializeStateValue(std::vector<unsigned> &Out,
                                  const StateValue &SV);

  static void serializeInitializedStateSlot(std::vector<unsigned> &Out,
                                            const InitializedStateSlot &ISS);
};

class DeoptStateBinaryDeserializer {
public:
  // Current implementation of the deserialize method relies on the Old
  // JVMStateBundle to deserialize from the operand bundle. We then transfer
  // data from old bundle to new bundle.
  static AbstractJVMState deserialize(const llvm::OperandBundleUse &OBU);

private:
  // Helper functions that translate from old bundle structs to new bundle
  // structs
  static AbstractJVMState
  buildFromOldJVMState(const llvm::AbstractJVMState &JVMS);

  static AbstractFrame buildFromOldFrame(const llvm::AbstractFrame &AF);

  static LazyObject buildFromOldLazyObject(const llvm::LazyObject &LO);

  static InitCommand buildFromOldInitCommand(const llvm::InitCommand &I);

  static InitializedStateSlot
  buildFromOldISS(const llvm::AbstractFrameBase::InitializedStateSlot ISS);

  static StateValue
  buildFromOldStateValue(const llvm::AbstractFrameBase::StateValue SV);
};

class DeoptIRSerializer {
  // Copied from JVMStateBundle.h
  struct LazyObjectRef {
    unsigned RelativeFrameOffset;
    unsigned LazyObjectID;

    LazyObjectRef(unsigned LazyObjectID, unsigned RelativeFrameOffset)
        : RelativeFrameOffset(RelativeFrameOffset), LazyObjectID(LazyObjectID) {
      unsigned MaxAllowedLazyObjectID = 1 << 16;
      unsigned MaxAllowedRelativeFrameOffset = 1 << 16;
      assert(RelativeFrameOffset < MaxAllowedRelativeFrameOffset &&
             "cannot reference objects that are more than 65536 frames away");
      assert(LazyObjectID < MaxAllowedLazyObjectID &&
             "can have only 65536 LazyObjects in a frame");
    }

    LazyObjectRef(unsigned LazyObjectRefAsInt) {
      RelativeFrameOffset = LazyObjectRefAsInt >> 16;
      LazyObjectID = LazyObjectRefAsInt & 0xFFFF;
    }

    unsigned getAsInt() {
      return (RelativeFrameOffset << 16) | (((uint32_t)LazyObjectID & 0xFFFF));
    }
  };

  // LazyObjectMaps remembers which LazyObject was created in which frame, so
  // that
  // References to a LazyObjectID in a previous frame can be mapped to a
  // materialized LazyObject
  struct LazyObjectMaps {
    llvm::SmallVector<llvm::DenseMap<unsigned, llvm::Value *>> MapList;
    void newFrame() {
      MapList.emplace_back(llvm::DenseMap<unsigned, llvm::Value *>());
    }
    // Adds given LazyObject to the last Map in MapList.
    void newLazyObject(unsigned LazyObjectID, llvm::Value *V) {
      MapList.back()[LazyObjectID] = V;
    }
    // Retrieve LazyObject given a cross-frame reference to it
    llvm::Value *getLazyObject(LazyObjectRef LR) const {
      return MapList[MapList.size() - LR.RelativeFrameOffset - 1]
          .at(LR.LazyObjectID);
    }
  };
  llvm::Module &M;

  // Cache Functions with known signatures, instead of querying module
  llvm::Function *JVMStateFrameFunc;
  llvm::Function *KnownValueFunc;

public:
  llvm::Function *serialize(const AbstractJVMState &JVMS);

  DeoptIRSerializer(llvm::Module &module);

private:
  void serializeFrame(llvm::IRBuilder<> &Builder, unsigned ValuesOffset,
                      LazyObjectMaps &LOM, const AbstractFrame &F);

  void serializeLazyObject(llvm::IRBuilder<> &Builder, unsigned ValuesOffset,
                           LazyObjectMaps &LOM, const LazyObject &LO);

  void serializeInitCommand(llvm::IRBuilder<> &Builder, unsigned ValuesOffset,
                            const LazyObjectMaps &LOM, llvm::Value *LO,
                            const InitCommand &IC);

  void serializeInitializedStateSlot(llvm::IRBuilder<> &Builder,
                                     unsigned ValuesOffset,
                                     const LazyObjectMaps &LOM,
                                     const AbstractFrame::ValueKind Kind,
                                     const InitializedStateSlot &ISS);

  llvm::Value *serializeStateValue(llvm::IRBuilder<> &Builder,
                                   unsigned ValuesOffset,
                                   const LazyObjectMaps &LOM,
                                   const StateValue &SV);
};

} // namespace jvmstate
} // namespace azul

#endif // DEOPT_STATE_SERIALIZATION_H
