//===-- Abstractions.h ------------------------------------------*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2018 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//
//
// This header describes the abstactions (of java bytecode semantics and where
// absolutely neccessary VM internal details), used by the optimizer.  Note
// that the frontend may have many more abstractions that it uses and tags with
// appropriate attributes and metadata.  This list is only the subset which the
// optimizer itself has to know about an understand.  In general, the review
// standard for adding anything to this file should be higher than normal. 
//
//===----------------------------------------------------------------------===//

#ifndef LLVM_AZUL_ABSTRACTIONS_H
#define LLVM_AZUL_ABSTRACTIONS_H

#include "llvm/ADT/StringRef.h"
#include "llvm/IR/Instructions.h"
#include "llvm/IR/Orca/Attributes.h"
#include "llvm/IR/Orca/JVMStateBundle.h"
#include <optional>

namespace llvm {
class AtomicMemTransferInst;
class AnyMemTransferInst;
class Function;
class Module;
class TargetLibraryInfo;
}

#define NUM_ARGS_UNKNOWN -1

/// Note this is a list of abstractions of which the optimizer needs to reason
/// about specifically.  There are many other abstractions not on this list.
/// An abstraction is merely something which starts with "azul." and uses the
/// late-inline mechanism to be lowered out of existance before codegen.
/// Please try to avoid adding things to this list.  The optimizer should only
/// know specific details about an abstraction as a last resort; we strongly
/// prefer the use of function attributes to convey optimization facts.

/// Macro F is expected to have three arguments. The first is the CamelCase name
/// for abstraction. It is usually used in the generated C++ names in Orca code. 
/// The second is the name of the abstraction function. The third is the number
/// of arguments of the abstraction.
#define DO_AZUL_ABSTRACTIONS(F)                                                \
  /* Java typing */                                                            \
  F(GetKlassID, "azul.get_klass_id", 1)                                        \
  F(GetJavaLangClass, "azul.get_java_lang_class", 1)                           \
  F(IsSubtypeOf, "azul.is_subtype_of", 2)                                      \
  /* Key fields - are these still needed? */                                   \
  F(ObjectArrayElementKlassID, "azul.objarray_element_kid", 1)                 \
  F(LoadKlass, "azul.load_klass", 1)                                           \
  /* Allocation */                                                             \
  F(NewObjectInstance, "azul.new_instance", 2)                                 \
  F(NewArray, "azul.new_array", 8)                                             \
  F(NewNDimObjectArray, "azul.multianewarray", 4)                              \
  F(NewArrayRealloc, "azul.new_array_realloc", 10)                             \
  F(FinalPublicationBarrier, "azul.final_publication_barrier", 1)              \
  /* Method resolution && devirtualization */                                  \
  F(GetCompiledEntry, "azul.get_compiled_entry", 1)                            \
  F(ResolveVirtual, "azul.resolve_virtual", 2)                                 \
  F(ResolveInterface, "azul.resolve_interface", 3)                             \
  F(ResolveMethodHandle, "azul.resolve_method_handle", 1)                      \
  /* Key language specific optimizations */                                    \
  F(MonitorEnter, "azul.monitorenter", 1)                                      \
  F(MonitorExit, "azul.monitorexit", 1)                                        \
  F(MonitorEnterThreadLocal, "azul.monitorenter.thread_local", 1)              \
  F(MonitorExitThreadLocal, "azul.monitorexit.thread_local", 1)                \
  F(HasFinalizer, "azul.has_finalizer", 1)                                     \
  F(RegisterFinalizer, "azul.register_finalizer", 2)                           \
  F(CloneObject, "azul.clone_object", 1)                                       \
  F(ObjectHashcode, "azul.object_hashcode", 2)                                 \
  /* atomic memcmp intrinsic */                                                \
  F(MemCmp, "azul.memcmp", 4)                                                  \
  /* Garbage Collection */                                                     \
  F(CHeapLVB_Scalar, "cHeapLvb", 2)                                            \
  F(CHeapLVB_Vec2, "cHeapLvb.2", 2)                                            \
  F(CHeapLVB_Vec4, "cHeapLvb.4", 2)                                            \
  F(CHeapLVB_Vec8, "cHeapLvb.8", 2)                                            \
  F(CHeapLVB_Vec16, "cHeapLvb.16", 2)                                          \
  F(CHeapLVB_Vec32, "cHeapLvb.32", 2)                                          \
  F(JHeapLVB_Scalar, "jHeapLvb", 2)                                            \
  F(JHeapLVB_Vec2, "jHeapLvb.2", 2)                                            \
  F(JHeapLVB_Vec4, "jHeapLvb.4", 2)                                            \
  F(JHeapLVB_Vec8, "jHeapLvb.8", 2)                                            \
  F(JHeapLVB_Vec16, "jHeapLvb.16", 2)                                          \
  F(JHeapLVB_Vec32, "jHeapLvb.32", 2)                                          \
  F(SVB_Scalar, "svb", 4)                                                      \
  F(SVB_Vec2, "svb.2", 4)                                                      \
  F(SVB_Vec4, "svb.4", 4)                                                      \
  F(SVB_Vec8, "svb.8", 4)                                                      \
  F(SVB_Vec16, "svb.16", 4)                                                    \
  F(SVB_Vec32, "svb.32", 4)                                                    \
  F(GenerationalCheck, "azul.stores_new_to_old", 2)                            \
  F(GenerationalCheck_Vec2, "azul.stores_new_to_old.2", 2)                     \
  F(GenerationalCheck_Vec4, "azul.stores_new_to_old.4", 2)                     \
  F(GenerationalCheck_Vec8, "azul.stores_new_to_old.8", 2)                     \
  F(GenerationalCheck_Vec16, "azul.stores_new_to_old.16", 2)                   \
  F(GenerationalCheck_Vec32, "azul.stores_new_to_old.32", 2)                   \
  F(GCSafepointPoll, "gc.safepoint_poll", 0)                                   \
  F(CompareAndSwapObject, "azul.compareAndSwapObject", 4)                      \
  /* Experimental transacational compilation */                                \
  F(ReportTransactionSuccess, "azul.report_transaction_success", 0)            \
  F(ReportTransactionFailure, "azul.report_transaction_failure", 1)            \
  F(MaybeDeoptimize, "azul.maybe_deoptimize", NUM_ARGS_UNKNOWN)                \
  F(IsKnownValue, "azul.is_known_value", 1)

// This macro defines helper function which check if `Call` is a
// direct call and forwards query \p name to called function object.
// Otherwise (indirect call) returns false.
#define DEFINE_CALLBASE_PROXY(type, name)         \
  inline type name(const CallBase &Call) {        \
    if (const auto *F = Call.getCalledFunction()) \
      return name(F);                             \
    return false;                                 \
  }                                               \

// This macro defines helper function which check if Value `V` is an
// instance of `CallBase` and forwards query \p name to that `CallBase`
// specialization. Otherwise returns false.
#define DEFINE_VALUE_PROXY(type, name)            \
  inline type name(const Value &V) {              \
    if (const auto *CS = dyn_cast<CallBase>(&V))  \
      return name(*CS);                           \
    return false;                                 \
  }                                               \


namespace azul {
using namespace llvm;

// Describes the meanings of the address spaces used.
enum AddressSpaceLayout : unsigned  {
  // Anything not in java heap or another known space.  Aliases w/ the TLS
  // address space, but not the JavaHeap ones.
  CHeapAddrSpace = 0,
  
  // Contains only objects managed by the GC; references are 64 bits.  May
  // alias with CompressedJavaHeapAddrSpace, but not other two.
  JavaHeapAddrSpace = 1,

  // Contains only objects managed by the GC; references are 32 bits.  May
  // alias with JavaHeapAddrSpace, but not other two.
  CompressedJavaHeapAddrSpace = 2,

  // Contains objects in the C-Heap which do not require LVBs and SVBs. No
  // guarantees on what it may alias with.
  CHeapNonLVBSVBAddrSpace = 3,
  
  // TLS Access (mapped to the JavaThread structure, analogous to the
  // standard usage).  Access through this address space are mapped to GS
  // register.  May alias structures in the CHeapAddressSpace.
  JavaThreadTLSAddrSpace = 256
};

// Currently for our JBA and custom vectorizable functions in VM, we limit the
// maximum vectorization factor and also expect the VF to be a power-of-two.
// TODO-PERF: While the latter is true today in LLVM, this may change at some
// point and we should address the missed optimization opportunity.
// TODO-PERF: The maximum vectorization factor need not be hardcoded to MAX_VF
// and is actually dependent on the "shape" of the vectorized function. The
// assumption here is that abstractions being vectorized has the widest
// parameter type being of 8 bytes (and VF is bounded by the widest type being
// vectorized).
namespace VectorConstantsForAbstractions {
  // Maximum vectorization factor we can support.
  constexpr static unsigned MAX_VF = 32;
  // The number of unique vectorization factors we can support.
  // Currently: { 2, 4, 8, 16, 32 }
  constexpr static unsigned NUM_VF_VALUES = 5;
  // This is a common API for both orca and VM side to define the vector
  // signature
  // for a scalar function.
  std::string
  generateVectorizedNameForCustomScalarFunction(const StringRef ScalarName,
                                                unsigned VF);

  // log_2(VF) is used as an index into FuncNames. Here FuncNames is expected to
  // be an array of StringRefs with exactly NUM_VF_VALUES.
  Function *getVectorFunctionFromVF(unsigned VF, const StringRef *FuncNames,
                                    Module *M);
	// Scalar and vector versions of unpoison function.
	const std::string UnpoisonName = "azul.unpoison.oop";
  const StringRef VectorizedUnpoisonNames[NUM_VF_VALUES] = {
    "azul.unpoison.oop.2", "azul.unpoison.oop.4", "azul.unpoison.oop.8",
    "azul.unpoison.oop.16", "azul.unpoison.oop.32" };

  const std::string PoisonName = "azul.poison.oop";
}

namespace CompressedOops {
const std::string CompressName = "azul.compress";
const std::string UncompressName = "azul.uncompress";
const StringRef
    VectorizedCompressNames[VectorConstantsForAbstractions::NUM_VF_VALUES] = {
        "azul.compress.v2", "azul.compress.v4", "azul.compress.v8",
        "azul.compress.v16", "azul.compress.v32"};
const StringRef
    VectorizedUncompressNames[VectorConstantsForAbstractions::NUM_VF_VALUES] = {
        "azul.uncompress.v2", "azul.uncompress.v4", "azul.uncompress.v8",
        "azul.uncompress.v16", "azul.uncompress.v32"};
}

#define DECL_ABSTRACTION(CamelCaseSym, name, numArgs)                          \
  bool is##CamelCaseSym(const Function *F);                                    \
  Function *get##CamelCaseSym(const Module *);                                 \
  DEFINE_CALLBASE_PROXY(bool, is##CamelCaseSym)                                \
  DEFINE_VALUE_PROXY(bool, is##CamelCaseSym)                                   \
  StringRef get##CamelCaseSym##Name();

DO_AZUL_ABSTRACTIONS(DECL_ABSTRACTION)

#undef DECL_ABSTRACTION

unsigned lookupAbstractionID(StringRef Name);

/// Returns true if and only if the type specified is a
/// pointer to a GC'd object which must be included in
/// barriers and safepoints.
bool isGCPointerType(const llvm::Type *Ty);
bool isGCPointer(const llvm::Value *V);
bool isGCPointer(const llvm::Value &V);

/// Return true if \p T is a compound type containing a GC pointer.
///
/// A "compound" type is a CompositeType that is not a pointer type == {
/// struct types, array types and vector types }.
bool isCompoundTypeContainingGCPointer(const llvm::Type *T);

/// Returns true if \T is a GC PointerType or of a compound type containing a GC
/// pointer.
bool isGCPointerScalarOrCompoundType(const llvm::Type *T);

bool isAbstraction(const Function &F);

// Returns true if CS is a new allocation site for any java-level entity
// (object, array, object-array).
bool isNewAllocation(const Function *F);
DEFINE_CALLBASE_PROXY(bool, isNewAllocation)
DEFINE_VALUE_PROXY(bool, isNewAllocation)

bool isLVB(const Function *F);
DEFINE_CALLBASE_PROXY(bool, isLVB)
DEFINE_VALUE_PROXY(bool, isLVB)

bool isSVB(const Function *F);
DEFINE_CALLBASE_PROXY(bool, isSVB)
DEFINE_VALUE_PROXY(bool, isSVB)

// API for getting the various operands of LVB.
Value* getLoadValueOperandLVB(const CallBase &CS);
Value *getAddrOperandLVB(const CallBase &CS);

// Return true if \p F will eventually *have* to be inlined.
bool isInliningMandatory(const Function *F);

/// Returns true if this callsite calls a function whose result depends only
/// on memory bits which are invariant once initialized.  A readonly call which
/// uses only invariant loads is trivially an invariant function, but a
/// function which reads memory where the entire value of the load is not
/// invariant, but the function's results only depends on the bits which are
/// invariant is also an invariant function.  This will likely be a new
/// attribute upstream once we find another similiar example or two.  Also,
/// note that invariantness does not imply dereferenceability!  Control
/// dependence must still be respected.
bool isInvariantCall(const CallBase &CS);

/// Returns true if this callsite calls a function whose result is a property
/// of an object which is known to be invariant throughout its lifetime.
/// Invariant object property function always returns the same value when called
/// with the same set of arguments.
/// Invariant object property functions have one of their arguments designated
/// as the "target object argument". These functions are always safe to
/// speculate as long as the target object is not null and it points to a fully
/// created object (in the sense as produced by "new" bytecode).
///
/// Note the difference between invariant object property and invariant call.
/// Invariant call result depends on memory bits which stay invariant once
/// initialized. Invariant object property is invariant once object is created.
/// A function which returns a value of an invariant java field can be marked as
/// invariant call, but not invariant object property as the field value can
/// change from zero to the invariant value after object creation. This extra
/// limitation on invariant object property calls enables optimizer to speculate
/// these calls aggressively.
bool isInvariantObjectProperty(const CallBase &CS);

/// Returns true if this callsite calls a function whose result is invariant
/// when called with the same arguments. This property enables commoning of
/// invariant calls, but doesn't imply speculatability!
bool isInvariant(const CallBase &CS);

/// If the given call is an invariant-object-property call, returns the
/// "target object argument" for the call. This is the object the call queries
/// a property of.
/// If the given call is not an invariant-object-property call, returns None.
std::optional<const Value *> getInvariantObjectPropertyArg(const CallBase &CS);
std::optional<const Use *> getInvariantObjectPropertyArgUse(const CallBase &CS);

/// Looks for the pattern and returns %method value if the call site matches.
/// %entry_point = @azul.get_compiled_entry(%method)
/// %entry_point.typed = bitcast %entry_point
/// call %entry_point.typed
std::optional<Value *> getIndirectJavaCallSiteTarget(const CallBase &CS);
bool isInlineCacheCall(const CallBase &CS);
bool isIndirectJavaCallSite(const CallBase &CS);

bool isSpecializableCallSite(const CallBase &CS);
std::optional<uint64_t> getSpecializableFunctionID(const CallBase &CS);

bool isFoldableCallSite(const CallBase &CS);

std::optional<double> getCallSiteFrequency(const CallBase &CS);

std::optional<uint64_t> getAlreadyCompiledSize(const CallBase &Call);

void setCallSiteFrequency(CallBase &CS, double value);

void removeCallSiteFrequency(CallBase &CS);

std::optional<uint64_t> getBytecodeSize(const Function *F);

std::optional<uint64_t> getFunctionID(const Function *F);

std::optional<uint64_t> getMustThrowKID(const CallBase &Call);

std::optional<uint64_t> getAttributeAsInt(AttributeList AS,
                                          StringRef AttributeName);

std::optional<uint64_t> getAttributeAsInt(const CallBase &CS,
                                          StringRef AttributeName);

std::optional<uint64_t> getAttributeAsInt(const Function *F,
                                          StringRef AttributeName);

class GetKlassID {
  const CallBase &CS;

public:
  GetKlassID(const CallBase &CS) : CS(CS) {}
  /// valid only if isGetKlassID is true.
  GetKlassID(const Value &V) : CS(cast<CallBase>(V)) {}

  Value *getValueArg() { return CS.getArgOperand(ValueArgIdx); }

  static const int ValueArgIdx = 0;
};

class GetJavaLangClass {
  const CallBase &CS;

public:
  GetJavaLangClass(const CallBase &CS) : CS(CS) {}
  /// valid only if isGetJavaLangClass is true.
  GetJavaLangClass(const Value &V) : CS(cast<CallBase>(V)) {}

  Value *getValueArg() { return CS.getArgOperand(ValueArgIdx); }

  static const int ValueArgIdx = 0;
};

class IsSubtypeOf {
  CallBase &CS;

public:
  IsSubtypeOf(CallBase &CS) : CS(CS) {}
  IsSubtypeOf(Value &V) : CS(cast<CallBase>(V)) {}

  Value *getParentKid() { return CS.getArgOperand(ParentKidArgIdx); }
  Value *getChildKid() { return CS.getArgOperand(ChildKidArgIdx); }
  void setParentKid(Value *kid) { CS.setOperand(ParentKidArgIdx, kid); }
  void setChildKid(Value *kid) { CS.setOperand(ChildKidArgIdx, kid); }

  static const int ParentKidArgIdx = 0;
  static const int ChildKidArgIdx = 1;
};

class ObjectArrayElementKlassID {
private:
  const CallBase &CS;

public:
  ObjectArrayElementKlassID(const CallBase &CS) : CS(CS) {}
  /// valid only if isObjectArrayElementKlassID is true.
  ObjectArrayElementKlassID(const Value &V) : CS(cast<CallBase>(V)) {}

  Value *getArray() { return CS.getArgOperand(0); }
};

class NewObjectInstance {
  const CallBase &CS;

public:
  NewObjectInstance(const CallBase &CS) : CS(CS) {}
  /// valid only if isNewObjectInstance is true.
  NewObjectInstance(const Value &V) : CS(cast<CallBase>(V)) {}

  Value *getThread() { return CS.getArgOperand(0); }
  Value *getKlassID() { return CS.getArgOperand(1); }
};
class NewArrayBase {
protected:
  const CallBase &CS;

  static bool isNewArrayRelated(const Function *F);
  static DEFINE_CALLBASE_PROXY(bool, isNewArrayRelated);
  static DEFINE_VALUE_PROXY(bool, isNewArrayRelated);

public:
  NewArrayBase(const CallBase &CS) : CS(CS) {
    assert(isNewArrayRelated(CS) && "Don't expect anything else");
  }
  /// valid only if isNewArray is true.
  NewArrayBase(const Value &V) : CS(cast<CallBase>(V)) {
    assert(isNewArrayRelated(CS) && "Don't expect anything else");
  }

  Value *getThread() { return CS.getArgOperand(ThreadArgIdx); }
  Value *getArrayKlassID() { return CS.getArgOperand(ArrayKlassIDArgIdx); }
  Value *getElementKid() { return CS.getArgOperand(ElementKidArgIdx); }
  Value *getPrimitiveType() { return CS.getArgOperand(PrimitiveTypeArgIdx); }
  Value *getLength() { return CS.getArgOperand(LengthArgIdx); }
  Value *getHeaderSize() { return CS.getArgOperand(HeaderSizeArgIdx); }
  Value *getElementShift() { return CS.getArgOperand(ElementShiftArgIdx); }

  static const int ThreadArgIdx = 0;
  static const int ArrayKlassIDArgIdx = 1;
  static const int ElementKidArgIdx = 2;
  static const int PrimitiveTypeArgIdx = 3;
  static const int LengthArgIdx = 4;
  static const int HeaderSizeArgIdx = 5;
  static const int ElementShiftArgIdx = 6;
};

class NewArray : public NewArrayBase {
public:
  NewArray(const CallBase &CS) : NewArrayBase(CS) {
    assert(isNewArray(CS) && "only valid for NewArray");
  }
  NewArray(const Value &V) : NewArrayBase(V) {
    assert(isNewArray(CS) && "only valid for NewArray");
  }

  // Returns offset in bytes.
  Value *getZeroInitializeFrom() {
    return CS.arg_size() <= ZeroInitializeFromArgIdx
               ? nullptr
               : CS.getArgOperand(ZeroInitializeFromArgIdx);
  }

  // Pessimistically returns false.
  bool isFullyInitialized() {
    auto *ZeroFrom = getZeroInitializeFrom();
    if (!ZeroFrom)
      return true; // Support tests with fewer arguments.

    if (auto *ZeroFromCI = dyn_cast<ConstantInt>(ZeroFrom))
      if (ZeroFromCI->getZExtValue() == 0)
        return true;

    return false;
  }

  static const int ZeroInitializeFromArgIdx = 7; // Offset in bytes.
};

class NewArrayRealloc : public NewArrayBase {
public:
  NewArrayRealloc(const CallBase &CS) : NewArrayBase(CS) {
    assert(isNewArrayRealloc(CS) && "only valid for NewArrayRealloc");
  }
  NewArrayRealloc(const Value &V) : NewArrayBase(V) {
    assert(isNewArrayRealloc(CS) && "only valid for NewArrayRealloc");
  }

  Value *getSrcArray() { return CS.getArgOperand(SrcArrayArgIdx); }
  Value *getOffset() { return CS.getArgOperand(OffsetArgIdx); }
  Value *getMemcpyLength() { return CS.getArgOperand(MemcpyLengthArgIdx); }

  static const int SrcArrayArgIdx = 7;
  static const int OffsetArgIdx = 8;
  static const int MemcpyLengthArgIdx = 9;
};

class NewNDimObjectArray {
  const CallBase &CS;

public:
  NewNDimObjectArray(const CallBase &CS) : CS(CS) {}
  /// valid only if isNewNDimObjectArray is true.
  NewNDimObjectArray(const Value &V) : CS(cast<CallBase>(V)) {}

  Value *getThread() { return CS.getArgOperand(0); }
  Value *getElementKlassID() { return CS.getArgOperand(1); }
  Value *getNumDimensions() { return CS.getArgOperand(2); }
  Value *getDimensionsArray() { return CS.getArgOperand(3); }
};
class GetCompiledEntry {
  const CallBase &CS;

public:
  GetCompiledEntry(const CallBase &CS) : CS(CS) {}
  /// valid only if isGetCompiledEntry is true.
  GetCompiledEntry(const Value &V) : CS(cast<CallBase>(V)) {}

  Value *getMethod() { return CS.getArgOperand(0); }
};
class MonitorBase {
protected:
  const CallBase &CS;

public:
  MonitorBase(const CallBase &CS) : CS(CS) {
    assert((isMonitorEnter(CS) || isMonitorEnterThreadLocal(CS) ||
            isMonitorExit(CS) || isMonitorExitThreadLocal(CS)) &&
           "only valid for "
           "MonitorEnter/MonitorEnterThreadLocal/MonitorExit/"
           "MonitorEnterThreadLocal");
  }
  MonitorBase(const Value &V) : MonitorBase(cast<CallBase>(V)) {}

  Value *getObject() { return CS.getArgOperand(getObjectArgIdx()); }
  const Use &getObjectUse() { return CS.getArgOperandUse(getObjectArgIdx()); }
  const CallBase *getCall() { return &CS; }

  unsigned getObjectArgIdx() const { return 0; }
};
class MonitorEnter : public MonitorBase {
public:
  MonitorEnter(const CallBase &CS) : MonitorBase(CS) {
    assert((isMonitorEnter(CS) || isMonitorEnterThreadLocal(CS)) &&
           "only valid for MonitorEnter or MonitorEnterThreadLocal");
  }
  MonitorEnter(const Value &V) : MonitorBase(cast<CallBase>(V)) {}
};
class MonitorExit : public MonitorBase {
public:
  MonitorExit(const CallBase &CS) : MonitorBase(CS) {
    assert((isMonitorExit(CS) || isMonitorExitThreadLocal(CS)) &&
           "only valid for MonitorExit or MonitorExitThreadLocal");
  }
  MonitorExit(const Value &V) : MonitorBase(cast<CallBase>(V)) {}
};
class HasFinalizer {
  const CallBase &CS;

public:
  HasFinalizer(const CallBase &CS) : CS(CS) {}
  /// valid only if isHasFinalizer is true.
  HasFinalizer(const Value &V) : CS(cast<CallBase>(V)) {}

  Value *getKlassID() { return CS.getArgOperand(KlassIDArgIdx); }

  static const int KlassIDArgIdx = 0;
};
class CloneObject {
  const CallBase &CS;

public:
  CloneObject(const CallBase &CS) : CS(CS) {}
  /// valid only if isCloneObject is true.
  CloneObject(const Value &V) : CS(cast<CallBase>(V)) {}

  Value *getSource() { return CS.getArgOperand(0); }
};
class LoadKlass {
  const CallBase &CS;

public:
  LoadKlass(const CallBase &CS) : CS(CS) {}
  /// valid only if isLoadKlass is true.
  LoadKlass(const Value &V) : CS(cast<CallBase>(V)) {}

  Value *getKlassID() { return CS.getArgOperand(0); }
};
class ObjectHashcode {
  const CallBase &CS;

public:
  ObjectHashcode(const CallBase &CS) : CS(CS) { assert(isObjectHashcode(CS)); }
  /// valid only if isObjectHashcode is true.
  ObjectHashcode(const Value &V) : ObjectHashcode(cast<CallBase>(V)) { }

  Value *getObject() { return CS.getArgOperand(ObjectArgIdx); }

  static const int ObjectArgIdx = 0;
};
class GenerationalCheck {
  const CallBase &CS;

public:
  GenerationalCheck(const CallBase &CS) : CS(CS) {
    assert(isGenerationalCheck(CS) ||
           isGenerationalCheck_Vec2(CS) ||
           isGenerationalCheck_Vec4(CS) ||
           isGenerationalCheck_Vec8(CS) ||
           isGenerationalCheck_Vec16(CS) ||
           isGenerationalCheck_Vec32(CS));
  }
  GenerationalCheck(const Value &V): GenerationalCheck(cast<CallBase>(V)) { }

  Value *getDestAddress() const { return CS.getArgOperand(0); }
  Value *getSourceValue() const { return CS.getArgOperand(1); }
};
class CallSiteWithVMState {
protected:
  const CallBase &CS;

  const AbstractJVMState &getJVMState() {
    if (!AS) {
      // Lazy initialize AS.
      auto DeoptOB = CS.getOperandBundle(LLVMContext::OB_deopt);
      auto CSAS = AbstractJVMState::Build(*DeoptOB);
      AS = std::optional<AbstractJVMState>(std::move(CSAS));
    }
    return *AS;
  }

private:
  // We use std::optional here for lazy initialization.
  // None means not yet initialized.
  std::optional<AbstractJVMState> AS;
  bool isASEmpty;

public:
  bool isVMStateEmpty() { return isASEmpty; }
  static bool hasVMState(const CallBase &CS) {
    return CS.getOperandBundle(LLVMContext::OB_deopt).has_value();
  }

  CallSiteWithVMState(const CallBase &CS) : CS(CS) {
    auto DeoptOB = CS.getOperandBundle(LLVMContext::OB_deopt);
    assert(DeoptOB && "Callsite should have vm state");
    isASEmpty = DeoptOB->Inputs.empty();
  }

  /// valid only if isCallSiteWithVMState is true.
  CallSiteWithVMState(const Value &V) : CallSiteWithVMState(cast<CallBase>(V)) { }

  uint64_t getCallerID() {
    assert(!isVMStateEmpty() && "must have JVM state");
    return getJVMState().getYoungestFrame().getCallerID();
  }
  uint64_t getCallerBCI() {
    assert(!isVMStateEmpty() && "must have JVM state");
    return getJVMState().getYoungestFrame().getBCI();
  }
};

const std::optional<AbstractJVMState>
getNonEmptyAbstractJVMState(const CallBase & CS);

class ResolveBase {
protected:
  const CallBase &CS;

public:
  ResolveBase(const CallBase &CS) : CS(CS) {}
  /// valid only if isResolveBase is true.
  ResolveBase(const Value &V) : ResolveBase(cast<CallBase>(V)) {}

  const Instruction *getInstruction() const { return &CS; }
  Value *getReceiver() const { return CS.getArgOperand(ReceiverArgIdx); }

  static const int ReceiverArgIdx = 0;
};
class ResolveVirtual : public ResolveBase {
public:
  ResolveVirtual(const CallBase &CS) : ResolveBase(CS) {
    assert(isResolveVirtual(CS));
  }
  /// valid only if isResolveVirtual is true.
  ResolveVirtual(const Value &V) : ResolveVirtual(cast<CallBase>(V)) { }

  Value *getVTableIndex() const { return CS.getArgOperand(1); }
};
class ResolveInterface : public ResolveBase {
public:
  ResolveInterface(const CallBase &CS) : ResolveBase(CS) {
    assert(isResolveInterface(CS));
  }
  /// valid only if isResolveInterface is true.
  ResolveInterface(const Value &V) : ResolveInterface(cast<CallBase>(V)) { }

  Value *getITableIndex() const { return CS.getArgOperand(2); }
};
class IndirectJavaCallSite : public CallSiteWithVMState {
private:
  Value *Target;
  bool IsVirtualOrInterface;

public:
  IndirectJavaCallSite(const CallBase &CS);
  /// valid only if IndirectJavaCallSite is true.
  IndirectJavaCallSite(const Value &V) : IndirectJavaCallSite(cast<CallBase>(V)) { }

  const CallBase *getInstruction() const { return &CS; }
  Value *getTarget() { return Target; };
  Value *getReceiver() { return CS.getArgOperand(0); }
  bool isVirtualOrInterface() { return IsVirtualOrInterface; }
};

class CompareAndSwapObject {
  const CallBase &CS;

public:
  CompareAndSwapObject(const CallBase &CS) : CS(CS) {}
  /// valid only if isCompareAndSwapObject is true.
  CompareAndSwapObject(const Value &V) : CS(cast<CallBase>(V)) {}

  const CallBase &getCall() const { return CS; }

  Value *getObject() const { return CS.getArgOperand(0); }
  Value *getOffset() const { return CS.getArgOperand(1); }
  Value *getExpectedValue() const { return CS.getArgOperand(2); }
  Value *getNewValue() const { return CS.getArgOperand(3); }

  const Use &getObjectUse() const { return CS.getArgOperandUse(0); }
  const Use &getOffsetUse() const { return CS.getArgOperandUse(1); }
  const Use &getExpectedValueUse() const { return CS.getArgOperandUse(2); }
  const Use &getNewValueUse() const { return CS.getArgOperandUse(3); }
};

class FinalPublicationBarrier {
  const CallBase &CS;

public:
  FinalPublicationBarrier(const CallBase &CS) : CS(CS) {}
  /// valid only if isFinalPublicationBarrier is true.
  FinalPublicationBarrier(const Value &V) : CS(cast<CallBase>(V)) {}

  Value *getValueArg() { return CS.getArgOperand(ValueArgIdx); }

  static const int ValueArgIdx = 0;
};

// An atomic memory comparison function.
// Returns 1 if length bytes from LHS and RHS are equal,
// otherwise returns 0.
class MemCmp {
  const CallBase &CS;
  static const int LHSIdx = 0;
  static const int RHSIdx = 1;
  static const int LengthIdx = 2;
  static const int ElementSizeIdx = 3;

public:
  MemCmp(const CallBase &CS) : CS(CS) {
    assert(isa<Constant>(getElementSize()) && "Element size must be constant");
  }
  /// valid only if isMemCmp is true.
  MemCmp(const Value &V) : MemCmp(cast<CallBase>(V)) {}

  Value *getLHS() { return CS.getArgOperand(LHSIdx); }
  Value *getRHS() { return CS.getArgOperand(RHSIdx); }
  // Length of comparison in bytes
  Value *getLength() { return CS.getArgOperand(LengthIdx); }
  // Element size in bytes. Must be constant.
  Value *getElementSize() { return CS.getArgOperand(ElementSizeIdx); }
};

// Queries about element types of AtomicMemory instructions.
// Must be guarded by azul::enableDownstreamChanges() at call sites.
Type *getSourceElementType(const AtomicMemTransferInst &AMI);
Type *getDestElementType(const AtomicMemTransferInst &AMI);
Type *getDestElementType(const AnyMemTransferInst &AMI);

inline bool isPtrArrayCopy(const AtomicMemTransferInst &AMI) {
  return getDestElementType(AMI)->isPointerTy();
}
inline bool isPtrArrayCopy(const AnyMemTransferInst &AMI) {
  return getDestElementType(AMI)->isPointerTy();
}

} // namespace azul

#endif
