//===- llvm/Analysis/Orca/EscapeAnalysis.h - EscapeAnalysis -----*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2018 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//
// \file
// This file provides the interface for Orca-specific escape analysis. 

// Escape analysis is similar to upstream CaptureTracking analysis with some 
// subtle differences. Informally:
// - "capture" - can anyone inspect the bits of this pointer?
// - "escape" - can anyone inspect the contents of this object?
// For example, reference comparison doesn't make an object escape (no one can
// inspect the content of the object via reference comparison), but might imply
// capturing. In general capturing is a more conservative property.
// 
// Having a downstream analysis helps us to exploit the difference between 
// capturing and escape more aggressively and also makes it easier to adjust
// the interface of the tracker to our downstream needs.
//
//===----------------------------------------------------------------------===//

#ifndef ESCAPEANALYSIS_H
#define ESCAPEANALYSIS_H

#include "llvm/IR/PassManager.h"
#include "llvm/Support/Debug.h"

namespace llvm {
class DataLayout;
class Value;
class Instruction;
class PtrToIntInst;
class Use;
template <typename T> class SmallVectorImpl;

/// Escape analysis printer pass. Iterates over the allocations in the F and
/// prints their uses with the UseEscapeKind tag. Used for testing and debugging
/// purposes.
struct EscapeAnalysisPrinterPass : PassInfoMixin<EscapeAnalysisPrinterPass> {
  raw_ostream &OS = dbgs();
  EscapeAnalysisPrinterPass() = default;
  PreservedAnalyses run(Function &F, FunctionAnalysisManager &FAM);
};
}

namespace azul {
  
/// Classification of the pointer uses from the escape analysis point of view.
enum UseEscapeKind {
  /// Object may escape through this use
  Escape, 
  /// Object can't escape through this use
  NoEscape, 
  /// Object can't escape through this use, but it produces a new value which
  /// needs to be tracked. E.g., getelementptr, bitcast, phi instructions.
  Alias
};

UseEscapeKind getUseEscapeKind(const llvm::Use *U);

struct EscapeTracker {
  virtual ~EscapeTracker() = default;
  
  /// Notifies the tracker about a transitive use of the pointer in question.
  /// Return true to stop the traversal or false to continue looking for more 
  /// transitive uses.
  virtual bool visitUse(const llvm::Use *U, UseEscapeKind Kind) = 0;
};

/// The tracker used by EscapeAnalysisPrinterPass to print uses and their
/// UseEscapeKinds
struct PrintEscapeTracker : public EscapeTracker {
  llvm::raw_ostream &OS;
  bool Escapes = false;

  /// If not true the tracker will print only "interesting" uses - escapes
  /// and aliases that may prevent allocation elimination.
  bool PrintAllUses = true;

  explicit PrintEscapeTracker(llvm::raw_ostream &OS, bool PrintAllUses = true) :
    OS(OS), PrintAllUses(PrintAllUses) {}

  bool visitUse(const llvm::Use *U, UseEscapeKind Kind) override;
};

/// Classifies the uses of the object pointer in the enclosing function (which 
/// is required to exist) from the escape analysis perspective. The tracker is 
/// notified about all the encountered uses (including transitive uses through 
/// recognized aliases) along with their type. The order of the traversal is not 
/// specified.
void ObjectMayEscape(const llvm::Value *V, EscapeTracker &T,
                     const llvm::DataLayout &DL);

/// Return true if the given object pointer may escape from the enclosing 
/// function (which is required to exist).
bool ObjectMayEscape(const llvm::Value *V, const llvm::DataLayout &DL);

/// Returns true if the given object pointer may escape from the enclosing
/// function (which is required to exist) before the `BeforeHere` Instruction.
/// If BeforeHere is not specified, we do a context insensitive analysis.
bool ObjectMayEscapeBefore(const llvm::Value *V, llvm::Instruction *BeforeHere,
                           const llvm::DataLayout &DL);
}

#endif /* ESCAPEANALYSIS_H */

