//===- llvm/Analysis/Orca/AvailableDefTracker.h - DefTracker ----*- C++ -*-===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
// Copyright 2013-2018 Azul Systems, Inc.  All Rights Reserved.
// http://www.azul.com
// Azul Systems is a contributor to the LLVM Team.
// Distributed under the same license terms detailed in LICENSE.TXT above.
//===----------------------------------------------------------------------===//

#ifndef AVAILABLEDEFTRACKER_H
#define AVAILABLEDEFTRACKER_H

#include "llvm/ADT/DenseMap.h"
#include "llvm/ADT/SmallPtrSet.h"
#include "llvm/IR/AssemblyAnnotationWriter.h"
#include "llvm/IR/PassManager.h"
#include "llvm/Support/Debug.h"

namespace llvm {
class AAResults;
struct AvailableDefs;
class BatchAAResults;
class CallBase;
class BasicBlock;
class Function;
class Value;
class InlineFunctionInfo;
class Instruction;
class MemorySSA;
class formatted_raw_ostream;
class raw_ostream;

struct AvailableDefPrinterPass : PassInfoMixin<AvailableDefPrinterPass> {
  raw_ostream &OS = dbgs();
  AvailableDefPrinterPass() = default;
  PreservedAnalyses run(Function &F, FunctionAnalysisManager &FAM);
};
}

namespace azul {
struct ADTState;
struct CallerDefs;
class AvailableDefTracker {
  friend struct llvm::AvailableDefPrinterPass;
  friend struct CallerDefs; // only for the below type aliases
private:
  /// A caller store that has not been clobbered yet.
  /// Extra bit in the pair indicates that the def is invariant.
  using CallerDef = llvm::PointerIntPair<llvm::Instruction *, 1>;
  using CallerDefSet = llvm::SmallPtrSet<CallerDef, 8>;

  llvm::Function *Caller;
  llvm::DenseMap<llvm::CallBase *, CallerDefSet> CallDefs;

  void invalidate();
  void analyzeCallerBB(llvm::BasicBlock *BB, llvm::BatchAAResults &CallerAA,
                       ADTState &State);
  void collectCallerDefsImpl(llvm::AAResults &CallerAA, ADTState &State);

public:
  AvailableDefTracker(llvm::Function *Caller) : Caller(Caller){};
  void collectCallerDefs(llvm::AAResults &CallerAA);
  void collectAvailableDefs(const llvm::CallBase *Call,
                            llvm::MemorySSA &CalleeMSSA,
                            llvm::AvailableDefs &Defs) const;

  /// Notifies tracker about inlining into the caller.
  void callSiteInlined(llvm::CallBase *Call, llvm::InlineFunctionInfo &IFI);
};

/// The caller state AvailableDefTracker computes and maintains during
/// analysis. Essentially contains the set of store instructions which have
/// not yet been clobbered thus can be used in callees as available defs.
struct CallerDefs {
  /// Set of stores which haven't been clobbered yet.
  llvm::SmallPtrSet<llvm::Instruction *, 4> Defs;
  /// An invariant subset of Defs set.
  llvm::SmallPtrSet<llvm::Instruction *, 4> InvariantDefs;

  /// Helper struct that builds set on top of llvm's TinyPtrVector and
  /// implements only API used by this pass.
  template<typename T>
  class TinyPtrVectorSet {
    llvm::TinyPtrVector<T> Data;

  public:
    void insert(const T &Elem) {
      // We can keep the Data sorted making there operations O(logn) but since
      // we don't expect for this set to ever have more than ~3 elements, linear
      // search should be fine (preferential even), while making things simpler.
      if (llvm::find(Data, Elem) == Data.end())
        Data.push_back(Elem);
    }

    void erase(const T &Elem) {
      llvm::erase(Data, Elem);
    }

    auto find(const T &Elem) {
      return llvm::find(Data, Elem);
    }

    auto begin() {
      return Data.begin();
    }
    auto end() {
      return Data.end();
    }
    auto size() const {
      return Data.size();
    }
  };

  /// A map from an address to the defs corresponding to this address
  /// There is only one def per the address unless dead code is involved.
  llvm::DenseMap<llvm::Value *, TinyPtrVectorSet<llvm::Instruction *>>
      TrackedPointers;

  void add(llvm::Instruction *I);
  void markInvariant(llvm::Instruction *I);
  void erase(llvm::Instruction *I);
  void clearNonInvariant();
  void meet(CallerDefs &Other);

  bool empty() const { return Defs.empty(); }
  AvailableDefTracker::CallerDefSet toCallerDefSet() const;

  void verify();
  void print(llvm::raw_ostream &OS);
};

/// Unoptimized (memory-wise) state used during the initial analysis phase
struct ADTState {
  llvm::DenseMap<llvm::CallBase *, CallerDefs> CallDefs;
  llvm::DenseMap<llvm::BasicBlock *, CallerDefs> BBEndDefs;
};

class AvailableDefAnnotatedWriter : public llvm::AssemblyAnnotationWriter {
  AvailableDefTracker &T;
  llvm::FunctionAnalysisManager &FAM;
  ADTState State;

public:
  AvailableDefAnnotatedWriter(AvailableDefTracker &T,
                              llvm::FunctionAnalysisManager &FAM)
      : T(T), FAM(FAM), State() {}
  AvailableDefAnnotatedWriter(AvailableDefTracker &T,
                              llvm::FunctionAnalysisManager &FAM,
                              const ADTState &State)
      : T(T), FAM(FAM), State(State) {}

  virtual void emitInstructionAnnot(const llvm::Instruction *I,
                                    llvm::formatted_raw_ostream &OS) override;
  virtual void emitBasicBlockEndAnnot(const llvm::BasicBlock *,
                                      llvm::formatted_raw_ostream &) override;

};
}

#endif /* AVAILABLEDEFTRACKER_H */
