# frozen_string_literal: true

# Class +Socket+ provides access to the underlying operating system
# socket implementations.  It can be used to provide more operating system
# specific functionality than the protocol-specific socket classes.
#
# The constants defined under Socket::Constants are also defined under
# Socket.  For example, Socket::AF_INET is usable as well as
# Socket::Constants::AF_INET.  See Socket::Constants for the list of
# constants.
#
# === What's a socket?
#
# Sockets are endpoints of a bidirectional communication channel.
# Sockets can communicate within a process, between processes on the same
# machine or between different machines.  There are many types of socket:
# TCPSocket, UDPSocket or UNIXSocket for example.
#
# Sockets have their own vocabulary:
#
# *domain:*
# The family of protocols:
# *    Socket::PF_INET
# *    Socket::PF_INET6
# *    Socket::PF_UNIX
# *    etc.
#
# *type:*
# The type of communications between the two endpoints, typically
# *    Socket::SOCK_STREAM
# *    Socket::SOCK_DGRAM.
#
# *protocol:*
# Typically _zero_.
# This may be used to identify a variant of a protocol.
#
# *hostname:*
# The identifier of a network interface:
# *    a string (hostname, IPv4 or IPv6 address or +broadcast+
#      which specifies a broadcast address)
# *    a zero-length string which specifies INADDR_ANY
# *    an integer (interpreted as binary address in host byte order).
#
# === Quick start
#
# Many of the classes, such as TCPSocket, UDPSocket or UNIXSocket,
# ease the use of sockets comparatively to the equivalent C programming interface.
#
# Let's create an internet socket using the IPv4 protocol in a C-like manner:
#
#   require 'socket'
#
#   s = Socket.new Socket::AF_INET, Socket::SOCK_STREAM
#   s.connect Socket.pack_sockaddr_in(80, 'example.com')
#
# You could also use the TCPSocket class:
#
#   s = TCPSocket.new 'example.com', 80
#
# A simple server might look like this:
#
#   require 'socket'
#
#   server = TCPServer.new 2000 # Server bound to port 2000
#
#   loop do
#     client = server.accept    # Wait for a client to connect
#     client.puts "Hello !"
#     client.puts "Time is #{Time.now}"
#     client.close
#   end
#
# A simple client may look like this:
#
#   require 'socket'
#
#   s = TCPSocket.new 'localhost', 2000
#
#   while line = s.gets # Read lines from socket
#     puts line         # and print them
#   end
#
#   s.close             # close socket when done
#
# === Exception Handling
#
# Ruby's Socket implementation raises exceptions based on the error
# generated by the system dependent implementation.  This is why the
# methods are documented in a way that isolate Unix-based system
# exceptions from Windows based exceptions. If more information on a
# particular exception is needed, please refer to the Unix manual pages or
# the Windows WinSock reference.
#
# === Convenience methods
#
# Although the general way to create socket is Socket.new,
# there are several methods of socket creation for most cases.
#
# TCP client socket::
#   Socket.tcp, TCPSocket.open
# TCP server socket::
#   Socket.tcp_server_loop, TCPServer.open
# UNIX client socket::
#   Socket.unix, UNIXSocket.open
# UNIX server socket::
#   Socket.unix_server_loop, UNIXServer.open
#
# === Documentation by
#
# * Zach Dennis
# * Sam Roberts
# * <em>Programming Ruby</em> from The Pragmatic Bookshelf.
#
# Much material in this documentation is taken with permission from
# <em>Programming Ruby</em> from The Pragmatic Bookshelf.
class Socket < BasicSocket
  # AppleTalk protocol
  AF_APPLETALK = _
  # Asynchronous Transfer Mode
  AF_ATM = _
  # AX.25 protocol
  AF_AX25 = _
  # CCITT (now ITU-T) protocols
  AF_CCITT = _
  # MIT CHAOS protocols
  AF_CHAOS = _
  # Computer Network Technology
  AF_CNT = _
  # Connection-oriented IP
  AF_COIP = _
  # Datakit protocol
  AF_DATAKIT = _
  # DECnet protocol
  AF_DEC = _
  # DEC Direct Data Link Interface protocol
  AF_DLI = _
  # CCITT (ITU-T) E.164 recommendation
  AF_E164 = _
  # European Computer Manufacturers protocols
  AF_ECMA = _
  # NSC Hyperchannel protocol
  AF_HYLINK = _
  # ARPANET IMP protocol
  AF_IMPLINK = _
  # IPv4 protocol
  AF_INET = _
  # IPv6 protocol
  AF_INET6 = _
  # IPX protocol
  AF_IPX = _
  # Integrated Services Digital Network
  AF_ISDN = _
  # ISO Open Systems Interconnection protocols
  AF_ISO = _
  # Local Area Transport protocol
  AF_LAT = _
  # Link layer interface
  AF_LINK = _
  # Host-internal protocols
  AF_LOCAL = _
  # Maximum address family for this platform
  AF_MAX = _
  # Native ATM access
  AF_NATM = _
  # Network driver raw access
  AF_NDRV = _
  # NetBIOS
  AF_NETBIOS = _
  # Netgraph sockets
  AF_NETGRAPH = _
  # XEROX NS protocols
  AF_NS = _
  # ISO Open Systems Interconnection protocols
  AF_OSI = _
  # Direct link-layer access
  AF_PACKET = _
  # Point-to-Point Protocol
  AF_PPP = _
  # PARC Universal Packet protocol
  AF_PUP = _
  # Internal routing protocol
  AF_ROUTE = _
  # Simple Internet Protocol
  AF_SIP = _
  # IBM SNA protocol
  AF_SNA = _
  AF_SYSTEM = _
  # UNIX sockets
  AF_UNIX = _
  # Unspecified protocol, any supported address family
  AF_UNSPEC = _
  # Accept only if any address is assigned
  AI_ADDRCONFIG = _
  # Allow all addresses
  AI_ALL = _
  # Fill in the canonical name
  AI_CANONNAME = _
  # Default flags for getaddrinfo
  AI_DEFAULT = _
  # Valid flag mask for getaddrinfo (not for application use)
  AI_MASK = _
  # Prevent host name resolution
  AI_NUMERICHOST = _
  # Prevent service name resolution
  AI_NUMERICSERV = _
  # Get address to use with bind()
  AI_PASSIVE = _
  # Accept IPv4-mapped IPv6 addresses
  AI_V4MAPPED = _
  # Accept IPv4 mapped addresses if the kernel supports it
  AI_V4MAPPED_CFG = _
  # Address family for hostname not supported
  EAI_ADDRFAMILY = _
  # Temporary failure in name resolution
  EAI_AGAIN = _
  # Invalid flags
  EAI_BADFLAGS = _
  # Invalid value for hints
  EAI_BADHINTS = _
  # Non-recoverable failure in name resolution
  EAI_FAIL = _
  # Address family not supported
  EAI_FAMILY = _
  # Maximum error code from getaddrinfo
  EAI_MAX = _
  # Memory allocation failure
  EAI_MEMORY = _
  # No address associated with hostname
  EAI_NODATA = _
  # Hostname nor servname, or not known
  EAI_NONAME = _
  # Argument buffer overflow
  EAI_OVERFLOW = _
  # Resolved protocol is unknown
  EAI_PROTOCOL = _
  # Servname not supported for socket type
  EAI_SERVICE = _
  # Socket type not supported
  EAI_SOCKTYPE = _
  # System error returned in errno
  EAI_SYSTEM = _
  # 802.1Q VLAN device
  IFF_802_1Q_VLAN = _
  # receive all multicast packets
  IFF_ALLMULTI = _
  # use alternate physical connection
  IFF_ALTPHYS = _
  # auto media select active
  IFF_AUTOMEDIA = _
  # bonding master or slave
  IFF_BONDING = _
  # device used as bridge port
  IFF_BRIDGE_PORT = _
  # broadcast address valid
  IFF_BROADCAST = _
  # flags not changeable
  IFF_CANTCHANGE = _
  # unconfigurable using ioctl(2)
  IFF_CANTCONFIG = _
  # turn on debugging
  IFF_DEBUG = _
  # disable netpoll at run-time
  IFF_DISABLE_NETPOLL = _
  # disallow bridging this ether dev
  IFF_DONT_BRIDGE = _
  # driver signals dormant
  IFF_DORMANT = _
  # tx hardware queue is full
  IFF_DRV_OACTIVE = _
  # resources allocated
  IFF_DRV_RUNNING = _
  # interface is winding down
  IFF_DYING = _
  # dialup device with changing addresses
  IFF_DYNAMIC = _
  # ethernet bridging device
  IFF_EBRIDGE = _
  # echo sent packets
  IFF_ECHO = _
  # ISATAP interface (RFC4214)
  IFF_ISATAP = _
  # per link layer defined bit 0
  IFF_LINK0 = _
  # per link layer defined bit 1
  IFF_LINK1 = _
  # per link layer defined bit 2
  IFF_LINK2 = _
  # hardware address change when it's running
  IFF_LIVE_ADDR_CHANGE = _
  # loopback net
  IFF_LOOPBACK = _
  # driver signals L1 up
  IFF_LOWER_UP = _
  # device used as macvlan port
  IFF_MACVLAN_PORT = _
  # master of a load balancer
  IFF_MASTER = _
  # bonding master, 802.3ad.
  IFF_MASTER_8023AD = _
  # bonding master, balance-alb.
  IFF_MASTER_ALB = _
  # bonding master, ARP mon in use
  IFF_MASTER_ARPMON = _
  # user-requested monitor mode
  IFF_MONITOR = _
  # supports multicast
  IFF_MULTICAST = _
  # no address resolution protocol
  IFF_NOARP = _
  # avoid use of trailers
  IFF_NOTRAILERS = _
  # transmission in progress
  IFF_OACTIVE = _
  # device used as Open vSwitch datapath port
  IFF_OVS_DATAPATH = _
  # point-to-point link
  IFF_POINTOPOINT = _
  # can set media type
  IFF_PORTSEL = _
  # user-requested promisc mode
  IFF_PPROMISC = _
  # receive all packets
  IFF_PROMISC = _
  # interface is being renamed
  IFF_RENAMING = _
  # routing entry installed
  IFF_ROUTE = _
  # resources allocated
  IFF_RUNNING = _
  # can't hear own transmissions
  IFF_SIMPLEX = _
  # slave of a load balancer
  IFF_SLAVE = _
  # bonding slave not the curr. active
  IFF_SLAVE_INACTIVE = _
  # need ARPs for validation
  IFF_SLAVE_NEEDARP = _
  # interface manages own routes
  IFF_SMART = _
  # static ARP
  IFF_STATICARP = _
  # sending custom FCS
  IFF_SUPP_NOFCS = _
  # used as team port
  IFF_TEAM_PORT = _
  # sharing skbs on transmit
  IFF_TX_SKB_SHARING = _
  # unicast filtering
  IFF_UNICAST_FLT = _
  # interface is up
  IFF_UP = _
  # volatile flags
  IFF_VOLATILE = _
  # WAN HDLC device
  IFF_WAN_HDLC = _
  # dev_hard_start_xmit() is allowed to release skb->dst
  IFF_XMIT_DST_RELEASE = _
  # Maximum interface name size
  IFNAMSIZ = _
  # Maximum interface name size
  IF_NAMESIZE = _
  # Multicast group for all systems on this subset
  INADDR_ALLHOSTS_GROUP = _
  # A socket bound to INADDR_ANY receives packets from all interfaces and sends from the default IP address
  INADDR_ANY = _
  # The network broadcast address
  INADDR_BROADCAST = _
  # The loopback address
  INADDR_LOOPBACK = _
  # The last local network multicast group
  INADDR_MAX_LOCAL_GROUP = _
  # A bitmask for matching no valid IP address
  INADDR_NONE = _
  # The reserved multicast group
  INADDR_UNSPEC_GROUP = _
  # Maximum length of an IPv6 address string
  INET6_ADDRSTRLEN = _
  # Maximum length of an IPv4 address string
  INET_ADDRSTRLEN = _
  # Default minimum address for bind or connect
  IPPORT_RESERVED = _
  # Default maximum address for bind or connect
  IPPORT_USERRESERVED = _
  # IP6 auth header
  IPPROTO_AH = _
  IPPROTO_BIP = _
  # IP6 destination option
  IPPROTO_DSTOPTS = _
  # Exterior Gateway Protocol
  IPPROTO_EGP = _
  # ISO cnlp
  IPPROTO_EON = _
  # IP6 Encapsulated Security Payload
  IPPROTO_ESP = _
  # IP6 fragmentation header
  IPPROTO_FRAGMENT = _
  # Gateway to Gateway Protocol
  IPPROTO_GGP = _
  # "hello" routing protocol
  IPPROTO_HELLO = _
  # IP6 hop-by-hop options
  IPPROTO_HOPOPTS = _
  # Control message protocol
  IPPROTO_ICMP = _
  # ICMP6
  IPPROTO_ICMPV6 = _
  # XNS IDP
  IPPROTO_IDP = _
  # Group Management Protocol
  IPPROTO_IGMP = _
  # Dummy protocol for IP
  IPPROTO_IP = _
  # IP6 header
  IPPROTO_IPV6 = _
  # Maximum IPPROTO constant
  IPPROTO_MAX = _
  # Sun net disk protocol
  IPPROTO_ND = _
  # IP6 no next header
  IPPROTO_NONE = _
  # PARC Universal Packet protocol
  IPPROTO_PUP = _
  # Raw IP packet
  IPPROTO_RAW = _
  # IP6 routing header
  IPPROTO_ROUTING = _
  # TCP
  IPPROTO_TCP = _
  # ISO transport protocol class 4
  IPPROTO_TP = _
  # UDP
  IPPROTO_UDP = _
  # Xpress Transport Protocol
  IPPROTO_XTP = _
  # Checksum offset for raw sockets
  IPV6_CHECKSUM = _
  # Don't fragment packets
  IPV6_DONTFRAG = _
  # Destination option
  IPV6_DSTOPTS = _
  # Hop limit
  IPV6_HOPLIMIT = _
  # Hop-by-hop option
  IPV6_HOPOPTS = _
  # Join a group membership
  IPV6_JOIN_GROUP = _
  # Leave a group membership
  IPV6_LEAVE_GROUP = _
  # IP6 multicast hops
  IPV6_MULTICAST_HOPS = _
  # IP6 multicast interface
  IPV6_MULTICAST_IF = _
  # IP6 multicast loopback
  IPV6_MULTICAST_LOOP = _
  # Next hop address
  IPV6_NEXTHOP = _
  # Retrieve current path MTU
  IPV6_PATHMTU = _
  # Receive packet information with datagram
  IPV6_PKTINFO = _
  # Receive all IP6 options for response
  IPV6_RECVDSTOPTS = _
  # Receive hop limit with datagram
  IPV6_RECVHOPLIMIT = _
  # Receive hop-by-hop options
  IPV6_RECVHOPOPTS = _
  # Receive current path MTU with datagram
  IPV6_RECVPATHMTU = _
  # Receive destination IP address and incoming interface
  IPV6_RECVPKTINFO = _
  # Receive routing header
  IPV6_RECVRTHDR = _
  # Receive traffic class
  IPV6_RECVTCLASS = _
  # Allows removal of sticky routing headers
  IPV6_RTHDR = _
  # Allows removal of sticky destination options header
  IPV6_RTHDRDSTOPTS = _
  # Routing header type 0
  IPV6_RTHDR_TYPE_0 = _
  # Specify the traffic class
  IPV6_TCLASS = _
  # IP6 unicast hops
  IPV6_UNICAST_HOPS = _
  # Use the minimum MTU size
  IPV6_USE_MIN_MTU = _
  # Only bind IPv6 with a wildcard bind
  IPV6_V6ONLY = _
  IPX_TYPE = _
  # Add a multicast group membership
  IP_ADD_MEMBERSHIP = _
  # Add a multicast group membership
  IP_ADD_SOURCE_MEMBERSHIP = _
  # Block IPv4 multicast packets with a give source address
  IP_BLOCK_SOURCE = _
  # Default multicast loopback
  IP_DEFAULT_MULTICAST_LOOP = _
  # Default multicast TTL
  IP_DEFAULT_MULTICAST_TTL = _
  # Don't fragment packets
  IP_DONTFRAG = _
  # Drop a multicast group membership
  IP_DROP_MEMBERSHIP = _
  # Drop a multicast group membership
  IP_DROP_SOURCE_MEMBERSHIP = _
  # Allow binding to nonexistent IP addresses
  IP_FREEBIND = _
  # Header is included with data
  IP_HDRINCL = _
  # IPsec security policy
  IP_IPSEC_POLICY = _
  # Maximum number multicast groups a socket can join
  IP_MAX_MEMBERSHIPS = _
  # Minimum TTL allowed for received packets
  IP_MINTTL = _
  # Multicast source filtering
  IP_MSFILTER = _
  # The Maximum Transmission Unit of the socket
  IP_MTU = _
  # Path MTU discovery
  IP_MTU_DISCOVER = _
  # IP multicast interface
  IP_MULTICAST_IF = _
  # IP multicast loopback
  IP_MULTICAST_LOOP = _
  # IP multicast TTL
  IP_MULTICAST_TTL = _
  # Force outgoing broadcast datagrams to have the undirected broadcast address
  IP_ONESBCAST = _
  # IP options to be included in packets
  IP_OPTIONS = _
  # Retrieve security context with datagram
  IP_PASSSEC = _
  # Receive packet information with datagrams
  IP_PKTINFO = _
  # Receive packet options with datagrams
  IP_PKTOPTIONS = _
  # Always send DF frames
  IP_PMTUDISC_DO = _
  # Never send DF frames
  IP_PMTUDISC_DONT = _
  # Use per-route hints
  IP_PMTUDISC_WANT = _
  # Set the port range for sockets with unspecified port numbers
  IP_PORTRANGE = _
  # Receive IP destination address with datagram
  IP_RECVDSTADDR = _
  # Enable extended reliable error message passing
  IP_RECVERR = _
  # Receive interface information with datagrams
  IP_RECVIF = _
  # Receive all IP options with datagram
  IP_RECVOPTS = _
  # Receive all IP options for response
  IP_RECVRETOPTS = _
  # Receive link-layer address with datagrams
  IP_RECVSLLA = _
  # Receive TOS with incoming packets
  IP_RECVTOS = _
  # Receive IP TTL with datagrams
  IP_RECVTTL = _
  # IP options to be included in datagrams
  IP_RETOPTS = _
  # Notify transit routers to more closely examine the contents of an IP packet
  IP_ROUTER_ALERT = _
  # Source address for outgoing UDP datagrams
  IP_SENDSRCADDR = _
  # IP type-of-service
  IP_TOS = _
  # Transparent proxy
  IP_TRANSPARENT = _
  # IP time-to-live
  IP_TTL = _
  # Unblock IPv4 multicast packets with a give source address
  IP_UNBLOCK_SOURCE = _
  IP_XFRM_POLICY = _
  # Connect blocks until accepted
  LOCAL_CONNWAIT = _
  # Pass credentials to receiver
  LOCAL_CREDS = _
  # Retrieve peer credentials
  LOCAL_PEERCRED = _
  # Block multicast packets from this source
  MCAST_BLOCK_SOURCE = _
  # Exclusive multicast source filter
  MCAST_EXCLUDE = _
  # Inclusive multicast source filter
  MCAST_INCLUDE = _
  # Join a multicast group
  MCAST_JOIN_GROUP = _
  # Join a multicast source group
  MCAST_JOIN_SOURCE_GROUP = _
  # Leave a multicast group
  MCAST_LEAVE_GROUP = _
  # Leave a multicast source group
  MCAST_LEAVE_SOURCE_GROUP = _
  # Multicast source filtering
  MCAST_MSFILTER = _
  # Unblock multicast packets from this source
  MCAST_UNBLOCK_SOURCE = _
  # End of record
  MSG_COMPAT = _
  # Confirm path validity
  MSG_CONFIRM = _
  # Control data lost before delivery
  MSG_CTRUNC = _
  # Send without using the routing tables
  MSG_DONTROUTE = _
  # This message should be non-blocking
  MSG_DONTWAIT = _
  # Data completes connection
  MSG_EOF = _
  # Data completes record
  MSG_EOR = _
  # Fetch message from error queue
  MSG_ERRQUEUE = _
  # Reduce step of the handshake process
  MSG_FASTOPEN = _
  MSG_FIN = _
  # Start of a hold sequence.  Dumps to so_temp
  MSG_FLUSH = _
  # Data ready to be read
  MSG_HAVEMORE = _
  # Hold fragment in so_temp
  MSG_HOLD = _
  # Sender will send more
  MSG_MORE = _
  # Do not generate SIGPIPE
  MSG_NOSIGNAL = _
  # Process out-of-band data
  MSG_OOB = _
  # Peek at incoming message
  MSG_PEEK = _
  # Wait for full request
  MSG_PROXY = _
  # Data remains in the current packet
  MSG_RCVMORE = _
  MSG_RST = _
  # Send the packet in so_temp
  MSG_SEND = _
  MSG_SYN = _
  # Data discarded before delivery
  MSG_TRUNC = _
  # Wait for full request or error
  MSG_WAITALL = _
  # The service specified is a datagram service (looks up UDP ports)
  NI_DGRAM = _
  # Maximum length of a hostname
  NI_MAXHOST = _
  # Maximum length of a service name
  NI_MAXSERV = _
  # A name is required
  NI_NAMEREQD = _
  # An FQDN is not required for local hosts, return only the local part
  NI_NOFQDN = _
  # Return a numeric address
  NI_NUMERICHOST = _
  # Return the service name as a digit string
  NI_NUMERICSERV = _
  # AppleTalk protocol
  PF_APPLETALK = _
  # Asynchronous Transfer Mode
  PF_ATM = _
  # AX.25 protocol
  PF_AX25 = _
  # CCITT (now ITU-T) protocols
  PF_CCITT = _
  # MIT CHAOS protocols
  PF_CHAOS = _
  # Computer Network Technology
  PF_CNT = _
  # Connection-oriented IP
  PF_COIP = _
  # Datakit protocol
  PF_DATAKIT = _
  # DECnet protocol
  PF_DEC = _
  # DEC Direct Data Link Interface protocol
  PF_DLI = _
  # European Computer Manufacturers protocols
  PF_ECMA = _
  # NSC Hyperchannel protocol
  PF_HYLINK = _
  # ARPANET IMP protocol
  PF_IMPLINK = _
  # IPv4 protocol
  PF_INET = _
  # IPv6 protocol
  PF_INET6 = _
  # IPX protocol
  PF_IPX = _
  # Integrated Services Digital Network
  PF_ISDN = _
  # ISO Open Systems Interconnection protocols
  PF_ISO = _
  PF_KEY = _
  # Local Area Transport protocol
  PF_LAT = _
  # Link layer interface
  PF_LINK = _
  # Host-internal protocols
  PF_LOCAL = _
  # Maximum address family for this platform
  PF_MAX = _
  # Native ATM access
  PF_NATM = _
  # Network driver raw access
  PF_NDRV = _
  # NetBIOS
  PF_NETBIOS = _
  # Netgraph sockets
  PF_NETGRAPH = _
  # XEROX NS protocols
  PF_NS = _
  # ISO Open Systems Interconnection protocols
  PF_OSI = _
  # Direct link-layer access
  PF_PACKET = _
  PF_PIP = _
  # Point-to-Point Protocol
  PF_PPP = _
  # PARC Universal Packet protocol
  PF_PUP = _
  # Internal routing protocol
  PF_ROUTE = _
  PF_RTIP = _
  # Simple Internet Protocol
  PF_SIP = _
  # IBM SNA protocol
  PF_SNA = _
  PF_SYSTEM = _
  # UNIX sockets
  PF_UNIX = _
  # Unspecified protocol, any supported address family
  PF_UNSPEC = _
  # eXpress Transfer Protocol
  PF_XTP = _
  # Timestamp (bintime)
  SCM_BINTIME = _
  # The sender's credentials
  SCM_CREDENTIALS = _
  # Process credentials
  SCM_CREDS = _
  # Access rights
  SCM_RIGHTS = _
  # Timestamp (timeval)
  SCM_TIMESTAMP = _
  # Timestamp (timespec list) (Linux 2.6.30)
  SCM_TIMESTAMPING = _
  # Timespec (timespec)
  SCM_TIMESTAMPNS = _
  # User credentials
  SCM_UCRED = _
  # Wifi status (Linux 3.3)
  SCM_WIFI_STATUS = _
  # Shut down the reading side of the socket
  SHUT_RD = _
  # Shut down the both sides of the socket
  SHUT_RDWR = _
  # Shut down the writing side of the socket
  SHUT_WR = _
  # A datagram socket provides connectionless, unreliable messaging
  SOCK_DGRAM = _
  # Device-level packet access
  SOCK_PACKET = _
  # A raw socket provides low-level access for direct access or implementing network protocols
  SOCK_RAW = _
  # A reliable datagram socket provides reliable delivery of messages
  SOCK_RDM = _
  # A sequential packet socket provides sequenced, reliable two-way connection for datagrams
  SOCK_SEQPACKET = _
  # A stream socket provides a sequenced, reliable two-way connection for a byte stream
  SOCK_STREAM = _
  # AppleTalk socket options
  SOL_ATALK = _
  # AX.25 socket options
  SOL_AX25 = _
  # IP socket options
  SOL_IP = _
  # IPX socket options
  SOL_IPX = _
  # Socket-level options
  SOL_SOCKET = _
  # TCP socket options
  SOL_TCP = _
  # UDP socket options
  SOL_UDP = _
  # Maximum connection requests that may be queued for a socket
  SOMAXCONN = _
  # Background socket priority
  SOPRI_BACKGROUND = _
  # Interactive socket priority
  SOPRI_INTERACTIVE = _
  # Normal socket priority
  SOPRI_NORMAL = _
  # Socket has had listen() called on it
  SO_ACCEPTCONN = _
  # There is an accept filter
  SO_ACCEPTFILTER = _
  # Bypass zone boundaries
  SO_ALLZONES = _
  # Attach an accept filter
  SO_ATTACH_FILTER = _
  # Only send packets from the given interface
  SO_BINDTODEVICE = _
  # Receive timestamp with datagrams (bintime)
  SO_BINTIME = _
  # Query supported BPF extensions (Linux 3.14)
  SO_BPF_EXTENSIONS = _
  # Permit sending of broadcast messages
  SO_BROADCAST = _
  # Set the threshold in microseconds for low latency polling (Linux 3.11)
  SO_BUSY_POLL = _
  # Debug info recording
  SO_DEBUG = _
  # Detach an accept filter
  SO_DETACH_FILTER = _
  # Domain given for socket() (Linux 2.6.32)
  SO_DOMAIN = _
  # Use interface addresses
  SO_DONTROUTE = _
  # Retain unread data
  SO_DONTTRUNC = _
  # Get and clear the error status
  SO_ERROR = _
  # Obtain filter set by SO_ATTACH_FILTER (Linux 3.8)
  SO_GET_FILTER = _
  # Keep connections alive
  SO_KEEPALIVE = _
  # Linger on close if data is present
  SO_LINGER = _
  # Lock the filter attached to a socket (Linux 3.9)
  SO_LOCK_FILTER = _
  # Mandatory Access Control exemption for unlabeled peers
  SO_MAC_EXEMPT = _
  # Set the mark for mark-based routing (Linux 2.6.25)
  SO_MARK = _
  # Cap the rate computed by transport layer. [bytes per second] (Linux 3.13)
  SO_MAX_PACING_RATE = _
  # Install socket-level Network Kernel Extension
  SO_NKE = _
  # Set netns of a socket (Linux 3.4)
  SO_NOFCS = _
  # Don't SIGPIPE on EPIPE
  SO_NOSIGPIPE = _
  # Disable checksums
  SO_NO_CHECK = _
  # Get first packet byte count
  SO_NREAD = _
  # Leave received out-of-band data in-line
  SO_OOBINLINE = _
  # Receive SCM_CREDENTIALS messages
  SO_PASSCRED = _
  # Toggle security context passing (Linux 2.6.18)
  SO_PASSSEC = _
  # Set the peek offset (Linux 3.4)
  SO_PEEK_OFF = _
  # The credentials of the foreign process connected to this socket
  SO_PEERCRED = _
  # Name of the connecting user
  SO_PEERNAME = _
  # Obtain the security credentials (Linux 2.6.2)
  SO_PEERSEC = _
  # The protocol-defined priority for all packets on this socket
  SO_PRIORITY = _
  # Protocol given for socket() (Linux 2.6.32)
  SO_PROTOCOL = _
  # Receive buffer size
  SO_RCVBUF = _
  # Receive buffer size without rmem_max limit (Linux 2.6.14)
  SO_RCVBUFFORCE = _
  # Receive low-water mark
  SO_RCVLOWAT = _
  # Receive timeout
  SO_RCVTIMEO = _
  # Receive user credentials with datagram
  SO_RECVUCRED = _
  # Allow local address reuse
  SO_REUSEADDR = _
  # Allow local address and port reuse
  SO_REUSEPORT = _
  # Toggle cmsg for number of packets dropped (Linux 2.6.33)
  SO_RXQ_OVFL = _
  SO_SECURITY_AUTHENTICATION = _
  SO_SECURITY_ENCRYPTION_NETWORK = _
  SO_SECURITY_ENCRYPTION_TRANSPORT = _
  # Make select() detect socket error queue with errorfds (Linux 3.10)
  SO_SELECT_ERR_QUEUE = _
  # Send buffer size
  SO_SNDBUF = _
  # Send buffer size without wmem_max limit (Linux 2.6.14)
  SO_SNDBUFFORCE = _
  # Send low-water mark
  SO_SNDLOWAT = _
  # Send timeout
  SO_SNDTIMEO = _
  # Receive timestamp with datagrams (timeval)
  SO_TIMESTAMP = _
  # Time stamping of incoming and outgoing packets (Linux 2.6.30)
  SO_TIMESTAMPING = _
  # Receive nanosecond timestamp with datagrams (timespec)
  SO_TIMESTAMPNS = _
  # Get the socket type
  SO_TYPE = _
  # Bypass hardware when possible
  SO_USELOOPBACK = _
  # Give a hint when more data is ready
  SO_WANTMORE = _
  # OOB data is wanted in MSG_FLAG on receive
  SO_WANTOOBFLAG = _
  # Toggle cmsg for wifi status (Linux 3.3)
  SO_WIFI_STATUS = _
  # TCP congestion control algorithm (Linux 2.6.13, glibc 2.6)
  TCP_CONGESTION = _
  # TCP Cookie Transactions (Linux 2.6.33, glibc 2.18)
  TCP_COOKIE_TRANSACTIONS = _
  # Don't send partial frames (Linux 2.2, glibc 2.2)
  TCP_CORK = _
  # Don't notify a listening socket until data is ready (Linux 2.4, glibc 2.2)
  TCP_DEFER_ACCEPT = _
  # Reduce step of the handshake process (Linux 3.7, glibc 2.18)
  TCP_FASTOPEN = _
  # Retrieve information about this socket (Linux 2.4, glibc 2.2)
  TCP_INFO = _
  # Maximum number of keepalive probes allowed before dropping a connection (Linux 2.4, glibc 2.2)
  TCP_KEEPCNT = _
  # Idle time before keepalive probes are sent (Linux 2.4, glibc 2.2)
  TCP_KEEPIDLE = _
  # Time between keepalive probes (Linux 2.4, glibc 2.2)
  TCP_KEEPINTVL = _
  # Lifetime of orphaned FIN_WAIT2 sockets (Linux 2.4, glibc 2.2)
  TCP_LINGER2 = _
  # Set maximum segment size
  TCP_MAXSEG = _
  # Use MD5 digests (RFC2385, Linux 2.6.20, glibc 2.7)
  TCP_MD5SIG = _
  # Don't delay sending to coalesce packets
  TCP_NODELAY = _
  # Don't use TCP options
  TCP_NOOPT = _
  # Don't push the last block of write
  TCP_NOPUSH = _
  # Sequence of a queue for repair mode (Linux 3.5, glibc 2.18)
  TCP_QUEUE_SEQ = _
  # Enable quickack mode (Linux 2.4.4, glibc 2.3)
  TCP_QUICKACK = _
  # Repair mode (Linux 3.5, glibc 2.18)
  TCP_REPAIR = _
  # Options for repair mode (Linux 3.5, glibc 2.18)
  TCP_REPAIR_OPTIONS = _
  # Queue for repair mode (Linux 3.5, glibc 2.18)
  TCP_REPAIR_QUEUE = _
  # Number of SYN retransmits before a connection is dropped (Linux 2.4, glibc 2.2)
  TCP_SYNCNT = _
  # Duplicated acknowledgments handling for thin-streams (Linux 2.6.34, glibc 2.18)
  TCP_THIN_DUPACK = _
  # Linear timeouts for thin-streams (Linux 2.6.34, glibc 2.18)
  TCP_THIN_LINEAR_TIMEOUTS = _
  # TCP timestamp (Linux 3.9, glibc 2.18)
  TCP_TIMESTAMP = _
  # Max timeout before a TCP connection is aborted (Linux 2.6.37, glibc 2.18)
  TCP_USER_TIMEOUT = _
  # Clamp the size of the advertised window (Linux 2.4, glibc 2.2)
  TCP_WINDOW_CLAMP = _
  # Don't send partial frames (Linux 2.5.44, glibc 2.11)
  UDP_CORK = _

  # Obtains address information for _nodename_:_servname_.
  #
  # Note that Addrinfo.getaddrinfo provides the same functionality in
  # an object oriented style.
  #
  # _family_ should be an address family such as: :INET, :INET6, etc.
  #
  # _socktype_ should be a socket type such as: :STREAM, :DGRAM, :RAW, etc.
  #
  # _protocol_ should be a protocol defined in the family,
  # and defaults to 0 for the family.
  #
  # _flags_ should be bitwise OR of Socket::AI_* constants.
  #
  #   Socket.getaddrinfo("www.ruby-lang.org", "http", nil, :STREAM)
  #   #=> [["AF_INET", 80, "carbon.ruby-lang.org", "221.186.184.68", 2, 1, 6]] # PF_INET/SOCK_STREAM/IPPROTO_TCP
  #
  #   Socket.getaddrinfo("localhost", nil)
  #   #=> [["AF_INET", 0, "localhost", "127.0.0.1", 2, 1, 6],  # PF_INET/SOCK_STREAM/IPPROTO_TCP
  #   #    ["AF_INET", 0, "localhost", "127.0.0.1", 2, 2, 17], # PF_INET/SOCK_DGRAM/IPPROTO_UDP
  #   #    ["AF_INET", 0, "localhost", "127.0.0.1", 2, 3, 0]]  # PF_INET/SOCK_RAW/IPPROTO_IP
  #
  # _reverse_lookup_ directs the form of the third element, and has to
  # be one of below.  If _reverse_lookup_ is omitted, the default value is +nil+.
  #
  #   +true+, +:hostname+:  hostname is obtained from numeric address using reverse lookup, which may take a time.
  #   +false+, +:numeric+:  hostname is same as numeric address.
  #   +nil+:              obey to the current +do_not_reverse_lookup+ flag.
  #
  # If Addrinfo object is preferred, use Addrinfo.getaddrinfo.
  def self.getaddrinfo(p1, p2, p3 = v3, p4 = v4, p5 = v5, p6 = v6, p7 = v7) end

  # Use Addrinfo#getnameinfo instead.
  # This method is deprecated for the following reasons:
  #
  # - Uncommon address representation:
  #   4/16-bytes binary string to represent IPv4/IPv6 address.
  # - gethostbyaddr() may take a long time and it may block other threads.
  #   (GVL cannot be released since gethostbyname() is not thread safe.)
  # - This method uses gethostbyname() function already removed from POSIX.
  #
  # This method obtains the host information for _address_.
  #
  #   p Socket.gethostbyaddr([221,186,184,68].pack("CCCC"))
  #   #=> ["carbon.ruby-lang.org", [], 2, "\xDD\xBA\xB8D"]
  #
  #   p Socket.gethostbyaddr([127,0,0,1].pack("CCCC"))
  #   ["localhost", [], 2, "\x7F\x00\x00\x01"]
  #   p Socket.gethostbyaddr(([0]*15+[1]).pack("C"*16))
  #   #=> ["localhost", ["ip6-localhost", "ip6-loopback"], 10,
  #        "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01"]
  def self.gethostbyaddr(p1, p2 = v2) end

  # Use Addrinfo.getaddrinfo instead.
  # This method is deprecated for the following reasons:
  #
  # - The 3rd element of the result is the address family of the first address.
  #   The address families of the rest of the addresses are not returned.
  # - Uncommon address representation:
  #   4/16-bytes binary string to represent IPv4/IPv6 address.
  # - gethostbyname() may take a long time and it may block other threads.
  #   (GVL cannot be released since gethostbyname() is not thread safe.)
  # - This method uses gethostbyname() function already removed from POSIX.
  #
  # This method obtains the host information for _hostname_.
  #
  #   p Socket.gethostbyname("hal") #=> ["localhost", ["hal"], 2, "\x7F\x00\x00\x01"]
  def self.gethostbyname(hostname) end

  # Returns the hostname.
  #
  #   p Socket.gethostname #=> "hal"
  #
  # Note that it is not guaranteed to be able to convert to IP address using gethostbyname, getaddrinfo, etc.
  # If you need local IP address, use Socket.ip_address_list.
  def self.gethostname; end

  # Returns an array of interface addresses.
  # An element of the array is an instance of Socket::Ifaddr.
  #
  # This method can be used to find multicast-enabled interfaces:
  #
  #   pp Socket.getifaddrs.reject {|ifaddr|
  #     !ifaddr.addr.ip? || (ifaddr.flags & Socket::IFF_MULTICAST == 0)
  #   }.map {|ifaddr| [ifaddr.name, ifaddr.ifindex, ifaddr.addr] }
  #   #=> [["eth0", 2, #<Addrinfo: 221.186.184.67>],
  #   #    ["eth0", 2, #<Addrinfo: fe80::216:3eff:fe95:88bb%eth0>]]
  #
  # Example result on GNU/Linux:
  #   pp Socket.getifaddrs
  #   #=> [#<Socket::Ifaddr lo UP,LOOPBACK,RUNNING,0x10000 PACKET[protocol=0 lo hatype=772 HOST hwaddr=00:00:00:00:00:00]>,
  #   #    #<Socket::Ifaddr eth0 UP,BROADCAST,RUNNING,MULTICAST,0x10000 PACKET[protocol=0 eth0 hatype=1 HOST hwaddr=00:16:3e:95:88:bb] broadcast=PACKET[protocol=0 eth0 hatype=1 HOST hwaddr=ff:ff:ff:ff:ff:ff]>,
  #   #    #<Socket::Ifaddr sit0 NOARP PACKET[protocol=0 sit0 hatype=776 HOST hwaddr=00:00:00:00]>,
  #   #    #<Socket::Ifaddr lo UP,LOOPBACK,RUNNING,0x10000 127.0.0.1 netmask=255.0.0.0>,
  #   #    #<Socket::Ifaddr eth0 UP,BROADCAST,RUNNING,MULTICAST,0x10000 221.186.184.67 netmask=255.255.255.240 broadcast=221.186.184.79>,
  #   #    #<Socket::Ifaddr lo UP,LOOPBACK,RUNNING,0x10000 ::1 netmask=ffff:ffff:ffff:ffff:ffff:ffff:ffff:ffff>,
  #   #    #<Socket::Ifaddr eth0 UP,BROADCAST,RUNNING,MULTICAST,0x10000 fe80::216:3eff:fe95:88bb%eth0 netmask=ffff:ffff:ffff:ffff::>]
  #
  # Example result on FreeBSD:
  #   pp Socket.getifaddrs
  #   #=> [#<Socket::Ifaddr usbus0 UP,0x10000 LINK[usbus0]>,
  #   #    #<Socket::Ifaddr re0 UP,BROADCAST,RUNNING,MULTICAST,0x800 LINK[re0 3a:d0:40:9a:fe:e8]>,
  #   #    #<Socket::Ifaddr re0 UP,BROADCAST,RUNNING,MULTICAST,0x800 10.250.10.18 netmask=255.255.255.? (7 bytes for 16 bytes sockaddr_in) broadcast=10.250.10.255>,
  #   #    #<Socket::Ifaddr re0 UP,BROADCAST,RUNNING,MULTICAST,0x800 fe80:2::38d0:40ff:fe9a:fee8 netmask=ffff:ffff:ffff:ffff::>,
  #   #    #<Socket::Ifaddr re0 UP,BROADCAST,RUNNING,MULTICAST,0x800 2001:2e8:408:10::12 netmask=UNSPEC>,
  #   #    #<Socket::Ifaddr plip0 POINTOPOINT,MULTICAST,0x800 LINK[plip0]>,
  #   #    #<Socket::Ifaddr lo0 UP,LOOPBACK,RUNNING,MULTICAST LINK[lo0]>,
  #   #    #<Socket::Ifaddr lo0 UP,LOOPBACK,RUNNING,MULTICAST ::1 netmask=ffff:ffff:ffff:ffff:ffff:ffff:ffff:ffff>,
  #   #    #<Socket::Ifaddr lo0 UP,LOOPBACK,RUNNING,MULTICAST fe80:4::1 netmask=ffff:ffff:ffff:ffff::>,
  #   #    #<Socket::Ifaddr lo0 UP,LOOPBACK,RUNNING,MULTICAST 127.0.0.1 netmask=255.?.?.? (5 bytes for 16 bytes sockaddr_in)>]
  def self.getifaddrs; end

  # Obtains name information for _sockaddr_.
  #
  # _sockaddr_ should be one of follows.
  # - packed sockaddr string such as Socket.sockaddr_in(80, "127.0.0.1")
  # - 3-elements array such as ["AF_INET", 80, "127.0.0.1"]
  # - 4-elements array such as ["AF_INET", 80, ignored, "127.0.0.1"]
  #
  # _flags_ should be bitwise OR of Socket::NI_* constants.
  #
  # Note:
  # The last form is compatible with IPSocket#addr and IPSocket#peeraddr.
  #
  #   Socket.getnameinfo(Socket.sockaddr_in(80, "127.0.0.1"))       #=> ["localhost", "www"]
  #   Socket.getnameinfo(["AF_INET", 80, "127.0.0.1"])              #=> ["localhost", "www"]
  #   Socket.getnameinfo(["AF_INET", 80, "localhost", "127.0.0.1"]) #=> ["localhost", "www"]
  #
  # If Addrinfo object is preferred, use Addrinfo#getnameinfo.
  def self.getnameinfo(p1, p2 = v2) end

  # Obtains the port number for _service_name_.
  #
  # If _protocol_name_ is not given, "tcp" is assumed.
  #
  #   Socket.getservbyname("smtp")          #=> 25
  #   Socket.getservbyname("shell")         #=> 514
  #   Socket.getservbyname("syslog", "udp") #=> 514
  def self.getservbyname(...) end

  # Obtains the port number for _port_.
  #
  # If _protocol_name_ is not given, "tcp" is assumed.
  #
  #   Socket.getservbyport(80)         #=> "www"
  #   Socket.getservbyport(514, "tcp") #=> "shell"
  #   Socket.getservbyport(514, "udp") #=> "syslog"
  def self.getservbyport(p1, p2 = v2) end

  # Returns local IP addresses as an array.
  #
  # The array contains Addrinfo objects.
  #
  #  pp Socket.ip_address_list
  #  #=> [#<Addrinfo: 127.0.0.1>,
  #       #<Addrinfo: 192.168.0.128>,
  #       #<Addrinfo: ::1>,
  #       ...]
  def self.ip_address_list; end

  # Packs _port_ and _host_ as an AF_INET/AF_INET6 sockaddr string.
  #
  #   Socket.sockaddr_in(80, "127.0.0.1")
  #   #=> "\x02\x00\x00P\x7F\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00"
  #
  #   Socket.sockaddr_in(80, "::1")
  #   #=> "\n\x00\x00P\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00"
  def self.pack_sockaddr_in(port, host) end

  # Packs _path_ as an AF_UNIX sockaddr string.
  #
  #   Socket.sockaddr_un("/tmp/sock") #=> "\x01\x00/tmp/sock\x00\x00..."
  def self.pack_sockaddr_un(path) end

  # Creates a pair of sockets connected each other.
  #
  # _domain_ should be a communications domain such as: :INET, :INET6, :UNIX, etc.
  #
  # _socktype_ should be a socket type such as: :STREAM, :DGRAM, :RAW, etc.
  #
  # _protocol_ should be a protocol defined in the domain,
  # defaults to 0 for the domain.
  #
  #   s1, s2 = Socket.pair(:UNIX, :STREAM, 0)
  #   s1.send "a", 0
  #   s1.send "b", 0
  #   s1.close
  #   p s2.recv(10) #=> "ab"
  #   p s2.recv(10) #=> ""
  #   p s2.recv(10) #=> ""
  #
  #   s1, s2 = Socket.pair(:UNIX, :DGRAM, 0)
  #   s1.send "a", 0
  #   s1.send "b", 0
  #   p s2.recv(10) #=> "a"
  #   p s2.recv(10) #=> "b"
  def self.pair(domain, type, protocol) end

  # Packs _port_ and _host_ as an AF_INET/AF_INET6 sockaddr string.
  #
  #   Socket.sockaddr_in(80, "127.0.0.1")
  #   #=> "\x02\x00\x00P\x7F\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00"
  #
  #   Socket.sockaddr_in(80, "::1")
  #   #=> "\n\x00\x00P\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00"
  def self.sockaddr_in(port, host) end

  # Packs _path_ as an AF_UNIX sockaddr string.
  #
  #   Socket.sockaddr_un("/tmp/sock") #=> "\x01\x00/tmp/sock\x00\x00..."
  def self.sockaddr_un(path) end

  # Creates a pair of sockets connected each other.
  #
  # _domain_ should be a communications domain such as: :INET, :INET6, :UNIX, etc.
  #
  # _socktype_ should be a socket type such as: :STREAM, :DGRAM, :RAW, etc.
  #
  # _protocol_ should be a protocol defined in the domain,
  # defaults to 0 for the domain.
  #
  #   s1, s2 = Socket.pair(:UNIX, :STREAM, 0)
  #   s1.send "a", 0
  #   s1.send "b", 0
  #   s1.close
  #   p s2.recv(10) #=> "ab"
  #   p s2.recv(10) #=> ""
  #   p s2.recv(10) #=> ""
  #
  #   s1, s2 = Socket.pair(:UNIX, :DGRAM, 0)
  #   s1.send "a", 0
  #   s1.send "b", 0
  #   p s2.recv(10) #=> "a"
  #   p s2.recv(10) #=> "b"
  def self.socketpair(domain, type, protocol) end

  # Unpacks _sockaddr_ into port and ip_address.
  #
  # _sockaddr_ should be a string or an addrinfo for AF_INET/AF_INET6.
  #
  #   sockaddr = Socket.sockaddr_in(80, "127.0.0.1")
  #   p sockaddr #=> "\x02\x00\x00P\x7F\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00"
  #   p Socket.unpack_sockaddr_in(sockaddr) #=> [80, "127.0.0.1"]
  def self.unpack_sockaddr_in(sockaddr) end

  # Unpacks _sockaddr_ into path.
  #
  # _sockaddr_ should be a string or an addrinfo for AF_UNIX.
  #
  #   sockaddr = Socket.sockaddr_un("/tmp/sock")
  #   p Socket.unpack_sockaddr_un(sockaddr) #=> "/tmp/sock"
  def self.unpack_sockaddr_un(sockaddr) end

  # Creates a new socket object.
  #
  # _domain_ should be a communications domain such as: :INET, :INET6, :UNIX, etc.
  #
  # _socktype_ should be a socket type such as: :STREAM, :DGRAM, :RAW, etc.
  #
  # _protocol_ is optional and should be a protocol defined in the domain.
  # If protocol is not given, 0 is used internally.
  #
  #   Socket.new(:INET, :STREAM) # TCP socket
  #   Socket.new(:INET, :DGRAM)  # UDP socket
  #   Socket.new(:UNIX, :STREAM) # UNIX stream socket
  #   Socket.new(:UNIX, :DGRAM)  # UNIX datagram socket
  def initialize(p1, p2, p3 = v3) end

  # Accepts a next connection.
  # Returns a new Socket object and Addrinfo object.
  #
  #   serv = Socket.new(:INET, :STREAM, 0)
  #   serv.listen(5)
  #   c = Socket.new(:INET, :STREAM, 0)
  #   c.connect(serv.connect_address)
  #   p serv.accept #=> [#<Socket:fd 6>, #<Addrinfo: 127.0.0.1:48555 TCP>]
  def accept; end

  # Binds to the given local address.
  #
  # === Parameter
  # * +local_sockaddr+ - the +struct+ sockaddr contained in a string or an Addrinfo object
  #
  # === Example
  #   require 'socket'
  #
  #   # use Addrinfo
  #   socket = Socket.new(:INET, :STREAM, 0)
  #   socket.bind(Addrinfo.tcp("127.0.0.1", 2222))
  #   p socket.local_address #=> #<Addrinfo: 127.0.0.1:2222 TCP>
  #
  #   # use struct sockaddr
  #   include Socket::Constants
  #   socket = Socket.new( AF_INET, SOCK_STREAM, 0 )
  #   sockaddr = Socket.pack_sockaddr_in( 2200, 'localhost' )
  #   socket.bind( sockaddr )
  #
  # === Unix-based Exceptions
  # On unix-based based systems the following system exceptions may be raised if
  # the call to _bind_ fails:
  # * Errno::EACCES - the specified _sockaddr_ is protected and the current
  #   user does not have permission to bind to it
  # * Errno::EADDRINUSE - the specified _sockaddr_ is already in use
  # * Errno::EADDRNOTAVAIL - the specified _sockaddr_ is not available from the
  #   local machine
  # * Errno::EAFNOSUPPORT - the specified _sockaddr_ is not a valid address for
  #   the family of the calling +socket+
  # * Errno::EBADF - the _sockaddr_ specified is not a valid file descriptor
  # * Errno::EFAULT - the _sockaddr_ argument cannot be accessed
  # * Errno::EINVAL - the +socket+ is already bound to an address, and the
  #   protocol does not support binding to the new _sockaddr_ or the +socket+
  #   has been shut down.
  # * Errno::EINVAL - the address length is not a valid length for the address
  #   family
  # * Errno::ENAMETOOLONG - the pathname resolved had a length which exceeded
  #   PATH_MAX
  # * Errno::ENOBUFS - no buffer space is available
  # * Errno::ENOSR - there were insufficient STREAMS resources available to
  #   complete the operation
  # * Errno::ENOTSOCK - the +socket+ does not refer to a socket
  # * Errno::EOPNOTSUPP - the socket type of the +socket+ does not support
  #   binding to an address
  #
  # On unix-based based systems if the address family of the calling +socket+ is
  # Socket::AF_UNIX the follow exceptions may be raised if the call to _bind_
  # fails:
  # * Errno::EACCES - search permission is denied for a component of the prefix
  #   path or write access to the +socket+ is denied
  # * Errno::EDESTADDRREQ - the _sockaddr_ argument is a null pointer
  # * Errno::EISDIR - same as Errno::EDESTADDRREQ
  # * Errno::EIO - an i/o error occurred
  # * Errno::ELOOP - too many symbolic links were encountered in translating
  #   the pathname in _sockaddr_
  # * Errno::ENAMETOOLLONG - a component of a pathname exceeded NAME_MAX
  #   characters, or an entire pathname exceeded PATH_MAX characters
  # * Errno::ENOENT - a component of the pathname does not name an existing file
  #   or the pathname is an empty string
  # * Errno::ENOTDIR - a component of the path prefix of the pathname in _sockaddr_
  #   is not a directory
  # * Errno::EROFS - the name would reside on a read only filesystem
  #
  # === Windows Exceptions
  # On Windows systems the following system exceptions may be raised if
  # the call to _bind_ fails:
  # * Errno::ENETDOWN-- the network is down
  # * Errno::EACCES - the attempt to connect the datagram socket to the
  #   broadcast address failed
  # * Errno::EADDRINUSE - the socket's local address is already in use
  # * Errno::EADDRNOTAVAIL - the specified address is not a valid address for this
  #   computer
  # * Errno::EFAULT - the socket's internal address or address length parameter
  #   is too small or is not a valid part of the user space addressed
  # * Errno::EINVAL - the +socket+ is already bound to an address
  # * Errno::ENOBUFS - no buffer space is available
  # * Errno::ENOTSOCK - the +socket+ argument does not refer to a socket
  #
  # === See
  # * bind manual pages on unix-based systems
  # * bind function in Microsoft's Winsock functions reference
  def bind(local_sockaddr) end

  # Requests a connection to be made on the given +remote_sockaddr+. Returns 0 if
  # successful, otherwise an exception is raised.
  #
  # === Parameter
  # * +remote_sockaddr+ - the +struct+ sockaddr contained in a string or Addrinfo object
  #
  # === Example:
  #   # Pull down Google's web page
  #   require 'socket'
  #   include Socket::Constants
  #   socket = Socket.new( AF_INET, SOCK_STREAM, 0 )
  #   sockaddr = Socket.pack_sockaddr_in( 80, 'www.google.com' )
  #   socket.connect( sockaddr )
  #   socket.write( "GET / HTTP/1.0\r\n\r\n" )
  #   results = socket.read
  #
  # === Unix-based Exceptions
  # On unix-based systems the following system exceptions may be raised if
  # the call to _connect_ fails:
  # * Errno::EACCES - search permission is denied for a component of the prefix
  #   path or write access to the +socket+ is denied
  # * Errno::EADDRINUSE - the _sockaddr_ is already in use
  # * Errno::EADDRNOTAVAIL - the specified _sockaddr_ is not available from the
  #   local machine
  # * Errno::EAFNOSUPPORT - the specified _sockaddr_ is not a valid address for
  #   the address family of the specified +socket+
  # * Errno::EALREADY - a connection is already in progress for the specified
  #   socket
  # * Errno::EBADF - the +socket+ is not a valid file descriptor
  # * Errno::ECONNREFUSED - the target _sockaddr_ was not listening for connections
  #   refused the connection request
  # * Errno::ECONNRESET - the remote host reset the connection request
  # * Errno::EFAULT - the _sockaddr_ cannot be accessed
  # * Errno::EHOSTUNREACH - the destination host cannot be reached (probably
  #   because the host is down or a remote router cannot reach it)
  # * Errno::EINPROGRESS - the O_NONBLOCK is set for the +socket+ and the
  #   connection cannot be immediately established; the connection will be
  #   established asynchronously
  # * Errno::EINTR - the attempt to establish the connection was interrupted by
  #   delivery of a signal that was caught; the connection will be established
  #   asynchronously
  # * Errno::EISCONN - the specified +socket+ is already connected
  # * Errno::EINVAL - the address length used for the _sockaddr_ is not a valid
  #   length for the address family or there is an invalid family in _sockaddr_
  # * Errno::ENAMETOOLONG - the pathname resolved had a length which exceeded
  #   PATH_MAX
  # * Errno::ENETDOWN - the local interface used to reach the destination is down
  # * Errno::ENETUNREACH - no route to the network is present
  # * Errno::ENOBUFS - no buffer space is available
  # * Errno::ENOSR - there were insufficient STREAMS resources available to
  #   complete the operation
  # * Errno::ENOTSOCK - the +socket+ argument does not refer to a socket
  # * Errno::EOPNOTSUPP - the calling +socket+ is listening and cannot be connected
  # * Errno::EPROTOTYPE - the _sockaddr_ has a different type than the socket
  #   bound to the specified peer address
  # * Errno::ETIMEDOUT - the attempt to connect time out before a connection
  #   was made.
  #
  # On unix-based systems if the address family of the calling +socket+ is
  # AF_UNIX the follow exceptions may be raised if the call to _connect_
  # fails:
  # * Errno::EIO - an i/o error occurred while reading from or writing to the
  #   file system
  # * Errno::ELOOP - too many symbolic links were encountered in translating
  #   the pathname in _sockaddr_
  # * Errno::ENAMETOOLLONG - a component of a pathname exceeded NAME_MAX
  #   characters, or an entire pathname exceeded PATH_MAX characters
  # * Errno::ENOENT - a component of the pathname does not name an existing file
  #   or the pathname is an empty string
  # * Errno::ENOTDIR - a component of the path prefix of the pathname in _sockaddr_
  #   is not a directory
  #
  # === Windows Exceptions
  # On Windows systems the following system exceptions may be raised if
  # the call to _connect_ fails:
  # * Errno::ENETDOWN - the network is down
  # * Errno::EADDRINUSE - the socket's local address is already in use
  # * Errno::EINTR - the socket was cancelled
  # * Errno::EINPROGRESS - a blocking socket is in progress or the service provider
  #   is still processing a callback function. Or a nonblocking connect call is
  #   in progress on the +socket+.
  # * Errno::EALREADY - see Errno::EINVAL
  # * Errno::EADDRNOTAVAIL - the remote address is not a valid address, such as
  #   ADDR_ANY TODO check ADDRANY TO INADDR_ANY
  # * Errno::EAFNOSUPPORT - addresses in the specified family cannot be used with
  #   with this +socket+
  # * Errno::ECONNREFUSED - the target _sockaddr_ was not listening for connections
  #   refused the connection request
  # * Errno::EFAULT - the socket's internal address or address length parameter
  #   is too small or is not a valid part of the user space address
  # * Errno::EINVAL - the +socket+ is a listening socket
  # * Errno::EISCONN - the +socket+ is already connected
  # * Errno::ENETUNREACH - the network cannot be reached from this host at this time
  # * Errno::EHOSTUNREACH - no route to the network is present
  # * Errno::ENOBUFS - no buffer space is available
  # * Errno::ENOTSOCK - the +socket+ argument does not refer to a socket
  # * Errno::ETIMEDOUT - the attempt to connect time out before a connection
  #   was made.
  # * Errno::EWOULDBLOCK - the socket is marked as nonblocking and the
  #   connection cannot be completed immediately
  # * Errno::EACCES - the attempt to connect the datagram socket to the
  #   broadcast address failed
  #
  # === See
  # * connect manual pages on unix-based systems
  # * connect function in Microsoft's Winsock functions reference
  def connect(remote_sockaddr) end

  # Listens for connections, using the specified +int+ as the backlog. A call
  # to _listen_ only applies if the +socket+ is of type SOCK_STREAM or
  # SOCK_SEQPACKET.
  #
  # === Parameter
  # * +backlog+ - the maximum length of the queue for pending connections.
  #
  # === Example 1
  #   require 'socket'
  #   include Socket::Constants
  #   socket = Socket.new( AF_INET, SOCK_STREAM, 0 )
  #   sockaddr = Socket.pack_sockaddr_in( 2200, 'localhost' )
  #   socket.bind( sockaddr )
  #   socket.listen( 5 )
  #
  # === Example 2 (listening on an arbitrary port, unix-based systems only):
  #   require 'socket'
  #   include Socket::Constants
  #   socket = Socket.new( AF_INET, SOCK_STREAM, 0 )
  #   socket.listen( 1 )
  #
  # === Unix-based Exceptions
  # On unix based systems the above will work because a new +sockaddr+ struct
  # is created on the address ADDR_ANY, for an arbitrary port number as handed
  # off by the kernel. It will not work on Windows, because Windows requires that
  # the +socket+ is bound by calling _bind_ before it can _listen_.
  #
  # If the _backlog_ amount exceeds the implementation-dependent maximum
  # queue length, the implementation's maximum queue length will be used.
  #
  # On unix-based based systems the following system exceptions may be raised if the
  # call to _listen_ fails:
  # * Errno::EBADF - the _socket_ argument is not a valid file descriptor
  # * Errno::EDESTADDRREQ - the _socket_ is not bound to a local address, and
  #   the protocol does not support listening on an unbound socket
  # * Errno::EINVAL - the _socket_ is already connected
  # * Errno::ENOTSOCK - the _socket_ argument does not refer to a socket
  # * Errno::EOPNOTSUPP - the _socket_ protocol does not support listen
  # * Errno::EACCES - the calling process does not have appropriate privileges
  # * Errno::EINVAL - the _socket_ has been shut down
  # * Errno::ENOBUFS - insufficient resources are available in the system to
  #   complete the call
  #
  # === Windows Exceptions
  # On Windows systems the following system exceptions may be raised if
  # the call to _listen_ fails:
  # * Errno::ENETDOWN - the network is down
  # * Errno::EADDRINUSE - the socket's local address is already in use. This
  #   usually occurs during the execution of _bind_ but could be delayed
  #   if the call to _bind_ was to a partially wildcard address (involving
  #   ADDR_ANY) and if a specific address needs to be committed at the
  #   time of the call to _listen_
  # * Errno::EINPROGRESS - a Windows Sockets 1.1 call is in progress or the
  #   service provider is still processing a callback function
  # * Errno::EINVAL - the +socket+ has not been bound with a call to _bind_.
  # * Errno::EISCONN - the +socket+ is already connected
  # * Errno::EMFILE - no more socket descriptors are available
  # * Errno::ENOBUFS - no buffer space is available
  # * Errno::ENOTSOC - +socket+ is not a socket
  # * Errno::EOPNOTSUPP - the referenced +socket+ is not a type that supports
  #   the _listen_ method
  #
  # === See
  # * listen manual pages on unix-based systems
  # * listen function in Microsoft's Winsock functions reference
  def listen(int) end

  # Receives up to _maxlen_ bytes from +socket+. _flags_ is zero or more
  # of the +MSG_+ options. The first element of the results, _mesg_, is the data
  # received. The second element, _sender_addrinfo_, contains protocol-specific
  # address information of the sender.
  #
  # === Parameters
  # * +maxlen+ - the maximum number of bytes to receive from the socket
  # * +flags+ - zero or more of the +MSG_+ options
  #
  # === Example
  #   # In one file, start this first
  #   require 'socket'
  #   include Socket::Constants
  #   socket = Socket.new( AF_INET, SOCK_STREAM, 0 )
  #   sockaddr = Socket.pack_sockaddr_in( 2200, 'localhost' )
  #   socket.bind( sockaddr )
  #   socket.listen( 5 )
  #   client, client_addrinfo = socket.accept
  #   data = client.recvfrom( 20 )[0].chomp
  #   puts "I only received 20 bytes '#{data}'"
  #   sleep 1
  #   socket.close
  #
  #   # In another file, start this second
  #   require 'socket'
  #   include Socket::Constants
  #   socket = Socket.new( AF_INET, SOCK_STREAM, 0 )
  #   sockaddr = Socket.pack_sockaddr_in( 2200, 'localhost' )
  #   socket.connect( sockaddr )
  #   socket.puts "Watch this get cut short!"
  #   socket.close
  #
  # === Unix-based Exceptions
  # On unix-based based systems the following system exceptions may be raised if the
  # call to _recvfrom_ fails:
  # * Errno::EAGAIN - the +socket+ file descriptor is marked as O_NONBLOCK and no
  #   data is waiting to be received; or MSG_OOB is set and no out-of-band data
  #   is available and either the +socket+ file descriptor is marked as
  #   O_NONBLOCK or the +socket+ does not support blocking to wait for
  #   out-of-band-data
  # * Errno::EWOULDBLOCK - see Errno::EAGAIN
  # * Errno::EBADF - the +socket+ is not a valid file descriptor
  # * Errno::ECONNRESET - a connection was forcibly closed by a peer
  # * Errno::EFAULT - the socket's internal buffer, address or address length
  #   cannot be accessed or written
  # * Errno::EINTR - a signal interrupted _recvfrom_ before any data was available
  # * Errno::EINVAL - the MSG_OOB flag is set and no out-of-band data is available
  # * Errno::EIO - an i/o error occurred while reading from or writing to the
  #   filesystem
  # * Errno::ENOBUFS - insufficient resources were available in the system to
  #   perform the operation
  # * Errno::ENOMEM - insufficient memory was available to fulfill the request
  # * Errno::ENOSR - there were insufficient STREAMS resources available to
  #   complete the operation
  # * Errno::ENOTCONN - a receive is attempted on a connection-mode socket that
  #   is not connected
  # * Errno::ENOTSOCK - the +socket+ does not refer to a socket
  # * Errno::EOPNOTSUPP - the specified flags are not supported for this socket type
  # * Errno::ETIMEDOUT - the connection timed out during connection establishment
  #   or due to a transmission timeout on an active connection
  #
  # === Windows Exceptions
  # On Windows systems the following system exceptions may be raised if
  # the call to _recvfrom_ fails:
  # * Errno::ENETDOWN - the network is down
  # * Errno::EFAULT - the internal buffer and from parameters on +socket+ are not
  #   part of the user address space, or the internal fromlen parameter is
  #   too small to accommodate the peer address
  # * Errno::EINTR - the (blocking) call was cancelled by an internal call to
  #   the WinSock function WSACancelBlockingCall
  # * Errno::EINPROGRESS - a blocking Windows Sockets 1.1 call is in progress or
  #   the service provider is still processing a callback function
  # * Errno::EINVAL - +socket+ has not been bound with a call to _bind_, or an
  #   unknown flag was specified, or MSG_OOB was specified for a socket with
  #   SO_OOBINLINE enabled, or (for byte stream-style sockets only) the internal
  #   len parameter on +socket+ was zero or negative
  # * Errno::EISCONN - +socket+ is already connected. The call to _recvfrom_ is
  #   not permitted with a connected socket on a socket that is connection
  #   oriented or connectionless.
  # * Errno::ENETRESET - the connection has been broken due to the keep-alive
  #   activity detecting a failure while the operation was in progress.
  # * Errno::EOPNOTSUPP - MSG_OOB was specified, but +socket+ is not stream-style
  #   such as type SOCK_STREAM. OOB data is not supported in the communication
  #   domain associated with +socket+, or +socket+ is unidirectional and
  #   supports only send operations
  # * Errno::ESHUTDOWN - +socket+ has been shutdown. It is not possible to
  #   call _recvfrom_ on a socket after _shutdown_ has been invoked.
  # * Errno::EWOULDBLOCK - +socket+ is marked as nonblocking and a  call to
  #   _recvfrom_ would block.
  # * Errno::EMSGSIZE - the message was too large to fit into the specified buffer
  #   and was truncated.
  # * Errno::ETIMEDOUT - the connection has been dropped, because of a network
  #   failure or because the system on the other end went down without
  #   notice
  # * Errno::ECONNRESET - the virtual circuit was reset by the remote side
  #   executing a hard or abortive close. The application should close the
  #   socket; it is no longer usable. On a UDP-datagram socket this error
  #   indicates a previous send operation resulted in an ICMP Port Unreachable
  #   message.
  def recvfrom(...) end

  # Accepts an incoming connection returning an array containing the (integer)
  # file descriptor for the incoming connection, _client_socket_fd_,
  # and an Addrinfo, _client_addrinfo_.
  #
  # === Example
  #   # In one script, start this first
  #   require 'socket'
  #   include Socket::Constants
  #   socket = Socket.new( AF_INET, SOCK_STREAM, 0 )
  #   sockaddr = Socket.pack_sockaddr_in( 2200, 'localhost' )
  #   socket.bind( sockaddr )
  #   socket.listen( 5 )
  #   client_fd, client_addrinfo = socket.sysaccept
  #   client_socket = Socket.for_fd( client_fd )
  #   puts "The client said, '#{client_socket.readline.chomp}'"
  #   client_socket.puts "Hello from script one!"
  #   socket.close
  #
  #   # In another script, start this second
  #   require 'socket'
  #   include Socket::Constants
  #   socket = Socket.new( AF_INET, SOCK_STREAM, 0 )
  #   sockaddr = Socket.pack_sockaddr_in( 2200, 'localhost' )
  #   socket.connect( sockaddr )
  #   socket.puts "Hello from script 2."
  #   puts "The server said, '#{socket.readline.chomp}'"
  #   socket.close
  #
  # Refer to Socket#accept for the exceptions that may be thrown if the call
  # to _sysaccept_ fails.
  #
  # === See
  # * Socket#accept
  def sysaccept; end

  # Socket::AncillaryData represents the ancillary data (control information)
  # used by sendmsg and recvmsg system call.  It contains socket #family,
  # control message (cmsg) #level, cmsg #type and cmsg #data.
  class AncillaryData
    # Creates a new Socket::AncillaryData object which contains a int as data.
    #
    # The size and endian is dependent on the host.
    #
    #   require 'socket'
    #
    #   p Socket::AncillaryData.int(:UNIX, :SOCKET, :RIGHTS, STDERR.fileno)
    #   #=> #<Socket::AncillaryData: UNIX SOCKET RIGHTS 2>
    def self.int(family, cmsg_level, cmsg_type, integer) end

    # Returns new ancillary data for IP_PKTINFO.
    #
    # If spec_dst is not given, addr is used.
    #
    # IP_PKTINFO is not standard.
    #
    # Supported platform: GNU/Linux
    #
    #   addr = Addrinfo.ip("127.0.0.1")
    #   ifindex = 0
    #   spec_dst = Addrinfo.ip("127.0.0.1")
    #   p Socket::AncillaryData.ip_pktinfo(addr, ifindex, spec_dst)
    #   #=> #<Socket::AncillaryData: INET IP PKTINFO 127.0.0.1 ifindex:0 spec_dst:127.0.0.1>
    def self.ip_pktinfo(...) end

    # Returns new ancillary data for IPV6_PKTINFO.
    #
    # IPV6_PKTINFO is defined by RFC 3542.
    #
    #   addr = Addrinfo.ip("::1")
    #   ifindex = 0
    #   p Socket::AncillaryData.ipv6_pktinfo(addr, ifindex)
    #   #=> #<Socket::AncillaryData: INET6 IPV6 PKTINFO ::1 ifindex:0>
    def self.ipv6_pktinfo(addr, ifindex) end

    # Creates a new Socket::AncillaryData object which contains file descriptors as data.
    #
    #   p Socket::AncillaryData.unix_rights(STDERR)
    #   #=> #<Socket::AncillaryData: UNIX SOCKET RIGHTS 2>
    def self.unix_rights(io1, io2, *args) end

    # _family_ should be an integer, a string or a symbol.
    # - Socket::AF_INET, "AF_INET", "INET", :AF_INET, :INET
    # - Socket::AF_UNIX, "AF_UNIX", "UNIX", :AF_UNIX, :UNIX
    # - etc.
    #
    # _cmsg_level_ should be an integer, a string or a symbol.
    # - Socket::SOL_SOCKET, "SOL_SOCKET", "SOCKET", :SOL_SOCKET and :SOCKET
    # - Socket::IPPROTO_IP, "IP" and :IP
    # - Socket::IPPROTO_IPV6, "IPV6" and :IPV6
    # - Socket::IPPROTO_TCP, "TCP" and :TCP
    # - etc.
    #
    # _cmsg_type_ should be an integer, a string or a symbol.
    # If a string/symbol is specified, it is interpreted depend on _cmsg_level_.
    # - Socket::SCM_RIGHTS, "SCM_RIGHTS", "RIGHTS", :SCM_RIGHTS, :RIGHTS for SOL_SOCKET
    # - Socket::IP_RECVTTL, "RECVTTL" and :RECVTTL for IPPROTO_IP
    # - Socket::IPV6_PKTINFO, "PKTINFO" and :PKTINFO for IPPROTO_IPV6
    # - etc.
    #
    # _cmsg_data_ should be a string.
    #
    #   p Socket::AncillaryData.new(:INET, :TCP, :NODELAY, "")
    #   #=> #<Socket::AncillaryData: INET TCP NODELAY "">
    #
    #   p Socket::AncillaryData.new(:INET6, :IPV6, :PKTINFO, "")
    #   #=> #<Socket::AncillaryData: INET6 IPV6 PKTINFO "">
    def initialize(family, cmsg_level, cmsg_type, cmsg_data) end

    # tests the level and type of _ancillarydata_.
    #
    #   ancdata = Socket::AncillaryData.new(:INET6, :IPV6, :PKTINFO, "")
    #   ancdata.cmsg_is?(Socket::IPPROTO_IPV6, Socket::IPV6_PKTINFO) #=> true
    #   ancdata.cmsg_is?(:IPV6, :PKTINFO)       #=> true
    #   ancdata.cmsg_is?(:IP, :PKTINFO)         #=> false
    #   ancdata.cmsg_is?(:SOCKET, :RIGHTS)      #=> false
    def cmsg_is?(level, type) end

    # returns the cmsg data as a string.
    #
    #   p Socket::AncillaryData.new(:INET6, :IPV6, :PKTINFO, "").data
    #   #=> ""
    def data; end

    # returns the socket family as an integer.
    #
    #   p Socket::AncillaryData.new(:INET6, :IPV6, :PKTINFO, "").family
    #   #=> 10
    def family; end

    # returns a string which shows ancillarydata in human-readable form.
    #
    #   p Socket::AncillaryData.new(:INET6, :IPV6, :PKTINFO, "").inspect
    #   #=> "#<Socket::AncillaryData: INET6 IPV6 PKTINFO \"\">"
    def inspect; end

    # Returns the data in _ancillarydata_ as an int.
    #
    # The size and endian is dependent on the host.
    #
    #   ancdata = Socket::AncillaryData.int(:UNIX, :SOCKET, :RIGHTS, STDERR.fileno)
    #   p ancdata.int #=> 2
    def int; end

    # Extracts addr, ifindex and spec_dst from IP_PKTINFO ancillary data.
    #
    # IP_PKTINFO is not standard.
    #
    # Supported platform: GNU/Linux
    #
    #   addr = Addrinfo.ip("127.0.0.1")
    #   ifindex = 0
    #   spec_dest = Addrinfo.ip("127.0.0.1")
    #   ancdata = Socket::AncillaryData.ip_pktinfo(addr, ifindex, spec_dest)
    #   p ancdata.ip_pktinfo
    #   #=> [#<Addrinfo: 127.0.0.1>, 0, #<Addrinfo: 127.0.0.1>]
    def ip_pktinfo; end

    # Extracts addr and ifindex from IPV6_PKTINFO ancillary data.
    #
    # IPV6_PKTINFO is defined by RFC 3542.
    #
    #   addr = Addrinfo.ip("::1")
    #   ifindex = 0
    #   ancdata = Socket::AncillaryData.ipv6_pktinfo(addr, ifindex)
    #   p ancdata.ipv6_pktinfo #=> [#<Addrinfo: ::1>, 0]
    def ipv6_pktinfo; end

    # Extracts addr from IPV6_PKTINFO ancillary data.
    #
    # IPV6_PKTINFO is defined by RFC 3542.
    #
    #   addr = Addrinfo.ip("::1")
    #   ifindex = 0
    #   ancdata = Socket::AncillaryData.ipv6_pktinfo(addr, ifindex)
    #   p ancdata.ipv6_pktinfo_addr #=> #<Addrinfo: ::1>
    def ipv6_pktinfo_addr; end

    # Extracts ifindex from IPV6_PKTINFO ancillary data.
    #
    # IPV6_PKTINFO is defined by RFC 3542.
    #
    #   addr = Addrinfo.ip("::1")
    #   ifindex = 0
    #   ancdata = Socket::AncillaryData.ipv6_pktinfo(addr, ifindex)
    #   p ancdata.ipv6_pktinfo_ifindex #=> 0
    def ipv6_pktinfo_ifindex; end

    # returns the cmsg level as an integer.
    #
    #   p Socket::AncillaryData.new(:INET6, :IPV6, :PKTINFO, "").level
    #   #=> 41
    def level; end

    # returns the timestamp as a time object.
    #
    # _ancillarydata_ should be one of following type:
    # - SOL_SOCKET/SCM_TIMESTAMP (microsecond) GNU/Linux, FreeBSD, NetBSD, OpenBSD, Solaris, MacOS X
    # - SOL_SOCKET/SCM_TIMESTAMPNS (nanosecond) GNU/Linux
    # - SOL_SOCKET/SCM_BINTIME (2**(-64) second) FreeBSD
    #
    #   Addrinfo.udp("127.0.0.1", 0).bind {|s1|
    #     Addrinfo.udp("127.0.0.1", 0).bind {|s2|
    #       s1.setsockopt(:SOCKET, :TIMESTAMP, true)
    #       s2.send "a", 0, s1.local_address
    #       ctl = s1.recvmsg.last
    #       p ctl    #=> #<Socket::AncillaryData: INET SOCKET TIMESTAMP 2009-02-24 17:35:46.775581>
    #       t = ctl.timestamp
    #       p t      #=> 2009-02-24 17:35:46 +0900
    #       p t.usec #=> 775581
    #       p t.nsec #=> 775581000
    #     }
    #   }
    def timestamp; end

    # returns the cmsg type as an integer.
    #
    #   p Socket::AncillaryData.new(:INET6, :IPV6, :PKTINFO, "").type
    #   #=> 2
    def type; end

    # returns the array of IO objects for SCM_RIGHTS control message in UNIX domain socket.
    #
    # The class of the IO objects in the array is IO or Socket.
    #
    # The array is attached to _ancillarydata_ when it is instantiated.
    # For example, BasicSocket#recvmsg attach the array when
    # receives a SCM_RIGHTS control message and :scm_rights=>true option is given.
    #
    #   # recvmsg needs :scm_rights=>true for unix_rights
    #   s1, s2 = UNIXSocket.pair
    #   p s1                                         #=> #<UNIXSocket:fd 3>
    #   s1.sendmsg "stdin and a socket", 0, nil, Socket::AncillaryData.unix_rights(STDIN, s1)
    #   _, _, _, ctl = s2.recvmsg(:scm_rights=>true)
    #   p ctl                                        #=> #<Socket::AncillaryData: UNIX SOCKET RIGHTS 6 7>
    #   p ctl.unix_rights                            #=> [#<IO:fd 6>, #<Socket:fd 7>]
    #   p File.identical?(STDIN, ctl.unix_rights[0]) #=> true
    #   p File.identical?(s1, ctl.unix_rights[1])    #=> true
    #
    #   # If :scm_rights=>true is not given, unix_rights returns nil
    #   s1, s2 = UNIXSocket.pair
    #   s1.sendmsg "stdin and a socket", 0, nil, Socket::AncillaryData.unix_rights(STDIN, s1)
    #   _, _, _, ctl = s2.recvmsg
    #   p ctl #=> #<Socket::AncillaryData: UNIX SOCKET RIGHTS 6 7>
    #   p ctl.unix_rights #=> nil
    def unix_rights; end
  end

  # Socket::Constants provides socket-related constants.  All possible
  # socket constants are listed in the documentation but they may not all
  # be present on your platform.
  #
  # If the underlying platform doesn't define a constant the corresponding
  # Ruby constant is not defined.
  module Constants
    # AppleTalk protocol
    AF_APPLETALK = _
    # Asynchronous Transfer Mode
    AF_ATM = _
    # AX.25 protocol
    AF_AX25 = _
    # CCITT (now ITU-T) protocols
    AF_CCITT = _
    # MIT CHAOS protocols
    AF_CHAOS = _
    # Computer Network Technology
    AF_CNT = _
    # Connection-oriented IP
    AF_COIP = _
    # Datakit protocol
    AF_DATAKIT = _
    # DECnet protocol
    AF_DEC = _
    # DEC Direct Data Link Interface protocol
    AF_DLI = _
    # CCITT (ITU-T) E.164 recommendation
    AF_E164 = _
    # European Computer Manufacturers protocols
    AF_ECMA = _
    # NSC Hyperchannel protocol
    AF_HYLINK = _
    # ARPANET IMP protocol
    AF_IMPLINK = _
    # IPv4 protocol
    AF_INET = _
    # IPv6 protocol
    AF_INET6 = _
    # IPX protocol
    AF_IPX = _
    # Integrated Services Digital Network
    AF_ISDN = _
    # ISO Open Systems Interconnection protocols
    AF_ISO = _
    # Local Area Transport protocol
    AF_LAT = _
    # Link layer interface
    AF_LINK = _
    # Host-internal protocols
    AF_LOCAL = _
    # Maximum address family for this platform
    AF_MAX = _
    # Native ATM access
    AF_NATM = _
    # Network driver raw access
    AF_NDRV = _
    # NetBIOS
    AF_NETBIOS = _
    # Netgraph sockets
    AF_NETGRAPH = _
    # XEROX NS protocols
    AF_NS = _
    # ISO Open Systems Interconnection protocols
    AF_OSI = _
    # Direct link-layer access
    AF_PACKET = _
    # Point-to-Point Protocol
    AF_PPP = _
    # PARC Universal Packet protocol
    AF_PUP = _
    # Internal routing protocol
    AF_ROUTE = _
    # Simple Internet Protocol
    AF_SIP = _
    # IBM SNA protocol
    AF_SNA = _
    AF_SYSTEM = _
    # UNIX sockets
    AF_UNIX = _
    # Unspecified protocol, any supported address family
    AF_UNSPEC = _
    # Accept only if any address is assigned
    AI_ADDRCONFIG = _
    # Allow all addresses
    AI_ALL = _
    # Fill in the canonical name
    AI_CANONNAME = _
    # Default flags for getaddrinfo
    AI_DEFAULT = _
    # Valid flag mask for getaddrinfo (not for application use)
    AI_MASK = _
    # Prevent host name resolution
    AI_NUMERICHOST = _
    # Prevent service name resolution
    AI_NUMERICSERV = _
    # Get address to use with bind()
    AI_PASSIVE = _
    # Accept IPv4-mapped IPv6 addresses
    AI_V4MAPPED = _
    # Accept IPv4 mapped addresses if the kernel supports it
    AI_V4MAPPED_CFG = _
    # Address family for hostname not supported
    EAI_ADDRFAMILY = _
    # Temporary failure in name resolution
    EAI_AGAIN = _
    # Invalid flags
    EAI_BADFLAGS = _
    # Invalid value for hints
    EAI_BADHINTS = _
    # Non-recoverable failure in name resolution
    EAI_FAIL = _
    # Address family not supported
    EAI_FAMILY = _
    # Maximum error code from getaddrinfo
    EAI_MAX = _
    # Memory allocation failure
    EAI_MEMORY = _
    # No address associated with hostname
    EAI_NODATA = _
    # Hostname nor servname, or not known
    EAI_NONAME = _
    # Argument buffer overflow
    EAI_OVERFLOW = _
    # Resolved protocol is unknown
    EAI_PROTOCOL = _
    # Servname not supported for socket type
    EAI_SERVICE = _
    # Socket type not supported
    EAI_SOCKTYPE = _
    # System error returned in errno
    EAI_SYSTEM = _
    # 802.1Q VLAN device
    IFF_802_1Q_VLAN = _
    # receive all multicast packets
    IFF_ALLMULTI = _
    # use alternate physical connection
    IFF_ALTPHYS = _
    # auto media select active
    IFF_AUTOMEDIA = _
    # bonding master or slave
    IFF_BONDING = _
    # device used as bridge port
    IFF_BRIDGE_PORT = _
    # broadcast address valid
    IFF_BROADCAST = _
    # flags not changeable
    IFF_CANTCHANGE = _
    # unconfigurable using ioctl(2)
    IFF_CANTCONFIG = _
    # turn on debugging
    IFF_DEBUG = _
    # disable netpoll at run-time
    IFF_DISABLE_NETPOLL = _
    # disallow bridging this ether dev
    IFF_DONT_BRIDGE = _
    # driver signals dormant
    IFF_DORMANT = _
    # tx hardware queue is full
    IFF_DRV_OACTIVE = _
    # resources allocated
    IFF_DRV_RUNNING = _
    # interface is winding down
    IFF_DYING = _
    # dialup device with changing addresses
    IFF_DYNAMIC = _
    # ethernet bridging device
    IFF_EBRIDGE = _
    # echo sent packets
    IFF_ECHO = _
    # ISATAP interface (RFC4214)
    IFF_ISATAP = _
    # per link layer defined bit 0
    IFF_LINK0 = _
    # per link layer defined bit 1
    IFF_LINK1 = _
    # per link layer defined bit 2
    IFF_LINK2 = _
    # hardware address change when it's running
    IFF_LIVE_ADDR_CHANGE = _
    # loopback net
    IFF_LOOPBACK = _
    # driver signals L1 up
    IFF_LOWER_UP = _
    # device used as macvlan port
    IFF_MACVLAN_PORT = _
    # master of a load balancer
    IFF_MASTER = _
    # bonding master, 802.3ad.
    IFF_MASTER_8023AD = _
    # bonding master, balance-alb.
    IFF_MASTER_ALB = _
    # bonding master, ARP mon in use
    IFF_MASTER_ARPMON = _
    # user-requested monitor mode
    IFF_MONITOR = _
    # supports multicast
    IFF_MULTICAST = _
    # no address resolution protocol
    IFF_NOARP = _
    # avoid use of trailers
    IFF_NOTRAILERS = _
    # transmission in progress
    IFF_OACTIVE = _
    # device used as Open vSwitch datapath port
    IFF_OVS_DATAPATH = _
    # point-to-point link
    IFF_POINTOPOINT = _
    # can set media type
    IFF_PORTSEL = _
    # user-requested promisc mode
    IFF_PPROMISC = _
    # receive all packets
    IFF_PROMISC = _
    # interface is being renamed
    IFF_RENAMING = _
    # routing entry installed
    IFF_ROUTE = _
    # resources allocated
    IFF_RUNNING = _
    # can't hear own transmissions
    IFF_SIMPLEX = _
    # slave of a load balancer
    IFF_SLAVE = _
    # bonding slave not the curr. active
    IFF_SLAVE_INACTIVE = _
    # need ARPs for validation
    IFF_SLAVE_NEEDARP = _
    # interface manages own routes
    IFF_SMART = _
    # static ARP
    IFF_STATICARP = _
    # sending custom FCS
    IFF_SUPP_NOFCS = _
    # used as team port
    IFF_TEAM_PORT = _
    # sharing skbs on transmit
    IFF_TX_SKB_SHARING = _
    # unicast filtering
    IFF_UNICAST_FLT = _
    # interface is up
    IFF_UP = _
    # volatile flags
    IFF_VOLATILE = _
    # WAN HDLC device
    IFF_WAN_HDLC = _
    # dev_hard_start_xmit() is allowed to release skb->dst
    IFF_XMIT_DST_RELEASE = _
    # Maximum interface name size
    IFNAMSIZ = _
    # Maximum interface name size
    IF_NAMESIZE = _
    # Multicast group for all systems on this subset
    INADDR_ALLHOSTS_GROUP = _
    # A socket bound to INADDR_ANY receives packets from all interfaces and sends from the default IP address
    INADDR_ANY = _
    # The network broadcast address
    INADDR_BROADCAST = _
    # The loopback address
    INADDR_LOOPBACK = _
    # The last local network multicast group
    INADDR_MAX_LOCAL_GROUP = _
    # A bitmask for matching no valid IP address
    INADDR_NONE = _
    # The reserved multicast group
    INADDR_UNSPEC_GROUP = _
    # Maximum length of an IPv6 address string
    INET6_ADDRSTRLEN = _
    # Maximum length of an IPv4 address string
    INET_ADDRSTRLEN = _
    # Default minimum address for bind or connect
    IPPORT_RESERVED = _
    # Default maximum address for bind or connect
    IPPORT_USERRESERVED = _
    # IP6 auth header
    IPPROTO_AH = _
    IPPROTO_BIP = _
    # IP6 destination option
    IPPROTO_DSTOPTS = _
    # Exterior Gateway Protocol
    IPPROTO_EGP = _
    # ISO cnlp
    IPPROTO_EON = _
    # IP6 Encapsulated Security Payload
    IPPROTO_ESP = _
    # IP6 fragmentation header
    IPPROTO_FRAGMENT = _
    # Gateway to Gateway Protocol
    IPPROTO_GGP = _
    # "hello" routing protocol
    IPPROTO_HELLO = _
    # IP6 hop-by-hop options
    IPPROTO_HOPOPTS = _
    # Control message protocol
    IPPROTO_ICMP = _
    # ICMP6
    IPPROTO_ICMPV6 = _
    # XNS IDP
    IPPROTO_IDP = _
    # Group Management Protocol
    IPPROTO_IGMP = _
    # Dummy protocol for IP
    IPPROTO_IP = _
    # IP6 header
    IPPROTO_IPV6 = _
    # Maximum IPPROTO constant
    IPPROTO_MAX = _
    # Sun net disk protocol
    IPPROTO_ND = _
    # IP6 no next header
    IPPROTO_NONE = _
    # PARC Universal Packet protocol
    IPPROTO_PUP = _
    # Raw IP packet
    IPPROTO_RAW = _
    # IP6 routing header
    IPPROTO_ROUTING = _
    # TCP
    IPPROTO_TCP = _
    # ISO transport protocol class 4
    IPPROTO_TP = _
    # UDP
    IPPROTO_UDP = _
    # Xpress Transport Protocol
    IPPROTO_XTP = _
    # Checksum offset for raw sockets
    IPV6_CHECKSUM = _
    # Don't fragment packets
    IPV6_DONTFRAG = _
    # Destination option
    IPV6_DSTOPTS = _
    # Hop limit
    IPV6_HOPLIMIT = _
    # Hop-by-hop option
    IPV6_HOPOPTS = _
    # Join a group membership
    IPV6_JOIN_GROUP = _
    # Leave a group membership
    IPV6_LEAVE_GROUP = _
    # IP6 multicast hops
    IPV6_MULTICAST_HOPS = _
    # IP6 multicast interface
    IPV6_MULTICAST_IF = _
    # IP6 multicast loopback
    IPV6_MULTICAST_LOOP = _
    # Next hop address
    IPV6_NEXTHOP = _
    # Retrieve current path MTU
    IPV6_PATHMTU = _
    # Receive packet information with datagram
    IPV6_PKTINFO = _
    # Receive all IP6 options for response
    IPV6_RECVDSTOPTS = _
    # Receive hop limit with datagram
    IPV6_RECVHOPLIMIT = _
    # Receive hop-by-hop options
    IPV6_RECVHOPOPTS = _
    # Receive current path MTU with datagram
    IPV6_RECVPATHMTU = _
    # Receive destination IP address and incoming interface
    IPV6_RECVPKTINFO = _
    # Receive routing header
    IPV6_RECVRTHDR = _
    # Receive traffic class
    IPV6_RECVTCLASS = _
    # Allows removal of sticky routing headers
    IPV6_RTHDR = _
    # Allows removal of sticky destination options header
    IPV6_RTHDRDSTOPTS = _
    # Routing header type 0
    IPV6_RTHDR_TYPE_0 = _
    # Specify the traffic class
    IPV6_TCLASS = _
    # IP6 unicast hops
    IPV6_UNICAST_HOPS = _
    # Use the minimum MTU size
    IPV6_USE_MIN_MTU = _
    # Only bind IPv6 with a wildcard bind
    IPV6_V6ONLY = _
    IPX_TYPE = _
    # Add a multicast group membership
    IP_ADD_MEMBERSHIP = _
    # Add a multicast group membership
    IP_ADD_SOURCE_MEMBERSHIP = _
    # Block IPv4 multicast packets with a give source address
    IP_BLOCK_SOURCE = _
    # Default multicast loopback
    IP_DEFAULT_MULTICAST_LOOP = _
    # Default multicast TTL
    IP_DEFAULT_MULTICAST_TTL = _
    # Don't fragment packets
    IP_DONTFRAG = _
    # Drop a multicast group membership
    IP_DROP_MEMBERSHIP = _
    # Drop a multicast group membership
    IP_DROP_SOURCE_MEMBERSHIP = _
    # Allow binding to nonexistent IP addresses
    IP_FREEBIND = _
    # Header is included with data
    IP_HDRINCL = _
    # IPsec security policy
    IP_IPSEC_POLICY = _
    # Maximum number multicast groups a socket can join
    IP_MAX_MEMBERSHIPS = _
    # Minimum TTL allowed for received packets
    IP_MINTTL = _
    # Multicast source filtering
    IP_MSFILTER = _
    # The Maximum Transmission Unit of the socket
    IP_MTU = _
    # Path MTU discovery
    IP_MTU_DISCOVER = _
    # IP multicast interface
    IP_MULTICAST_IF = _
    # IP multicast loopback
    IP_MULTICAST_LOOP = _
    # IP multicast TTL
    IP_MULTICAST_TTL = _
    # Force outgoing broadcast datagrams to have the undirected broadcast address
    IP_ONESBCAST = _
    # IP options to be included in packets
    IP_OPTIONS = _
    # Retrieve security context with datagram
    IP_PASSSEC = _
    # Receive packet information with datagrams
    IP_PKTINFO = _
    # Receive packet options with datagrams
    IP_PKTOPTIONS = _
    # Always send DF frames
    IP_PMTUDISC_DO = _
    # Never send DF frames
    IP_PMTUDISC_DONT = _
    # Use per-route hints
    IP_PMTUDISC_WANT = _
    # Set the port range for sockets with unspecified port numbers
    IP_PORTRANGE = _
    # Receive IP destination address with datagram
    IP_RECVDSTADDR = _
    # Enable extended reliable error message passing
    IP_RECVERR = _
    # Receive interface information with datagrams
    IP_RECVIF = _
    # Receive all IP options with datagram
    IP_RECVOPTS = _
    # Receive all IP options for response
    IP_RECVRETOPTS = _
    # Receive link-layer address with datagrams
    IP_RECVSLLA = _
    # Receive TOS with incoming packets
    IP_RECVTOS = _
    # Receive IP TTL with datagrams
    IP_RECVTTL = _
    # IP options to be included in datagrams
    IP_RETOPTS = _
    # Notify transit routers to more closely examine the contents of an IP packet
    IP_ROUTER_ALERT = _
    # Source address for outgoing UDP datagrams
    IP_SENDSRCADDR = _
    # IP type-of-service
    IP_TOS = _
    # Transparent proxy
    IP_TRANSPARENT = _
    # IP time-to-live
    IP_TTL = _
    # Unblock IPv4 multicast packets with a give source address
    IP_UNBLOCK_SOURCE = _
    IP_XFRM_POLICY = _
    # Connect blocks until accepted
    LOCAL_CONNWAIT = _
    # Pass credentials to receiver
    LOCAL_CREDS = _
    # Retrieve peer credentials
    LOCAL_PEERCRED = _
    # Block multicast packets from this source
    MCAST_BLOCK_SOURCE = _
    # Exclusive multicast source filter
    MCAST_EXCLUDE = _
    # Inclusive multicast source filter
    MCAST_INCLUDE = _
    # Join a multicast group
    MCAST_JOIN_GROUP = _
    # Join a multicast source group
    MCAST_JOIN_SOURCE_GROUP = _
    # Leave a multicast group
    MCAST_LEAVE_GROUP = _
    # Leave a multicast source group
    MCAST_LEAVE_SOURCE_GROUP = _
    # Multicast source filtering
    MCAST_MSFILTER = _
    # Unblock multicast packets from this source
    MCAST_UNBLOCK_SOURCE = _
    # End of record
    MSG_COMPAT = _
    # Confirm path validity
    MSG_CONFIRM = _
    # Control data lost before delivery
    MSG_CTRUNC = _
    # Send without using the routing tables
    MSG_DONTROUTE = _
    # This message should be non-blocking
    MSG_DONTWAIT = _
    # Data completes connection
    MSG_EOF = _
    # Data completes record
    MSG_EOR = _
    # Fetch message from error queue
    MSG_ERRQUEUE = _
    # Reduce step of the handshake process
    MSG_FASTOPEN = _
    MSG_FIN = _
    # Start of a hold sequence.  Dumps to so_temp
    MSG_FLUSH = _
    # Data ready to be read
    MSG_HAVEMORE = _
    # Hold fragment in so_temp
    MSG_HOLD = _
    # Sender will send more
    MSG_MORE = _
    # Do not generate SIGPIPE
    MSG_NOSIGNAL = _
    # Process out-of-band data
    MSG_OOB = _
    # Peek at incoming message
    MSG_PEEK = _
    # Wait for full request
    MSG_PROXY = _
    # Data remains in the current packet
    MSG_RCVMORE = _
    MSG_RST = _
    # Send the packet in so_temp
    MSG_SEND = _
    MSG_SYN = _
    # Data discarded before delivery
    MSG_TRUNC = _
    # Wait for full request or error
    MSG_WAITALL = _
    # The service specified is a datagram service (looks up UDP ports)
    NI_DGRAM = _
    # Maximum length of a hostname
    NI_MAXHOST = _
    # Maximum length of a service name
    NI_MAXSERV = _
    # A name is required
    NI_NAMEREQD = _
    # An FQDN is not required for local hosts, return only the local part
    NI_NOFQDN = _
    # Return a numeric address
    NI_NUMERICHOST = _
    # Return the service name as a digit string
    NI_NUMERICSERV = _
    # AppleTalk protocol
    PF_APPLETALK = _
    # Asynchronous Transfer Mode
    PF_ATM = _
    # AX.25 protocol
    PF_AX25 = _
    # CCITT (now ITU-T) protocols
    PF_CCITT = _
    # MIT CHAOS protocols
    PF_CHAOS = _
    # Computer Network Technology
    PF_CNT = _
    # Connection-oriented IP
    PF_COIP = _
    # Datakit protocol
    PF_DATAKIT = _
    # DECnet protocol
    PF_DEC = _
    # DEC Direct Data Link Interface protocol
    PF_DLI = _
    # European Computer Manufacturers protocols
    PF_ECMA = _
    # NSC Hyperchannel protocol
    PF_HYLINK = _
    # ARPANET IMP protocol
    PF_IMPLINK = _
    # IPv4 protocol
    PF_INET = _
    # IPv6 protocol
    PF_INET6 = _
    # IPX protocol
    PF_IPX = _
    # Integrated Services Digital Network
    PF_ISDN = _
    # ISO Open Systems Interconnection protocols
    PF_ISO = _
    PF_KEY = _
    # Local Area Transport protocol
    PF_LAT = _
    # Link layer interface
    PF_LINK = _
    # Host-internal protocols
    PF_LOCAL = _
    # Maximum address family for this platform
    PF_MAX = _
    # Native ATM access
    PF_NATM = _
    # Network driver raw access
    PF_NDRV = _
    # NetBIOS
    PF_NETBIOS = _
    # Netgraph sockets
    PF_NETGRAPH = _
    # XEROX NS protocols
    PF_NS = _
    # ISO Open Systems Interconnection protocols
    PF_OSI = _
    # Direct link-layer access
    PF_PACKET = _
    PF_PIP = _
    # Point-to-Point Protocol
    PF_PPP = _
    # PARC Universal Packet protocol
    PF_PUP = _
    # Internal routing protocol
    PF_ROUTE = _
    PF_RTIP = _
    # Simple Internet Protocol
    PF_SIP = _
    # IBM SNA protocol
    PF_SNA = _
    PF_SYSTEM = _
    # UNIX sockets
    PF_UNIX = _
    # Unspecified protocol, any supported address family
    PF_UNSPEC = _
    # eXpress Transfer Protocol
    PF_XTP = _
    # Timestamp (bintime)
    SCM_BINTIME = _
    # The sender's credentials
    SCM_CREDENTIALS = _
    # Process credentials
    SCM_CREDS = _
    # Access rights
    SCM_RIGHTS = _
    # Timestamp (timeval)
    SCM_TIMESTAMP = _
    # Timestamp (timespec list) (Linux 2.6.30)
    SCM_TIMESTAMPING = _
    # Timespec (timespec)
    SCM_TIMESTAMPNS = _
    # User credentials
    SCM_UCRED = _
    # Wifi status (Linux 3.3)
    SCM_WIFI_STATUS = _
    # Shut down the reading side of the socket
    SHUT_RD = _
    # Shut down the both sides of the socket
    SHUT_RDWR = _
    # Shut down the writing side of the socket
    SHUT_WR = _
    # A datagram socket provides connectionless, unreliable messaging
    SOCK_DGRAM = _
    # Device-level packet access
    SOCK_PACKET = _
    # A raw socket provides low-level access for direct access or implementing network protocols
    SOCK_RAW = _
    # A reliable datagram socket provides reliable delivery of messages
    SOCK_RDM = _
    # A sequential packet socket provides sequenced, reliable two-way connection for datagrams
    SOCK_SEQPACKET = _
    # A stream socket provides a sequenced, reliable two-way connection for a byte stream
    SOCK_STREAM = _
    # AppleTalk socket options
    SOL_ATALK = _
    # AX.25 socket options
    SOL_AX25 = _
    # IP socket options
    SOL_IP = _
    # IPX socket options
    SOL_IPX = _
    # Socket-level options
    SOL_SOCKET = _
    # TCP socket options
    SOL_TCP = _
    # UDP socket options
    SOL_UDP = _
    # Maximum connection requests that may be queued for a socket
    SOMAXCONN = _
    # Background socket priority
    SOPRI_BACKGROUND = _
    # Interactive socket priority
    SOPRI_INTERACTIVE = _
    # Normal socket priority
    SOPRI_NORMAL = _
    # Socket has had listen() called on it
    SO_ACCEPTCONN = _
    # There is an accept filter
    SO_ACCEPTFILTER = _
    # Bypass zone boundaries
    SO_ALLZONES = _
    # Attach an accept filter
    SO_ATTACH_FILTER = _
    # Only send packets from the given interface
    SO_BINDTODEVICE = _
    # Receive timestamp with datagrams (bintime)
    SO_BINTIME = _
    # Query supported BPF extensions (Linux 3.14)
    SO_BPF_EXTENSIONS = _
    # Permit sending of broadcast messages
    SO_BROADCAST = _
    # Set the threshold in microseconds for low latency polling (Linux 3.11)
    SO_BUSY_POLL = _
    # Debug info recording
    SO_DEBUG = _
    # Detach an accept filter
    SO_DETACH_FILTER = _
    # Domain given for socket() (Linux 2.6.32)
    SO_DOMAIN = _
    # Use interface addresses
    SO_DONTROUTE = _
    # Retain unread data
    SO_DONTTRUNC = _
    # Get and clear the error status
    SO_ERROR = _
    # Obtain filter set by SO_ATTACH_FILTER (Linux 3.8)
    SO_GET_FILTER = _
    # Keep connections alive
    SO_KEEPALIVE = _
    # Linger on close if data is present
    SO_LINGER = _
    # Lock the filter attached to a socket (Linux 3.9)
    SO_LOCK_FILTER = _
    # Mandatory Access Control exemption for unlabeled peers
    SO_MAC_EXEMPT = _
    # Set the mark for mark-based routing (Linux 2.6.25)
    SO_MARK = _
    # Cap the rate computed by transport layer. [bytes per second] (Linux 3.13)
    SO_MAX_PACING_RATE = _
    # Install socket-level Network Kernel Extension
    SO_NKE = _
    # Set netns of a socket (Linux 3.4)
    SO_NOFCS = _
    # Don't SIGPIPE on EPIPE
    SO_NOSIGPIPE = _
    # Disable checksums
    SO_NO_CHECK = _
    # Get first packet byte count
    SO_NREAD = _
    # Leave received out-of-band data in-line
    SO_OOBINLINE = _
    # Receive SCM_CREDENTIALS messages
    SO_PASSCRED = _
    # Toggle security context passing (Linux 2.6.18)
    SO_PASSSEC = _
    # Set the peek offset (Linux 3.4)
    SO_PEEK_OFF = _
    # The credentials of the foreign process connected to this socket
    SO_PEERCRED = _
    # Name of the connecting user
    SO_PEERNAME = _
    # Obtain the security credentials (Linux 2.6.2)
    SO_PEERSEC = _
    # The protocol-defined priority for all packets on this socket
    SO_PRIORITY = _
    # Protocol given for socket() (Linux 2.6.32)
    SO_PROTOCOL = _
    # Receive buffer size
    SO_RCVBUF = _
    # Receive buffer size without rmem_max limit (Linux 2.6.14)
    SO_RCVBUFFORCE = _
    # Receive low-water mark
    SO_RCVLOWAT = _
    # Receive timeout
    SO_RCVTIMEO = _
    # Receive user credentials with datagram
    SO_RECVUCRED = _
    # Allow local address reuse
    SO_REUSEADDR = _
    # Allow local address and port reuse
    SO_REUSEPORT = _
    # Toggle cmsg for number of packets dropped (Linux 2.6.33)
    SO_RXQ_OVFL = _
    SO_SECURITY_AUTHENTICATION = _
    SO_SECURITY_ENCRYPTION_NETWORK = _
    SO_SECURITY_ENCRYPTION_TRANSPORT = _
    # Make select() detect socket error queue with errorfds (Linux 3.10)
    SO_SELECT_ERR_QUEUE = _
    # Send buffer size
    SO_SNDBUF = _
    # Send buffer size without wmem_max limit (Linux 2.6.14)
    SO_SNDBUFFORCE = _
    # Send low-water mark
    SO_SNDLOWAT = _
    # Send timeout
    SO_SNDTIMEO = _
    # Receive timestamp with datagrams (timeval)
    SO_TIMESTAMP = _
    # Time stamping of incoming and outgoing packets (Linux 2.6.30)
    SO_TIMESTAMPING = _
    # Receive nanosecond timestamp with datagrams (timespec)
    SO_TIMESTAMPNS = _
    # Get the socket type
    SO_TYPE = _
    # Bypass hardware when possible
    SO_USELOOPBACK = _
    # Give a hint when more data is ready
    SO_WANTMORE = _
    # OOB data is wanted in MSG_FLAG on receive
    SO_WANTOOBFLAG = _
    # Toggle cmsg for wifi status (Linux 3.3)
    SO_WIFI_STATUS = _
    # TCP congestion control algorithm (Linux 2.6.13, glibc 2.6)
    TCP_CONGESTION = _
    # TCP Cookie Transactions (Linux 2.6.33, glibc 2.18)
    TCP_COOKIE_TRANSACTIONS = _
    # Don't send partial frames (Linux 2.2, glibc 2.2)
    TCP_CORK = _
    # Don't notify a listening socket until data is ready (Linux 2.4, glibc 2.2)
    TCP_DEFER_ACCEPT = _
    # Reduce step of the handshake process (Linux 3.7, glibc 2.18)
    TCP_FASTOPEN = _
    # Retrieve information about this socket (Linux 2.4, glibc 2.2)
    TCP_INFO = _
    # Maximum number of keepalive probes allowed before dropping a connection (Linux 2.4, glibc 2.2)
    TCP_KEEPCNT = _
    # Idle time before keepalive probes are sent (Linux 2.4, glibc 2.2)
    TCP_KEEPIDLE = _
    # Time between keepalive probes (Linux 2.4, glibc 2.2)
    TCP_KEEPINTVL = _
    # Lifetime of orphaned FIN_WAIT2 sockets (Linux 2.4, glibc 2.2)
    TCP_LINGER2 = _
    # Set maximum segment size
    TCP_MAXSEG = _
    # Use MD5 digests (RFC2385, Linux 2.6.20, glibc 2.7)
    TCP_MD5SIG = _
    # Don't delay sending to coalesce packets
    TCP_NODELAY = _
    # Don't use TCP options
    TCP_NOOPT = _
    # Don't push the last block of write
    TCP_NOPUSH = _
    # Sequence of a queue for repair mode (Linux 3.5, glibc 2.18)
    TCP_QUEUE_SEQ = _
    # Enable quickack mode (Linux 2.4.4, glibc 2.3)
    TCP_QUICKACK = _
    # Repair mode (Linux 3.5, glibc 2.18)
    TCP_REPAIR = _
    # Options for repair mode (Linux 3.5, glibc 2.18)
    TCP_REPAIR_OPTIONS = _
    # Queue for repair mode (Linux 3.5, glibc 2.18)
    TCP_REPAIR_QUEUE = _
    # Number of SYN retransmits before a connection is dropped (Linux 2.4, glibc 2.2)
    TCP_SYNCNT = _
    # Duplicated acknowledgments handling for thin-streams (Linux 2.6.34, glibc 2.18)
    TCP_THIN_DUPACK = _
    # Linear timeouts for thin-streams (Linux 2.6.34, glibc 2.18)
    TCP_THIN_LINEAR_TIMEOUTS = _
    # TCP timestamp (Linux 3.9, glibc 2.18)
    TCP_TIMESTAMP = _
    # Max timeout before a TCP connection is aborted (Linux 2.6.37, glibc 2.18)
    TCP_USER_TIMEOUT = _
    # Clamp the size of the advertised window (Linux 2.4, glibc 2.2)
    TCP_WINDOW_CLAMP = _
    # Don't send partial frames (Linux 2.5.44, glibc 2.11)
    UDP_CORK = _
  end

  # Socket::Ifaddr represents a result of getifaddrs() function.
  class Ifaddr < Data
    # Returns the address of _ifaddr_.
    # nil is returned if address is not available in _ifaddr_.
    def addr; end

    # Returns the broadcast address of _ifaddr_.
    # nil is returned if the flags doesn't have IFF_BROADCAST.
    def broadaddr; end

    # Returns the destination address of _ifaddr_.
    # nil is returned if the flags doesn't have IFF_POINTOPOINT.
    def dstaddr; end

    # Returns the flags of _ifaddr_.
    def flags; end

    # Returns the interface index of _ifaddr_.
    def ifindex; end

    # Returns a string to show contents of _ifaddr_.
    def inspect; end

    # Returns the interface name of _ifaddr_.
    def name; end

    # Returns the netmask address of _ifaddr_.
    # nil is returned if netmask is not available in _ifaddr_.
    def netmask; end

    # Returns the vhid address of _ifaddr_.
    # nil is returned if there is no vhid.
    def vhid; end
  end

  # Socket::Option represents a socket option used by
  # BasicSocket#getsockopt and BasicSocket#setsockopt.  A socket option
  # contains the socket #family, protocol #level, option name #optname and
  # option value #data.
  class Option
    # Creates a new Socket::Option object which contains boolean as data.
    # Actually 0 or 1 as int is used.
    #
    #   require 'socket'
    #
    #   p Socket::Option.bool(:INET, :SOCKET, :KEEPALIVE, true)
    #   #=> #<Socket::Option: INET SOCKET KEEPALIVE 1>
    #
    #   p Socket::Option.bool(:INET, :SOCKET, :KEEPALIVE, false)
    #   #=> #<Socket::Option: AF_INET SOCKET KEEPALIVE 0>
    def self.bool(family, level, optname, bool) end

    # Creates a new Socket::Option object which contains a byte as data.
    #
    #   p Socket::Option.byte(:INET, :SOCKET, :KEEPALIVE, 1)
    #   #=> #<Socket::Option: INET SOCKET KEEPALIVE 1>
    def self.byte(family, level, optname, integer) end

    # Creates a new Socket::Option object which contains an int as data.
    #
    # The size and endian is dependent on the platform.
    #
    #   p Socket::Option.int(:INET, :SOCKET, :KEEPALIVE, 1)
    #   #=> #<Socket::Option: INET SOCKET KEEPALIVE 1>
    def self.int(family, level, optname, integer) end

    # Creates a new Socket::Option object for IP_MULTICAST_LOOP.
    #
    # The size is dependent on the platform.
    #
    #   sockopt = Socket::Option.int(:INET, :IPPROTO_IP, :IP_MULTICAST_LOOP, 1)
    #   p sockopt.int => 1
    #
    #   p Socket::Option.ipv4_multicast_loop(10)
    #   #=> #<Socket::Option: INET IP MULTICAST_LOOP 10>
    def self.ipv4_multicast_loop(integer) end

    # Creates a new Socket::Option object for IP_MULTICAST_TTL.
    #
    # The size is dependent on the platform.
    #
    #   p Socket::Option.ipv4_multicast_ttl(10)
    #   #=> #<Socket::Option: INET IP MULTICAST_TTL 10>
    def self.ipv4_multicast_ttl(integer) end

    # Creates a new Socket::Option object for SOL_SOCKET/SO_LINGER.
    #
    # _onoff_ should be an integer or a boolean.
    #
    # _secs_ should be the number of seconds.
    #
    #   p Socket::Option.linger(true, 10)
    #   #=> #<Socket::Option: UNSPEC SOCKET LINGER on 10sec>
    def self.linger(onoff, secs) end

    # Returns a new Socket::Option object.
    #
    #   sockopt = Socket::Option.new(:INET, :SOCKET, :KEEPALIVE, [1].pack("i"))
    #   p sockopt #=> #<Socket::Option: INET SOCKET KEEPALIVE 1>
    def initialize(family, level, optname, data) end

    # Returns the data in _sockopt_ as an boolean value.
    #
    #   sockopt = Socket::Option.int(:INET, :SOCKET, :KEEPALIVE, 1)
    #   p sockopt.bool => true
    def bool; end

    # Returns the data in _sockopt_ as an byte.
    #
    #   sockopt = Socket::Option.byte(:INET, :SOCKET, :KEEPALIVE, 1)
    #   p sockopt.byte => 1
    def byte; end

    # returns the socket option data as a string.
    #
    #   p Socket::Option.new(:INET6, :IPV6, :RECVPKTINFO, [1].pack("i!")).data
    #   #=> "\x01\x00\x00\x00"
    def data; end
    alias to_s data

    # returns the socket family as an integer.
    #
    #   p Socket::Option.new(:INET6, :IPV6, :RECVPKTINFO, [1].pack("i!")).family
    #   #=> 10
    def family; end

    # Returns a string which shows sockopt in human-readable form.
    #
    #   p Socket::Option.new(:INET, :SOCKET, :KEEPALIVE, [1].pack("i")).inspect
    #   #=> "#<Socket::Option: INET SOCKET KEEPALIVE 1>"
    def inspect; end

    # Returns the data in _sockopt_ as an int.
    #
    # The size and endian is dependent on the platform.
    #
    #   sockopt = Socket::Option.int(:INET, :SOCKET, :KEEPALIVE, 1)
    #   p sockopt.int => 1
    def int; end

    # Returns the ipv4_multicast_loop data in _sockopt_ as an integer.
    #
    #   sockopt = Socket::Option.ipv4_multicast_loop(10)
    #   p sockopt.ipv4_multicast_loop => 10
    def ipv4_multicast_loop; end

    # Returns the ipv4_multicast_ttl data in _sockopt_ as an integer.
    #
    #   sockopt = Socket::Option.ipv4_multicast_ttl(10)
    #   p sockopt.ipv4_multicast_ttl => 10
    def ipv4_multicast_ttl; end

    # returns the socket level as an integer.
    #
    #   p Socket::Option.new(:INET6, :IPV6, :RECVPKTINFO, [1].pack("i!")).level
    #   #=> 41
    def level; end

    # Returns the linger data in _sockopt_ as a pair of boolean and integer.
    #
    #   sockopt = Socket::Option.linger(true, 10)
    #   p sockopt.linger => [true, 10]
    def linger; end

    # returns the socket option name as an integer.
    #
    #   p Socket::Option.new(:INET6, :IPV6, :RECVPKTINFO, [1].pack("i!")).optname
    #   #=> 2
    def optname; end

    # Calls String#unpack on sockopt.data.
    #
    #   sockopt = Socket::Option.new(:INET, :SOCKET, :KEEPALIVE, [1].pack("i"))
    #   p sockopt.unpack("i")      #=> [1]
    #   p sockopt.data.unpack("i") #=> [1]
    def unpack(template) end
  end
end
