# frozen_string_literal: true

# Class +Socket+ provides access to the underlying operating system
# socket implementations.  It can be used to provide more operating system
# specific functionality than the protocol-specific socket classes.
#
# The constants defined under Socket::Constants are also defined under
# Socket.  For example, Socket::AF_INET is usable as well as
# Socket::Constants::AF_INET.  See Socket::Constants for the list of
# constants.
#
# === What's a socket?
#
# Sockets are endpoints of a bidirectionnal communication channel.
# Sockets can communicate within a process, between processes on the same
# machine or between different machines.  There are many types of socket:
# TCPSocket, UDPSocket or UNIXSocket for example.
#
# Sockets have their own vocabulary:
#
# *domain:*
# The family of protocols:
# *    Socket::PF_INET
# *    Socket::PF_INET6
# *    Socket::PF_UNIX
# *    etc.
#
# *type:*
# The type of communications between the two endpoints, typically
# *    Socket::SOCK_STREAM
# *    Socket::SOCK_DGRAM.
#
# *protocol:*
# Typically _zero_.
# This may be used to identify a variant of a protocol.
#
# *hostname:*
# The identifier of a network interface:
# *    a string (hostname, IPv4 or IPv6 adress or +broadcast+
#      which specifies a broadcast address)
# *    a zero-length string which specifies INADDR_ANY
# *    an integer (interpreted as binary address in host byte order).
#
# === Quick start
#
# Many of the classes, such as TCPSocket, UDPSocket or UNIXSocket,
# ease the use of sockets comparatively to the equivalent C programming interface.
#
# Let's create an internet socket using the IPv4 protocol in a C-like manner:
#
#   s = Socket.new Socket::AF_INET, Socket::SOCK_STREAM
#   s.connect Socket.pack_sockaddr_in(80, 'example.com')
#
# You could also use the TCPSocket class:
#
#   s = TCPSocket.new 'example.com', 80
#
# A simple server might look like this:
#
#   require 'socket'
#
#   server = TCPServer.new 2000 # Server bound to port 2000
#
#   loop do
#     client = server.accept    # Wait for a client to connect
#     client.puts "Hello !"
#     client.puts "Time is #{Time.now}"
#     client.close
#   end
#
# A simple client may look like this:
#
#   require 'socket'
#
#   s = TCPSocket.new 'localhost', 2000
#
#   while line = s.gets # Read lines from socket
#     puts line         # and print them
#   end
#
#   s.close             # close socket when done
#
# === Exception Handling
#
# Ruby's Socket implementation raises exceptions based on the error
# generated by the system dependent implementation.  This is why the
# methods are documented in a way that isolate Unix-based system
# exceptions from Windows based exceptions. If more information on a
# particular exception is needed, please refer to the Unix manual pages or
# the Windows WinSock reference.
#
# === Convenience methods
#
# Although the general way to create socket is Socket.new,
# there are several methods of socket creation for most cases.
#
# TCP client socket::
#   Socket.tcp, TCPSocket.open
# TCP server socket::
#   Socket.tcp_server_loop, TCPServer.open
# UNIX client socket::
#   Socket.unix, UNIXSocket.open
# UNIX server socket::
#   Socket.unix_server_loop, UNIXServer.open
#
# === Documentation by
#
# * Zach Dennis
# * Sam Roberts
# * <em>Programming Ruby</em> from The Pragmatic Bookshelf.
#
# Much material in this documentation is taken with permission from
# <em>Programming Ruby</em> from The Pragmatic Bookshelf.
class Socket < BasicSocket
  # Obtains address information for _nodename_:_servname_.
  #
  # _family_ should be an address family such as: :INET, :INET6, :UNIX, etc.
  #
  # _socktype_ should be a socket type such as: :STREAM, :DGRAM, :RAW, etc.
  #
  # _protocol_ should be a protocol defined in the family,
  # and defaults to 0 for the family.
  #
  # _flags_ should be bitwise OR of Socket::AI_* constants.
  #
  #   Socket.getaddrinfo("www.ruby-lang.org", "http", nil, :STREAM)
  #   #=> [["AF_INET", 80, "carbon.ruby-lang.org", "221.186.184.68", 2, 1, 6]] # PF_INET/SOCK_STREAM/IPPROTO_TCP
  #
  #   Socket.getaddrinfo("localhost", nil)
  #   #=> [["AF_INET", 0, "localhost", "127.0.0.1", 2, 1, 6],  # PF_INET/SOCK_STREAM/IPPROTO_TCP
  #   #    ["AF_INET", 0, "localhost", "127.0.0.1", 2, 2, 17], # PF_INET/SOCK_DGRAM/IPPROTO_UDP
  #   #    ["AF_INET", 0, "localhost", "127.0.0.1", 2, 3, 0]]  # PF_INET/SOCK_RAW/IPPROTO_IP
  #
  # _reverse_lookup_ directs the form of the third element, and has to
  # be one of below.  If _reverse_lookup_ is omitted, the default value is +nil+.
  #
  #   +true+, +:hostname+:  hostname is obtained from numeric address using reverse lookup, which may take a time.
  #   +false+, +:numeric+:  hostname is same as numeric address.
  #   +nil+:              obey to the current +do_not_reverse_lookup+ flag.
  #
  # If Addrinfo object is preferred, use Addrinfo.getaddrinfo.
  def self.getaddrinfo(p1, p2, p3 = v3, p4 = v4, p5 = v5, p6 = v6, p7 = v7) end

  # Obtains the host information for _address_.
  #
  #   p Socket.gethostbyaddr([221,186,184,68].pack("CCCC"))
  #   #=> ["carbon.ruby-lang.org", [], 2, "\xDD\xBA\xB8D"]
  def self.gethostbyaddr(p1, p2 = v2) end

  # Obtains the host information for _hostname_.
  #
  #   p Socket.gethostbyname("hal") #=> ["localhost", ["hal"], 2, "\x7F\x00\x00\x01"]
  def self.gethostbyname(hostname) end

  # Returns the hostname.
  #
  #   p Socket.gethostname #=> "hal"
  #
  # Note that it is not guaranteed to be able to convert to IP address using gethostbyname, getaddrinfo, etc.
  # If you need local IP address, use Socket.ip_address_list.
  def self.gethostname; end

  # Returns an array of interface addresses.
  # An element of the array is an instance of Socket::Ifaddr.
  #
  # This method can be used to find multicast-enabled interfaces:
  #
  #   pp Socket.getifaddrs.reject {|ifaddr|
  #     !ifaddr.addr.ip? || (ifaddr.flags & Socket::IFF_MULTICAST == 0)
  #   }.map {|ifaddr| [ifaddr.name, ifaddr.ifindex, ifaddr.addr] }
  #   #=> [["eth0", 2, #<Addrinfo: 221.186.184.67>],
  #   #    ["eth0", 2, #<Addrinfo: fe80::216:3eff:fe95:88bb%eth0>]]
  #
  # Example result on GNU/Linux:
  #   pp Socket.getifaddrs
  #   #=> [#<Socket::Ifaddr lo UP,LOOPBACK,RUNNING,0x10000 PACKET[protocol=0 lo hatype=772 HOST hwaddr=00:00:00:00:00:00]>,
  #   #    #<Socket::Ifaddr eth0 UP,BROADCAST,RUNNING,MULTICAST,0x10000 PACKET[protocol=0 eth0 hatype=1 HOST hwaddr=00:16:3e:95:88:bb] broadcast=PACKET[protocol=0 eth0 hatype=1 HOST hwaddr=ff:ff:ff:ff:ff:ff]>,
  #   #    #<Socket::Ifaddr sit0 NOARP PACKET[protocol=0 sit0 hatype=776 HOST hwaddr=00:00:00:00]>,
  #   #    #<Socket::Ifaddr lo UP,LOOPBACK,RUNNING,0x10000 127.0.0.1 netmask=255.0.0.0>,
  #   #    #<Socket::Ifaddr eth0 UP,BROADCAST,RUNNING,MULTICAST,0x10000 221.186.184.67 netmask=255.255.255.240 broadcast=221.186.184.79>,
  #   #    #<Socket::Ifaddr lo UP,LOOPBACK,RUNNING,0x10000 ::1 netmask=ffff:ffff:ffff:ffff:ffff:ffff:ffff:ffff>,
  #   #    #<Socket::Ifaddr eth0 UP,BROADCAST,RUNNING,MULTICAST,0x10000 fe80::216:3eff:fe95:88bb%eth0 netmask=ffff:ffff:ffff:ffff::>]
  #
  # Example result on FreeBSD:
  #   pp Socket.getifaddrs
  #   #=> [#<Socket::Ifaddr usbus0 UP,0x10000 LINK[usbus0]>,
  #   #    #<Socket::Ifaddr re0 UP,BROADCAST,RUNNING,MULTICAST,0x800 LINK[re0 3a:d0:40:9a:fe:e8]>,
  #   #    #<Socket::Ifaddr re0 UP,BROADCAST,RUNNING,MULTICAST,0x800 10.250.10.18 netmask=255.255.255.? (7 bytes for 16 bytes sockaddr_in) broadcast=10.250.10.255>,
  #   #    #<Socket::Ifaddr re0 UP,BROADCAST,RUNNING,MULTICAST,0x800 fe80:2::38d0:40ff:fe9a:fee8 netmask=ffff:ffff:ffff:ffff::>,
  #   #    #<Socket::Ifaddr re0 UP,BROADCAST,RUNNING,MULTICAST,0x800 2001:2e8:408:10::12 netmask=UNSPEC>,
  #   #    #<Socket::Ifaddr plip0 POINTOPOINT,MULTICAST,0x800 LINK[plip0]>,
  #   #    #<Socket::Ifaddr lo0 UP,LOOPBACK,RUNNING,MULTICAST LINK[lo0]>,
  #   #    #<Socket::Ifaddr lo0 UP,LOOPBACK,RUNNING,MULTICAST ::1 netmask=ffff:ffff:ffff:ffff:ffff:ffff:ffff:ffff>,
  #   #    #<Socket::Ifaddr lo0 UP,LOOPBACK,RUNNING,MULTICAST fe80:4::1 netmask=ffff:ffff:ffff:ffff::>,
  #   #    #<Socket::Ifaddr lo0 UP,LOOPBACK,RUNNING,MULTICAST 127.0.0.1 netmask=255.?.?.? (5 bytes for 16 bytes sockaddr_in)>]
  def self.getifaddrs; end

  # Obtains name information for _sockaddr_.
  #
  # _sockaddr_ should be one of follows.
  # - packed sockaddr string such as Socket.sockaddr_in(80, "127.0.0.1")
  # - 3-elements array such as ["AF_INET", 80, "127.0.0.1"]
  # - 4-elements array such as ["AF_INET", 80, ignored, "127.0.0.1"]
  #
  # _flags_ should be bitwise OR of Socket::NI_* constants.
  #
  # Note:
  # The last form is compatible with IPSocket#addr and IPSocket#peeraddr.
  #
  #   Socket.getnameinfo(Socket.sockaddr_in(80, "127.0.0.1"))       #=> ["localhost", "www"]
  #   Socket.getnameinfo(["AF_INET", 80, "127.0.0.1"])              #=> ["localhost", "www"]
  #   Socket.getnameinfo(["AF_INET", 80, "localhost", "127.0.0.1"]) #=> ["localhost", "www"]
  #
  # If Addrinfo object is preferred, use Addrinfo#getnameinfo.
  def self.getnameinfo(p1, p2 = v2) end

  # Obtains the port number for _service_name_.
  #
  # If _protocol_name_ is not given, "tcp" is assumed.
  #
  #   Socket.getservbyname("smtp")          #=> 25
  #   Socket.getservbyname("shell")         #=> 514
  #   Socket.getservbyname("syslog", "udp") #=> 514
  def self.getservbyname(*several_variants) end

  # Obtains the port number for _port_.
  #
  # If _protocol_name_ is not given, "tcp" is assumed.
  #
  #   Socket.getservbyport(80)         #=> "www"
  #   Socket.getservbyport(514, "tcp") #=> "shell"
  #   Socket.getservbyport(514, "udp") #=> "syslog"
  def self.getservbyport(p1, p2 = v2) end

  # Returns local IP addresses as an array.
  #
  # The array contains Addrinfo objects.
  #
  #  pp Socket.ip_address_list
  #  #=> [#<Addrinfo: 127.0.0.1>,
  #       #<Addrinfo: 192.168.0.128>,
  #       #<Addrinfo: ::1>,
  #       ...]
  def self.ip_address_list; end

  # Packs _port_ and _host_ as an AF_INET/AF_INET6 sockaddr string.
  #
  #   Socket.sockaddr_in(80, "127.0.0.1")
  #   #=> "\x02\x00\x00P\x7F\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00"
  #
  #   Socket.sockaddr_in(80, "::1")
  #   #=> "\n\x00\x00P\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00"
  def self.pack_sockaddr_in(port, host) end

  # Packs _path_ as an AF_UNIX sockaddr string.
  #
  #   Socket.sockaddr_un("/tmp/sock") #=> "\x01\x00/tmp/sock\x00\x00..."
  def self.pack_sockaddr_un(path) end

  # Creates a pair of sockets connected each other.
  #
  # _domain_ should be a communications domain such as: :INET, :INET6, :UNIX, etc.
  #
  # _socktype_ should be a socket type such as: :STREAM, :DGRAM, :RAW, etc.
  #
  # _protocol_ should be a protocol defined in the domain,
  # defaults to 0 for the domain.
  #
  #   s1, s2 = Socket.pair(:UNIX, :STREAM, 0)
  #   s1.send "a", 0
  #   s1.send "b", 0
  #   s1.close
  #   p s2.recv(10) #=> "ab"
  #   p s2.recv(10) #=> ""
  #   p s2.recv(10) #=> ""
  #
  #   s1, s2 = Socket.pair(:UNIX, :DGRAM, 0)
  #   s1.send "a", 0
  #   s1.send "b", 0
  #   p s2.recv(10) #=> "a"
  #   p s2.recv(10) #=> "b"
  def self.pair(domain, type, protocol) end

  # Packs _port_ and _host_ as an AF_INET/AF_INET6 sockaddr string.
  #
  #   Socket.sockaddr_in(80, "127.0.0.1")
  #   #=> "\x02\x00\x00P\x7F\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00"
  #
  #   Socket.sockaddr_in(80, "::1")
  #   #=> "\n\x00\x00P\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00"
  def self.sockaddr_in(port, host) end

  # Packs _path_ as an AF_UNIX sockaddr string.
  #
  #   Socket.sockaddr_un("/tmp/sock") #=> "\x01\x00/tmp/sock\x00\x00..."
  def self.sockaddr_un(path) end

  # Creates a pair of sockets connected each other.
  #
  # _domain_ should be a communications domain such as: :INET, :INET6, :UNIX, etc.
  #
  # _socktype_ should be a socket type such as: :STREAM, :DGRAM, :RAW, etc.
  #
  # _protocol_ should be a protocol defined in the domain,
  # defaults to 0 for the domain.
  #
  #   s1, s2 = Socket.pair(:UNIX, :STREAM, 0)
  #   s1.send "a", 0
  #   s1.send "b", 0
  #   s1.close
  #   p s2.recv(10) #=> "ab"
  #   p s2.recv(10) #=> ""
  #   p s2.recv(10) #=> ""
  #
  #   s1, s2 = Socket.pair(:UNIX, :DGRAM, 0)
  #   s1.send "a", 0
  #   s1.send "b", 0
  #   p s2.recv(10) #=> "a"
  #   p s2.recv(10) #=> "b"
  def self.socketpair(domain, type, protocol) end

  # Unpacks _sockaddr_ into port and ip_address.
  #
  # _sockaddr_ should be a string or an addrinfo for AF_INET/AF_INET6.
  #
  #   sockaddr = Socket.sockaddr_in(80, "127.0.0.1")
  #   p sockaddr #=> "\x02\x00\x00P\x7F\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00"
  #   p Socket.unpack_sockaddr_in(sockaddr) #=> [80, "127.0.0.1"]
  def self.unpack_sockaddr_in(sockaddr) end

  # Unpacks _sockaddr_ into path.
  #
  # _sockaddr_ should be a string or an addrinfo for AF_UNIX.
  #
  #   sockaddr = Socket.sockaddr_un("/tmp/sock")
  #   p Socket.unpack_sockaddr_un(sockaddr) #=> "/tmp/sock"
  def self.unpack_sockaddr_un(sockaddr) end

  # Creates a new socket object.
  #
  # _domain_ should be a communications domain such as: :INET, :INET6, :UNIX, etc.
  #
  # _socktype_ should be a socket type such as: :STREAM, :DGRAM, :RAW, etc.
  #
  # _protocol_ is optional and should be a protocol defined in the domain.
  # If protocol is not given, 0 is used internally.
  #
  #   Socket.new(:INET, :STREAM) # TCP socket
  #   Socket.new(:INET, :DGRAM)  # UDP socket
  #   Socket.new(:UNIX, :STREAM) # UNIX stream socket
  #   Socket.new(:UNIX, :DGRAM)  # UNIX datagram socket
  def initialize(p1, p2, p3 = v3) end

  # Accepts a next connection.
  # Returns a new Socket object and Addrinfo object.
  #
  #   serv = Socket.new(:INET, :STREAM, 0)
  #   serv.listen(5)
  #   c = Socket.new(:INET, :STREAM, 0)
  #   c.connect(serv.connect_address)
  #   p serv.accept #=> [#<Socket:fd 6>, #<Addrinfo: 127.0.0.1:48555 TCP>]
  def accept; end

  # Accepts an incoming connection using accept(2) after
  # O_NONBLOCK is set for the underlying file descriptor.
  # It returns an array containing the accepted socket
  # for the incoming connection, _client_socket_,
  # and an Addrinfo, _client_addrinfo_.
  #
  # === Example
  #      # In one script, start this first
  #      require 'socket'
  #      include Socket::Constants
  #      socket = Socket.new(AF_INET, SOCK_STREAM, 0)
  #      sockaddr = Socket.sockaddr_in(2200, 'localhost')
  #      socket.bind(sockaddr)
  #      socket.listen(5)
  #      begin # emulate blocking accept
  #        client_socket, client_addrinfo = socket.accept_nonblock
  #      rescue IO::WaitReadable, Errno::EINTR
  #        IO.select([socket])
  #        retry
  #      end
  #      puts "The client said, '#{client_socket.readline.chomp}'"
  #      client_socket.puts "Hello from script one!"
  #      socket.close
  #
  #      # In another script, start this second
  #      require 'socket'
  #      include Socket::Constants
  #      socket = Socket.new(AF_INET, SOCK_STREAM, 0)
  #      sockaddr = Socket.sockaddr_in(2200, 'localhost')
  #      socket.connect(sockaddr)
  #      socket.puts "Hello from script 2."
  #      puts "The server said, '#{socket.readline.chomp}'"
  #      socket.close
  #
  # Refer to Socket#accept for the exceptions that may be thrown if the call
  # to _accept_nonblock_ fails.
  #
  # Socket#accept_nonblock may raise any error corresponding to accept(2) failure,
  # including Errno::EWOULDBLOCK.
  #
  # If the exception is Errno::EWOULDBLOCK, Errno::AGAIN, Errno::ECONNABORTED or Errno::EPROTO,
  # it is extended by IO::WaitReadable.
  # So IO::WaitReadable can be used to rescue the exceptions for retrying accept_nonblock.
  #
  # === See
  # * Socket#accept
  def accept_nonblock; end

  # Binds to the given local address.
  #
  # === Parameter
  # * +local_sockaddr+ - the +struct+ sockaddr contained in a string or an Addrinfo object
  #
  # === Example
  #      require 'socket'
  #
  #      # use Addrinfo
  #      socket = Socket.new(:INET, :STREAM, 0)
  #      socket.bind(Addrinfo.tcp("127.0.0.1", 2222))
  #      p socket.local_address #=> #<Addrinfo: 127.0.0.1:2222 TCP>
  #
  #      # use struct sockaddr
  #      include Socket::Constants
  #      socket = Socket.new( AF_INET, SOCK_STREAM, 0 )
  #      sockaddr = Socket.pack_sockaddr_in( 2200, 'localhost' )
  #      socket.bind( sockaddr )
  #
  # === Unix-based Exceptions
  # On unix-based based systems the following system exceptions may be raised if
  # the call to _bind_ fails:
  # * Errno::EACCES - the specified _sockaddr_ is protected and the current
  #   user does not have permission to bind to it
  # * Errno::EADDRINUSE - the specified _sockaddr_ is already in use
  # * Errno::EADDRNOTAVAIL - the specified _sockaddr_ is not available from the
  #   local machine
  # * Errno::EAFNOSUPPORT - the specified _sockaddr_ is not a valid address for
  #   the family of the calling +socket+
  # * Errno::EBADF - the _sockaddr_ specified is not a valid file descriptor
  # * Errno::EFAULT - the _sockaddr_ argument cannot be accessed
  # * Errno::EINVAL - the +socket+ is already bound to an address, and the
  #   protocol does not support binding to the new _sockaddr_ or the +socket+
  #   has been shut down.
  # * Errno::EINVAL - the address length is not a valid length for the address
  #   family
  # * Errno::ENAMETOOLONG - the pathname resolved had a length which exceeded
  #   PATH_MAX
  # * Errno::ENOBUFS - no buffer space is available
  # * Errno::ENOSR - there were insufficient STREAMS resources available to
  #   complete the operation
  # * Errno::ENOTSOCK - the +socket+ does not refer to a socket
  # * Errno::EOPNOTSUPP - the socket type of the +socket+ does not support
  #   binding to an address
  #
  # On unix-based based systems if the address family of the calling +socket+ is
  # Socket::AF_UNIX the follow exceptions may be raised if the call to _bind_
  # fails:
  # * Errno::EACCES - search permission is denied for a component of the prefix
  #   path or write access to the +socket+ is denied
  # * Errno::EDESTADDRREQ - the _sockaddr_ argument is a null pointer
  # * Errno::EISDIR - same as Errno::EDESTADDRREQ
  # * Errno::EIO - an i/o error occurred
  # * Errno::ELOOP - too many symbolic links were encountered in translating
  #   the pathname in _sockaddr_
  # * Errno::ENAMETOOLLONG - a component of a pathname exceeded NAME_MAX
  #   characters, or an entire pathname exceeded PATH_MAX characters
  # * Errno::ENOENT - a component of the pathname does not name an existing file
  #   or the pathname is an empty string
  # * Errno::ENOTDIR - a component of the path prefix of the pathname in _sockaddr_
  #   is not a directory
  # * Errno::EROFS - the name would reside on a read only filesystem
  #
  # === Windows Exceptions
  # On Windows systems the following system exceptions may be raised if
  # the call to _bind_ fails:
  # * Errno::ENETDOWN-- the network is down
  # * Errno::EACCES - the attempt to connect the datagram socket to the
  #   broadcast address failed
  # * Errno::EADDRINUSE - the socket's local address is already in use
  # * Errno::EADDRNOTAVAIL - the specified address is not a valid address for this
  #   computer
  # * Errno::EFAULT - the socket's internal address or address length parameter
  #   is too small or is not a valid part of the user space addressed
  # * Errno::EINVAL - the +socket+ is already bound to an address
  # * Errno::ENOBUFS - no buffer space is available
  # * Errno::ENOTSOCK - the +socket+ argument does not refer to a socket
  #
  # === See
  # * bind manual pages on unix-based systems
  # * bind function in Microsoft's Winsock functions reference
  def bind(local_sockaddr) end

  # Requests a connection to be made on the given +remote_sockaddr+. Returns 0 if
  # successful, otherwise an exception is raised.
  #
  # === Parameter
  # * +remote_sockaddr+ - the +struct+ sockaddr contained in a string or Addrinfo object
  #
  # === Example:
  #      # Pull down Google's web page
  #      require 'socket'
  #      include Socket::Constants
  #      socket = Socket.new( AF_INET, SOCK_STREAM, 0 )
  #      sockaddr = Socket.pack_sockaddr_in( 80, 'www.google.com' )
  #      socket.connect( sockaddr )
  #      socket.write( "GET / HTTP/1.0\r\n\r\n" )
  #      results = socket.read
  #
  # === Unix-based Exceptions
  # On unix-based systems the following system exceptions may be raised if
  # the call to _connect_ fails:
  # * Errno::EACCES - search permission is denied for a component of the prefix
  #   path or write access to the +socket+ is denied
  # * Errno::EADDRINUSE - the _sockaddr_ is already in use
  # * Errno::EADDRNOTAVAIL - the specified _sockaddr_ is not available from the
  #   local machine
  # * Errno::EAFNOSUPPORT - the specified _sockaddr_ is not a valid address for
  #   the address family of the specified +socket+
  # * Errno::EALREADY - a connection is already in progress for the specified
  #   socket
  # * Errno::EBADF - the +socket+ is not a valid file descriptor
  # * Errno::ECONNREFUSED - the target _sockaddr_ was not listening for connections
  #   refused the connection request
  # * Errno::ECONNRESET - the remote host reset the connection request
  # * Errno::EFAULT - the _sockaddr_ cannot be accessed
  # * Errno::EHOSTUNREACH - the destination host cannot be reached (probably
  #   because the host is down or a remote router cannot reach it)
  # * Errno::EINPROGRESS - the O_NONBLOCK is set for the +socket+ and the
  #   connection cannot be immediately established; the connection will be
  #   established asynchronously
  # * Errno::EINTR - the attempt to establish the connection was interrupted by
  #   delivery of a signal that was caught; the connection will be established
  #   asynchronously
  # * Errno::EISCONN - the specified +socket+ is already connected
  # * Errno::EINVAL - the address length used for the _sockaddr_ is not a valid
  #   length for the address family or there is an invalid family in _sockaddr_
  # * Errno::ENAMETOOLONG - the pathname resolved had a length which exceeded
  #   PATH_MAX
  # * Errno::ENETDOWN - the local interface used to reach the destination is down
  # * Errno::ENETUNREACH - no route to the network is present
  # * Errno::ENOBUFS - no buffer space is available
  # * Errno::ENOSR - there were insufficient STREAMS resources available to
  #   complete the operation
  # * Errno::ENOTSOCK - the +socket+ argument does not refer to a socket
  # * Errno::EOPNOTSUPP - the calling +socket+ is listening and cannot be connected
  # * Errno::EPROTOTYPE - the _sockaddr_ has a different type than the socket
  #   bound to the specified peer address
  # * Errno::ETIMEDOUT - the attempt to connect time out before a connection
  #   was made.
  #
  # On unix-based systems if the address family of the calling +socket+ is
  # AF_UNIX the follow exceptions may be raised if the call to _connect_
  # fails:
  # * Errno::EIO - an i/o error occurred while reading from or writing to the
  #   file system
  # * Errno::ELOOP - too many symbolic links were encountered in translating
  #   the pathname in _sockaddr_
  # * Errno::ENAMETOOLLONG - a component of a pathname exceeded NAME_MAX
  #   characters, or an entire pathname exceeded PATH_MAX characters
  # * Errno::ENOENT - a component of the pathname does not name an existing file
  #   or the pathname is an empty string
  # * Errno::ENOTDIR - a component of the path prefix of the pathname in _sockaddr_
  #   is not a directory
  #
  # === Windows Exceptions
  # On Windows systems the following system exceptions may be raised if
  # the call to _connect_ fails:
  # * Errno::ENETDOWN - the network is down
  # * Errno::EADDRINUSE - the socket's local address is already in use
  # * Errno::EINTR - the socket was cancelled
  # * Errno::EINPROGRESS - a blocking socket is in progress or the service provider
  #   is still processing a callback function. Or a nonblocking connect call is
  #   in progress on the +socket+.
  # * Errno::EALREADY - see Errno::EINVAL
  # * Errno::EADDRNOTAVAIL - the remote address is not a valid address, such as
  #   ADDR_ANY TODO check ADDRANY TO INADDR_ANY
  # * Errno::EAFNOSUPPORT - addresses in the specified family cannot be used with
  #   with this +socket+
  # * Errno::ECONNREFUSED - the target _sockaddr_ was not listening for connections
  #   refused the connection request
  # * Errno::EFAULT - the socket's internal address or address length parameter
  #   is too small or is not a valid part of the user space address
  # * Errno::EINVAL - the +socket+ is a listening socket
  # * Errno::EISCONN - the +socket+ is already connected
  # * Errno::ENETUNREACH - the network cannot be reached from this host at this time
  # * Errno::EHOSTUNREACH - no route to the network is present
  # * Errno::ENOBUFS - no buffer space is available
  # * Errno::ENOTSOCK - the +socket+ argument does not refer to a socket
  # * Errno::ETIMEDOUT - the attempt to connect time out before a connection
  #   was made.
  # * Errno::EWOULDBLOCK - the socket is marked as nonblocking and the
  #   connection cannot be completed immediately
  # * Errno::EACCES - the attempt to connect the datagram socket to the
  #   broadcast address failed
  #
  # === See
  # * connect manual pages on unix-based systems
  # * connect function in Microsoft's Winsock functions reference
  def connect(remote_sockaddr) end

  # Requests a connection to be made on the given +remote_sockaddr+ after
  # O_NONBLOCK is set for the underlying file descriptor.
  # Returns 0 if successful, otherwise an exception is raised.
  #
  # === Parameter
  # * +remote_sockaddr+ - the +struct+ sockaddr contained in a string or Addrinfo object
  #
  # === Example:
  #      # Pull down Google's web page
  #      require 'socket'
  #      include Socket::Constants
  #      socket = Socket.new(AF_INET, SOCK_STREAM, 0)
  #      sockaddr = Socket.sockaddr_in(80, 'www.google.com')
  #      begin # emulate blocking connect
  #        socket.connect_nonblock(sockaddr)
  #      rescue IO::WaitWritable
  #        IO.select(nil, [socket]) # wait 3-way handshake completion
  #        begin
  #          socket.connect_nonblock(sockaddr) # check connection failure
  #        rescue Errno::EISCONN
  #        end
  #      end
  #      socket.write("GET / HTTP/1.0\r\n\r\n")
  #      results = socket.read
  #
  # Refer to Socket#connect for the exceptions that may be thrown if the call
  # to _connect_nonblock_ fails.
  #
  # Socket#connect_nonblock may raise any error corresponding to connect(2) failure,
  # including Errno::EINPROGRESS.
  #
  # If the exception is Errno::EINPROGRESS,
  # it is extended by IO::WaitWritable.
  # So IO::WaitWritable can be used to rescue the exceptions for retrying connect_nonblock.
  #
  # === See
  # * Socket#connect
  def connect_nonblock(remote_sockaddr) end

  # Listens for connections, using the specified +int+ as the backlog. A call
  # to _listen_ only applies if the +socket+ is of type SOCK_STREAM or
  # SOCK_SEQPACKET.
  #
  # === Parameter
  # * +backlog+ - the maximum length of the queue for pending connections.
  #
  # === Example 1
  #      require 'socket'
  #      include Socket::Constants
  #      socket = Socket.new( AF_INET, SOCK_STREAM, 0 )
  #      sockaddr = Socket.pack_sockaddr_in( 2200, 'localhost' )
  #      socket.bind( sockaddr )
  #      socket.listen( 5 )
  #
  # === Example 2 (listening on an arbitrary port, unix-based systems only):
  #      require 'socket'
  #      include Socket::Constants
  #      socket = Socket.new( AF_INET, SOCK_STREAM, 0 )
  #      socket.listen( 1 )
  #
  # === Unix-based Exceptions
  # On unix based systems the above will work because a new +sockaddr+ struct
  # is created on the address ADDR_ANY, for an arbitrary port number as handed
  # off by the kernel. It will not work on Windows, because Windows requires that
  # the +socket+ is bound by calling _bind_ before it can _listen_.
  #
  # If the _backlog_ amount exceeds the implementation-dependent maximum
  # queue length, the implementation's maximum queue length will be used.
  #
  # On unix-based based systems the following system exceptions may be raised if the
  # call to _listen_ fails:
  # * Errno::EBADF - the _socket_ argument is not a valid file descriptor
  # * Errno::EDESTADDRREQ - the _socket_ is not bound to a local address, and
  #   the protocol does not support listening on an unbound socket
  # * Errno::EINVAL - the _socket_ is already connected
  # * Errno::ENOTSOCK - the _socket_ argument does not refer to a socket
  # * Errno::EOPNOTSUPP - the _socket_ protocol does not support listen
  # * Errno::EACCES - the calling process does not have appropriate privileges
  # * Errno::EINVAL - the _socket_ has been shut down
  # * Errno::ENOBUFS - insufficient resources are available in the system to
  #   complete the call
  #
  # === Windows Exceptions
  # On Windows systems the following system exceptions may be raised if
  # the call to _listen_ fails:
  # * Errno::ENETDOWN - the network is down
  # * Errno::EADDRINUSE - the socket's local address is already in use. This
  #   usually occurs during the execution of _bind_ but could be delayed
  #   if the call to _bind_ was to a partially wildcard address (involving
  #   ADDR_ANY) and if a specific address needs to be committed at the
  #   time of the call to _listen_
  # * Errno::EINPROGRESS - a Windows Sockets 1.1 call is in progress or the
  #   service provider is still processing a callback function
  # * Errno::EINVAL - the +socket+ has not been bound with a call to _bind_.
  # * Errno::EISCONN - the +socket+ is already connected
  # * Errno::EMFILE - no more socket descriptors are available
  # * Errno::ENOBUFS - no buffer space is available
  # * Errno::ENOTSOC - +socket+ is not a socket
  # * Errno::EOPNOTSUPP - the referenced +socket+ is not a type that supports
  #   the _listen_ method
  #
  # === See
  # * listen manual pages on unix-based systems
  # * listen function in Microsoft's Winsock functions reference
  def listen(int) end

  # Receives up to _maxlen_ bytes from +socket+. _flags_ is zero or more
  # of the +MSG_+ options. The first element of the results, _mesg_, is the data
  # received. The second element, _sender_addrinfo_, contains protocol-specific
  # address information of the sender.
  #
  # === Parameters
  # * +maxlen+ - the maximum number of bytes to receive from the socket
  # * +flags+ - zero or more of the +MSG_+ options
  #
  # === Example
  #      # In one file, start this first
  #      require 'socket'
  #      include Socket::Constants
  #      socket = Socket.new( AF_INET, SOCK_STREAM, 0 )
  #      sockaddr = Socket.pack_sockaddr_in( 2200, 'localhost' )
  #      socket.bind( sockaddr )
  #      socket.listen( 5 )
  #      client, client_addrinfo = socket.accept
  #      data = client.recvfrom( 20 )[0].chomp
  #      puts "I only received 20 bytes '#{data}'"
  #      sleep 1
  #      socket.close
  #
  #      # In another file, start this second
  #      require 'socket'
  #      include Socket::Constants
  #      socket = Socket.new( AF_INET, SOCK_STREAM, 0 )
  #      sockaddr = Socket.pack_sockaddr_in( 2200, 'localhost' )
  #      socket.connect( sockaddr )
  #      socket.puts "Watch this get cut short!"
  #      socket.close
  #
  # === Unix-based Exceptions
  # On unix-based based systems the following system exceptions may be raised if the
  # call to _recvfrom_ fails:
  # * Errno::EAGAIN - the +socket+ file descriptor is marked as O_NONBLOCK and no
  #   data is waiting to be received; or MSG_OOB is set and no out-of-band data
  #   is available and either the +socket+ file descriptor is marked as
  #   O_NONBLOCK or the +socket+ does not support blocking to wait for
  #   out-of-band-data
  # * Errno::EWOULDBLOCK - see Errno::EAGAIN
  # * Errno::EBADF - the +socket+ is not a valid file descriptor
  # * Errno::ECONNRESET - a connection was forcibly closed by a peer
  # * Errno::EFAULT - the socket's internal buffer, address or address length
  #   cannot be accessed or written
  # * Errno::EINTR - a signal interrupted _recvfrom_ before any data was available
  # * Errno::EINVAL - the MSG_OOB flag is set and no out-of-band data is available
  # * Errno::EIO - an i/o error occurred while reading from or writing to the
  #   filesystem
  # * Errno::ENOBUFS - insufficient resources were available in the system to
  #   perform the operation
  # * Errno::ENOMEM - insufficient memory was available to fulfill the request
  # * Errno::ENOSR - there were insufficient STREAMS resources available to
  #   complete the operation
  # * Errno::ENOTCONN - a receive is attempted on a connection-mode socket that
  #   is not connected
  # * Errno::ENOTSOCK - the +socket+ does not refer to a socket
  # * Errno::EOPNOTSUPP - the specified flags are not supported for this socket type
  # * Errno::ETIMEDOUT - the connection timed out during connection establishment
  #   or due to a transmission timeout on an active connection
  #
  # === Windows Exceptions
  # On Windows systems the following system exceptions may be raised if
  # the call to _recvfrom_ fails:
  # * Errno::ENETDOWN - the network is down
  # * Errno::EFAULT - the internal buffer and from parameters on +socket+ are not
  #   part of the user address space, or the internal fromlen parameter is
  #   too small to accommodate the peer address
  # * Errno::EINTR - the (blocking) call was cancelled by an internal call to
  #   the WinSock function WSACancelBlockingCall
  # * Errno::EINPROGRESS - a blocking Windows Sockets 1.1 call is in progress or
  #   the service provider is still processing a callback function
  # * Errno::EINVAL - +socket+ has not been bound with a call to _bind_, or an
  #   unknown flag was specified, or MSG_OOB was specified for a socket with
  #   SO_OOBINLINE enabled, or (for byte stream-style sockets only) the internal
  #   len parameter on +socket+ was zero or negative
  # * Errno::EISCONN - +socket+ is already connected. The call to _recvfrom_ is
  #   not permitted with a connected socket on a socket that is connection
  #   oriented or connectionless.
  # * Errno::ENETRESET - the connection has been broken due to the keep-alive
  #   activity detecting a failure while the operation was in progress.
  # * Errno::EOPNOTSUPP - MSG_OOB was specified, but +socket+ is not stream-style
  #   such as type SOCK_STREAM. OOB data is not supported in the communication
  #   domain associated with +socket+, or +socket+ is unidirectional and
  #   supports only send operations
  # * Errno::ESHUTDOWN - +socket+ has been shutdown. It is not possible to
  #   call _recvfrom_ on a socket after _shutdown_ has been invoked.
  # * Errno::EWOULDBLOCK - +socket+ is marked as nonblocking and a  call to
  #   _recvfrom_ would block.
  # * Errno::EMSGSIZE - the message was too large to fit into the specified buffer
  #   and was truncated.
  # * Errno::ETIMEDOUT - the connection has been dropped, because of a network
  #   failure or because the system on the other end went down without
  #   notice
  # * Errno::ECONNRESET - the virtual circuit was reset by the remote side
  #   executing a hard or abortive close. The application should close the
  #   socket; it is no longer usable. On a UDP-datagram socket this error
  #   indicates a previous send operation resulted in an ICMP Port Unreachable
  #   message.
  def recvfrom(*several_variants) end

  # Receives up to _maxlen_ bytes from +socket+ using recvfrom(2) after
  # O_NONBLOCK is set for the underlying file descriptor.
  # _flags_ is zero or more of the +MSG_+ options.
  # The first element of the results, _mesg_, is the data received.
  # The second element, _sender_addrinfo_, contains protocol-specific address
  # information of the sender.
  #
  # When recvfrom(2) returns 0, Socket#recvfrom_nonblock returns
  # an empty string as data.
  # The meaning depends on the socket: EOF on TCP, empty packet on UDP, etc.
  #
  # === Parameters
  # * +maxlen+ - the maximum number of bytes to receive from the socket
  # * +flags+ - zero or more of the +MSG_+ options
  #
  # === Example
  #      # In one file, start this first
  #      require 'socket'
  #      include Socket::Constants
  #      socket = Socket.new(AF_INET, SOCK_STREAM, 0)
  #      sockaddr = Socket.sockaddr_in(2200, 'localhost')
  #      socket.bind(sockaddr)
  #      socket.listen(5)
  #      client, client_addrinfo = socket.accept
  #      begin # emulate blocking recvfrom
  #        pair = client.recvfrom_nonblock(20)
  #      rescue IO::WaitReadable
  #        IO.select([client])
  #        retry
  #      end
  #      data = pair[0].chomp
  #      puts "I only received 20 bytes '#{data}'"
  #      sleep 1
  #      socket.close
  #
  #      # In another file, start this second
  #      require 'socket'
  #      include Socket::Constants
  #      socket = Socket.new(AF_INET, SOCK_STREAM, 0)
  #      sockaddr = Socket.sockaddr_in(2200, 'localhost')
  #      socket.connect(sockaddr)
  #      socket.puts "Watch this get cut short!"
  #      socket.close
  #
  # Refer to Socket#recvfrom for the exceptions that may be thrown if the call
  # to _recvfrom_nonblock_ fails.
  #
  # Socket#recvfrom_nonblock may raise any error corresponding to recvfrom(2) failure,
  # including Errno::EWOULDBLOCK.
  #
  # If the exception is Errno::EWOULDBLOCK or Errno::AGAIN,
  # it is extended by IO::WaitReadable.
  # So IO::WaitReadable can be used to rescue the exceptions for retrying recvfrom_nonblock.
  #
  # === See
  # * Socket#recvfrom
  def recvfrom_nonblock(*several_variants) end

  # Accepts an incoming connection returning an array containing the (integer)
  # file descriptor for the incoming connection, _client_socket_fd_,
  # and an Addrinfo, _client_addrinfo_.
  #
  # === Example
  #      # In one script, start this first
  #      require 'socket'
  #      include Socket::Constants
  #      socket = Socket.new( AF_INET, SOCK_STREAM, 0 )
  #      sockaddr = Socket.pack_sockaddr_in( 2200, 'localhost' )
  #      socket.bind( sockaddr )
  #      socket.listen( 5 )
  #      client_fd, client_addrinfo = socket.sysaccept
  #      client_socket = Socket.for_fd( client_fd )
  #      puts "The client said, '#{client_socket.readline.chomp}'"
  #      client_socket.puts "Hello from script one!"
  #      socket.close
  #
  #      # In another script, start this second
  #      require 'socket'
  #      include Socket::Constants
  #      socket = Socket.new( AF_INET, SOCK_STREAM, 0 )
  #      sockaddr = Socket.pack_sockaddr_in( 2200, 'localhost' )
  #      socket.connect( sockaddr )
  #      socket.puts "Hello from script 2."
  #      puts "The server said, '#{socket.readline.chomp}'"
  #      socket.close
  #
  # Refer to Socket#accept for the exceptions that may be thrown if the call
  # to _sysaccept_ fails.
  #
  # === See
  # * Socket#accept
  def sysaccept; end

  # Socket::AncillaryData represents the ancillary data (control information)
  # used by sendmsg and recvmsg system call.  It contains socket #family,
  # control message (cmsg) #level, cmsg #type and cmsg #data.
  class AncillaryData
    # Creates a new Socket::AncillaryData object which contains a int as data.
    #
    # The size and endian is dependent on the host.
    #
    #   p Socket::AncillaryData.int(:UNIX, :SOCKET, :RIGHTS, STDERR.fileno)
    #   #=> #<Socket::AncillaryData: UNIX SOCKET RIGHTS 2>
    def self.int(family, cmsg_level, cmsg_type, integer) end

    # Returns new ancillary data for IP_PKTINFO.
    #
    # If spec_dst is not given, addr is used.
    #
    # IP_PKTINFO is not standard.
    #
    # Supported platform: GNU/Linux
    #
    #   addr = Addrinfo.ip("127.0.0.1")
    #   ifindex = 0
    #   spec_dst = Addrinfo.ip("127.0.0.1")
    #   p Socket::AncillaryData.ip_pktinfo(addr, ifindex, spec_dst)
    #   #=> #<Socket::AncillaryData: INET IP PKTINFO 127.0.0.1 ifindex:0 spec_dst:127.0.0.1>
    def self.ip_pktinfo(*several_variants) end

    # Returns new ancillary data for IPV6_PKTINFO.
    #
    # IPV6_PKTINFO is defined by RFC 3542.
    #
    #   addr = Addrinfo.ip("::1")
    #   ifindex = 0
    #   p Socket::AncillaryData.ipv6_pktinfo(addr, ifindex)
    #   #=> #<Socket::AncillaryData: INET6 IPV6 PKTINFO ::1 ifindex:0>
    def self.ipv6_pktinfo(addr, ifindex) end

    # Creates a new Socket::AncillaryData object which contains file descriptors as data.
    #
    #   p Socket::AncillaryData.unix_rights(STDERR)
    #   #=> #<Socket::AncillaryData: UNIX SOCKET RIGHTS 2>
    def self.unix_rights(io1, io2, *args) end

    # _family_ should be an integer, a string or a symbol.
    # - Socket::AF_INET, "AF_INET", "INET", :AF_INET, :INET
    # - Socket::AF_UNIX, "AF_UNIX", "UNIX", :AF_UNIX, :UNIX
    # - etc.
    #
    # _cmsg_level_ should be an integer, a string or a symbol.
    # - Socket::SOL_SOCKET, "SOL_SOCKET", "SOCKET", :SOL_SOCKET and :SOCKET
    # - Socket::IPPROTO_IP, "IP" and :IP
    # - Socket::IPPROTO_IPV6, "IPV6" and :IPV6
    # - Socket::IPPROTO_TCP, "TCP" and :TCP
    # - etc.
    #
    # _cmsg_type_ should be an integer, a string or a symbol.
    # If a string/symbol is specified, it is interpreted depend on _cmsg_level_.
    # - Socket::SCM_RIGHTS, "SCM_RIGHTS", "RIGHTS", :SCM_RIGHTS, :RIGHTS for SOL_SOCKET
    # - Socket::IP_RECVTTL, "RECVTTL" and :RECVTTL for IPPROTO_IP
    # - Socket::IPV6_PKTINFO, "PKTINFO" and :PKTINFO for IPPROTO_IPV6
    # - etc.
    #
    # _cmsg_data_ should be a string.
    #
    #   p Socket::AncillaryData.new(:INET, :TCP, :NODELAY, "")
    #   #=> #<Socket::AncillaryData: INET TCP NODELAY "">
    #
    #   p Socket::AncillaryData.new(:INET6, :IPV6, :PKTINFO, "")
    #   #=> #<Socket::AncillaryData: INET6 IPV6 PKTINFO "">
    def initialize(family, cmsg_level, cmsg_type, cmsg_data) end

    # tests the level and type of _ancillarydata_.
    #
    #   ancdata = Socket::AncillaryData.new(:INET6, :IPV6, :PKTINFO, "")
    #   ancdata.cmsg_is?(Socket::IPPROTO_IPV6, Socket::IPV6_PKTINFO) #=> true
    #   ancdata.cmsg_is?(:IPV6, :PKTINFO)       #=> true
    #   ancdata.cmsg_is?(:IP, :PKTINFO)         #=> false
    #   ancdata.cmsg_is?(:SOCKET, :RIGHTS)      #=> false
    def cmsg_is?(level, type) end

    # returns the cmsg data as a string.
    #
    #   p Socket::AncillaryData.new(:INET6, :IPV6, :PKTINFO, "").data
    #   #=> ""
    def data; end

    # returns the socket family as an integer.
    #
    #   p Socket::AncillaryData.new(:INET6, :IPV6, :PKTINFO, "").family
    #   #=> 10
    def family; end

    # returns a string which shows ancillarydata in human-readable form.
    #
    #   p Socket::AncillaryData.new(:INET6, :IPV6, :PKTINFO, "").inspect
    #   #=> "#<Socket::AncillaryData: INET6 IPV6 PKTINFO \"\">"
    def inspect; end

    # Returns the data in _ancillarydata_ as an int.
    #
    # The size and endian is dependent on the host.
    #
    #   ancdata = Socket::AncillaryData.int(:UNIX, :SOCKET, :RIGHTS, STDERR.fileno)
    #   p ancdata.int #=> 2
    def int; end

    # Extracts addr, ifindex and spec_dst from IP_PKTINFO ancillary data.
    #
    # IP_PKTINFO is not standard.
    #
    # Supported platform: GNU/Linux
    #
    #   addr = Addrinfo.ip("127.0.0.1")
    #   ifindex = 0
    #   spec_dest = Addrinfo.ip("127.0.0.1")
    #   ancdata = Socket::AncillaryData.ip_pktinfo(addr, ifindex, spec_dest)
    #   p ancdata.ip_pktinfo
    #   #=> [#<Addrinfo: 127.0.0.1>, 0, #<Addrinfo: 127.0.0.1>]
    def ip_pktinfo; end

    # Extracts addr and ifindex from IPV6_PKTINFO ancillary data.
    #
    # IPV6_PKTINFO is defined by RFC 3542.
    #
    #   addr = Addrinfo.ip("::1")
    #   ifindex = 0
    #   ancdata = Socket::AncillaryData.ipv6_pktinfo(addr, ifindex)
    #   p ancdata.ipv6_pktinfo #=> [#<Addrinfo: ::1>, 0]
    def ipv6_pktinfo; end

    # Extracts addr from IPV6_PKTINFO ancillary data.
    #
    # IPV6_PKTINFO is defined by RFC 3542.
    #
    #   addr = Addrinfo.ip("::1")
    #   ifindex = 0
    #   ancdata = Socket::AncillaryData.ipv6_pktinfo(addr, ifindex)
    #   p ancdata.ipv6_pktinfo_addr #=> #<Addrinfo: ::1>
    def ipv6_pktinfo_addr; end

    # Extracts ifindex from IPV6_PKTINFO ancillary data.
    #
    # IPV6_PKTINFO is defined by RFC 3542.
    #
    #   addr = Addrinfo.ip("::1")
    #   ifindex = 0
    #   ancdata = Socket::AncillaryData.ipv6_pktinfo(addr, ifindex)
    #   p ancdata.ipv6_pktinfo_ifindex #=> 0
    def ipv6_pktinfo_ifindex; end

    # returns the cmsg level as an integer.
    #
    #   p Socket::AncillaryData.new(:INET6, :IPV6, :PKTINFO, "").level
    #   #=> 41
    def level; end

    # returns the timestamp as a time object.
    #
    # _ancillarydata_ should be one of following type:
    # - SOL_SOCKET/SCM_TIMESTAMP (microsecond) GNU/Linux, FreeBSD, NetBSD, OpenBSD, Solaris, MacOS X
    # - SOL_SOCKET/SCM_TIMESTAMPNS (nanosecond) GNU/Linux
    # - SOL_SOCKET/SCM_BINTIME (2**(-64) second) FreeBSD
    #
    #   Addrinfo.udp("127.0.0.1", 0).bind {|s1|
    #     Addrinfo.udp("127.0.0.1", 0).bind {|s2|
    #       s1.setsockopt(:SOCKET, :TIMESTAMP, true)
    #       s2.send "a", 0, s1.local_address
    #       ctl = s1.recvmsg.last
    #       p ctl    #=> #<Socket::AncillaryData: INET SOCKET TIMESTAMP 2009-02-24 17:35:46.775581>
    #       t = ctl.timestamp
    #       p t      #=> 2009-02-24 17:35:46 +0900
    #       p t.usec #=> 775581
    #       p t.nsec #=> 775581000
    #     }
    #   }
    def timestamp; end

    # returns the cmsg type as an integer.
    #
    #   p Socket::AncillaryData.new(:INET6, :IPV6, :PKTINFO, "").type
    #   #=> 2
    def type; end

    # returns the array of IO objects for SCM_RIGHTS control message in UNIX domain socket.
    #
    # The class of the IO objects in the array is IO or Socket.
    #
    # The array is attached to _ancillarydata_ when it is instantiated.
    # For example, BasicSocket#recvmsg attach the array when
    # receives a SCM_RIGHTS control message and :scm_rights=>true option is given.
    #
    #   # recvmsg needs :scm_rights=>true for unix_rights
    #   s1, s2 = UNIXSocket.pair
    #   p s1                                         #=> #<UNIXSocket:fd 3>
    #   s1.sendmsg "stdin and a socket", 0, nil, Socket::AncillaryData.unix_rights(STDIN, s1)
    #   _, _, _, ctl = s2.recvmsg(:scm_rights=>true)
    #   p ctl                                        #=> #<Socket::AncillaryData: UNIX SOCKET RIGHTS 6 7>
    #   p ctl.unix_rights                            #=> [#<IO:fd 6>, #<Socket:fd 7>]
    #   p File.identical?(STDIN, ctl.unix_rights[0]) #=> true
    #   p File.identical?(s1, ctl.unix_rights[1])    #=> true
    #
    #   # If :scm_rights=>true is not given, unix_rights returns nil
    #   s1, s2 = UNIXSocket.pair
    #   s1.sendmsg "stdin and a socket", 0, nil, Socket::AncillaryData.unix_rights(STDIN, s1)
    #   _, _, _, ctl = s2.recvmsg
    #   p ctl #=> #<Socket::AncillaryData: UNIX SOCKET RIGHTS 6 7>
    #   p ctl.unix_rights #=> nil
    def unix_rights; end
  end

  # Socket::Ifaddr represents a result of getifaddrs() function.
  class Ifaddr < Data
    # Returns the address of _ifaddr_.
    # nil is returned if address is not available in _ifaddr_.
    def addr; end

    # Returns the broadcast address of _ifaddr_.
    # nil is returned if the flags doesn't have IFF_BROADCAST.
    def broadaddr; end

    # Returns the destination address of _ifaddr_.
    # nil is returned if the flags doesn't have IFF_POINTOPOINT.
    def dstaddr; end

    # Returns the flags of _ifaddr_.
    def flags; end

    # Returns the interface index of _ifaddr_.
    def ifindex; end

    # Returns a string to show contents of _ifaddr_.
    def inspect; end

    # Returns the interface name of _ifaddr_.
    def name; end

    # Returns the netmask address of _ifaddr_.
    # nil is returned if netmask is not available in _ifaddr_.
    def netmask; end
  end

  # Socket::Option represents a socket option used by
  # BasicSocket#getsockopt and BasicSocket#setsockopt.  A socket option
  # contains the socket #family, protocol #level, option name #optname and
  # option value #data.
  class Option
    # Creates a new Socket::Option object which contains boolean as data.
    # Actually 0 or 1 as int is used.
    #
    #   p Socket::Option.bool(:INET, :SOCKET, :KEEPALIVE, true)
    #   #=> #<Socket::Option: INET SOCKET KEEPALIVE 1>
    #
    #   p Socket::Option.bool(:INET, :SOCKET, :KEEPALIVE, false)
    #   #=> #<Socket::Option: AF_INET SOCKET KEEPALIVE 0>
    def self.bool(family, level, optname, bool) end

    # Creates a new Socket::Option object which contains a byte as data.
    #
    # The size and endian is dependent on the platform.
    #
    #   p Socket::Option.byte(:INET, :SOCKET, :KEEPALIVE, 1)
    #   #=> #<Socket::Option: INET SOCKET KEEPALIVE 1>
    def self.byte(family, level, optname, integer) end

    # Creates a new Socket::Option object which contains an int as data.
    #
    # The size and endian is dependent on the platform.
    #
    #   p Socket::Option.int(:INET, :SOCKET, :KEEPALIVE, 1)
    #   #=> #<Socket::Option: INET SOCKET KEEPALIVE 1>
    def self.int(family, level, optname, integer) end

    # Creates a new Socket::Option object for IP_MULTICAST_LOOP.
    #
    # The size is dependent on the platform.
    #
    #   sockopt = Socket::Option.int(:INET, :IPPROTO_IP, :IP_MULTICAST_LOOP, 1)
    #   p sockopt.int => 1
    #
    #   p Socket::Option.ipv4_multicast_loop(10)
    #   #=> #<Socket::Option: INET IP MULTICAST_LOOP 10>
    def self.ipv4_multicast_loop(integer) end

    # Creates a new Socket::Option object for IP_MULTICAST_TTL.
    #
    # The size is dependent on the platform.
    #
    #   p Socket::Option.ipv4_multicast_ttl(10)
    #   #=> #<Socket::Option: INET IP MULTICAST_TTL 10>
    def self.ipv4_multicast_ttl(integer) end

    # Creates a new Socket::Option object for SOL_SOCKET/SO_LINGER.
    #
    # _onoff_ should be an integer or a boolean.
    #
    # _secs_ should be the number of seconds.
    #
    #   p Socket::Option.linger(true, 10)
    #   #=> #<Socket::Option: UNSPEC SOCKET LINGER on 10sec>
    def self.linger(onoff, secs) end

    # Returns a new Socket::Option object.
    #
    #   sockopt = Socket::Option.new(:INET, :SOCKET, :KEEPALIVE, [1].pack("i"))
    #   p sockopt #=> #<Socket::Option: INET SOCKET KEEPALIVE 1>
    def initialize(family, level, optname, data) end

    # Returns the data in _sockopt_ as an boolean value.
    #
    #   sockopt = Socket::Option.int(:INET, :SOCKET, :KEEPALIVE, 1)
    #   p sockopt.bool => true
    def bool; end

    # Returns the data in _sockopt_ as an byte.
    #
    # The size and endian is dependent on the platform.
    #
    #   sockopt = Socket::Option.byte(:INET, :SOCKET, :KEEPALIVE, 1)
    #   p sockopt.byte => 1
    def byte; end

    # returns the socket option data as a string.
    #
    #   p Socket::Option.new(:INET6, :IPV6, :RECVPKTINFO, [1].pack("i!")).data
    #   #=> "\x01\x00\x00\x00"
    def data; end
    alias to_s data

    # returns the socket family as an integer.
    #
    #   p Socket::Option.new(:INET6, :IPV6, :RECVPKTINFO, [1].pack("i!")).family
    #   #=> 10
    def family; end

    # Returns a string which shows sockopt in human-readable form.
    #
    #   p Socket::Option.new(:INET, :SOCKET, :KEEPALIVE, [1].pack("i")).inspect
    #   #=> "#<Socket::Option: INET SOCKET KEEPALIVE 1>"
    def inspect; end

    # Returns the data in _sockopt_ as an int.
    #
    # The size and endian is dependent on the platform.
    #
    #   sockopt = Socket::Option.int(:INET, :SOCKET, :KEEPALIVE, 1)
    #   p sockopt.int => 1
    def int; end

    # Returns the ipv4_multicast_loop data in _sockopt_ as a integer.
    #
    #   sockopt = Socket::Option.ipv4_multicast_loop(10)
    #   p sockopt.ipv4_multicast_loop => 10
    def ipv4_multicast_loop; end

    # Returns the ipv4_multicast_ttl data in _sockopt_ as a integer.
    #
    #   sockopt = Socket::Option.ipv4_multicast_ttl(10)
    #   p sockopt.ipv4_multicast_ttl => 10
    def ipv4_multicast_ttl; end

    # returns the socket level as an integer.
    #
    #   p Socket::Option.new(:INET6, :IPV6, :RECVPKTINFO, [1].pack("i!")).level
    #   #=> 41
    def level; end

    # Returns the linger data in _sockopt_ as a pair of boolean and integer.
    #
    #   sockopt = Socket::Option.linger(true, 10)
    #   p sockopt.linger => [true, 10]
    def linger; end

    # returns the socket option name as an integer.
    #
    #   p Socket::Option.new(:INET6, :IPV6, :RECVPKTINFO, [1].pack("i!")).optname
    #   #=> 2
    def optname; end

    # Calls String#unpack on sockopt.data.
    #
    #   sockopt = Socket::Option.new(:INET, :SOCKET, :KEEPALIVE, [1].pack("i"))
    #   p sockopt.unpack("i")      #=> [1]
    #   p sockopt.data.unpack("i") #=> [1]
    def unpack(template) end
  end
end
