extends RefCounted
class_name WebRTCPeerConnection

## Interface to a WebRTC peer connection.
##
## A WebRTC connection between the local computer and a remote peer. Provides an interface to connect, maintain and monitor the connection.
## Setting up a WebRTC connection between two peers may not seem a trivial task, but it can be broken down into 3 main steps:
## - The peer that wants to initiate the connection ([code]A[/code] from now on) creates an offer and send it to the other peer ([code]B[/code] from now on).
## - [code]B[/code] receives the offer, generate and answer, and sends it to [code]A[/code]).
## - [code]A[/code] and [code]B[/code] then generates and exchange ICE candidates with each other.
## After these steps, the connection should become connected. Keep on reading or look into the tutorial for more information.


## Emitted when a new in-band channel is received, i.e. when the channel was created with [code]negotiated: false[/code] (default).
## The object will be an instance of [WebRTCDataChannel]. You must keep a reference of it or it will be closed automatically. See [method create_data_channel].
signal data_channel_received(channel: WebRTCDataChannel)
## Emitted when a new ICE candidate has been created. The three parameters are meant to be passed to the remote peer over the signaling server.
signal ice_candidate_created(media: String, index: int, name: String)
## Emitted after a successful call to [method create_offer] or [method set_remote_description] (when it generates an answer). The parameters are meant to be passed to [method set_local_description] on this object, and sent to the remote peer over the signaling server.
signal session_description_created(type: String, sdp: String)
## The connection is new, data channels and an offer can be created in this state.
## The peer is connecting, ICE is in progress, none of the transports has failed.
## The peer is connected, all ICE transports are connected.
## At least one ICE transport is disconnected.
## One or more of the ICE transports failed.
## The peer connection is closed (after calling [method close] for example).
## The peer connection was just created and hasn't done any networking yet.
## The ICE agent is in the process of gathering candidates for the connection.
## The ICE agent has finished gathering candidates. If something happens that requires collecting new candidates, such as a new interface being added or the addition of a new ICE server, the state will revert to gathering to gather those candidates.
## There is no ongoing exchange of offer and answer underway. This may mean that the [WebRTCPeerConnection] is new ([constant STATE_NEW]) or that negotiation is complete and a connection has been established ([constant STATE_CONNECTED]).
## The local peer has called [method set_local_description], passing in SDP representing an offer (usually created by calling [method create_offer]), and the offer has been applied successfully.
## The remote peer has created an offer and used the signaling server to deliver it to the local peer, which has set the offer as the remote description by calling [method set_remote_description].
## The offer sent by the remote peer has been applied and an answer has been created and applied by calling [method set_local_description]. This provisional answer describes the supported media formats and so forth, but may not have a complete set of ICE candidates included. Further candidates will be delivered separately later.
## A provisional answer has been received and successfully applied in response to an offer previously sent and established by calling [method set_local_description].
## The [WebRTCPeerConnection] has been closed.

#enum ConnectionState
enum {
    STATE_NEW = 0,
    STATE_CONNECTING = 1,
    STATE_CONNECTED = 2,
    STATE_DISCONNECTED = 3,
    STATE_FAILED = 4,
    STATE_CLOSED = 5,
}
#enum GatheringState
enum {
    GATHERING_STATE_NEW = 0,
    GATHERING_STATE_GATHERING = 1,
    GATHERING_STATE_COMPLETE = 2,
}
#enum SignalingState
enum {
    SIGNALING_STATE_STABLE = 0,
    SIGNALING_STATE_HAVE_LOCAL_OFFER = 1,
    SIGNALING_STATE_HAVE_REMOTE_OFFER = 2,
    SIGNALING_STATE_HAVE_LOCAL_PRANSWER = 3,
    SIGNALING_STATE_HAVE_REMOTE_PRANSWER = 4,
    SIGNALING_STATE_CLOSED = 5,
}


## Add an ice candidate generated by a remote peer (and received over the signaling server). See [signal ice_candidate_created].
func add_ice_candidate(media: String, index: int, name: String) -> int:
	pass;

## Close the peer connection and all data channels associated with it.
## [b]Note:[/b] You cannot reuse this object for a new connection unless you call [method initialize].
func close() -> void:
	pass;

## Returns a new [WebRTCDataChannel] (or [code]null[/code] on failure) with given [param label] and optionally configured via the [param options] dictionary. This method can only be called when the connection is in state [constant STATE_NEW].
## There are two ways to create a working data channel: either call [method create_data_channel] on only one of the peer and listen to [signal data_channel_received] on the other, or call [method create_data_channel] on both peers, with the same values, and the [code]"negotiated"[/code] option set to [code]true[/code].
## Valid [param options] are:
## [codeblock]
## {
## "negotiated": true, # When set to true (default off), means the channel is negotiated out of band. "id" must be set too. "data_channel_received" will not be called.
## "id": 1, # When "negotiated" is true this value must also be set to the same value on both peer.
## # Only one of maxRetransmits and maxPacketLifeTime can be specified, not both. They make the channel unreliable (but also better at real time).
## "maxRetransmits": 1, # Specify the maximum number of attempt the peer will make to retransmits packets if they are not acknowledged.
## "maxPacketLifeTime": 100, # Specify the maximum amount of time before giving up retransmitions of unacknowledged packets (in milliseconds).
## "ordered": true, # When in unreliable mode (i.e. either "maxRetransmits" or "maxPacketLifetime" is set), "ordered" (true by default) specify if packet ordering is to be enforced.
## "protocol": "my-custom-protocol", # A custom sub-protocol string for this channel.
## }
## [/codeblock]
## [b]Note:[/b] You must keep a reference to channels created this way, or it will be closed.
func create_data_channel(label: String, options: Dictionary = {}) -> WebRTCDataChannel:
	pass;

## Creates a new SDP offer to start a WebRTC connection with a remote peer. At least one [WebRTCDataChannel] must have been created before calling this method.
## If this functions returns [constant OK], [signal session_description_created] will be called when the session is ready to be sent.
func create_offer() -> int:
	pass;

## Returns the connection state.
func get_connection_state() -> int:
	pass;

## Returns the ICE [enum GatheringState] of the connection. This lets you detect, for example, when collection of ICE candidates has finished.
func get_gathering_state() -> int:
	pass;

## Returns the signaling state on the local end of the connection while connecting or reconnecting to another peer.
func get_signaling_state() -> int:
	pass;

## Re-initialize this peer connection, closing any previously active connection, and going back to state [constant STATE_NEW]. A dictionary of [param configuration] options can be passed to configure the peer connection.
## Valid [param configuration] options are:
## [codeblock]
## {
## "iceServers": [
## {
## "urls": [ "stun:stun.example.com:3478" ], # One or more STUN servers.
## },
## {
## "urls": [ "turn:turn.example.com:3478" ], # One or more TURN servers.
## "username": "a_username", # Optional username for the TURN server.
## "credential": "a_password", # Optional password for the TURN server.
## }
## ]
## }
## [/codeblock]
func initialize(configuration: Dictionary = {}) -> int:
	pass;

## Call this method frequently (e.g. in [method Node._process] or [method Node._physics_process]) to properly receive signals.
func poll() -> int:
	pass;

## Sets the [param extension_class] as the default [WebRTCPeerConnectionExtension] returned when creating a new [WebRTCPeerConnection].
static func set_default_extension(extension_class: StringName) -> void:
	pass;

## Sets the SDP description of the local peer. This should be called in response to [signal session_description_created].
## After calling this function the peer will start emitting [signal ice_candidate_created] (unless an [enum Error] different from [constant OK] is returned).
func set_local_description(type: String, sdp: String) -> int:
	pass;

## Sets the SDP description of the remote peer. This should be called with the values generated by a remote peer and received over the signaling server.
## If [param type] is [code]"offer"[/code] the peer will emit [signal session_description_created] with the appropriate answer.
## If [param type] is [code]"answer"[/code] the peer will start emitting [signal ice_candidate_created].
func set_remote_description(type: String, sdp: String) -> int:
	pass;


