class_name Vector2

## A 2D vector using floating-point coordinates.
##
## A 2-element structure that can be used to represent 2D coordinates or any other pair of numeric values.
## It uses floating-point coordinates. By default, these floating-point values use 32-bit precision, unlike [float] which is always 64-bit. If double precision is needed, compile the engine with the option [code]precision=double[/code].
## See [Vector2i] for its integer counterpart.
## [b]Note:[/b] In a boolean context, a Vector2 will evaluate to [code]false[/code] if it's equal to [code]Vector2(0, 0)[/code]. Otherwise, a Vector2 will always evaluate to [code]true[/code].
##
## @tutorial(Math documentation index): https://docs.godotengine.org/en/stable/tutorials/math/index.html
## @tutorial(Vector math): https://docs.godotengine.org/en/stable/tutorials/math/vector_math.html
## @tutorial(Advanced vector math): https://docs.godotengine.org/en/stable/tutorials/math/vectors_advanced.html
## @tutorial(3Blue1Brown Essence of Linear Algebra): https://www.youtube.com/playlist?list=PLZHQObOWTQDPD3MizzM2xVFitgF8hE_ab
## @tutorial(Matrix Transform Demo): https://godotengine.org/asset-library/asset/2787
## @tutorial(All 2D Demos): https://github.com/godotengine/godot-demo-projects/tree/master/2d


## Enumerated value for the X axis. Returned by [method max_axis_index] and [method min_axis_index].
## Enumerated value for the Y axis. Returned by [method max_axis_index] and [method min_axis_index].
## Zero vector, a vector with all components set to [code]0[/code].
const ZERO = Vector2(0, 0);

## One vector, a vector with all components set to [code]1[/code].
const ONE = Vector2(1, 1);

## Infinity vector, a vector with all components set to [constant @GDScript.INF].
const INF = Vector2(inf, inf);

## Left unit vector. Represents the direction of left.
const LEFT = Vector2(-1, 0);

## Right unit vector. Represents the direction of right.
const RIGHT = Vector2(1, 0);

## Up unit vector. Y is down in 2D, so this vector points -Y.
const UP = Vector2(0, -1);

## Down unit vector. Y is down in 2D, so this vector points +Y.
const DOWN = Vector2(0, 1);


#enum Axis
enum {
    AXIS_X = 0,
    AXIS_Y = 1,
}
## The vector's X component. Also accessible by using the index position [code][0][/code].
var x: float

## The vector's Y component. Also accessible by using the index position [code][1][/code].
var y: float


## Constructs a default-initialized [Vector2] with all components set to [code]0[/code].
func Vector2() -> Vector2:
	pass;

## Constructs a [Vector2] as a copy of the given [Vector2].
func Vector2(from: Vector2) -> Vector2:
	pass;

## Constructs a new [Vector2] from [Vector2i].
func Vector2(from: Vector2i) -> Vector2:
	pass;

## Constructs a new [Vector2] from the given [param x] and [param y].
func Vector2(x: float, y: float) -> Vector2:
	pass;


## Returns a new vector with all components in absolute values (i.e. positive).
func abs() -> Vector2:
	pass;

## Returns this vector's angle with respect to the positive X axis, or [code](1, 0)[/code] vector, in radians.
## For example, [code]Vector2.RIGHT.angle()[/code] will return zero, [code]Vector2.DOWN.angle()[/code] will return [code]PI / 2[/code] (a quarter turn, or 90 degrees), and [code]Vector2(1, -1).angle()[/code] will return [code]-PI / 4[/code] (a negative eighth turn, or -45 degrees).
## [url=https://raw.githubusercontent.com/godotengine/godot-docs/master/img/vector2_angle.png]Illustration of the returned angle.[/url]
## Equivalent to the result of [method @GlobalScope.atan2] when called with the vector's [member y] and [member x] as parameters: [code]atan2(y, x)[/code].
func angle() -> float:
	pass;

## Returns the signed angle to the given vector, in radians.
## [url=https://raw.githubusercontent.com/godotengine/godot-docs/master/img/vector2_angle_to.png]Illustration of the returned angle.[/url]
func angle_to(to: Vector2) -> float:
	pass;

## Returns the angle between the line connecting the two points and the X axis, in radians.
## [code]a.angle_to_point(b)[/code] is equivalent of doing [code](b - a).angle()[/code].
## [url=https://raw.githubusercontent.com/godotengine/godot-docs/master/img/vector2_angle_to_point.png]Illustration of the returned angle.[/url]
func angle_to_point(to: Vector2) -> float:
	pass;

## Returns the aspect ratio of this vector, the ratio of [member x] to [member y].
func aspect() -> float:
	pass;

## Returns the derivative at the given [param t] on the [url=https://en.wikipedia.org/wiki/B%C3%A9zier_curve]Bézier curve[/url] defined by this vector and the given [param control_1], [param control_2], and [param end] points.
func bezier_derivative(control_1: Vector2, control_2: Vector2, end: Vector2, t: float) -> Vector2:
	pass;

## Returns the point at the given [param t] on the [url=https://en.wikipedia.org/wiki/B%C3%A9zier_curve]Bézier curve[/url] defined by this vector and the given [param control_1], [param control_2], and [param end] points.
func bezier_interpolate(control_1: Vector2, control_2: Vector2, end: Vector2, t: float) -> Vector2:
	pass;

## Returns the vector "bounced off" from a line defined by the given normal [param n] perpendicular to the line.
## [b]Note:[/b] [method bounce] performs the operation that most engines and frameworks call [code skip-lint]reflect()[/code].
func bounce(n: Vector2) -> Vector2:
	pass;

## Returns a new vector with all components rounded up (towards positive infinity).
func ceil() -> Vector2:
	pass;

## Returns a new vector with all components clamped between the components of [param min] and [param max], by running [method @GlobalScope.clamp] on each component.
func clamp(min: Vector2, max: Vector2) -> Vector2:
	pass;

## Returns a new vector with all components clamped between [param min] and [param max], by running [method @GlobalScope.clamp] on each component.
func clampf(min: float, max: float) -> Vector2:
	pass;

## Returns the 2D analog of the cross product for this vector and [param with].
## This is the signed area of the parallelogram formed by the two vectors. If the second vector is clockwise from the first vector, then the cross product is the positive area. If counter-clockwise, the cross product is the negative area. If the two vectors are parallel this returns zero, making it useful for testing if two vectors are parallel.
## [b]Note:[/b] Cross product is not defined in 2D mathematically. This method embeds the 2D vectors in the XY plane of 3D space and uses their cross product's Z component as the analog.
func cross(with: Vector2) -> float:
	pass;

## Performs a cubic interpolation between this vector and [param b] using [param pre_a] and [param post_b] as handles, and returns the result at position [param weight]. [param weight] is on the range of 0.0 to 1.0, representing the amount of interpolation.
func cubic_interpolate(b: Vector2, pre_a: Vector2, post_b: Vector2, weight: float) -> Vector2:
	pass;

## Performs a cubic interpolation between this vector and [param b] using [param pre_a] and [param post_b] as handles, and returns the result at position [param weight]. [param weight] is on the range of 0.0 to 1.0, representing the amount of interpolation.
## It can perform smoother interpolation than [method cubic_interpolate] by the time values.
func cubic_interpolate_in_time(b: Vector2, pre_a: Vector2, post_b: Vector2, weight: float, b_t: float, pre_a_t: float, post_b_t: float) -> Vector2:
	pass;

## Returns the normalized vector pointing from this vector to [param to]. This is equivalent to using [code](b - a).normalized()[/code].
func direction_to(to: Vector2) -> Vector2:
	pass;

## Returns the squared distance between this vector and [param to].
## This method runs faster than [method distance_to], so prefer it if you need to compare vectors or need the squared distance for some formula.
func distance_squared_to(to: Vector2) -> float:
	pass;

## Returns the distance between this vector and [param to].
func distance_to(to: Vector2) -> float:
	pass;

## Returns the dot product of this vector and [param with]. This can be used to compare the angle between two vectors. For example, this can be used to determine whether an enemy is facing the player.
## The dot product will be [code]0[/code] for a right angle (90 degrees), greater than 0 for angles narrower than 90 degrees and lower than 0 for angles wider than 90 degrees.
## When using unit (normalized) vectors, the result will always be between [code]-1.0[/code] (180 degree angle) when the vectors are facing opposite directions, and [code]1.0[/code] (0 degree angle) when the vectors are aligned.
## [b]Note:[/b] [code]a.dot(b)[/code] is equivalent to [code]b.dot(a)[/code].
func dot(with: Vector2) -> float:
	pass;

## Returns a new vector with all components rounded down (towards negative infinity).
func floor() -> Vector2:
	pass;

## Creates a [Vector2] rotated to the given [param angle] in radians. This is equivalent to doing [code]Vector2(cos(angle), sin(angle))[/code] or [code]Vector2.RIGHT.rotated(angle)[/code].
## [codeblock]
## print(Vector2.from_angle(0)) # Prints (1.0, 0.0)
## print(Vector2(1, 0).angle()) # Prints 0.0, which is the angle used above.
## print(Vector2.from_angle(PI / 2)) # Prints (0.0, 1.0)
## [/codeblock]
## [b]Note:[/b] The length of the returned [Vector2] is [i]approximately[/i] [code]1.0[/code], but is is not guaranteed to be exactly [code]1.0[/code] due to floating-point precision issues. Call [method normalized] on the returned [Vector2] if you require a unit vector.
static func from_angle(angle: float) -> Vector2:
	pass;

## Returns [code]true[/code] if this vector and [param to] are approximately equal, by running [method @GlobalScope.is_equal_approx] on each component.
func is_equal_approx(to: Vector2) -> bool:
	pass;

## Returns [code]true[/code] if this vector is finite, by calling [method @GlobalScope.is_finite] on each component.
func is_finite() -> bool:
	pass;

## Returns [code]true[/code] if the vector is normalized, i.e. its length is approximately equal to 1.
func is_normalized() -> bool:
	pass;

## Returns [code]true[/code] if this vector's values are approximately zero, by running [method @GlobalScope.is_zero_approx] on each component.
## This method is faster than using [method is_equal_approx] with one value as a zero vector.
func is_zero_approx() -> bool:
	pass;

## Returns the length (magnitude) of this vector.
func length() -> float:
	pass;

## Returns the squared length (squared magnitude) of this vector.
## This method runs faster than [method length], so prefer it if you need to compare vectors or need the squared distance for some formula.
func length_squared() -> float:
	pass;

## Returns the result of the linear interpolation between this vector and [param to] by amount [param weight]. [param weight] is on the range of [code]0.0[/code] to [code]1.0[/code], representing the amount of interpolation.
func lerp(to: Vector2, weight: float) -> Vector2:
	pass;

## Returns the vector with a maximum length by limiting its length to [param length]. If the vector is non-finite, the result is undefined.
func limit_length(length: float = 1.0) -> Vector2:
	pass;

## Returns the component-wise maximum of this and [param with], equivalent to [code]Vector2(maxf(x, with.x), maxf(y, with.y))[/code].
func max(with: Vector2) -> Vector2:
	pass;

## Returns the axis of the vector's highest value. See [code]AXIS_*[/code] constants. If all components are equal, this method returns [constant AXIS_X].
func max_axis_index() -> int:
	pass;

## Returns the component-wise maximum of this and [param with], equivalent to [code]Vector2(maxf(x, with), maxf(y, with))[/code].
func maxf(with: float) -> Vector2:
	pass;

## Returns the component-wise minimum of this and [param with], equivalent to [code]Vector2(minf(x, with.x), minf(y, with.y))[/code].
func min(with: Vector2) -> Vector2:
	pass;

## Returns the axis of the vector's lowest value. See [code]AXIS_*[/code] constants. If all components are equal, this method returns [constant AXIS_Y].
func min_axis_index() -> int:
	pass;

## Returns the component-wise minimum of this and [param with], equivalent to [code]Vector2(minf(x, with), minf(y, with))[/code].
func minf(with: float) -> Vector2:
	pass;

## Returns a new vector moved toward [param to] by the fixed [param delta] amount. Will not go past the final value.
func move_toward(to: Vector2, delta: float) -> Vector2:
	pass;

## Returns the result of scaling the vector to unit length. Equivalent to [code]v / v.length()[/code]. Returns [code](0, 0)[/code] if [code]v.length() == 0[/code]. See also [method is_normalized].
## [b]Note:[/b] This function may return incorrect values if the input vector length is near zero.
func normalized() -> Vector2:
	pass;

## Returns a perpendicular vector rotated 90 degrees counter-clockwise compared to the original, with the same length.
func orthogonal() -> Vector2:
	pass;

## Returns a vector composed of the [method @GlobalScope.fposmod] of this vector's components and [param mod].
func posmod(mod: float) -> Vector2:
	pass;

## Returns a vector composed of the [method @GlobalScope.fposmod] of this vector's components and [param modv]'s components.
func posmodv(modv: Vector2) -> Vector2:
	pass;

## Returns a new vector resulting from projecting this vector onto the given vector [param b]. The resulting new vector is parallel to [param b]. See also [method slide].
## [b]Note:[/b] If the vector [param b] is a zero vector, the components of the resulting new vector will be [constant @GDScript.NAN].
func project(b: Vector2) -> Vector2:
	pass;

## Returns the result of reflecting the vector from a line defined by the given direction vector [param line].
## [b]Note:[/b] [method reflect] differs from what other engines and frameworks call [code skip-lint]reflect()[/code]. In other engines, [code skip-lint]reflect()[/code] takes a normal direction which is a direction perpendicular to the line. In Godot, you specify the direction of the line directly. See also [method bounce] which does what most engines call [code skip-lint]reflect()[/code].
func reflect(line: Vector2) -> Vector2:
	pass;

## Returns the result of rotating this vector by [param angle] (in radians). See also [method @GlobalScope.deg_to_rad].
func rotated(angle: float) -> Vector2:
	pass;

## Returns a new vector with all components rounded to the nearest integer, with halfway cases rounded away from zero.
func round() -> Vector2:
	pass;

## Returns a new vector with each component set to [code]1.0[/code] if it's positive, [code]-1.0[/code] if it's negative, and [code]0.0[/code] if it's zero. The result is identical to calling [method @GlobalScope.sign] on each component.
func sign() -> Vector2:
	pass;

## Returns the result of spherical linear interpolation between this vector and [param to], by amount [param weight]. [param weight] is on the range of 0.0 to 1.0, representing the amount of interpolation.
## This method also handles interpolating the lengths if the input vectors have different lengths. For the special case of one or both input vectors having zero length, this method behaves like [method lerp].
func slerp(to: Vector2, weight: float) -> Vector2:
	pass;

## Returns a new vector resulting from sliding this vector along a line with normal [param n]. The resulting new vector is perpendicular to [param n], and is equivalent to this vector minus its projection on [param n]. See also [method project].
## [b]Note:[/b] The vector [param n] must be normalized. See also [method normalized].
func slide(n: Vector2) -> Vector2:
	pass;

## Returns a new vector with each component snapped to the nearest multiple of the corresponding component in [param step]. This can also be used to round the components to an arbitrary number of decimals.
func snapped(step: Vector2) -> Vector2:
	pass;

## Returns a new vector with each component snapped to the nearest multiple of [param step]. This can also be used to round the components to an arbitrary number of decimals.
func snappedf(step: float) -> Vector2:
	pass;


