extends Node3D
class_name OpenXRCompositionLayer

## The parent class of all OpenXR composition layer nodes.
##
## Composition layers allow 2D viewports to be displayed inside of the headset by the XR compositor through special projections that retain their quality. This allows for rendering clear text while keeping the layer at a native resolution.
## [b]Note:[/b] If the OpenXR runtime doesn't support the given composition layer type, a fallback mesh can be generated with a [ViewportTexture], in order to emulate the composition layer.


## Perform nearest-neighbor filtering when sampling the texture.
## Perform linear filtering when sampling the texture.
## Perform cubic filtering when sampling the texture.
## Disable mipmapping.
## [b]Note:[/b] Mipmapping can only be disabled in the compatibility renderer.
## Use the mipmap of the nearest resolution.
## Use linear interpolation of the two mipmaps of the nearest resolution.
## Clamp the texture to its specified border color.
## Clamp the texture to its edge color.
## Repeat the texture infinitely.
## Repeat the texture infinitely, mirroring it on each repeat.
## Mirror the texture once and then clamp the texture to its edge color.
## [b]Note:[/b] This wrap mode is not available in the compatibility renderer.
## Maps a color channel to the value of the red channel.
## Maps a color channel to the value of the green channel.
## Maps a color channel to the value of the blue channel.
## Maps a color channel to the value of the alpha channel.
## Maps a color channel to the value of zero.
## Maps a color channel to the value of one.

#enum Filter
enum {
    FILTER_NEAREST = 0,
    FILTER_LINEAR = 1,
    FILTER_CUBIC = 2,
}
#enum MipmapMode
enum {
    MIPMAP_MODE_DISABLED = 0,
    MIPMAP_MODE_NEAREST = 1,
    MIPMAP_MODE_LINEAR = 2,
}
#enum Wrap
enum {
    WRAP_CLAMP_TO_BORDER = 0,
    WRAP_CLAMP_TO_EDGE = 1,
    WRAP_REPEAT = 2,
    WRAP_MIRRORED_REPEAT = 3,
    WRAP_MIRROR_CLAMP_TO_EDGE = 4,
}
#enum Swizzle
enum {
    SWIZZLE_RED = 0,
    SWIZZLE_GREEN = 1,
    SWIZZLE_BLUE = 2,
    SWIZZLE_ALPHA = 3,
    SWIZZLE_ZERO = 4,
    SWIZZLE_ONE = 5,
}
## Enables the blending the layer using its alpha channel.
## Can be combined with [member Viewport.transparent_bg] to give the layer a transparent background.
var alpha_blend: bool:
	get = get_alpha_blend, set = set_alpha_blend

## The size of the Android surface to create if [member use_android_surface] is enabled.
var android_surface_size: Vector2i:
	get = get_android_surface_size, set = set_android_surface_size

## Enables a technique called "hole punching", which allows putting the composition layer behind the main projection layer (i.e. setting [member sort_order] to a negative value) while "punching a hole" through everything rendered by Godot so that the layer is still visible.
## This can be used to create the illusion that the composition layer exists in the same 3D space as everything rendered by Godot, allowing objects to appear to pass both behind or in front of the composition layer.
var enable_hole_punch: bool:
	get = get_enable_hole_punch, set = set_enable_hole_punch

## The [SubViewport] to render on the composition layer.
var layer_viewport: SubViewport:
	get = get_layer_viewport, set = set_layer_viewport

## The sort order for this composition layer. Higher numbers will be shown in front of lower numbers.
## [b]Note:[/b] This will have no effect if a fallback mesh is being used.
var sort_order: int:
	get = get_sort_order, set = set_sort_order

## The swizzle value for the alpha channel of the swapchain state.
## [b]Note:[/b] This property only has an effect on devices that support the OpenXR XR_FB_swapchain_update_state OpenGLES/Vulkan extensions.
var swapchain_state_alpha_swizzle: int:
	get = get_alpha_swizzle, set = set_alpha_swizzle

## The swizzle value for the blue channel of the swapchain state.
## [b]Note:[/b] This property only has an effect on devices that support the OpenXR XR_FB_swapchain_update_state OpenGLES/Vulkan extensions.
var swapchain_state_blue_swizzle: int:
	get = get_blue_swizzle, set = set_blue_swizzle

## The border color of the swapchain state that is used when the wrap mode clamps to the border.
## [b]Note:[/b] This property only has an effect on devices that support the OpenXR XR_FB_swapchain_update_state OpenGLES/Vulkan extensions.
var swapchain_state_border_color: Color:
	get = get_border_color, set = set_border_color

## The swizzle value for the green channel of the swapchain state.
## [b]Note:[/b] This property only has an effect on devices that support the OpenXR XR_FB_swapchain_update_state OpenGLES/Vulkan extensions.
var swapchain_state_green_swizzle: int:
	get = get_green_swizzle, set = set_green_swizzle

## The horizontal wrap mode of the swapchain state.
## [b]Note:[/b] This property only has an effect on devices that support the OpenXR XR_FB_swapchain_update_state OpenGLES/Vulkan extensions.
var swapchain_state_horizontal_wrap: int:
	get = get_horizontal_wrap, set = set_horizontal_wrap

## The magnification filter of the swapchain state.
## [b]Note:[/b] This property only has an effect on devices that support the OpenXR XR_FB_swapchain_update_state OpenGLES/Vulkan extensions.
var swapchain_state_mag_filter: int:
	get = get_mag_filter, set = set_mag_filter

## The max anisotropy of the swapchain state.
## [b]Note:[/b] This property only has an effect on devices that support the OpenXR XR_FB_swapchain_update_state OpenGLES/Vulkan extensions.
var swapchain_state_max_anisotropy: float:
	get = get_max_anisotropy, set = set_max_anisotropy

## The minification filter of the swapchain state.
## [b]Note:[/b] This property only has an effect on devices that support the OpenXR XR_FB_swapchain_update_state OpenGLES/Vulkan extensions.
var swapchain_state_min_filter: int:
	get = get_min_filter, set = set_min_filter

## The mipmap mode of the swapchain state.
## [b]Note:[/b] This property only has an effect on devices that support the OpenXR XR_FB_swapchain_update_state OpenGLES/Vulkan extensions.
var swapchain_state_mipmap_mode: int:
	get = get_mipmap_mode, set = set_mipmap_mode

## The swizzle value for the red channel of the swapchain state.
## [b]Note:[/b] This property only has an effect on devices that support the OpenXR XR_FB_swapchain_update_state OpenGLES/Vulkan extensions.
var swapchain_state_red_swizzle: int:
	get = get_red_swizzle, set = set_red_swizzle

## The vertical wrap mode of the swapchain state.
## [b]Note:[/b] This property only has an effect on devices that support the OpenXR XR_FB_swapchain_update_state OpenGLES/Vulkan extensions.
var swapchain_state_vertical_wrap: int:
	get = get_vertical_wrap, set = set_vertical_wrap

## If enabled, an Android surface will be created (with the dimensions from [member android_surface_size]) which will provide the 2D content for the composition layer, rather than using [member layer_viewport].
## See [method get_android_surface] for information about how to get the surface so that your application can draw to it.
## [b]Note:[/b] This will only work in Android builds.
var use_android_surface: bool:
	get = get_use_android_surface, set = set_use_android_surface



## Returns a [JavaObject] representing an [code]android.view.Surface[/code] if [member use_android_surface] is enabled and OpenXR has created the surface. Otherwise, this will return [code]null[/code].
## [b]Note:[/b] The surface can only be created during an active OpenXR session. So, if [member use_android_surface] is enabled outside of an OpenXR session, it won't be created until a new session fully starts.
func get_android_surface() -> JavaObject:
	pass;

## Returns UV coordinates where the given ray intersects with the composition layer. [param origin] and [param direction] must be in global space.
## Returns [code]Vector2(-1.0, -1.0)[/code] if the ray doesn't intersect.
func intersects_ray(origin: Vector3, direction: Vector3) -> Vector2:
	pass;

## Returns [code]true[/code] if the OpenXR runtime natively supports this composition layer type.
## [b]Note:[/b] This will only return an accurate result after the OpenXR session has started.
func is_natively_supported() -> bool:
	pass;


func get_alpha_blend() -> bool:
	return alpha_blend

func set_alpha_blend(value: bool) -> void:
	alpha_blend = value

func get_android_surface_size() -> Vector2i:
	return android_surface_size

func set_android_surface_size(value: Vector2i) -> void:
	android_surface_size = value

func get_enable_hole_punch() -> bool:
	return enable_hole_punch

func set_enable_hole_punch(value: bool) -> void:
	enable_hole_punch = value

func get_layer_viewport() -> SubViewport:
	return layer_viewport

func set_layer_viewport(value: SubViewport) -> void:
	layer_viewport = value

func get_sort_order() -> int:
	return sort_order

func set_sort_order(value: int) -> void:
	sort_order = value

func get_alpha_swizzle() -> int:
	return swapchain_state_alpha_swizzle

func set_alpha_swizzle(value: int) -> void:
	swapchain_state_alpha_swizzle = value

func get_blue_swizzle() -> int:
	return swapchain_state_blue_swizzle

func set_blue_swizzle(value: int) -> void:
	swapchain_state_blue_swizzle = value

func get_border_color() -> Color:
	return swapchain_state_border_color

func set_border_color(value: Color) -> void:
	swapchain_state_border_color = value

func get_green_swizzle() -> int:
	return swapchain_state_green_swizzle

func set_green_swizzle(value: int) -> void:
	swapchain_state_green_swizzle = value

func get_horizontal_wrap() -> int:
	return swapchain_state_horizontal_wrap

func set_horizontal_wrap(value: int) -> void:
	swapchain_state_horizontal_wrap = value

func get_mag_filter() -> int:
	return swapchain_state_mag_filter

func set_mag_filter(value: int) -> void:
	swapchain_state_mag_filter = value

func get_max_anisotropy() -> float:
	return swapchain_state_max_anisotropy

func set_max_anisotropy(value: float) -> void:
	swapchain_state_max_anisotropy = value

func get_min_filter() -> int:
	return swapchain_state_min_filter

func set_min_filter(value: int) -> void:
	swapchain_state_min_filter = value

func get_mipmap_mode() -> int:
	return swapchain_state_mipmap_mode

func set_mipmap_mode(value: int) -> void:
	swapchain_state_mipmap_mode = value

func get_red_swizzle() -> int:
	return swapchain_state_red_swizzle

func set_red_swizzle(value: int) -> void:
	swapchain_state_red_swizzle = value

func get_vertical_wrap() -> int:
	return swapchain_state_vertical_wrap

func set_vertical_wrap(value: int) -> void:
	swapchain_state_vertical_wrap = value

func get_use_android_surface() -> bool:
	return use_android_surface

func set_use_android_surface(value: bool) -> void:
	use_android_surface = value

