class_name Object

## Base class for all other classes in the engine.
##
## An advanced [Variant] type. All classes in the engine inherit from Object. Each class may define new properties, methods or signals, which are available to all inheriting classes. For example, a [Sprite2D] instance is able to call [method Node.add_child] because it inherits from [Node].
## You can create new instances, using [code]Object.new()[/code] in GDScript, or [code]new GodotObject[/code] in C#.
## To delete an Object instance, call [method free]. This is necessary for most classes inheriting Object, because they do not manage memory on their own, and will otherwise cause memory leaks when no longer in use. There are a few classes that perform memory management. For example, [RefCounted] (and by extension [Resource]) deletes itself when no longer referenced, and [Node] deletes its children when freed.
## Objects can have a [Script] attached to them. Once the [Script] is instantiated, it effectively acts as an extension to the base class, allowing it to define and inherit new properties, methods and signals.
## Inside a [Script], [method _get_property_list] may be overridden to customize properties in several ways. This allows them to be available to the editor, display as lists of options, sub-divide into groups, save on disk, etc. Scripting languages offer easier ways to customize properties, such as with the [annotation @GDScript.@export] annotation.
## Godot is very dynamic. An object's script, and therefore its properties, methods and signals, can be changed at run-time. Because of this, there can be occasions where, for example, a property required by a method may not exist. To prevent run-time errors, see methods such as [method set], [method get], [method call], [method has_method], [method has_signal], etc. Note that these methods are [b]much[/b] slower than direct references.
## In GDScript, you can also check if a given property, method, or signal name exists in an object with the [code]in[/code] operator:
## [codeblock]
## var node = Node.new()
## print("name" in node)         # Prints true
## print("get_parent" in node)   # Prints true
## print("tree_entered" in node) # Prints true
## print("unknown" in node)      # Prints false
## [/codeblock]
## Notifications are [int] constants commonly sent and received by objects. For example, on every rendered frame, the [SceneTree] notifies nodes inside the tree with a [constant Node.NOTIFICATION_PROCESS]. The nodes receive it and may call [method Node._process] to update. To make use of notifications, see [method notification] and [method _notification].
## Lastly, every object can also contain metadata (data about data). [method set_meta] can be useful to store information that the object itself does not depend on. To keep your code clean, making excessive use of metadata is discouraged.
## [b]Note:[/b] Unlike references to a [RefCounted], references to an object stored in a variable can become invalid without being set to [code]null[/code]. To check if an object has been deleted, do [i]not[/i] compare it against [code]null[/code]. Instead, use [method @GlobalScope.is_instance_valid]. It's also recommended to inherit from [RefCounted] for classes storing data instead of [Object].
## [b]Note:[/b] The [code]script[/code] is not exposed like most properties. To set or get an object's [Script] in code, use [method set_script] and [method get_script], respectively.
## [b]Note:[/b] In a boolean context, an [Object] will evaluate to [code]false[/code] if it is equal to [code]null[/code] or it has been freed. Otherwise, an [Object] will always evaluate to [code]true[/code]. See also [method @GlobalScope.is_instance_valid].
##
## @tutorial(Object class introduction): https://docs.godotengine.org/en/stable/contributing/development/core_and_modules/object_class.html
## @tutorial(When and how to avoid using nodes for everything): https://docs.godotengine.org/en/stable/tutorials/best_practices/node_alternatives.html
## @tutorial(Object notifications): https://docs.godotengine.org/en/stable/tutorials/best_practices/godot_notifications.html


## Emitted when [method notify_property_list_changed] is called.
signal property_list_changed
## Emitted when the object's script is changed.
## [b]Note:[/b] When this signal is emitted, the new script is not initialized yet. If you need to access the new script, defer connections to this signal with [constant CONNECT_DEFERRED].
signal script_changed
## Notification received when the object is initialized, before its script is attached. Used internally.
const NOTIFICATION_POSTINITIALIZE = 0;

## Notification received when the object is about to be deleted. Can be used like destructors in object-oriented programming languages.
const NOTIFICATION_PREDELETE = 1;

## Notification received when the object finishes hot reloading. This notification is only sent for extensions classes and derived.
const NOTIFICATION_EXTENSION_RELOADED = 2;

## Deferred connections trigger their [Callable]s on idle time (at the end of the frame), rather than instantly.
## Persisting connections are stored when the object is serialized (such as when using [method PackedScene.pack]). In the editor, connections created through the Node dock are always persisting.
## One-shot connections disconnect themselves after emission.
## Reference-counted connections can be assigned to the same [Callable] multiple times. Each disconnection decreases the internal counter. The signal fully disconnects only when the counter reaches 0.
## The source object is automatically bound when a [PackedScene] is instantiated. If this flag bit is enabled, the source object will be appended right after the original arguments of the signal.

#enum ConnectFlags
enum {
    CONNECT_DEFERRED = 1,
    CONNECT_PERSIST = 2,
    CONNECT_ONE_SHOT = 4,
    CONNECT_REFERENCE_COUNTED = 8,
    CONNECT_APPEND_SOURCE_OBJECT = 16,
}


## Override this method to customize the behavior of [method get]. Should return the given [param property]'s value, or [code]null[/code] if the [param property] should be handled normally.
## Combined with [method _set] and [method _get_property_list], this method allows defining custom properties, which is particularly useful for editor plugins. Note that a property must be present in [method get_property_list], otherwise this method will not be called.
## [codeblocks]
## [gdscript]
## func _get(property):
## if property == "fake_property":
## print("Getting my property!")
## return 4
## func _get_property_list():
## return [
## { "name": "fake_property", "type": TYPE_INT }
## ]
## [/gdscript]
## [csharp]
## public override Variant _Get(StringName property)
## {
## if (property == "FakeProperty")
## {
## GD.Print("Getting my property!");
## return 4;
## }
## return default;
## }
## public override Godot.Collections.Array<Godot.Collections.Dictionary> _GetPropertyList()
## {
## return
## [
## new Godot.Collections.Dictionary()
## {
## { "name", "FakeProperty" },
## { "type", (int)Variant.Type.Int },
## },
## ];
## }
## [/csharp]
## [/codeblocks]
func _get(property: StringName) -> Variant:
	pass;

## Override this method to provide a custom list of additional properties to handle by the engine.
## Should return a property list, as an [Array] of dictionaries. The result is added to the array of [method get_property_list], and should be formatted in the same way. Each [Dictionary] must at least contain the [code]name[/code] and [code]type[/code] entries.
## You can use [method _property_can_revert] and [method _property_get_revert] to customize the default values of the properties added by this method.
## The example below displays a list of numbers shown as words going from [code]ZERO[/code] to [code]FIVE[/code], with [code]number_count[/code] controlling the size of the list:
## [codeblocks]
## [gdscript]
## @tool
## extends Node
## @export var number_count = 3:
## set(nc):
## number_count = nc
## numbers.resize(number_count)
## notify_property_list_changed()
## var numbers = PackedInt32Array([0, 0, 0])
## func _get_property_list():
## var properties = []
## for i in range(number_count):
## properties.append({
## "name": "number_%d" % i,
## "type": TYPE_INT,
## "hint": PROPERTY_HINT_ENUM,
## "hint_string": "ZERO,ONE,TWO,THREE,FOUR,FIVE",
## })
## return properties
## func _get(property):
## if property.begins_with("number_"):
## var index = property.get_slice("_", 1).to_int()
## return numbers[index]
## func _set(property, value):
## if property.begins_with("number_"):
## var index = property.get_slice("_", 1).to_int()
## numbers[index] = value
## return true
## return false
## [/gdscript]
## [csharp]
## [Tool]
## public partial class MyNode : Node
## {
## private int _numberCount;
## [Export]
## public int NumberCount
## {
## get => _numberCount;
## set
## {
## _numberCount = value;
## _numbers.Resize(_numberCount);
## NotifyPropertyListChanged();
## }
## }
## private Godot.Collections.Array<int> _numbers = [];
## public override Godot.Collections.Array<Godot.Collections.Dictionary> _GetPropertyList()
## {
## Godot.Collections.Array<Godot.Collections.Dictionary> properties = [];
## for (int i = 0; i < _numberCount; i++)
## {
## properties.Add(new Godot.Collections.Dictionary()
## {
## { "name", $"number_{i}" },
## { "type", (int)Variant.Type.Int },
## { "hint", (int)PropertyHint.Enum },
## { "hint_string", "Zero,One,Two,Three,Four,Five" },
## });
## }
## return properties;
## }
## public override Variant _Get(StringName property)
## {
## string propertyName = property.ToString();
## if (propertyName.StartsWith("number_"))
## {
## int index = int.Parse(propertyName.Substring("number_".Length));
## return _numbers[index];
## }
## return default;
## }
## public override bool _Set(StringName property, Variant value)
## {
## string propertyName = property.ToString();
## if (propertyName.StartsWith("number_"))
## {
## int index = int.Parse(propertyName.Substring("number_".Length));
## _numbers[index] = value.As<int>();
## return true;
## }
## return false;
## }
## }
## [/csharp]
## [/codeblocks]
## [b]Note:[/b] This method is intended for advanced purposes. For most common use cases, the scripting languages offer easier ways to handle properties. See [annotation @GDScript.@export], [annotation @GDScript.@export_enum], [annotation @GDScript.@export_group], etc. If you want to customize exported properties, use [method _validate_property].
## [b]Note:[/b] If the object's script is not [annotation @GDScript.@tool], this method will not be called in the editor.
func _get_property_list() -> Array[Dictionary]:
	pass;

## Called when the object's script is instantiated, oftentimes after the object is initialized in memory (through [code]Object.new()[/code] in GDScript, or [code]new GodotObject[/code] in C#). It can be also defined to take in parameters. This method is similar to a constructor in most programming languages.
## [b]Note:[/b] If [method _init] is defined with [i]required[/i] parameters, the Object with script may only be created directly. If any other means (such as [method PackedScene.instantiate] or [method Node.duplicate]) are used, the script's initialization will fail.
func _init() -> void:
	pass;

## Returns the current iterable value. [param iter] stores the iteration state, but unlike [method _iter_init] and [method _iter_next] the state is supposed to be read-only, so there is no [Array] wrapper.
## [b]Tip:[/b] In GDScript, you can use a subtype of [Variant] as the return type for [method _iter_get]. The specified type will be used to set the type of the iterator variable in [code]for[/code] loops, enhancing type safety.
func _iter_get(iter: Variant) -> Variant:
	pass;

## Initializes the iterator. [param iter] stores the iteration state. Since GDScript does not support passing arguments by reference, a single-element array is used as a wrapper. Returns [code]true[/code] so long as the iterator has not reached the end.
## [codeblock]
## class MyRange:
## var _from
## var _to
## func _init(from, to):
## assert(from <= to)
## _from = from
## _to = to
## func _iter_init(iter):
## iter[0] = _from
## return iter[0] < _to
## func _iter_next(iter):
## iter[0] += 1
## return iter[0] < _to
## func _iter_get(iter):
## return iter
## func _ready():
## var my_range = MyRange.new(2, 5)
## for x in my_range:
## print(x) # Prints 2, 3, 4.
## [/codeblock]
## [b]Note:[/b] Alternatively, you can ignore [param iter] and use the object's state instead, see [url=$DOCS_URL/tutorials/scripting/gdscript/gdscript_advanced.html#custom-iterators]online docs[/url] for an example. Note that in this case you will not be able to reuse the same iterator instance in nested loops. Also, make sure you reset the iterator state in this method if you want to reuse the same instance multiple times.
func _iter_init(iter: Array) -> bool:
	pass;

## Moves the iterator to the next iteration. [param iter] stores the iteration state. Since GDScript does not support passing arguments by reference, a single-element array is used as a wrapper. Returns [code]true[/code] so long as the iterator has not reached the end.
func _iter_next(iter: Array) -> bool:
	pass;

## Called when the object receives a notification, which can be identified in [param what] by comparing it with a constant. See also [method notification].
## [codeblocks]
## [gdscript]
## func _notification(what):
## if what == NOTIFICATION_PREDELETE:
## print("Goodbye!")
## [/gdscript]
## [csharp]
## public override void _Notification(int what)
## {
## if (what == NotificationPredelete)
## {
## GD.Print("Goodbye!");
## }
## }
## [/csharp]
## [/codeblocks]
## [b]Note:[/b] The base [Object] defines a few notifications ([constant NOTIFICATION_POSTINITIALIZE] and [constant NOTIFICATION_PREDELETE]). Inheriting classes such as [Node] define a lot more notifications, which are also received by this method.
func _notification(what: int) -> void:
	pass;

## Override this method to customize the given [param property]'s revert behavior. Should return [code]true[/code] if the [param property] has a custom default value and is revertible in the Inspector dock. Use [method _property_get_revert] to specify the [param property]'s default value.
## [b]Note:[/b] This method must return consistently, regardless of the current value of the [param property].
func _property_can_revert(property: StringName) -> bool:
	pass;

## Override this method to customize the given [param property]'s revert behavior. Should return the default value for the [param property]. If the default value differs from the [param property]'s current value, a revert icon is displayed in the Inspector dock.
## [b]Note:[/b] [method _property_can_revert] must also be overridden for this method to be called.
func _property_get_revert(property: StringName) -> Variant:
	pass;

## Override this method to customize the behavior of [method set]. Should set the [param property] to [param value] and return [code]true[/code], or [code]false[/code] if the [param property] should be handled normally. The [i]exact[/i] way to set the [param property] is up to this method's implementation.
## Combined with [method _get] and [method _get_property_list], this method allows defining custom properties, which is particularly useful for editor plugins. Note that a property [i]must[/i] be present in [method get_property_list], otherwise this method will not be called.
## [codeblocks]
## [gdscript]
## var internal_data = {}
## func _set(property, value):
## if property == "fake_property":
## # Storing the value in the fake property.
## internal_data["fake_property"] = value
## return true
## return false
## func _get_property_list():
## return [
## { "name": "fake_property", "type": TYPE_INT }
## ]
## [/gdscript]
## [csharp]
## private Godot.Collections.Dictionary _internalData = new Godot.Collections.Dictionary();
## public override bool _Set(StringName property, Variant value)
## {
## if (property == "FakeProperty")
## {
## // Storing the value in the fake property.
## _internalData["FakeProperty"] = value;
## return true;
## }
## return false;
## }
## public override Godot.Collections.Array<Godot.Collections.Dictionary> _GetPropertyList()
## {
## return
## [
## new Godot.Collections.Dictionary()
## {
## { "name", "FakeProperty" },
## { "type", (int)Variant.Type.Int },
## },
## ];
## }
## [/csharp]
## [/codeblocks]
func _set(property: StringName, value: Variant) -> bool:
	pass;

## Override this method to customize the return value of [method to_string], and therefore the object's representation as a [String].
## [codeblock]
## func _to_string():
## return "Welcome to Godot 4!"
## func _init():
## print(self)       # Prints "Welcome to Godot 4!"
## var a = str(self) # a is "Welcome to Godot 4!"
## [/codeblock]
func _to_string() -> String:
	pass;

## Override this method to customize existing properties. Every property info goes through this method, except properties added with [method _get_property_list]. The dictionary contents is the same as in [method _get_property_list].
## [codeblocks]
## [gdscript]
## @tool
## extends Node
## @export var is_number_editable: bool:
## set(value):
## is_number_editable = value
## notify_property_list_changed()
## @export var number: int
## func _validate_property(property: Dictionary):
## if property.name == "number" and not is_number_editable:
## property.usage |= PROPERTY_USAGE_READ_ONLY
## [/gdscript]
## [csharp]
## [Tool]
## public partial class MyNode : Node
## {
## private bool _isNumberEditable;
## [Export]
## public bool IsNumberEditable
## {
## get => _isNumberEditable;
## set
## {
## _isNumberEditable = value;
## NotifyPropertyListChanged();
## }
## }
## [Export]
## public int Number { get; set; }
## public override void _ValidateProperty(Godot.Collections.Dictionary property)
## {
## if (property["name"].AsStringName() == PropertyName.Number && !IsNumberEditable)
## {
## var usage = property["usage"].As<PropertyUsageFlags>() | PropertyUsageFlags.ReadOnly;
## property["usage"] = (int)usage;
## }
## }
## }
## [/csharp]
## [/codeblocks]
func _validate_property(property: Dictionary) -> void:
	pass;

## Adds a user-defined signal named [param signal]. Optional arguments for the signal can be added as an [Array] of dictionaries, each defining a [code]name[/code] [String] and a [code]type[/code] [int] (see [enum Variant.Type]). See also [method has_user_signal] and [method remove_user_signal].
## [codeblocks]
## [gdscript]
## add_user_signal("hurt", [
## { "name": "damage", "type": TYPE_INT },
## { "name": "source", "type": TYPE_OBJECT }
## ])
## [/gdscript]
## [csharp]
## AddUserSignal("Hurt",
## [
## new Godot.Collections.Dictionary()
## {
## { "name", "damage" },
## { "type", (int)Variant.Type.Int },
## },
## new Godot.Collections.Dictionary()
## {
## { "name", "source" },
## { "type", (int)Variant.Type.Object },
## },
## ]);
## [/csharp]
## [/codeblocks]
func add_user_signal(signal: String, arguments: Array = []) -> void:
	pass;

## Calls the [param method] on the object and returns the result. This method supports a variable number of arguments, so parameters can be passed as a comma separated list.
## [codeblocks]
## [gdscript]
## var node = Node3D.new()
## node.call("rotate", Vector3(1.0, 0.0, 0.0), 1.571)
## [/gdscript]
## [csharp]
## var node = new Node3D();
## node.Call(Node3D.MethodName.Rotate, new Vector3(1f, 0f, 0f), 1.571f);
## [/csharp]
## [/codeblocks]
## [b]Note:[/b] In C#, [param method] must be in snake_case when referring to built-in Godot methods. Prefer using the names exposed in the [code]MethodName[/code] class to avoid allocating a new [StringName] on each call.
vararg func call(method: StringName) -> Variant:
	pass;

## Calls the [param method] on the object during idle time. Always returns [code]null[/code], [b]not[/b] the method's result.
## Idle time happens mainly at the end of process and physics frames. In it, deferred calls will be run until there are none left, which means you can defer calls from other deferred calls and they'll still be run in the current idle time cycle. This means you should not call a method deferred from itself (or from a method called by it), as this causes infinite recursion the same way as if you had called the method directly.
## This method supports a variable number of arguments, so parameters can be passed as a comma separated list.
## [codeblocks]
## [gdscript]
## var node = Node3D.new()
## node.call_deferred("rotate", Vector3(1.0, 0.0, 0.0), 1.571)
## [/gdscript]
## [csharp]
## var node = new Node3D();
## node.CallDeferred(Node3D.MethodName.Rotate, new Vector3(1f, 0f, 0f), 1.571f);
## [/csharp]
## [/codeblocks]
## See also [method Callable.call_deferred].
## [b]Note:[/b] In C#, [param method] must be in snake_case when referring to built-in Godot methods. Prefer using the names exposed in the [code]MethodName[/code] class to avoid allocating a new [StringName] on each call.
## [b]Note:[/b] If you're looking to delay the function call by a frame, refer to the [signal SceneTree.process_frame] and [signal SceneTree.physics_frame] signals.
## [codeblock]
## var node = Node3D.new()
## # Make a Callable and bind the arguments to the node's rotate() call.
## var callable = node.rotate.bind(Vector3(1.0, 0.0, 0.0), 1.571)
## # Connect the callable to the process_frame signal, so it gets called in the next process frame.
## # CONNECT_ONE_SHOT makes sure it only gets called once instead of every frame.
## get_tree().process_frame.connect(callable, CONNECT_ONE_SHOT)
## [/codeblock]
vararg func call_deferred(method: StringName) -> Variant:
	pass;

## Calls the [param method] on the object and returns the result. Unlike [method call], this method expects all parameters to be contained inside [param arg_array].
## [codeblocks]
## [gdscript]
## var node = Node3D.new()
## node.callv("rotate", [Vector3(1.0, 0.0, 0.0), 1.571])
## [/gdscript]
## [csharp]
## var node = new Node3D();
## node.Callv(Node3D.MethodName.Rotate, [new Vector3(1f, 0f, 0f), 1.571f]);
## [/csharp]
## [/codeblocks]
## [b]Note:[/b] In C#, [param method] must be in snake_case when referring to built-in Godot methods. Prefer using the names exposed in the [code]MethodName[/code] class to avoid allocating a new [StringName] on each call.
func callv(method: StringName, arg_array: Array) -> Variant:
	pass;

## Returns [code]true[/code] if the object is allowed to translate messages with [method tr] and [method tr_n]. See also [method set_message_translation].
func can_translate_messages() -> bool:
	pass;

## If this method is called during [constant NOTIFICATION_PREDELETE], this object will reject being freed and will remain allocated. This is mostly an internal function used for error handling to avoid the user from freeing objects when they are not intended to.
func cancel_free() -> void:
	pass;

## Connects a [param signal] by name to a [param callable]. Optional [param flags] can be also added to configure the connection's behavior (see [enum ConnectFlags] constants).
## A signal can only be connected once to the same [Callable]. If the signal is already connected, this method returns [constant ERR_INVALID_PARAMETER] and generates an error, unless the signal is connected with [constant CONNECT_REFERENCE_COUNTED]. To prevent this, use [method is_connected] first to check for existing connections.
## [b]Note:[/b] If the [param callable]'s object is freed, the connection will be lost.
## [b]Note:[/b] In GDScript, it is generally recommended to connect signals with [method Signal.connect] instead.
## [b]Note:[/b] This operation (and all other signal related operations) is thread-safe.
func connect(signal: StringName, callable: Callable, flags: int = 0) -> int:
	pass;

## Disconnects a [param signal] by name from a given [param callable]. If the connection does not exist, generates an error. Use [method is_connected] to make sure that the connection exists.
func disconnect(signal: StringName, callable: Callable) -> void:
	pass;

## Emits the given [param signal] by name. The signal must exist, so it should be a built-in signal of this class or one of its inherited classes, or a user-defined signal (see [method add_user_signal]). This method supports a variable number of arguments, so parameters can be passed as a comma separated list.
## Returns [constant ERR_UNAVAILABLE] if [param signal] does not exist or the parameters are invalid.
## [codeblocks]
## [gdscript]
## emit_signal("hit", "sword", 100)
## emit_signal("game_over")
## [/gdscript]
## [csharp]
## EmitSignal(SignalName.Hit, "sword", 100);
## EmitSignal(SignalName.GameOver);
## [/csharp]
## [/codeblocks]
## [b]Note:[/b] In C#, [param signal] must be in snake_case when referring to built-in Godot signals. Prefer using the names exposed in the [code]SignalName[/code] class to avoid allocating a new [StringName] on each call.
vararg func emit_signal(signal: StringName) -> int:
	pass;

## Deletes the object from memory. Pre-existing references to the object become invalid, and any attempt to access them will result in a runtime error. Checking the references with [method @GlobalScope.is_instance_valid] will return [code]false[/code]. This is equivalent to the [code]memdelete[/code] function in GDExtension C++.
func free() -> void:
	pass;

## Returns the [Variant] value of the given [param property]. If the [param property] does not exist, this method returns [code]null[/code].
## [codeblocks]
## [gdscript]
## var node = Node2D.new()
## node.rotation = 1.5
## var a = node.get("rotation") # a is 1.5
## [/gdscript]
## [csharp]
## var node = new Node2D();
## node.Rotation = 1.5f;
## var a = node.Get(Node2D.PropertyName.Rotation); // a is 1.5
## [/csharp]
## [/codeblocks]
## [b]Note:[/b] In C#, [param property] must be in snake_case when referring to built-in Godot properties. Prefer using the names exposed in the [code]PropertyName[/code] class to avoid allocating a new [StringName] on each call.
func get(property: StringName) -> Variant:
	pass;

## Returns the object's built-in class name, as a [String]. See also [method is_class].
## [b]Note:[/b] This method ignores [code]class_name[/code] declarations. If this object's script has defined a [code]class_name[/code], the base, built-in class name is returned instead.
func get_class() -> String:
	pass;

## Returns an [Array] of signal connections received by this object. Each connection is represented as a [Dictionary] that contains three entries:
## - [code]signal[/code] is a reference to the [Signal];
## - [code]callable[/code] is a reference to the [Callable];
## - [code]flags[/code] is a combination of [enum ConnectFlags].
func get_incoming_connections() -> Array[Dictionary]:
	pass;

## Gets the object's property indexed by the given [param property_path]. The path should be a [NodePath] relative to the current object and can use the colon character ([code]:[/code]) to access nested properties.
## [b]Examples:[/b] [code]"position:x"[/code] or [code]"material:next_pass:blend_mode"[/code].
## [codeblocks]
## [gdscript]
## var node = Node2D.new()
## node.position = Vector2(5, -10)
## var a = node.get_indexed("position")   # a is Vector2(5, -10)
## var b = node.get_indexed("position:y") # b is -10
## [/gdscript]
## [csharp]
## var node = new Node2D();
## node.Position = new Vector2(5, -10);
## var a = node.GetIndexed("position");   // a is Vector2(5, -10)
## var b = node.GetIndexed("position:y"); // b is -10
## [/csharp]
## [/codeblocks]
## [b]Note:[/b] In C#, [param property_path] must be in snake_case when referring to built-in Godot properties. Prefer using the names exposed in the [code]PropertyName[/code] class to avoid allocating a new [StringName] on each call.
## [b]Note:[/b] This method does not support actual paths to nodes in the [SceneTree], only sub-property paths. In the context of nodes, use [method Node.get_node_and_resource] instead.
func get_indexed(property_path: NodePath) -> Variant:
	pass;

## Returns the object's unique instance ID. This ID can be saved in [EncodedObjectAsID], and can be used to retrieve this object instance with [method @GlobalScope.instance_from_id].
## [b]Note:[/b] This ID is only useful during the current session. It won't correspond to a similar object if the ID is sent over a network, or loaded from a file at a later time.
func get_instance_id() -> int:
	pass;

## Returns the object's metadata value for the given entry [param name]. If the entry does not exist, returns [param default]. If [param default] is [code]null[/code], an error is also generated.
## [b]Note:[/b] A metadata's name must be a valid identifier as per [method StringName.is_valid_identifier] method.
## [b]Note:[/b] Metadata that has a name starting with an underscore ([code]_[/code]) is considered editor-only. Editor-only metadata is not displayed in the Inspector and should not be edited, although it can still be found by this method.
func get_meta(name: StringName, default: Variant = null) -> Variant:
	pass;

## Returns the object's metadata entry names as an [Array] of [StringName]s.
func get_meta_list() -> Array[StringName]:
	pass;

## Returns the number of arguments of the given [param method] by name.
## [b]Note:[/b] In C#, [param method] must be in snake_case when referring to built-in Godot methods. Prefer using the names exposed in the [code]MethodName[/code] class to avoid allocating a new [StringName] on each call.
func get_method_argument_count(method: StringName) -> int:
	pass;

## Returns this object's methods and their signatures as an [Array] of dictionaries. Each [Dictionary] contains the following entries:
## - [code]name[/code] is the name of the method, as a [String];
## - [code]args[/code] is an [Array] of dictionaries representing the arguments;
## - [code]default_args[/code] is the default arguments as an [Array] of variants;
## - [code]flags[/code] is a combination of [enum MethodFlags];
## - [code]id[/code] is the method's internal identifier [int];
## - [code]return[/code] is the returned value, as a [Dictionary];
## [b]Note:[/b] The dictionaries of [code]args[/code] and [code]return[/code] are formatted identically to the results of [method get_property_list], although not all entries are used.
func get_method_list() -> Array[Dictionary]:
	pass;

## Returns the object's property list as an [Array] of dictionaries. Each [Dictionary] contains the following entries:
## - [code]name[/code] is the property's name, as a [String];
## - [code]class_name[/code] is an empty [StringName], unless the property is [constant TYPE_OBJECT] and it inherits from a class;
## - [code]type[/code] is the property's type, as an [int] (see [enum Variant.Type]);
## - [code]hint[/code] is [i]how[/i] the property is meant to be edited (see [enum PropertyHint]);
## - [code]hint_string[/code] depends on the hint (see [enum PropertyHint]);
## - [code]usage[/code] is a combination of [enum PropertyUsageFlags].
## [b]Note:[/b] In GDScript, all class members are treated as properties. In C# and GDExtension, it may be necessary to explicitly mark class members as Godot properties using decorators or attributes.
func get_property_list() -> Array[Dictionary]:
	pass;

## Returns the object's [Script] instance, or [code]null[/code] if no script is attached.
func get_script() -> Variant:
	pass;

## Returns an [Array] of connections for the given [param signal] name. Each connection is represented as a [Dictionary] that contains three entries:
## - [code skip-lint]signal[/code] is a reference to the [Signal];
## - [code]callable[/code] is a reference to the connected [Callable];
## - [code]flags[/code] is a combination of [enum ConnectFlags].
func get_signal_connection_list(signal: StringName) -> Array[Dictionary]:
	pass;

## Returns the list of existing signals as an [Array] of dictionaries.
## [b]Note:[/b] Due of the implementation, each [Dictionary] is formatted very similarly to the returned values of [method get_method_list].
func get_signal_list() -> Array[Dictionary]:
	pass;

## Returns the name of the translation domain used by [method tr] and [method tr_n]. See also [TranslationServer].
func get_translation_domain() -> StringName:
	pass;

## Returns [code]true[/code] if any connection exists on the given [param signal] name.
## [b]Note:[/b] In C#, [param signal] must be in snake_case when referring to built-in Godot methods. Prefer using the names exposed in the [code]SignalName[/code] class to avoid allocating a new [StringName] on each call.
func has_connections(signal: StringName) -> bool:
	pass;

## Returns [code]true[/code] if a metadata entry is found with the given [param name]. See also [method get_meta], [method set_meta] and [method remove_meta].
## [b]Note:[/b] A metadata's name must be a valid identifier as per [method StringName.is_valid_identifier] method.
## [b]Note:[/b] Metadata that has a name starting with an underscore ([code]_[/code]) is considered editor-only. Editor-only metadata is not displayed in the Inspector and should not be edited, although it can still be found by this method.
func has_meta(name: StringName) -> bool:
	pass;

## Returns [code]true[/code] if the given [param method] name exists in the object.
## [b]Note:[/b] In C#, [param method] must be in snake_case when referring to built-in Godot methods. Prefer using the names exposed in the [code]MethodName[/code] class to avoid allocating a new [StringName] on each call.
func has_method(method: StringName) -> bool:
	pass;

## Returns [code]true[/code] if the given [param signal] name exists in the object.
## [b]Note:[/b] In C#, [param signal] must be in snake_case when referring to built-in Godot signals. Prefer using the names exposed in the [code]SignalName[/code] class to avoid allocating a new [StringName] on each call.
func has_signal(signal: StringName) -> bool:
	pass;

## Returns [code]true[/code] if the given user-defined [param signal] name exists. Only signals added with [method add_user_signal] are included. See also [method remove_user_signal].
func has_user_signal(signal: StringName) -> bool:
	pass;

## Returns [code]true[/code] if the object is blocking its signals from being emitted. See [method set_block_signals].
func is_blocking_signals() -> bool:
	pass;

## Returns [code]true[/code] if the object inherits from the given [param class]. See also [method get_class].
## [codeblocks]
## [gdscript]
## var sprite2d = Sprite2D.new()
## sprite2d.is_class("Sprite2D") # Returns true
## sprite2d.is_class("Node")     # Returns true
## sprite2d.is_class("Node3D")   # Returns false
## [/gdscript]
## [csharp]
## var sprite2D = new Sprite2D();
## sprite2D.IsClass("Sprite2D"); // Returns true
## sprite2D.IsClass("Node");     // Returns true
## sprite2D.IsClass("Node3D");   // Returns false
## [/csharp]
## [/codeblocks]
## [b]Note:[/b] This method ignores [code]class_name[/code] declarations in the object's script.
func is_class(class: String) -> bool:
	pass;

## Returns [code]true[/code] if a connection exists between the given [param signal] name and [param callable].
## [b]Note:[/b] In C#, [param signal] must be in snake_case when referring to built-in Godot signals. Prefer using the names exposed in the [code]SignalName[/code] class to avoid allocating a new [StringName] on each call.
func is_connected(signal: StringName, callable: Callable) -> bool:
	pass;

## Returns [code]true[/code] if the [method Node.queue_free] method was called for the object.
func is_queued_for_deletion() -> bool:
	pass;

## Sends the given [param what] notification to all classes inherited by the object, triggering calls to [method _notification], starting from the highest ancestor (the [Object] class) and going down to the object's script.
## If [param reversed] is [code]true[/code], the call order is reversed.
## [codeblocks]
## [gdscript]
## var player = Node2D.new()
## player.set_script(load("res://player.gd"))
## player.notification(NOTIFICATION_ENTER_TREE)
## # The call order is Object -> Node -> Node2D -> player.gd.
## player.notification(NOTIFICATION_ENTER_TREE, true)
## # The call order is player.gd -> Node2D -> Node -> Object.
## [/gdscript]
## [csharp]
## var player = new Node2D();
## player.SetScript(GD.Load("res://player.gd"));
## player.Notification(NotificationEnterTree);
## // The call order is GodotObject -> Node -> Node2D -> player.gd.
## player.Notification(NotificationEnterTree, true);
## // The call order is player.gd -> Node2D -> Node -> GodotObject.
## [/csharp]
## [/codeblocks]
func notification(what: int, reversed: bool = false) -> void:
	pass;

## Emits the [signal property_list_changed] signal. This is mainly used to refresh the editor, so that the Inspector and editor plugins are properly updated.
func notify_property_list_changed() -> void:
	pass;

## Returns [code]true[/code] if the given [param property] has a custom default value. Use [method property_get_revert] to get the [param property]'s default value.
## [b]Note:[/b] This method is used by the Inspector dock to display a revert icon. The object must implement [method _property_can_revert] to customize the default value. If [method _property_can_revert] is not implemented, this method returns [code]false[/code].
func property_can_revert(property: StringName) -> bool:
	pass;

## Returns the custom default value of the given [param property]. Use [method property_can_revert] to check if the [param property] has a custom default value.
## [b]Note:[/b] This method is used by the Inspector dock to display a revert icon. The object must implement [method _property_get_revert] to customize the default value. If [method _property_get_revert] is not implemented, this method returns [code]null[/code].
func property_get_revert(property: StringName) -> Variant:
	pass;

## Removes the given entry [param name] from the object's metadata. See also [method has_meta], [method get_meta] and [method set_meta].
## [b]Note:[/b] A metadata's name must be a valid identifier as per [method StringName.is_valid_identifier] method.
## [b]Note:[/b] Metadata that has a name starting with an underscore ([code]_[/code]) is considered editor-only. Editor-only metadata is not displayed in the Inspector and should not be edited, although it can still be found by this method.
func remove_meta(name: StringName) -> void:
	pass;

## Removes the given user signal [param signal] from the object. See also [method add_user_signal] and [method has_user_signal].
func remove_user_signal(signal: StringName) -> void:
	pass;

## Assigns [param value] to the given [param property]. If the property does not exist or the given [param value]'s type doesn't match, nothing happens.
## [codeblocks]
## [gdscript]
## var node = Node2D.new()
## node.set("global_scale", Vector2(8, 2.5))
## print(node.global_scale) # Prints (8.0, 2.5)
## [/gdscript]
## [csharp]
## var node = new Node2D();
## node.Set(Node2D.PropertyName.GlobalScale, new Vector2(8, 2.5f));
## GD.Print(node.GlobalScale); // Prints (8, 2.5)
## [/csharp]
## [/codeblocks]
## [b]Note:[/b] In C#, [param property] must be in snake_case when referring to built-in Godot properties. Prefer using the names exposed in the [code]PropertyName[/code] class to avoid allocating a new [StringName] on each call.
func set(property: StringName, value: Variant) -> void:
	pass;

## If set to [code]true[/code], the object becomes unable to emit signals. As such, [method emit_signal] and signal connections will not work, until it is set to [code]false[/code].
func set_block_signals(enable: bool) -> void:
	pass;

## Assigns [param value] to the given [param property], at the end of the current frame. This is equivalent to calling [method set] through [method call_deferred].
## [codeblocks]
## [gdscript]
## var node = Node2D.new()
## add_child(node)
## node.rotation = 1.5
## node.set_deferred("rotation", 3.0)
## print(node.rotation) # Prints 1.5
## await get_tree().process_frame
## print(node.rotation) # Prints 3.0
## [/gdscript]
## [csharp]
## var node = new Node2D();
## node.Rotation = 1.5f;
## node.SetDeferred(Node2D.PropertyName.Rotation, 3f);
## GD.Print(node.Rotation); // Prints 1.5
## await ToSignal(GetTree(), SceneTree.SignalName.ProcessFrame);
## GD.Print(node.Rotation); // Prints 3.0
## [/csharp]
## [/codeblocks]
## [b]Note:[/b] In C#, [param property] must be in snake_case when referring to built-in Godot properties. Prefer using the names exposed in the [code]PropertyName[/code] class to avoid allocating a new [StringName] on each call.
func set_deferred(property: StringName, value: Variant) -> void:
	pass;

## Assigns a new [param value] to the property identified by the [param property_path]. The path should be a [NodePath] relative to this object, and can use the colon character ([code]:[/code]) to access nested properties.
## [codeblocks]
## [gdscript]
## var node = Node2D.new()
## node.set_indexed("position", Vector2(42, 0))
## node.set_indexed("position:y", -10)
## print(node.position) # Prints (42.0, -10.0)
## [/gdscript]
## [csharp]
## var node = new Node2D();
## node.SetIndexed("position", new Vector2(42, 0));
## node.SetIndexed("position:y", -10);
## GD.Print(node.Position); // Prints (42, -10)
## [/csharp]
## [/codeblocks]
## [b]Note:[/b] In C#, [param property_path] must be in snake_case when referring to built-in Godot properties. Prefer using the names exposed in the [code]PropertyName[/code] class to avoid allocating a new [StringName] on each call.
func set_indexed(property_path: NodePath, value: Variant) -> void:
	pass;

## If set to [code]true[/code], allows the object to translate messages with [method tr] and [method tr_n]. Enabled by default. See also [method can_translate_messages].
func set_message_translation(enable: bool) -> void:
	pass;

## Adds or changes the entry [param name] inside the object's metadata. The metadata [param value] can be any [Variant], although some types cannot be serialized correctly.
## If [param value] is [code]null[/code], the entry is removed. This is the equivalent of using [method remove_meta]. See also [method has_meta] and [method get_meta].
## [b]Note:[/b] A metadata's name must be a valid identifier as per [method StringName.is_valid_identifier] method.
## [b]Note:[/b] Metadata that has a name starting with an underscore ([code]_[/code]) is considered editor-only. Editor-only metadata is not displayed in the Inspector and should not be edited, although it can still be found by this method.
func set_meta(name: StringName, value: Variant) -> void:
	pass;

## Attaches [param script] to the object, and instantiates it. As a result, the script's [method _init] is called. A [Script] is used to extend the object's functionality.
## If a script already exists, its instance is detached, and its property values and state are lost. Built-in property values are still kept.
func set_script(script: Variant) -> void:
	pass;

## Sets the name of the translation domain used by [method tr] and [method tr_n]. See also [TranslationServer].
func set_translation_domain(domain: StringName) -> void:
	pass;

## Returns a [String] representing the object. Defaults to [code]"<ClassName#RID>"[/code]. Override [method _to_string] to customize the string representation of the object.
func to_string() -> String:
	pass;

## Translates a [param message], using the translation catalogs configured in the Project Settings. Further [param context] can be specified to help with the translation. Note that most [Control] nodes automatically translate their strings, so this method is mostly useful for formatted strings or custom drawn text.
## If [method can_translate_messages] is [code]false[/code], or no translation is available, this method returns the [param message] without changes. See [method set_message_translation].
## For detailed examples, see [url=$DOCS_URL/tutorials/i18n/internationalizing_games.html]Internationalizing games[/url].
## [b]Note:[/b] This method can't be used without an [Object] instance, as it requires the [method can_translate_messages] method. To translate strings in a static context, use [method TranslationServer.translate].
func tr(message: StringName, context: StringName = &"") -> String:
	pass;

## Translates a [param message] or [param plural_message], using the translation catalogs configured in the Project Settings. Further [param context] can be specified to help with the translation.
## If [method can_translate_messages] is [code]false[/code], or no translation is available, this method returns [param message] or [param plural_message], without changes. See [method set_message_translation].
## The [param n] is the number, or amount, of the message's subject. It is used by the translation system to fetch the correct plural form for the current language.
## For detailed examples, see [url=$DOCS_URL/tutorials/i18n/localization_using_gettext.html]Localization using gettext[/url].
## [b]Note:[/b] Negative and [float] numbers may not properly apply to some countable subjects. It's recommended to handle these cases with [method tr].
## [b]Note:[/b] This method can't be used without an [Object] instance, as it requires the [method can_translate_messages] method. To translate strings in a static context, use [method TranslationServer.translate_plural].
func tr_n(message: StringName, plural_message: StringName, n: int, context: StringName = &"") -> String:
	pass;


