class_name NodePath

## A pre-parsed scene tree path.
##
## The [NodePath] built-in [Variant] type represents a path to a node or property in a hierarchy of nodes. It is designed to be efficiently passed into many built-in methods (such as [method Node.get_node], [method Object.set_indexed], [method Tween.tween_property], etc.) without a hard dependence on the node or property they point to.
## A node path is represented as a [String] composed of slash-separated ([code]/[/code]) node names and colon-separated ([code]:[/code]) property names (also called "subnames"). Similar to a filesystem path, [code]".."[/code] and [code]"."[/code] are special node names. They refer to the parent node and the current node, respectively.
## The following examples are paths relative to the current node:
## [codeblock]
## ^"A"     # Points to the direct child A.
## ^"A/B"   # Points to A's child B.
## ^"."     # Points to the current node.
## ^".."    # Points to the parent node.
## ^"../C"  # Points to the sibling node C.
## ^"../.." # Points to the grandparent node.
## [/codeblock]
## A leading slash means the path is absolute, and begins from the [SceneTree]:
## [codeblock]
## ^"/root"            # Points to the SceneTree's root Window.
## ^"/root/Title"      # May point to the main scene's root node named "Title".
## ^"/root/Global"     # May point to an autoloaded node or scene named "Global".
## [/codeblock]
## Despite their name, node paths may also point to a property:
## [codeblock]
## ^":position"           # Points to this object's position.
## ^":position:x"         # Points to this object's position in the x axis.
## ^"Camera3D:rotation:y" # Points to the child Camera3D and its y rotation.
## ^"/root:size:x"        # Points to the root Window and its width.
## [/codeblock]
## In some situations, it's possible to omit the leading [code]:[/code] when pointing to an object's property. As an example, this is the case with [method Object.set_indexed] and [method Tween.tween_property], as those methods call [method NodePath.get_as_property_path] under the hood. However, it's generally recommended to keep the [code]:[/code] prefix.
## Node paths cannot check whether they are valid and may point to nodes or properties that do not exist. Their meaning depends entirely on the context in which they're used.
## You usually do not have to worry about the [NodePath] type, as strings are automatically converted to the type when necessary. There are still times when defining node paths is useful. For example, exported [NodePath] properties allow you to easily select any node within the currently edited scene. They are also automatically updated when moving, renaming or deleting nodes in the scene tree editor. See also [annotation @GDScript.@export_node_path].
## See also [StringName], which is a similar type designed for optimized strings.
## [b]Note:[/b] In a boolean context, a [NodePath] will evaluate to [code]false[/code] if it is empty ([code]NodePath("")[/code]). Otherwise, a [NodePath] will always evaluate to [code]true[/code].
##
## @tutorial(2D Role Playing Game (RPG) Demo): https://godotengine.org/asset-library/asset/2729




## Constructs an empty [NodePath].
func NodePath() -> NodePath:
	pass;

## Constructs a [NodePath] as a copy of the given [NodePath].
func NodePath(from: NodePath) -> NodePath:
	pass;

## Constructs a [NodePath] from a [String]. The created path is absolute if prefixed with a slash (see [method is_absolute]).
## The "subnames" optionally included after the path to the target node can point to properties, and can also be nested.
## The following strings can be valid node paths:
## [codeblock]
## # Points to the Sprite2D node.
## "Level/RigidBody2D/Sprite2D"
## # Points to the Sprite2D node and its "texture" resource.
## # get_node() would retrieve the Sprite2D, while get_node_and_resource()
## # would retrieve both the Sprite2D node and the "texture" resource.
## "Level/RigidBody2D/Sprite2D:texture"
## # Points to the Sprite2D node and its "position" property.
## "Level/RigidBody2D/Sprite2D:position"
## # Points to the Sprite2D node and the "x" component of its "position" property.
## "Level/RigidBody2D/Sprite2D:position:x"
## # Points to the RigidBody2D node as an absolute path beginning from the SceneTree.
## "/root/Level/RigidBody2D"
## [/codeblock]
## [b]Note:[/b] In GDScript, it's also possible to convert a constant string into a node path by prefixing it with [code]^[/code]. [code]^"path/to/node"[/code] is equivalent to [code]NodePath("path/to/node")[/code].
func NodePath(from: String) -> NodePath:
	pass;


## Returns a copy of this node path with a colon character ([code]:[/code]) prefixed, transforming it to a pure property path with no node names (relative to the current node).
## [codeblocks]
## [gdscript]
## # node_path points to the "x" property of the child node named "position".
## var node_path = ^"position:x"
## # property_path points to the "position" in the "x" axis of this node.
## var property_path = node_path.get_as_property_path()
## print(property_path) # Prints ":position:x"
## [/gdscript]
## [csharp]
## // nodePath points to the "x" property of the child node named "position".
## var nodePath = new NodePath("position:x");
## // propertyPath points to the "position" in the "x" axis of this node.
## NodePath propertyPath = nodePath.GetAsPropertyPath();
## GD.Print(propertyPath); // Prints ":position:x"
## [/csharp]
## [/codeblocks]
func get_as_property_path() -> NodePath:
	pass;

## Returns all node names concatenated with a slash character ([code]/[/code]) as a single [StringName].
func get_concatenated_names() -> StringName:
	pass;

## Returns all property subnames concatenated with a colon character ([code]:[/code]) as a single [StringName].
## [codeblocks]
## [gdscript]
## var node_path = ^"Sprite2D:texture:resource_name"
## print(node_path.get_concatenated_subnames()) # Prints "texture:resource_name"
## [/gdscript]
## [csharp]
## var nodePath = new NodePath("Sprite2D:texture:resource_name");
## GD.Print(nodePath.GetConcatenatedSubnames()); // Prints "texture:resource_name"
## [/csharp]
## [/codeblocks]
func get_concatenated_subnames() -> StringName:
	pass;

## Returns the node name indicated by [param idx], starting from 0. If [param idx] is out of bounds, an error is generated. See also [method get_subname_count] and [method get_name_count].
## [codeblocks]
## [gdscript]
## var sprite_path = NodePath("../RigidBody2D/Sprite2D")
## print(sprite_path.get_name(0)) # Prints ".."
## print(sprite_path.get_name(1)) # Prints "RigidBody2D"
## print(sprite_path.get_name(2)) # Prints "Sprite"
## [/gdscript]
## [csharp]
## var spritePath = new NodePath("../RigidBody2D/Sprite2D");
## GD.Print(spritePath.GetName(0)); // Prints ".."
## GD.Print(spritePath.GetName(1)); // Prints "PathFollow2D"
## GD.Print(spritePath.GetName(2)); // Prints "Sprite"
## [/csharp]
## [/codeblocks]
func get_name(idx: int) -> StringName:
	pass;

## Returns the number of node names in the path. Property subnames are not included.
## For example, [code]"../RigidBody2D/Sprite2D:texture"[/code] contains 3 node names.
func get_name_count() -> int:
	pass;

## Returns the property name indicated by [param idx], starting from 0. If [param idx] is out of bounds, an error is generated. See also [method get_subname_count].
## [codeblocks]
## [gdscript]
## var path_to_name = NodePath("Sprite2D:texture:resource_name")
## print(path_to_name.get_subname(0)) # Prints "texture"
## print(path_to_name.get_subname(1)) # Prints "resource_name"
## [/gdscript]
## [csharp]
## var pathToName = new NodePath("Sprite2D:texture:resource_name");
## GD.Print(pathToName.GetSubname(0)); // Prints "texture"
## GD.Print(pathToName.GetSubname(1)); // Prints "resource_name"
## [/csharp]
## [/codeblocks]
func get_subname(idx: int) -> StringName:
	pass;

## Returns the number of property names ("subnames") in the path. Each subname in the node path is listed after a colon character ([code]:[/code]).
## For example, [code]"Level/RigidBody2D/Sprite2D:texture:resource_name"[/code] contains 2 subnames.
func get_subname_count() -> int:
	pass;

## Returns the 32-bit hash value representing the node path's contents.
## [b]Note:[/b] Node paths with equal hash values are [i]not[/i] guaranteed to be the same, as a result of hash collisions. Node paths with different hash values are guaranteed to be different.
func hash() -> int:
	pass;

## Returns [code]true[/code] if the node path is absolute. Unlike a relative path, an absolute path is represented by a leading slash character ([code]/[/code]) and always begins from the [SceneTree]. It can be used to reliably access nodes from the root node (e.g. [code]"/root/Global"[/code] if an autoload named "Global" exists).
func is_absolute() -> bool:
	pass;

## Returns [code]true[/code] if the node path has been constructed from an empty [String] ([code]""[/code]).
func is_empty() -> bool:
	pass;

## Returns the slice of the [NodePath], from [param begin] (inclusive) to [param end] (exclusive), as a new [NodePath].
## The absolute value of [param begin] and [param end] will be clamped to the sum of [method get_name_count] and [method get_subname_count], so the default value for [param end] makes it slice to the end of the [NodePath] by default (i.e. [code]path.slice(1)[/code] is a shorthand for [code]path.slice(1, path.get_name_count() + path.get_subname_count())[/code]).
## If either [param begin] or [param end] are negative, they will be relative to the end of the [NodePath] (i.e. [code]path.slice(0, -2)[/code] is a shorthand for [code]path.slice(0, path.get_name_count() + path.get_subname_count() - 2)[/code]).
func slice(begin: int, end: int = 2147483647) -> NodePath:
	pass;


