extends Node
class_name Node3D

## Base object in 3D space, inherited by all 3D nodes.
##
## The [Node3D] node is the base representation of a node in 3D space. All other 3D nodes inherit from this class.
## Affine operations (translation, rotation, scale) are calculated in the coordinate system relative to the parent, unless the [Node3D]'s [member top_level] is [code]true[/code]. In this coordinate system, affine operations correspond to direct affine operations on the [Node3D]'s [member transform]. The term [i]parent space[/i] refers to this coordinate system. The coordinate system that is attached to the [Node3D] itself is referred to as object-local coordinate system, or [i]local space[/i].
## [b]Note:[/b] Unless otherwise specified, all methods that need angle parameters must receive angles in [i]radians[/i]. To convert degrees to radians, use [method @GlobalScope.deg_to_rad].
## [b]Note:[/b] In Godot 3 and older, [Node3D] was named [i]Spatial[/i].
##
## @tutorial(Introduction to 3D): https://docs.godotengine.org/en/stable/tutorials/3d/introduction_to_3d.html
## @tutorial(All 3D Demos): https://github.com/godotengine/godot-demo-projects/tree/master/3d


## Emitted when this node's visibility changes (see [member visible] and [method is_visible_in_tree]).
## This signal is emitted [i]after[/i] the related [constant NOTIFICATION_VISIBILITY_CHANGED] notification.
signal visibility_changed
## Notification received when this node's [member global_transform] changes, if [method is_transform_notification_enabled] is [code]true[/code]. See also [method set_notify_transform].
## [b]Note:[/b] Most 3D nodes such as [VisualInstance3D] or [CollisionObject3D] automatically enable this to function correctly.
## [b]Note:[/b] In the editor, nodes will propagate this notification to their children if a gizmo is attached (see [method add_gizmo]).
const NOTIFICATION_TRANSFORM_CHANGED = 2000;

## Notification received when this node is registered to a new [World3D] (see [method get_world_3d]).
const NOTIFICATION_ENTER_WORLD = 41;

## Notification received when this node is unregistered from the current [World3D] (see [method get_world_3d]).
const NOTIFICATION_EXIT_WORLD = 42;

## Notification received when this node's visibility changes (see [member visible] and [method is_visible_in_tree]).
## This notification is received [i]before[/i] the related [signal visibility_changed] signal.
const NOTIFICATION_VISIBILITY_CHANGED = 43;

## Notification received when this node's [member transform] changes, if [method is_local_transform_notification_enabled] is [code]true[/code]. This is not received when a parent [Node3D]'s [member transform] changes. See also [method set_notify_local_transform].
## [b]Note:[/b] Some 3D nodes such as [CSGShape3D] or [CollisionShape3D] automatically enable this to function correctly.
const NOTIFICATION_LOCAL_TRANSFORM_CHANGED = 44;

## The rotation is edited using a [Vector3] in [url=https://en.wikipedia.org/wiki/Euler_angles]Euler angles[/url].
## The rotation is edited using a [Quaternion].
## The rotation is edited using a [Basis]. In this mode, the raw [member basis]'s axes can be freely modified, but the [member scale] property is not available.

#enum RotationEditMode
enum {
    ROTATION_EDIT_MODE_EULER = 0,
    ROTATION_EDIT_MODE_QUATERNION = 1,
    ROTATION_EDIT_MODE_BASIS = 2,
}
## Basis of the [member transform] property. Represents the rotation, scale, and shear of this node in parent space (relative to the parent node).
var basis: Basis:
	get = get_basis, set = set_basis

## Basis of the [member global_transform] property. Represents the rotation, scale, and shear of this node in global space (relative to the world).
## [b]Note:[/b] If the node is not inside the tree, getting this property fails and returns [constant Basis.IDENTITY].
var global_basis: Basis:
	get = get_global_basis, set = set_global_basis

## Global position (translation) of this node in global space (relative to the world). This is equivalent to the [member global_transform]'s [member Transform3D.origin].
## [b]Note:[/b] If the node is not inside the tree, getting this property fails and returns [constant Vector3.ZERO].
var global_position: Vector3:
	get = get_global_position, set = set_global_position

## Global rotation of this node as [url=https://en.wikipedia.org/wiki/Euler_angles]Euler angles[/url], in radians and in global space (relative to the world). This value is obtained from [member global_basis]'s rotation.
## - The [member Vector3.x] is the angle around the global X axis (pitch);
## - The [member Vector3.y] is the angle around the global Y axis (yaw);
## - The [member Vector3.z] is the angle around the global Z axis (roll).
## [b]Note:[/b] Unlike [member rotation], this property always follows the YXZ convention ([constant EULER_ORDER_YXZ]).
## [b]Note:[/b] If the node is not inside the tree, getting this property fails and returns [constant Vector3.ZERO].
var global_rotation: Vector3:
	get = get_global_rotation, set = set_global_rotation

## The [member global_rotation] of this node, in degrees instead of radians.
## [b]Note:[/b] If the node is not inside the tree, getting this property fails and returns [constant Vector3.ZERO].
var global_rotation_degrees: Vector3:
	get = get_global_rotation_degrees, set = set_global_rotation_degrees

## The transformation of this node, in global space (relative to the world). Contains and represents this node's [member global_position], [member global_rotation], and global scale.
## [b]Note:[/b] If the node is not inside the tree, getting this property fails and returns [constant Transform3D.IDENTITY].
var global_transform: Transform3D:
	get = get_global_transform, set = set_global_transform

## Position (translation) of this node in parent space (relative to the parent node). This is equivalent to the [member transform]'s [member Transform3D.origin].
var position: Vector3:
	get = get_position, set = set_position

## Rotation of this node represented as a [Quaternion] in parent space (relative to the parent node). This value is obtained from [member basis]'s rotation.
## [b]Note:[/b] Quaternions are much more suitable for 3D math but are less intuitive. Setting this property can be useful for interpolation (see [method Quaternion.slerp]).
var quaternion: Quaternion:
	get = get_quaternion, set = set_quaternion

## Rotation of this node as [url=https://en.wikipedia.org/wiki/Euler_angles]Euler angles[/url], in radians and in parent space (relative to the parent node). This value is obtained from [member basis]'s rotation.
## - The [member Vector3.x] is the angle around the local X axis (pitch);
## - The [member Vector3.y] is the angle around the local Y axis (yaw);
## - The [member Vector3.z] is the angle around the local Z axis (roll).
## The order of each consecutive rotation can be changed with [member rotation_order] (see [enum EulerOrder] constants). By default, the YXZ convention is used ([constant EULER_ORDER_YXZ]).
## [b]Note:[/b] This property is edited in degrees in the inspector. If you want to use degrees in a script, use [member rotation_degrees].
var rotation: Vector3:
	get = get_rotation, set = set_rotation

## The [member rotation] of this node, in degrees instead of radians.
## [b]Note:[/b] This is [b]not[/b] the property available in the Inspector dock.
var rotation_degrees: Vector3:
	get = get_rotation_degrees, set = set_rotation_degrees

## How this node's rotation and scale are displayed in the Inspector dock.
var rotation_edit_mode: int:
	get = get_rotation_edit_mode, set = set_rotation_edit_mode

## The axis rotation order of the [member rotation] property. The final orientation is calculated by rotating around the local X, Y, and Z axis in this order.
var rotation_order: int:
	get = get_rotation_order, set = set_rotation_order

## Scale of this node in local space (relative to this node). This value is obtained from [member basis]'s scale.
## [b]Note:[/b] The behavior of some 3D node types is not affected by this property. These include [Light3D], [Camera3D], [AudioStreamPlayer3D], and more.
## [b]Warning:[/b] The scale's components must either be all positive or all negative, and [b]not[/b] exactly [code]0.0[/code]. Otherwise, it won't be possible to obtain the scale from the [member basis]. This may cause the intended scale to be lost when reloaded from disk, and potentially other unstable behavior.
var scale: Vector3:
	get = get_scale, set = set_scale

## If [code]true[/code], the node does not inherit its transformations from its parent. As such, node transformations will only be in global space, which also means that [member global_transform] and [member transform] will be identical.
var top_level: bool:
	get = is_set_as_top_level, set = set_as_top_level

## The local transformation of this node, in parent space (relative to the parent node). Contains and represents this node's [member position], [member rotation], and [member scale].
var transform: Transform3D:
	get = get_transform, set = set_transform

## Path to the visibility range parent for this node and its descendants. The visibility parent must be a [GeometryInstance3D].
## Any visual instance will only be visible if the visibility parent (and all of its visibility ancestors) is hidden by being closer to the camera than its own [member GeometryInstance3D.visibility_range_begin]. Nodes hidden via the [member Node3D.visible] property are essentially removed from the visibility dependency tree, so dependent instances will not take the hidden node or its descendants into account.
var visibility_parent: NodePath:
	get = get_visibility_parent, set = set_visibility_parent

## If [code]true[/code], this node can be visible. The node is only rendered when all of its ancestors are visible, as well. That means [method is_visible_in_tree] must return [code]true[/code].
var visible: bool:
	get = is_visible, set = set_visible



## Attaches the given [param gizmo] to this node. Only works in the editor.
## [b]Note:[/b] [param gizmo] should be an [EditorNode3DGizmo]. The argument type is [Node3DGizmo] to avoid depending on editor classes in [Node3D].
func add_gizmo(gizmo: Node3DGizmo) -> void:
	pass;

## Clears all [EditorNode3DGizmo] objects attached to this node. Only works in the editor.
func clear_gizmos() -> void:
	pass;

## Deselects all subgizmos for this node. Useful to call when the selected subgizmo may no longer exist after a property change. Only works in the editor.
func clear_subgizmo_selection() -> void:
	pass;

## Forces the node's [member global_transform] to update, by sending [constant NOTIFICATION_TRANSFORM_CHANGED]. Fails if the node is not inside the tree.
## [b]Note:[/b] For performance reasons, transform changes are usually accumulated and applied [i]once[/i] at the end of the frame. The update propagates through [Node3D] children, as well. Therefore, use this method only when you need an up-to-date transform (such as during physics operations).
func force_update_transform() -> void:
	pass;

## Returns all the [EditorNode3DGizmo] objects attached to this node. Only works in the editor.
func get_gizmos() -> Array[Node3DGizmo]:
	pass;

## When using physics interpolation, there will be circumstances in which you want to know the interpolated (displayed) transform of a node rather than the standard transform (which may only be accurate to the most recent physics tick).
## This is particularly important for frame-based operations that take place in [method Node._process], rather than [method Node._physics_process]. Examples include [Camera3D]s focusing on a node, or finding where to fire lasers from on a frame rather than physics tick.
## [b]Note:[/b] This function creates an interpolation pump on the [Node3D] the first time it is called, which can respond to physics interpolation resets. If you get problems with "streaking" when initially following a [Node3D], be sure to call [method get_global_transform_interpolated] at least once [i]before[/i] resetting the [Node3D] physics interpolation.
func get_global_transform_interpolated() -> Transform3D:
	pass;

## Returns the parent [Node3D] that directly affects this node's [member global_transform]. Returns [code]null[/code] if no parent exists, the parent is not a [Node3D], or [member top_level] is [code]true[/code].
## [b]Note:[/b] This method is not always equivalent to [method Node.get_parent], which does not take [member top_level] into account.
func get_parent_node_3d() -> Node3D:
	pass;

## Returns the [World3D] this node is registered to.
## Usually, this is the same as the world used by this node's viewport (see [method Node.get_viewport] and [method Viewport.find_world_3d]).
func get_world_3d() -> World3D:
	pass;

## Rotates this node's [member global_basis] around the global [param axis] by the given [param angle], in radians. This operation is calculated in global space (relative to the world) and preserves the [member global_position].
func global_rotate(axis: Vector3, angle: float) -> void:
	pass;

## Scales this node's [member global_basis] by the given [param scale] factor. This operation is calculated in global space (relative to the world) and preserves the [member global_position].
## [b]Note:[/b] This method is not to be confused with the [member scale] property.
func global_scale(scale: Vector3) -> void:
	pass;

## Adds the given translation [param offset] to the node's [member global_position] in global space (relative to the world).
func global_translate(offset: Vector3) -> void:
	pass;

## Prevents this node from being rendered. Equivalent to setting [member visible] to [code]false[/code]. This is the opposite of [method show].
func hide() -> void:
	pass;

## Returns [code]true[/code] if the node receives [constant NOTIFICATION_LOCAL_TRANSFORM_CHANGED] whenever [member transform] changes. This is enabled with [method set_notify_local_transform].
func is_local_transform_notification_enabled() -> bool:
	pass;

## Returns [code]true[/code] if this node's [member global_transform] is automatically orthonormalized. This results in this node not appearing distorted, as if its global scale were set to [constant Vector3.ONE] (or its negative counterpart). See also [method set_disable_scale] and [method orthonormalize].
## [b]Note:[/b] [member transform] is not affected by this setting.
func is_scale_disabled() -> bool:
	pass;

## Returns [code]true[/code] if the node receives [constant NOTIFICATION_TRANSFORM_CHANGED] whenever [member global_transform] changes. This is enabled with [method set_notify_transform].
func is_transform_notification_enabled() -> bool:
	pass;

## Returns [code]true[/code] if this node is inside the scene tree and the [member visible] property is [code]true[/code] for this node and all of its [Node3D] ancestors [i]in sequence[/i]. An ancestor of any other type (such as [Node] or [Node2D]) breaks the sequence. See also [method Node.get_parent].
## [b]Note:[/b] This method cannot take [member VisualInstance3D.layers] into account, so even if this method returns [code]true[/code], the node may not be rendered.
func is_visible_in_tree() -> bool:
	pass;

## Rotates the node so that the local forward axis (-Z, [constant Vector3.FORWARD]) points toward the [param target] position. This operation is calculated in global space (relative to the world).
## The local up axis (+Y) points as close to the [param up] vector as possible while staying perpendicular to the local forward axis. The resulting transform is orthogonal, and the scale is preserved. Non-uniform scaling may not work correctly.
## The [param target] position cannot be the same as the node's position, the [param up] vector cannot be [constant Vector3.ZERO]. Furthermore, the direction from the node's position to the [param target] position cannot be parallel to the [param up] vector, to avoid an unintended rotation around the local Z axis.
## If [param use_model_front] is [code]true[/code], the +Z axis (asset front) is treated as forward (implies +X is left) and points toward the [param target] position. By default, the -Z axis (camera forward) is treated as forward (implies +X is right).
## [b]Note:[/b] This method fails if the node is not in the scene tree. If necessary, use [method look_at_from_position] instead.
func look_at(target: Vector3, up: Vector3 = Vector3(0, 1, 0), use_model_front: bool = false) -> void:
	pass;

## Moves the node to the specified [param position], then rotates the node to point toward the [param target] position, similar to [method look_at]. This operation is calculated in global space (relative to the world).
func look_at_from_position(position: Vector3, target: Vector3, up: Vector3 = Vector3(0, 1, 0), use_model_front: bool = false) -> void:
	pass;

## Orthonormalizes this node's [member basis]. This method sets this node's [member scale] to [constant Vector3.ONE] (or its negative counterpart), but preserves the [member position] and [member rotation]. See also [method Transform3D.orthonormalized].
func orthonormalize() -> void:
	pass;

## Rotates this node's [member basis] around the [param axis] by the given [param angle], in radians. This operation is calculated in parent space (relative to the parent) and preserves the [member position].
func rotate(axis: Vector3, angle: float) -> void:
	pass;

## Rotates this node's [member basis] around the [param axis] by the given [param angle], in radians. This operation is calculated in local space (relative to this node) and preserves the [member position].
func rotate_object_local(axis: Vector3, angle: float) -> void:
	pass;

## Rotates this node's [member basis] around the X axis by the given [param angle], in radians. This operation is calculated in parent space (relative to the parent) and preserves the [member position].
func rotate_x(angle: float) -> void:
	pass;

## Rotates this node's [member basis] around the Y axis by the given [param angle], in radians. This operation is calculated in parent space (relative to the parent) and preserves the [member position].
func rotate_y(angle: float) -> void:
	pass;

## Rotates this node's [member basis] around the Z axis by the given [param angle], in radians. This operation is calculated in parent space (relative to the parent) and preserves the [member position].
func rotate_z(angle: float) -> void:
	pass;

## Scales this node's [member basis] by the given [param scale] factor. This operation is calculated in local space (relative to this node) and preserves the [member position].
func scale_object_local(scale: Vector3) -> void:
	pass;

## If [code]true[/code], this node's [member global_transform] is automatically orthonormalized. This results in this node not appearing distorted, as if its global scale were set to [constant Vector3.ONE] (or its negative counterpart). See also [method is_scale_disabled] and [method orthonormalize].
## [b]Note:[/b] [member transform] is not affected by this setting.
func set_disable_scale(disable: bool) -> void:
	pass;

## Sets this node's [member transform] to [constant Transform3D.IDENTITY], which resets all transformations in parent space ([member position], [member rotation], and [member scale]).
func set_identity() -> void:
	pass;

## If [code]true[/code], the node will not receive [constant NOTIFICATION_TRANSFORM_CHANGED] or [constant NOTIFICATION_LOCAL_TRANSFORM_CHANGED].
## It may useful to call this method when handling these notifications to prevent infinite recursion.
func set_ignore_transform_notification(enabled: bool) -> void:
	pass;

## If [code]true[/code], the node will receive [constant NOTIFICATION_LOCAL_TRANSFORM_CHANGED] whenever [member transform] changes.
## [b]Note:[/b] Some 3D nodes such as [CSGShape3D] or [CollisionShape3D] automatically enable this to function correctly.
func set_notify_local_transform(enable: bool) -> void:
	pass;

## If [code]true[/code], the node will receive [constant NOTIFICATION_TRANSFORM_CHANGED] whenever [member global_transform] changes.
## [b]Note:[/b] Most 3D nodes such as [VisualInstance3D] or [CollisionObject3D] automatically enable this to function correctly.
## [b]Note:[/b] In the editor, nodes will propagate this notification to their children if a gizmo is attached (see [method add_gizmo]).
func set_notify_transform(enable: bool) -> void:
	pass;

## Selects the [param gizmo]'s subgizmo with the given [param id] and sets its transform. Only works in the editor.
## [b]Note:[/b] The gizmo object would typically be an instance of [EditorNode3DGizmo], but the argument type is kept generic to avoid creating a dependency on editor classes in [Node3D].
func set_subgizmo_selection(gizmo: Node3DGizmo, id: int, transform: Transform3D) -> void:
	pass;

## Allows this node to be rendered. Equivalent to setting [member visible] to [code]true[/code]. This is the opposite of [method hide].
func show() -> void:
	pass;

## Returns the [param local_point] converted from this node's local space to global space. This is the opposite of [method to_local].
func to_global(local_point: Vector3) -> Vector3:
	pass;

## Returns the [param global_point] converted from global space to this node's local space. This is the opposite of [method to_global].
func to_local(global_point: Vector3) -> Vector3:
	pass;

## Adds the given translation [param offset] to the node's position, in local space (relative to this node).
## [b]Note:[/b] Prefer using [method translate_object_local], instead, as this method may be changed in a future release.
## [b]Note:[/b] Despite the naming convention, this operation is [b]not[/b] calculated in parent space for compatibility reasons. To translate in parent space, add [param offset] to the [member position] ([code]node_3d.position += offset[/code]).
func translate(offset: Vector3) -> void:
	pass;

## Adds the given translation [param offset] to the node's position, in local space (relative to this node).
func translate_object_local(offset: Vector3) -> void:
	pass;

## Updates all the [EditorNode3DGizmo] objects attached to this node. Only works in the editor.
func update_gizmos() -> void:
	pass;


func get_basis() -> Basis:
	return basis

func set_basis(value: Basis) -> void:
	basis = value

func get_global_basis() -> Basis:
	return global_basis

func set_global_basis(value: Basis) -> void:
	global_basis = value

func get_global_position() -> Vector3:
	return global_position

func set_global_position(value: Vector3) -> void:
	global_position = value

func get_global_rotation() -> Vector3:
	return global_rotation

func set_global_rotation(value: Vector3) -> void:
	global_rotation = value

func get_global_rotation_degrees() -> Vector3:
	return global_rotation_degrees

func set_global_rotation_degrees(value: Vector3) -> void:
	global_rotation_degrees = value

func get_global_transform() -> Transform3D:
	return global_transform

func set_global_transform(value: Transform3D) -> void:
	global_transform = value

func get_position() -> Vector3:
	return position

func set_position(value: Vector3) -> void:
	position = value

func get_quaternion() -> Quaternion:
	return quaternion

func set_quaternion(value: Quaternion) -> void:
	quaternion = value

func get_rotation() -> Vector3:
	return rotation

func set_rotation(value: Vector3) -> void:
	rotation = value

func get_rotation_degrees() -> Vector3:
	return rotation_degrees

func set_rotation_degrees(value: Vector3) -> void:
	rotation_degrees = value

func get_rotation_edit_mode() -> int:
	return rotation_edit_mode

func set_rotation_edit_mode(value: int) -> void:
	rotation_edit_mode = value

func get_rotation_order() -> int:
	return rotation_order

func set_rotation_order(value: int) -> void:
	rotation_order = value

func get_scale() -> Vector3:
	return scale

func set_scale(value: Vector3) -> void:
	scale = value

func is_set_as_top_level() -> bool:
	return top_level

func set_as_top_level(value: bool) -> void:
	top_level = value

func get_transform() -> Transform3D:
	return transform

func set_transform(value: Transform3D) -> void:
	transform = value

func get_visibility_parent() -> NodePath:
	return visibility_parent

func set_visibility_parent(value: NodePath) -> void:
	visibility_parent = value

func is_visible() -> bool:
	return visible

func set_visible(value: bool) -> void:
	visible = value

