extends Resource
class_name NavigationPolygon

## A 2D navigation mesh that describes a traversable surface for pathfinding.
##
## A navigation mesh can be created either by baking it with the help of the [NavigationServer2D], or by adding vertices and convex polygon indices arrays manually.
## To bake a navigation mesh at least one outline needs to be added that defines the outer bounds of the baked area.
## [codeblocks]
## [gdscript]
## var new_navigation_mesh = NavigationPolygon.new()
## var bounding_outline = PackedVector2Array([Vector2(0, 0), Vector2(0, 50), Vector2(50, 50), Vector2(50, 0)])
## new_navigation_mesh.add_outline(bounding_outline)
## NavigationServer2D.bake_from_source_geometry_data(new_navigation_mesh, NavigationMeshSourceGeometryData2D.new());
## $NavigationRegion2D.navigation_polygon = new_navigation_mesh
## [/gdscript]
## [csharp]
## var newNavigationMesh = new NavigationPolygon();
## Vector2[] boundingOutline = [new Vector2(0, 0), new Vector2(0, 50), new Vector2(50, 50), new Vector2(50, 0)];
## newNavigationMesh.AddOutline(boundingOutline);
## NavigationServer2D.BakeFromSourceGeometryData(newNavigationMesh, new NavigationMeshSourceGeometryData2D());
## GetNode<NavigationRegion2D>("NavigationRegion2D").NavigationPolygon = newNavigationMesh;
## [/csharp]
## [/codeblocks]
## Adding vertices and polygon indices manually.
## [codeblocks]
## [gdscript]
## var new_navigation_mesh = NavigationPolygon.new()
## var new_vertices = PackedVector2Array([Vector2(0, 0), Vector2(0, 50), Vector2(50, 50), Vector2(50, 0)])
## new_navigation_mesh.vertices = new_vertices
## var new_polygon_indices = PackedInt32Array([0, 1, 2, 3])
## new_navigation_mesh.add_polygon(new_polygon_indices)
## $NavigationRegion2D.navigation_polygon = new_navigation_mesh
## [/gdscript]
## [csharp]
## var newNavigationMesh = new NavigationPolygon();
## Vector2[] newVertices = [new Vector2(0, 0), new Vector2(0, 50), new Vector2(50, 50), new Vector2(50, 0)];
## newNavigationMesh.Vertices = newVertices;
## int[] newPolygonIndices = [0, 1, 2, 3];
## newNavigationMesh.AddPolygon(newPolygonIndices);
## GetNode<NavigationRegion2D>("NavigationRegion2D").NavigationPolygon = newNavigationMesh;
## [/csharp]
## [/codeblocks]
##
## @tutorial(Using NavigationMeshes): https://docs.godotengine.org/en/stable/tutorials/navigation/navigation_using_navigationmeshes.html
## @tutorial(Navigation Polygon 2D Demo): https://godotengine.org/asset-library/asset/2722


## Convex partitioning that yields navigation mesh with convex polygons.
## Triangulation partitioning that yields navigation mesh with triangle polygons.
## Represents the size of the [enum SamplePartitionType] enum.
## Parses mesh instances as obstruction geometry. This includes [Polygon2D], [MeshInstance2D], [MultiMeshInstance2D], and [TileMap] nodes.
## Meshes are only parsed when they use a 2D vertices surface format.
## Parses [StaticBody2D] and [TileMap] colliders as obstruction geometry. The collider should be in any of the layers specified by [member parsed_collision_mask].
## Both [constant PARSED_GEOMETRY_MESH_INSTANCES] and [constant PARSED_GEOMETRY_STATIC_COLLIDERS].
## Represents the size of the [enum ParsedGeometryType] enum.
## Scans the child nodes of the root node recursively for geometry.
## Scans nodes in a group and their child nodes recursively for geometry. The group is specified by [member source_geometry_group_name].
## Uses nodes in a group for geometry. The group is specified by [member source_geometry_group_name].
## Represents the size of the [enum SourceGeometryMode] enum.

#enum SamplePartitionType
enum {
    SAMPLE_PARTITION_CONVEX_PARTITION = 0,
    SAMPLE_PARTITION_TRIANGULATE = 1,
    SAMPLE_PARTITION_MAX = 2,
}
#enum ParsedGeometryType
enum {
    PARSED_GEOMETRY_MESH_INSTANCES = 0,
    PARSED_GEOMETRY_STATIC_COLLIDERS = 1,
    PARSED_GEOMETRY_BOTH = 2,
    PARSED_GEOMETRY_MAX = 3,
}
#enum SourceGeometryMode
enum {
    SOURCE_GEOMETRY_ROOT_NODE_CHILDREN = 0,
    SOURCE_GEOMETRY_GROUPS_WITH_CHILDREN = 1,
    SOURCE_GEOMETRY_GROUPS_EXPLICIT = 2,
    SOURCE_GEOMETRY_MAX = 3,
}
## The distance to erode/shrink the walkable surface when baking the navigation mesh.
var agent_radius: float:
	get = get_agent_radius, set = set_agent_radius

## If the baking [Rect2] has an area the navigation mesh baking will be restricted to its enclosing area.
var baking_rect: Rect2:
	get = get_baking_rect, set = set_baking_rect

## The position offset applied to the [member baking_rect] [Rect2].
var baking_rect_offset: Vector2:
	get = get_baking_rect_offset, set = set_baking_rect_offset

## The size of the non-navigable border around the bake bounding area defined by the [member baking_rect] [Rect2].
## In conjunction with the [member baking_rect] the border size can be used to bake tile aligned navigation meshes without the tile edges being shrunk by [member agent_radius].
var border_size: float:
	get = get_border_size, set = set_border_size

## The cell size used to rasterize the navigation mesh vertices. Must match with the cell size on the navigation map.
var cell_size: float:
	get = get_cell_size, set = set_cell_size

## The physics layers to scan for static colliders.
## Only used when [member parsed_geometry_type] is [constant PARSED_GEOMETRY_STATIC_COLLIDERS] or [constant PARSED_GEOMETRY_BOTH].
var parsed_collision_mask: int:
	get = get_parsed_collision_mask, set = set_parsed_collision_mask

## Determines which type of nodes will be parsed as geometry.
var parsed_geometry_type: int:
	get = get_parsed_geometry_type, set = set_parsed_geometry_type

## Partitioning algorithm for creating the navigation mesh polys.
var sample_partition_type: int:
	get = get_sample_partition_type, set = set_sample_partition_type

## The group name of nodes that should be parsed for baking source geometry.
## Only used when [member source_geometry_mode] is [constant SOURCE_GEOMETRY_GROUPS_WITH_CHILDREN] or [constant SOURCE_GEOMETRY_GROUPS_EXPLICIT].
var source_geometry_group_name: StringName:
	get = get_source_geometry_group_name, set = set_source_geometry_group_name

## The source of the geometry used when baking.
var source_geometry_mode: int:
	get = get_source_geometry_mode, set = set_source_geometry_mode



## Appends a [PackedVector2Array] that contains the vertices of an outline to the internal array that contains all the outlines.
func add_outline(outline: PackedVector2Array) -> void:
	pass;

## Adds a [PackedVector2Array] that contains the vertices of an outline to the internal array that contains all the outlines at a fixed position.
func add_outline_at_index(outline: PackedVector2Array, index: int) -> void:
	pass;

## Adds a polygon using the indices of the vertices you get when calling [method get_vertices].
func add_polygon(polygon: PackedInt32Array) -> void:
	pass;

## Clears the internal arrays for vertices and polygon indices.
func clear() -> void:
	pass;

## Clears the array of the outlines, but it doesn't clear the vertices and the polygons that were created by them.
func clear_outlines() -> void:
	pass;

## Clears the array of polygons, but it doesn't clear the array of outlines and vertices.
func clear_polygons() -> void:
	pass;

## Returns the [NavigationMesh] resulting from this navigation polygon. This navigation mesh can be used to update the navigation mesh of a region with the [method NavigationServer3D.region_set_navigation_mesh] API directly (as 2D uses the 3D server behind the scene).
func get_navigation_mesh() -> NavigationMesh:
	pass;

## Returns a [PackedVector2Array] containing the vertices of an outline that was created in the editor or by script.
func get_outline(idx: int) -> PackedVector2Array:
	pass;

## Returns the number of outlines that were created in the editor or by script.
func get_outline_count() -> int:
	pass;

## Returns whether or not the specified layer of the [member parsed_collision_mask] is enabled, given a [param layer_number] between 1 and 32.
func get_parsed_collision_mask_value(layer_number: int) -> bool:
	pass;

## Returns a [PackedInt32Array] containing the indices of the vertices of a created polygon.
func get_polygon(idx: int) -> PackedInt32Array:
	pass;

## Returns the count of all polygons.
func get_polygon_count() -> int:
	pass;

## Returns a [PackedVector2Array] containing all the vertices being used to create the polygons.
func get_vertices() -> PackedVector2Array:
	pass;

## Creates polygons from the outlines added in the editor or by script.
func make_polygons_from_outlines() -> void:
	pass;

## Removes an outline created in the editor or by script. You have to call [method make_polygons_from_outlines] for the polygons to update.
func remove_outline(idx: int) -> void:
	pass;

## Changes an outline created in the editor or by script. You have to call [method make_polygons_from_outlines] for the polygons to update.
func set_outline(idx: int, outline: PackedVector2Array) -> void:
	pass;

## Based on [param value], enables or disables the specified layer in the [member parsed_collision_mask], given a [param layer_number] between 1 and 32.
func set_parsed_collision_mask_value(layer_number: int, value: bool) -> void:
	pass;

## Sets the vertices that can be then indexed to create polygons with the [method add_polygon] method.
func set_vertices(vertices: PackedVector2Array) -> void:
	pass;


func get_agent_radius() -> float:
	return agent_radius

func set_agent_radius(value: float) -> void:
	agent_radius = value

func get_baking_rect() -> Rect2:
	return baking_rect

func set_baking_rect(value: Rect2) -> void:
	baking_rect = value

func get_baking_rect_offset() -> Vector2:
	return baking_rect_offset

func set_baking_rect_offset(value: Vector2) -> void:
	baking_rect_offset = value

func get_border_size() -> float:
	return border_size

func set_border_size(value: float) -> void:
	border_size = value

func get_cell_size() -> float:
	return cell_size

func set_cell_size(value: float) -> void:
	cell_size = value

func get_parsed_collision_mask() -> int:
	return parsed_collision_mask

func set_parsed_collision_mask(value: int) -> void:
	parsed_collision_mask = value

func get_parsed_geometry_type() -> int:
	return parsed_geometry_type

func set_parsed_geometry_type(value: int) -> void:
	parsed_geometry_type = value

func get_sample_partition_type() -> int:
	return sample_partition_type

func set_sample_partition_type(value: int) -> void:
	sample_partition_type = value

func get_source_geometry_group_name() -> StringName:
	return source_geometry_group_name

func set_source_geometry_group_name(value: StringName) -> void:
	source_geometry_group_name = value

func get_source_geometry_mode() -> int:
	return source_geometry_mode

func set_source_geometry_mode(value: int) -> void:
	source_geometry_mode = value

