extends RefCounted
class_name NavigationPathQueryParameters2D

## Provides parameters for 2D navigation path queries.
##
## By changing various properties of this object, such as the start and target position, you can configure path queries to the [NavigationServer2D].
##
## @tutorial(Using NavigationPathQueryObjects): https://docs.godotengine.org/en/stable/tutorials/navigation/navigation_using_navigationpathqueryobjects.html


## The path query uses the default A* pathfinding algorithm.
## Applies a funnel algorithm to the raw path corridor found by the pathfinding algorithm. This will result in the shortest path possible inside the path corridor. This postprocessing very much depends on the navigation mesh polygon layout and the created corridor. Especially tile- or gridbased layouts can face artificial corners with diagonal movement due to a jagged path corridor imposed by the cell shapes.
## Centers every path position in the middle of the traveled navigation mesh polygon edge. This creates better paths for tile- or gridbased layouts that restrict the movement to the cells center.
## Applies no postprocessing and returns the raw path corridor as found by the pathfinding algorithm.
## Don't include any additional metadata about the returned path.
## Include the type of navigation primitive (region or link) that each point of the path goes through.
## Include the [RID]s of the regions and links that each point of the path goes through.
## Include the [code]ObjectID[/code]s of the [Object]s which manage the regions and links each point of the path goes through.
## Include all available metadata about the returned path.

#enum PathfindingAlgorithm
enum {
    PATHFINDING_ALGORITHM_ASTAR = 0,
}
#enum PathPostProcessing
enum {
    PATH_POSTPROCESSING_CORRIDORFUNNEL = 0,
    PATH_POSTPROCESSING_EDGECENTERED = 1,
    PATH_POSTPROCESSING_NONE = 2,
}
#enum PathMetadataFlags
enum {
    PATH_METADATA_INCLUDE_NONE = 0,
    PATH_METADATA_INCLUDE_TYPES = 1,
    PATH_METADATA_INCLUDE_RIDS = 2,
    PATH_METADATA_INCLUDE_OWNERS = 4,
    PATH_METADATA_INCLUDE_ALL = 7,
}
## The list of region [RID]s that will be excluded from the path query. Use [method NavigationRegion2D.get_rid] to get the [RID] associated with a [NavigationRegion2D] node.
## [b]Note:[/b] The returned array is copied and any changes to it will not update the original property value. To update the value you need to modify the returned array, and then set it to the property again.
var excluded_regions: Array[RID]:
	get = get_excluded_regions, set = set_excluded_regions

## The list of region [RID]s that will be included by the path query. Use [method NavigationRegion2D.get_rid] to get the [RID] associated with a [NavigationRegion2D] node. If left empty all regions are included. If a region ends up being both included and excluded at the same time it will be excluded.
## [b]Note:[/b] The returned array is copied and any changes to it will not update the original property value. To update the value you need to modify the returned array, and then set it to the property again.
var included_regions: Array[RID]:
	get = get_included_regions, set = set_included_regions

## The navigation map [RID] used in the path query.
var map: RID:
	get = get_map, set = set_map

## Additional information to include with the navigation path.
var metadata_flags: int:
	get = get_metadata_flags, set = set_metadata_flags

## The navigation layers the query will use (as a bitmask).
var navigation_layers: int:
	get = get_navigation_layers, set = set_navigation_layers

## The path postprocessing applied to the raw path corridor found by the [member pathfinding_algorithm].
var path_postprocessing: int:
	get = get_path_postprocessing, set = set_path_postprocessing

## The pathfinding algorithm used in the path query.
var pathfinding_algorithm: int:
	get = get_pathfinding_algorithm, set = set_pathfinding_algorithm

## The path simplification amount in worlds units.
var simplify_epsilon: float:
	get = get_simplify_epsilon, set = set_simplify_epsilon

## If [code]true[/code] a simplified version of the path will be returned with less critical path points removed. The simplification amount is controlled by [member simplify_epsilon]. The simplification uses a variant of Ramer-Douglas-Peucker algorithm for curve point decimation.
## Path simplification can be helpful to mitigate various path following issues that can arise with certain agent types and script behaviors. E.g. "steering" agents or avoidance in "open fields".
var simplify_path: bool:
	get = get_simplify_path, set = set_simplify_path

## The pathfinding start position in global coordinates.
var start_position: Vector2:
	get = get_start_position, set = set_start_position

## The pathfinding target position in global coordinates.
var target_position: Vector2:
	get = get_target_position, set = set_target_position




func get_excluded_regions() -> Array[RID]:
	return excluded_regions

func set_excluded_regions(value: Array[RID]) -> void:
	excluded_regions = value

func get_included_regions() -> Array[RID]:
	return included_regions

func set_included_regions(value: Array[RID]) -> void:
	included_regions = value

func get_map() -> RID:
	return map

func set_map(value: RID) -> void:
	map = value

func get_metadata_flags() -> int:
	return metadata_flags

func set_metadata_flags(value: int) -> void:
	metadata_flags = value

func get_navigation_layers() -> int:
	return navigation_layers

func set_navigation_layers(value: int) -> void:
	navigation_layers = value

func get_path_postprocessing() -> int:
	return path_postprocessing

func set_path_postprocessing(value: int) -> void:
	path_postprocessing = value

func get_pathfinding_algorithm() -> int:
	return pathfinding_algorithm

func set_pathfinding_algorithm(value: int) -> void:
	pathfinding_algorithm = value

func get_simplify_epsilon() -> float:
	return simplify_epsilon

func set_simplify_epsilon(value: float) -> void:
	simplify_epsilon = value

func get_simplify_path() -> bool:
	return simplify_path

func set_simplify_path(value: bool) -> void:
	simplify_path = value

func get_start_position() -> Vector2:
	return start_position

func set_start_position(value: Vector2) -> void:
	start_position = value

func get_target_position() -> Vector2:
	return target_position

func set_target_position(value: Vector2) -> void:
	target_position = value

