extends GraphElement
class_name GraphNode

## A container with connection ports, representing a node in a [GraphEdit].
##
## [GraphNode] allows to create nodes for a [GraphEdit] graph with customizable content based on its child controls. [GraphNode] is derived from [Container] and it is responsible for placing its children on screen. This works similar to [VBoxContainer]. Children, in turn, provide [GraphNode] with so-called slots, each of which can have a connection port on either side.
## Each [GraphNode] slot is defined by its index and can provide the node with up to two ports: one on the left, and one on the right. By convention the left port is also referred to as the [b]input port[/b] and the right port is referred to as the [b]output port[/b]. Each port can be enabled and configured individually, using different type and color. The type is an arbitrary value that you can define using your own considerations. The parent [GraphEdit] will receive this information on each connect and disconnect request.
## Slots can be configured in the Inspector dock once you add at least one child [Control]. The properties are grouped by each slot's index in the "Slot" section.
## [b]Note:[/b] While GraphNode is set up using slots and slot indices, connections are made between the ports which are enabled. Because of that [GraphEdit] uses the port's index and not the slot's index. You can use [method get_input_port_slot] and [method get_output_port_slot] to get the slot index from the port index.


## Emitted when any slot's size might have changed.
signal slot_sizes_changed
## Emitted when any GraphNode's slot is updated.
signal slot_updated(slot_index: int)

var focus_mode: int:
	get = get_focus_mode, set = set_focus_mode

## If [code]true[/code], you can connect ports with different types, even if the connection was not explicitly allowed in the parent [GraphEdit].
var ignore_invalid_connection_type: bool:
	get = is_ignoring_valid_connection_type, set = set_ignore_invalid_connection_type

var mouse_filter: int:
	get = get_mouse_filter, set = set_mouse_filter

## The text displayed in the GraphNode's title bar.
var title: String:
	get = get_title, set = set_title



func _draw_port(slot_index: int, position: Vector2i, left: bool, color: Color) -> void:
	pass;

## Disables all slots of the GraphNode. This will remove all input/output ports from the GraphNode.
func clear_all_slots() -> void:
	pass;

## Disables the slot with the given [param slot_index]. This will remove the corresponding input and output port from the GraphNode.
func clear_slot(slot_index: int) -> void:
	pass;

## Returns the [Color] of the input port with the given [param port_idx].
func get_input_port_color(port_idx: int) -> Color:
	pass;

## Returns the number of slots with an enabled input port.
func get_input_port_count() -> int:
	pass;

## Returns the position of the input port with the given [param port_idx].
func get_input_port_position(port_idx: int) -> Vector2:
	pass;

## Returns the corresponding slot index of the input port with the given [param port_idx].
func get_input_port_slot(port_idx: int) -> int:
	pass;

## Returns the type of the input port with the given [param port_idx].
func get_input_port_type(port_idx: int) -> int:
	pass;

## Returns the [Color] of the output port with the given [param port_idx].
func get_output_port_color(port_idx: int) -> Color:
	pass;

## Returns the number of slots with an enabled output port.
func get_output_port_count() -> int:
	pass;

## Returns the position of the output port with the given [param port_idx].
func get_output_port_position(port_idx: int) -> Vector2:
	pass;

## Returns the corresponding slot index of the output port with the given [param port_idx].
func get_output_port_slot(port_idx: int) -> int:
	pass;

## Returns the type of the output port with the given [param port_idx].
func get_output_port_type(port_idx: int) -> int:
	pass;

## Returns the left (input) [Color] of the slot with the given [param slot_index].
func get_slot_color_left(slot_index: int) -> Color:
	pass;

## Returns the right (output) [Color] of the slot with the given [param slot_index].
func get_slot_color_right(slot_index: int) -> Color:
	pass;

## Returns the left (input) custom [Texture2D] of the slot with the given [param slot_index].
func get_slot_custom_icon_left(slot_index: int) -> Texture2D:
	pass;

## Returns the right (output) custom [Texture2D] of the slot with the given [param slot_index].
func get_slot_custom_icon_right(slot_index: int) -> Texture2D:
	pass;

## Returns the left (input) type of the slot with the given [param slot_index].
func get_slot_type_left(slot_index: int) -> int:
	pass;

## Returns the right (output) type of the slot with the given [param slot_index].
func get_slot_type_right(slot_index: int) -> int:
	pass;

## Returns the [HBoxContainer] used for the title bar, only containing a [Label] for displaying the title by default. This can be used to add custom controls to the title bar such as option or close buttons.
func get_titlebar_hbox() -> HBoxContainer:
	pass;

## Returns [code]true[/code] if the background [StyleBox] of the slot with the given [param slot_index] is drawn.
func is_slot_draw_stylebox(slot_index: int) -> bool:
	pass;

## Returns [code]true[/code] if left (input) side of the slot with the given [param slot_index] is enabled.
func is_slot_enabled_left(slot_index: int) -> bool:
	pass;

## Returns [code]true[/code] if right (output) side of the slot with the given [param slot_index] is enabled.
func is_slot_enabled_right(slot_index: int) -> bool:
	pass;

## Sets properties of the slot with the given [param slot_index].
## If [param enable_left_port]/[param enable_right_port] is [code]true[/code], a port will appear and the slot will be able to be connected from this side.
## With [param type_left]/[param type_right] an arbitrary type can be assigned to each port. Two ports can be connected if they share the same type, or if the connection between their types is allowed in the parent [GraphEdit] (see [method GraphEdit.add_valid_connection_type]). Keep in mind that the [GraphEdit] has the final say in accepting the connection. Type compatibility simply allows the [signal GraphEdit.connection_request] signal to be emitted.
## Ports can be further customized using [param color_left]/[param color_right] and [param custom_icon_left]/[param custom_icon_right]. The color parameter adds a tint to the icon. The custom icon can be used to override the default port dot.
## Additionally, [param draw_stylebox] can be used to enable or disable drawing of the background stylebox for each slot. See [theme_item slot].
## Individual properties can also be set using one of the [code]set_slot_*[/code] methods.
## [b]Note:[/b] This method only sets properties of the slot. To create the slot itself, add a [Control]-derived child to the GraphNode.
func set_slot(slot_index: int, enable_left_port: bool, type_left: int, color_left: Color, enable_right_port: bool, type_right: int, color_right: Color, custom_icon_left: Texture2D = null, custom_icon_right: Texture2D = null, draw_stylebox: bool = true) -> void:
	pass;

## Sets the [Color] of the left (input) side of the slot with the given [param slot_index] to [param color].
func set_slot_color_left(slot_index: int, color: Color) -> void:
	pass;

## Sets the [Color] of the right (output) side of the slot with the given [param slot_index] to [param color].
func set_slot_color_right(slot_index: int, color: Color) -> void:
	pass;

## Sets the custom [Texture2D] of the left (input) side of the slot with the given [param slot_index] to [param custom_icon].
func set_slot_custom_icon_left(slot_index: int, custom_icon: Texture2D) -> void:
	pass;

## Sets the custom [Texture2D] of the right (output) side of the slot with the given [param slot_index] to [param custom_icon].
func set_slot_custom_icon_right(slot_index: int, custom_icon: Texture2D) -> void:
	pass;

## Toggles the background [StyleBox] of the slot with the given [param slot_index].
func set_slot_draw_stylebox(slot_index: int, enable: bool) -> void:
	pass;

## Toggles the left (input) side of the slot with the given [param slot_index]. If [param enable] is [code]true[/code], a port will appear on the left side and the slot will be able to be connected from this side.
func set_slot_enabled_left(slot_index: int, enable: bool) -> void:
	pass;

## Toggles the right (output) side of the slot with the given [param slot_index]. If [param enable] is [code]true[/code], a port will appear on the right side and the slot will be able to be connected from this side.
func set_slot_enabled_right(slot_index: int, enable: bool) -> void:
	pass;

## Sets the left (input) type of the slot with the given [param slot_index] to [param type]. If the value is negative, all connections will be disallowed to be created via user inputs.
func set_slot_type_left(slot_index: int, type: int) -> void:
	pass;

## Sets the right (output) type of the slot with the given [param slot_index] to [param type]. If the value is negative, all connections will be disallowed to be created via user inputs.
func set_slot_type_right(slot_index: int, type: int) -> void:
	pass;


func get_focus_mode() -> int:
	return focus_mode

func set_focus_mode(value: int) -> void:
	focus_mode = value

func is_ignoring_valid_connection_type() -> bool:
	return ignore_invalid_connection_type

func set_ignore_invalid_connection_type(value: bool) -> void:
	ignore_invalid_connection_type = value

func get_mouse_filter() -> int:
	return mouse_filter

func set_mouse_filter(value: int) -> void:
	mouse_filter = value

func get_title() -> String:
	return title

func set_title(value: String) -> void:
	title = value

