extends RefCounted
class_name GLTFObjectModelProperty

## Describes how to access a property as defined in the glTF object model.
##
## GLTFObjectModelProperty defines a mapping between a property in the glTF object model and a NodePath in the Godot scene tree. This can be used to animate properties in a glTF file using the [code]KHR_animation_pointer[/code] extension, or to access them through an engine-agnostic script such as a behavior graph as defined by the [code]KHR_interactivity[/code] extension.
## The glTF property is identified by JSON pointer(s) stored in [member json_pointers], while the Godot property it maps to is defined by [member node_paths]. In most cases [member json_pointers] and [member node_paths] will each only have one item, but in some cases a single glTF JSON pointer will map to multiple Godot properties, or a single Godot property will be mapped to multiple glTF JSON pointers, or it might be a many-to-many relationship.
## [Expression] objects can be used to define conversions between the data, such as when glTF defines an angle in radians and Godot uses degrees. The [member object_model_type] property defines the type of data stored in the glTF file as defined by the object model, see [enum GLTFObjectModelType] for possible values.
##
## @tutorial(GLTF Object Model): https://github.com/KhronosGroup/glTF/blob/main/specification/2.0/ObjectModel.adoc
## @tutorial(KHR_animation_pointer GLTF extension): https://github.com/KhronosGroup/glTF/tree/main/extensions/2.0/Khronos/KHR_animation_pointer


## Unknown or not set object model type. If the object model type is set to this value, the real type still needs to be determined.
## Object model type "bool". Represented in the glTF JSON as a boolean, and encoded in a [GLTFAccessor] as "SCALAR". When encoded in an accessor, a value of [code]0[/code] is [code]false[/code], and any other value is [code]true[/code].
## Object model type "float". Represented in the glTF JSON as a number, and encoded in a [GLTFAccessor] as "SCALAR".
## Object model type "float[lb][rb]". Represented in the glTF JSON as an array of numbers, and encoded in a [GLTFAccessor] as "SCALAR".
## Object model type "float2". Represented in the glTF JSON as an array of two numbers, and encoded in a [GLTFAccessor] as "VEC2".
## Object model type "float3". Represented in the glTF JSON as an array of three numbers, and encoded in a [GLTFAccessor] as "VEC3".
## Object model type "float4". Represented in the glTF JSON as an array of four numbers, and encoded in a [GLTFAccessor] as "VEC4".
## Object model type "float2x2". Represented in the glTF JSON as an array of four numbers, and encoded in a [GLTFAccessor] as "MAT2".
## Object model type "float3x3". Represented in the glTF JSON as an array of nine numbers, and encoded in a [GLTFAccessor] as "MAT3".
## Object model type "float4x4". Represented in the glTF JSON as an array of sixteen numbers, and encoded in a [GLTFAccessor] as "MAT4".
## Object model type "int". Represented in the glTF JSON as a number, and encoded in a [GLTFAccessor] as "SCALAR". The range of values is limited to signed integers. For [code]KHR_interactivity[/code], only 32-bit integers are supported.

#enum GLTFObjectModelType
enum {
    GLTF_OBJECT_MODEL_TYPE_UNKNOWN = 0,
    GLTF_OBJECT_MODEL_TYPE_BOOL = 1,
    GLTF_OBJECT_MODEL_TYPE_FLOAT = 2,
    GLTF_OBJECT_MODEL_TYPE_FLOAT_ARRAY = 3,
    GLTF_OBJECT_MODEL_TYPE_FLOAT2 = 4,
    GLTF_OBJECT_MODEL_TYPE_FLOAT3 = 5,
    GLTF_OBJECT_MODEL_TYPE_FLOAT4 = 6,
    GLTF_OBJECT_MODEL_TYPE_FLOAT2X2 = 7,
    GLTF_OBJECT_MODEL_TYPE_FLOAT3X3 = 8,
    GLTF_OBJECT_MODEL_TYPE_FLOAT4X4 = 9,
    GLTF_OBJECT_MODEL_TYPE_INT = 10,
}
## If set, this [Expression] will be used to convert the property value from the glTF object model to the value expected by the Godot property. This is useful when the glTF object model uses a different unit system, or when the data needs to be transformed in some way. If [code]null[/code], the value will be copied as-is.
var gltf_to_godot_expression: Expression:
	get = get_gltf_to_godot_expression, set = set_gltf_to_godot_expression

## If set, this [Expression] will be used to convert the property value from the Godot property to the value expected by the glTF object model. This is useful when the glTF object model uses a different unit system, or when the data needs to be transformed in some way. If [code]null[/code], the value will be copied as-is.
var godot_to_gltf_expression: Expression:
	get = get_godot_to_gltf_expression, set = set_godot_to_gltf_expression

## The glTF object model JSON pointers used to identify the property in the glTF object model. In most cases, there will be only one item in this array, but specific cases may require multiple pointers. The items are themselves arrays which represent the JSON pointer split into its components.
var json_pointers: Array[PackedStringArray]:
	get = get_json_pointers, set = set_json_pointers

## An array of [NodePath]s that point to a property, or multiple properties, in the Godot scene tree. On import, this will either be set by [GLTFDocument], or by a [GLTFDocumentExtension] class. For simple cases, use [method append_path_to_property] to add properties to this array.
## In most cases [member node_paths] will only have one item, but in some cases a single glTF JSON pointer will map to multiple Godot properties. For example, a [GLTFCamera] or [GLTFLight] used on multiple glTF nodes will be represented by multiple Godot nodes.
var node_paths: Array[NodePath]:
	get = get_node_paths, set = set_node_paths

## The type of data stored in the glTF file as defined by the object model. This is a superset of the available accessor types, and determines the accessor type.
var object_model_type: int:
	get = get_object_model_type, set = set_object_model_type

## The type of data stored in the Godot property. This is the type of the property that the [member node_paths] point to.
var variant_type: int:
	get = get_variant_type, set = set_variant_type



## Appends a [NodePath] to [member node_paths]. This can be used by [GLTFDocumentExtension] classes to define how a glTF object model property maps to a Godot property, or multiple Godot properties. Prefer using [method append_path_to_property] for simple cases. Be sure to also call [method set_types] once (the order does not matter).
func append_node_path(node_path: NodePath) -> void:
	pass;

## High-level wrapper over [method append_node_path] that handles the most common cases. It constructs a new [NodePath] using [param node_path] as a base and appends [param prop_name] to the subpath. Be sure to also call [method set_types] once (the order does not matter).
func append_path_to_property(node_path: NodePath, prop_name: StringName) -> void:
	pass;

## The GLTF accessor type associated with this property's [member object_model_type]. See [member GLTFAccessor.accessor_type] for possible values, and see [enum GLTFObjectModelType] for how the object model type maps to accessor types.
func get_accessor_type() -> int:
	pass;

## Returns [code]true[/code] if [member json_pointers] is not empty. This is used during export to determine if a [GLTFObjectModelProperty] can handle converting a Godot property to a glTF object model property.
func has_json_pointers() -> bool:
	pass;

## Returns [code]true[/code] if [member node_paths] is not empty. This is used during import to determine if a [GLTFObjectModelProperty] can handle converting a glTF object model property to a Godot property.
func has_node_paths() -> bool:
	pass;

## Sets the [member variant_type] and [member object_model_type] properties. This is a convenience method to set both properties at once, since they are almost always known at the same time. This method should be called once. Calling it again with the same values will have no effect.
func set_types(variant_type: int, obj_model_type: int) -> void:
	pass;


func get_gltf_to_godot_expression() -> Expression:
	return gltf_to_godot_expression

func set_gltf_to_godot_expression(value: Expression) -> void:
	gltf_to_godot_expression = value

func get_godot_to_gltf_expression() -> Expression:
	return godot_to_gltf_expression

func set_godot_to_gltf_expression(value: Expression) -> void:
	godot_to_gltf_expression = value

func get_json_pointers() -> Array[PackedStringArray]:
	return json_pointers

func set_json_pointers(value: Array[PackedStringArray]) -> void:
	json_pointers = value

func get_node_paths() -> Array[NodePath]:
	return node_paths

func set_node_paths(value: Array[NodePath]) -> void:
	node_paths = value

func get_object_model_type() -> int:
	return object_model_type

func set_object_model_type(value: int) -> void:
	object_model_type = value

func get_variant_type() -> int:
	return variant_type

func set_variant_type(value: int) -> void:
	variant_type = value

