extends Resource
class_name GLTFDocument

## Class for importing and exporting glTF files in and out of Godot.
##
## GLTFDocument supports reading data from a glTF file, buffer, or Godot scene. This data can then be written to the filesystem, buffer, or used to create a Godot scene.
## All of the data in a glTF scene is stored in the [GLTFState] class. GLTFDocument processes state objects, but does not contain any scene data itself. GLTFDocument has member variables to store export configuration settings such as the image format, but is otherwise stateless. Multiple scenes can be processed with the same settings using the same GLTFDocument object and different [GLTFState] objects.
## GLTFDocument can be extended with arbitrary functionality by extending the [GLTFDocumentExtension] class and registering it with GLTFDocument via [method register_gltf_document_extension]. This allows for custom data to be imported and exported.
##
## @tutorial(Runtime file loading and saving): https://docs.godotengine.org/en/stable/tutorials/io/runtime_file_loading_and_saving.html
## @tutorial(glTF 'What the duck?' guide): https://www.khronos.org/files/gltf20-reference-guide.pdf
## @tutorial(Khronos glTF specification): https://registry.khronos.org/glTF/


## Treat the Godot scene's root node as the root node of the glTF file, and mark it as the single root node via the [code]GODOT_single_root[/code] glTF extension. This will be parsed the same as [constant ROOT_NODE_MODE_KEEP_ROOT] if the implementation does not support [code]GODOT_single_root[/code].
## Treat the Godot scene's root node as the root node of the glTF file, but do not mark it as anything special. An extra root node will be generated when importing into Godot. This uses only vanilla glTF features. This is equivalent to the behavior in Godot 4.1 and earlier.
## Treat the Godot scene's root node as the name of the glTF scene, and add all of its children as root nodes of the glTF file. This uses only vanilla glTF features. This avoids an extra root node, but only the name of the Godot scene's root node will be preserved, as it will not be saved as a node.
## If the scene contains any non-visible nodes, include them, mark them as non-visible with [code]KHR_node_visibility[/code], and require that importers respect their non-visibility. Downside: If the importer does not support [code]KHR_node_visibility[/code], the file cannot be imported.
## If the scene contains any non-visible nodes, include them, mark them as non-visible with [code]KHR_node_visibility[/code], and do not impose any requirements on importers. Downside: If the importer does not support [code]KHR_node_visibility[/code], invisible objects will be visible.
## If the scene contains any non-visible nodes, do not include them in the export. This is the same as the behavior in Godot 4.4 and earlier. Downside: Invisible nodes will not exist in the exported file.

#enum RootNodeMode
enum {
    ROOT_NODE_MODE_SINGLE_ROOT = 0,
    ROOT_NODE_MODE_KEEP_ROOT = 1,
    ROOT_NODE_MODE_MULTI_ROOT = 2,
}
#enum VisibilityMode
enum {
    VISIBILITY_MODE_INCLUDE_REQUIRED = 0,
    VISIBILITY_MODE_INCLUDE_OPTIONAL = 1,
    VISIBILITY_MODE_EXCLUDE = 2,
}
## The user-friendly name of the fallback image format. This is used when exporting the glTF file, including writing to a file and writing to a byte array.
## This property may only be one of "None", "PNG", or "JPEG", and is only used when the [member image_format] is not one of "None", "PNG", or "JPEG". If having multiple extension image formats is desired, that can be done using a [GLTFDocumentExtension] class - this property only covers the use case of providing a base glTF fallback image when using a custom image format.
var fallback_image_format: String:
	get = get_fallback_image_format, set = set_fallback_image_format

## The quality of the fallback image, if any. For PNG files, this downscales the image on both dimensions by this factor. For JPEG files, this is the lossy quality of the image. A low value is recommended, since including multiple high quality images in a glTF file defeats the file size gains of using a more efficient image format.
var fallback_image_quality: float:
	get = get_fallback_image_quality, set = set_fallback_image_quality

## The user-friendly name of the export image format. This is used when exporting the glTF file, including writing to a file and writing to a byte array.
## By default, Godot allows the following options: "None", "PNG", "JPEG", "Lossless WebP", and "Lossy WebP". Support for more image formats can be added in [GLTFDocumentExtension] classes. A single extension class can provide multiple options for the specific format to use, or even an option that uses multiple formats at once.
var image_format: String:
	get = get_image_format, set = set_image_format

## If [member image_format] is a lossy image format, this determines the lossy quality of the image. On a range of [code]0.0[/code] to [code]1.0[/code], where [code]0.0[/code] is the lowest quality and [code]1.0[/code] is the highest quality. A lossy quality of [code]1.0[/code] is not the same as lossless.
var lossy_quality: float:
	get = get_lossy_quality, set = set_lossy_quality

## How to process the root node during export. The default and recommended value is [constant ROOT_NODE_MODE_SINGLE_ROOT].
## [b]Note:[/b] Regardless of how the glTF file is exported, when importing, the root node type and name can be overridden in the scene import settings tab.
var root_node_mode: int:
	get = get_root_node_mode, set = set_root_node_mode

## How to deal with node visibility during export. This setting does nothing if all nodes are visible. The default and recommended value is [constant VISIBILITY_MODE_INCLUDE_REQUIRED], which uses the [code]KHR_node_visibility[/code] extension.
var visibility_mode: int:
	get = get_visibility_mode, set = set_visibility_mode



## Takes a [PackedByteArray] defining a glTF and imports the data to the given [GLTFState] object through the [param state] parameter.
## [b]Note:[/b] The [param base_path] tells [method append_from_buffer] where to find dependencies and can be empty.
func append_from_buffer(bytes: PackedByteArray, base_path: String, state: GLTFState, flags: int = 0) -> int:
	pass;

## Takes a path to a glTF file and imports the data at that file path to the given [GLTFState] object through the [param state] parameter.
## [b]Note:[/b] The [param base_path] tells [method append_from_file] where to find dependencies and can be empty.
func append_from_file(path: String, state: GLTFState, flags: int = 0, base_path: String = "") -> int:
	pass;

## Takes a Godot Engine scene node and exports it and its descendants to the given [GLTFState] object through the [param state] parameter.
func append_from_scene(node: Node, state: GLTFState, flags: int = 0) -> int:
	pass;

## Determines a mapping between the given Godot [param node_path] and the corresponding glTF Object Model JSON pointer(s) in the generated glTF file. The details of this mapping are returned in a [GLTFObjectModelProperty] object. Additional mappings can be supplied via the [method GLTFDocumentExtension._import_object_model_property] callback method.
static func export_object_model_property(state: GLTFState, node_path: NodePath, godot_node: Node, gltf_node_index: int) -> GLTFObjectModelProperty:
	pass;

## Takes a [GLTFState] object through the [param state] parameter and returns a glTF [PackedByteArray].
func generate_buffer(state: GLTFState) -> PackedByteArray:
	pass;

## Takes a [GLTFState] object through the [param state] parameter and returns a Godot Engine scene node.
## The [param bake_fps] parameter overrides the bake_fps in [param state].
func generate_scene(state: GLTFState, bake_fps: float = 30, trimming: bool = false, remove_immutable_tracks: bool = true) -> Node:
	pass;

## Returns a list of all support glTF extensions, including extensions supported directly by the engine, and extensions supported by user plugins registering [GLTFDocumentExtension] classes.
## [b]Note:[/b] If this method is run before a GLTFDocumentExtension is registered, its extensions won't be included in the list. Be sure to only run this method after all extensions are registered. If you run this when the engine starts, consider waiting a frame before calling this method to ensure all extensions are registered.
static func get_supported_gltf_extensions() -> PackedStringArray:
	pass;

## Determines a mapping between the given glTF Object Model [param json_pointer] and the corresponding Godot node path(s) in the generated Godot scene. The details of this mapping are returned in a [GLTFObjectModelProperty] object. Additional mappings can be supplied via the [method GLTFDocumentExtension._export_object_model_property] callback method.
static func import_object_model_property(state: GLTFState, json_pointer: String) -> GLTFObjectModelProperty:
	pass;

## Registers the given [GLTFDocumentExtension] instance with GLTFDocument. If [param first_priority] is [code]true[/code], this extension will be run first. Otherwise, it will be run last.
## [b]Note:[/b] Like GLTFDocument itself, all GLTFDocumentExtension classes must be stateless in order to function properly. If you need to store data, use the [code]set_additional_data[/code] and [code]get_additional_data[/code] methods in [GLTFState] or [GLTFNode].
static func register_gltf_document_extension(extension: GLTFDocumentExtension, first_priority: bool = false) -> void:
	pass;

## Unregisters the given [GLTFDocumentExtension] instance.
static func unregister_gltf_document_extension(extension: GLTFDocumentExtension) -> void:
	pass;

## Takes a [GLTFState] object through the [param state] parameter and writes a glTF file to the filesystem.
## [b]Note:[/b] The extension of the glTF file determines if it is a .glb binary file or a .gltf text file.
func write_to_filesystem(state: GLTFState, path: String) -> int:
	pass;


func get_fallback_image_format() -> String:
	return fallback_image_format

func set_fallback_image_format(value: String) -> void:
	fallback_image_format = value

func get_fallback_image_quality() -> float:
	return fallback_image_quality

func set_fallback_image_quality(value: float) -> void:
	fallback_image_quality = value

func get_image_format() -> String:
	return image_format

func set_image_format(value: String) -> void:
	image_format = value

func get_lossy_quality() -> float:
	return lossy_quality

func set_lossy_quality(value: float) -> void:
	lossy_quality = value

func get_root_node_mode() -> int:
	return root_node_mode

func set_root_node_mode(value: int) -> void:
	root_node_mode = value

func get_visibility_mode() -> int:
	return visibility_mode

func set_visibility_mode(value: int) -> void:
	visibility_mode = value

