extends Resource
class_name GLTFAccessor

## Represents a glTF accessor.
##
## GLTFAccessor is a data structure representing a glTF [code]accessor[/code] that would be found in the [code]"accessors"[/code] array. A buffer is a blob of binary data. A buffer view is a slice of a buffer. An accessor is a typed interpretation of the data in a buffer view.
## Most custom data stored in glTF does not need accessors, only buffer views (see [GLTFBufferView]). Accessors are for more advanced use cases such as interleaved mesh data encoded for the GPU.
##
## @tutorial(Buffers, BufferViews, and Accessors in Khronos glTF specification): https://github.com/KhronosGroup/glTF-Tutorials/blob/master/gltfTutorial/gltfTutorial_005_BuffersBufferViewsAccessors.md
## @tutorial(Runtime file loading and saving): https://docs.godotengine.org/en/stable/tutorials/io/runtime_file_loading_and_saving.html


## Accessor type "SCALAR". For the glTF object model, this can be used to map to a single float, int, or bool value, or a float array.
## Accessor type "VEC2". For the glTF object model, this maps to "float2", represented in the glTF JSON as an array of two floats.
## Accessor type "VEC3". For the glTF object model, this maps to "float3", represented in the glTF JSON as an array of three floats.
## Accessor type "VEC4". For the glTF object model, this maps to "float4", represented in the glTF JSON as an array of four floats.
## Accessor type "MAT2". For the glTF object model, this maps to "float2x2", represented in the glTF JSON as an array of four floats.
## Accessor type "MAT3". For the glTF object model, this maps to "float3x3", represented in the glTF JSON as an array of nine floats.
## Accessor type "MAT4". For the glTF object model, this maps to "float4x4", represented in the glTF JSON as an array of sixteen floats.
## Component type "NONE". This is not a valid component type, and is used to indicate that the component type is not set.
## Component type "BYTE". The value is [code]0x1400[/code] which comes from OpenGL. This indicates data is stored in 1-byte or 8-bit signed integers. This is a core part of the glTF specification.
## Component type "UNSIGNED_BYTE". The value is [code]0x1401[/code] which comes from OpenGL. This indicates data is stored in 1-byte or 8-bit unsigned integers. This is a core part of the glTF specification.
## Component type "SHORT". The value is [code]0x1402[/code] which comes from OpenGL. This indicates data is stored in 2-byte or 16-bit signed integers. This is a core part of the glTF specification.
## Component type "UNSIGNED_SHORT". The value is [code]0x1403[/code] which comes from OpenGL. This indicates data is stored in 2-byte or 16-bit unsigned integers. This is a core part of the glTF specification.
## Component type "INT". The value is [code]0x1404[/code] which comes from OpenGL. This indicates data is stored in 4-byte or 32-bit signed integers. This is NOT a core part of the glTF specification, and may not be supported by all glTF importers. May be used by some extensions including [code]KHR_interactivity[/code].
## Component type "UNSIGNED_INT". The value is [code]0x1405[/code] which comes from OpenGL. This indicates data is stored in 4-byte or 32-bit unsigned integers. This is a core part of the glTF specification.
## Component type "FLOAT". The value is [code]0x1406[/code] which comes from OpenGL. This indicates data is stored in 4-byte or 32-bit floating-point numbers. This is a core part of the glTF specification.
## Component type "DOUBLE". The value is [code]0x140A[/code] which comes from OpenGL. This indicates data is stored in 8-byte or 64-bit floating-point numbers. This is NOT a core part of the glTF specification, and may not be supported by all glTF importers. May be used by some extensions including [code]KHR_interactivity[/code].
## Component type "HALF_FLOAT". The value is [code]0x140B[/code] which comes from OpenGL. This indicates data is stored in 2-byte or 16-bit floating-point numbers. This is NOT a core part of the glTF specification, and may not be supported by all glTF importers. May be used by some extensions including [code]KHR_interactivity[/code].
## Component type "LONG". The value is [code]0x140E[/code] which comes from OpenGL. This indicates data is stored in 8-byte or 64-bit signed integers. This is NOT a core part of the glTF specification, and may not be supported by all glTF importers. May be used by some extensions including [code]KHR_interactivity[/code].
## Component type "UNSIGNED_LONG". The value is [code]0x140F[/code] which comes from OpenGL. This indicates data is stored in 8-byte or 64-bit unsigned integers. This is NOT a core part of the glTF specification, and may not be supported by all glTF importers. May be used by some extensions including [code]KHR_interactivity[/code].

#enum GLTFAccessorType
enum {
    TYPE_SCALAR = 0,
    TYPE_VEC2 = 1,
    TYPE_VEC3 = 2,
    TYPE_VEC4 = 3,
    TYPE_MAT2 = 4,
    TYPE_MAT3 = 5,
    TYPE_MAT4 = 6,
}
#enum GLTFComponentType
enum {
    COMPONENT_TYPE_NONE = 0,
    COMPONENT_TYPE_SIGNED_BYTE = 5120,
    COMPONENT_TYPE_UNSIGNED_BYTE = 5121,
    COMPONENT_TYPE_SIGNED_SHORT = 5122,
    COMPONENT_TYPE_UNSIGNED_SHORT = 5123,
    COMPONENT_TYPE_SIGNED_INT = 5124,
    COMPONENT_TYPE_UNSIGNED_INT = 5125,
    COMPONENT_TYPE_SINGLE_FLOAT = 5126,
    COMPONENT_TYPE_DOUBLE_FLOAT = 5130,
    COMPONENT_TYPE_HALF_FLOAT = 5131,
    COMPONENT_TYPE_SIGNED_LONG = 5134,
    COMPONENT_TYPE_UNSIGNED_LONG = 5135,
}
## The glTF accessor type, as an enum.
var accessor_type: int:
	get = get_accessor_type, set = set_accessor_type

## The index of the buffer view this accessor is referencing. If [code]-1[/code], this accessor is not referencing any buffer view.
var buffer_view: int:
	get = get_buffer_view, set = set_buffer_view

## The offset relative to the start of the buffer view in bytes.
var byte_offset: int:
	get = get_byte_offset, set = set_byte_offset

## The glTF component type as an enum. See [enum GLTFComponentType] for possible values. Within the core glTF specification, a value of 5125 or "UNSIGNED_INT" must not be used for any accessor that is not referenced by mesh.primitive.indices.
var component_type: int:
	get = get_component_type, set = set_component_type

## The number of elements referenced by this accessor.
var count: int:
	get = get_count, set = set_count

## Maximum value of each component in this accessor.
var max: PackedFloat64Array:
	get = get_max, set = set_max

## Minimum value of each component in this accessor.
var min: PackedFloat64Array:
	get = get_min, set = set_min

## Specifies whether integer data values are normalized before usage.
var normalized: bool:
	get = get_normalized, set = set_normalized

## Number of deviating accessor values stored in the sparse array.
var sparse_count: int:
	get = get_sparse_count, set = set_sparse_count

## The index of the buffer view with sparse indices. The referenced buffer view MUST NOT have its target or byteStride properties defined. The buffer view and the optional byteOffset MUST be aligned to the componentType byte length.
var sparse_indices_buffer_view: int:
	get = get_sparse_indices_buffer_view, set = set_sparse_indices_buffer_view

## The offset relative to the start of the buffer view in bytes.
var sparse_indices_byte_offset: int:
	get = get_sparse_indices_byte_offset, set = set_sparse_indices_byte_offset

## The indices component data type as an enum. Possible values are 5121 for "UNSIGNED_BYTE", 5123 for "UNSIGNED_SHORT", and 5125 for "UNSIGNED_INT".
var sparse_indices_component_type: int:
	get = get_sparse_indices_component_type, set = set_sparse_indices_component_type

## The index of the bufferView with sparse values. The referenced buffer view MUST NOT have its target or byteStride properties defined.
var sparse_values_buffer_view: int:
	get = get_sparse_values_buffer_view, set = set_sparse_values_buffer_view

## The offset relative to the start of the bufferView in bytes.
var sparse_values_byte_offset: int:
	get = get_sparse_values_byte_offset, set = set_sparse_values_byte_offset

## The glTF accessor type, as an [int]. Possible values are [code]0[/code] for "SCALAR", [code]1[/code] for "VEC2", [code]2[/code] for "VEC3", [code]3[/code] for "VEC4", [code]4[/code] for "MAT2", [code]5[/code] for "MAT3", and [code]6[/code] for "MAT4".
var type: int:
	get = get_type, set = set_type




func get_accessor_type() -> int:
	return accessor_type

func set_accessor_type(value: int) -> void:
	accessor_type = value

func get_buffer_view() -> int:
	return buffer_view

func set_buffer_view(value: int) -> void:
	buffer_view = value

func get_byte_offset() -> int:
	return byte_offset

func set_byte_offset(value: int) -> void:
	byte_offset = value

func get_component_type() -> int:
	return component_type

func set_component_type(value: int) -> void:
	component_type = value

func get_count() -> int:
	return count

func set_count(value: int) -> void:
	count = value

func get_max() -> PackedFloat64Array:
	return max

func set_max(value: PackedFloat64Array) -> void:
	max = value

func get_min() -> PackedFloat64Array:
	return min

func set_min(value: PackedFloat64Array) -> void:
	min = value

func get_normalized() -> bool:
	return normalized

func set_normalized(value: bool) -> void:
	normalized = value

func get_sparse_count() -> int:
	return sparse_count

func set_sparse_count(value: int) -> void:
	sparse_count = value

func get_sparse_indices_buffer_view() -> int:
	return sparse_indices_buffer_view

func set_sparse_indices_buffer_view(value: int) -> void:
	sparse_indices_buffer_view = value

func get_sparse_indices_byte_offset() -> int:
	return sparse_indices_byte_offset

func set_sparse_indices_byte_offset(value: int) -> void:
	sparse_indices_byte_offset = value

func get_sparse_indices_component_type() -> int:
	return sparse_indices_component_type

func set_sparse_indices_component_type(value: int) -> void:
	sparse_indices_component_type = value

func get_sparse_values_buffer_view() -> int:
	return sparse_values_buffer_view

func set_sparse_values_buffer_view(value: int) -> void:
	sparse_values_buffer_view = value

func get_sparse_values_byte_offset() -> int:
	return sparse_values_byte_offset

func set_sparse_values_byte_offset(value: int) -> void:
	sparse_values_byte_offset = value

func get_type() -> int:
	return type

func set_type(value: int) -> void:
	type = value

