extends RefCounted
class_name EditorExportPlatform

## Identifies a supported export platform, and internally provides the functionality of exporting to that platform.
##
## Base resource that provides the functionality of exporting a release build of a project to a platform, from the editor. Stores platform-specific metadata such as the name and supported features of the platform, and performs the exporting of projects, PCK files, and ZIP files. Uses an export template for the platform provided at the time of project exporting.
## Used in scripting by [EditorExportPlugin] to configure platform-specific customization of scenes and resources. See [method EditorExportPlugin._begin_customize_scenes] and [method EditorExportPlugin._begin_customize_resources] for more details.
##
## @tutorial(Console support in Godot): https://docs.godotengine.org/en/stable/tutorials/platform/consoles.html


## Invalid message type used as the default value when no type is specified.
## Message type for informational messages that have no effect on the export.
## Message type for warning messages that should be addressed but still allow to complete the export.
## Message type for error messages that must be addressed and fail the export.
## Flag is set if remotely debugged project is expected to use remote file system. If set, [method gen_export_flags] will add [code]--remote-fs[/code] and [code]--remote-fs-password[/code] (if password is set in the editor settings) command line arguments to the list.
## Flag is set if remote debug is enabled. If set, [method gen_export_flags] will add [code]--remote-debug[/code] and [code]--breakpoints[/code] (if breakpoints are selected in the script editor or added by the plugin) command line arguments to the list.
## Flag is set if remotely debugged project is running on the localhost. If set, [method gen_export_flags] will use [code]localhost[/code] instead of [member EditorSettings.network/debug/remote_host] as remote debugger host.
## Flag is set if "Visible Collision Shapes" remote debug option is enabled. If set, [method gen_export_flags] will add [code]--debug-collisions[/code] command line arguments to the list.
## Flag is set if Visible Navigation" remote debug option is enabled. If set, [method gen_export_flags] will add [code]--debug-navigation[/code] command line arguments to the list.

#enum ExportMessageType
enum {
    EXPORT_MESSAGE_NONE = 0,
    EXPORT_MESSAGE_INFO = 1,
    EXPORT_MESSAGE_WARNING = 2,
    EXPORT_MESSAGE_ERROR = 3,
}
#enum DebugFlags
enum {
    DEBUG_FLAG_DUMB_CLIENT = 1,
    DEBUG_FLAG_REMOTE_DEBUG = 2,
    DEBUG_FLAG_REMOTE_DEBUG_LOCALHOST = 4,
    DEBUG_FLAG_VIEW_COLLISIONS = 8,
    DEBUG_FLAG_VIEW_NAVIGATION = 16,
}


## Adds a message to the export log that will be displayed when exporting ends.
func add_message(type: int, category: String, message: String) -> void:
	pass;

## Clears the export log.
func clear_messages() -> void:
	pass;

## Create a new preset for this platform.
func create_preset() -> EditorExportPreset:
	pass;

## Creates a PCK archive at [param path] for the specified [param preset].
func export_pack(preset: EditorExportPreset, debug: bool, path: String, flags: int = 0) -> int:
	pass;

## Creates a patch PCK archive at [param path] for the specified [param preset], containing only the files that have changed since the last patch.
## [b]Note:[/b] [param patches] is an optional override of the set of patches defined in the export preset. When empty the patches defined in the export preset will be used instead.
func export_pack_patch(preset: EditorExportPreset, debug: bool, path: String, patches: PackedStringArray = PackedStringArray(), flags: int = 0) -> int:
	pass;

## Creates a full project at [param path] for the specified [param preset].
func export_project(preset: EditorExportPreset, debug: bool, path: String, flags: int = 0) -> int:
	pass;

## Exports project files for the specified preset. This method can be used to implement custom export format, other than PCK and ZIP. One of the callbacks is called for each exported file.
## [param save_cb] is called for all exported files and have the following arguments: [code]file_path: String[/code], [code]file_data: PackedByteArray[/code], [code]file_index: int[/code], [code]file_count: int[/code], [code]encryption_include_filters: PackedStringArray[/code], [code]encryption_exclude_filters: PackedStringArray[/code], [code]encryption_key: PackedByteArray[/code].
## [param shared_cb] is called for exported native shared/static libraries and have the following arguments: [code]file_path: String[/code], [code]tags: PackedStringArray[/code], [code]target_folder: String[/code].
## [b]Note:[/b] [code]file_index[/code] and [code]file_count[/code] are intended for progress tracking only and aren't necessarily unique and precise.
func export_project_files(preset: EditorExportPreset, debug: bool, save_cb: Callable, shared_cb: Callable = Callable()) -> int:
	pass;

## Create a ZIP archive at [param path] for the specified [param preset].
func export_zip(preset: EditorExportPreset, debug: bool, path: String, flags: int = 0) -> int:
	pass;

## Create a patch ZIP archive at [param path] for the specified [param preset], containing only the files that have changed since the last patch.
## [b]Note:[/b] [param patches] is an optional override of the set of patches defined in the export preset. When empty the patches defined in the export preset will be used instead.
func export_zip_patch(preset: EditorExportPreset, debug: bool, path: String, patches: PackedStringArray = PackedStringArray(), flags: int = 0) -> int:
	pass;

## Locates export template for the platform, and returns [Dictionary] with the following keys: [code]path: String[/code] and [code]error: String[/code]. This method is provided for convenience and custom export platforms aren't required to use it or keep export templates stored in the same way official templates are.
func find_export_template(template_file_name: String) -> Dictionary:
	pass;

## Generates array of command line arguments for the default export templates for the debug flags and editor settings.
func gen_export_flags(flags: int) -> PackedStringArray:
	pass;

## Returns array of [EditorExportPreset]s for this platform.
func get_current_presets() -> Array:
	pass;

## Returns array of core file names that always should be exported regardless of preset config.
static func get_forced_export_files(preset: EditorExportPreset) -> PackedStringArray:
	pass;

## Returns additional files that should always be exported regardless of preset configuration, and are not part of the project source. The returned [Dictionary] contains filename keys ([String]) and their corresponding raw data ([PackedByteArray]).
func get_internal_export_files(preset: EditorExportPreset, debug: bool) -> Dictionary:
	pass;

## Returns message category, for the message with [param index].
func get_message_category(index: int) -> String:
	pass;

## Returns number of messages in the export log.
func get_message_count() -> int:
	pass;

## Returns message text, for the message with [param index].
func get_message_text(index: int) -> String:
	pass;

## Returns message type, for the message with [param index].
func get_message_type(index: int) -> int:
	pass;

## Returns the name of the export operating system handled by this [EditorExportPlatform] class, as a friendly string. Possible return values are [code]Windows[/code], [code]Linux[/code], [code]macOS[/code], [code]Android[/code], [code]iOS[/code], and [code]Web[/code].
func get_os_name() -> String:
	pass;

## Returns most severe message type currently present in the export log.
func get_worst_message_type() -> int:
	pass;

## Saves PCK archive and returns [Dictionary] with the following keys: [code]result: Error[/code], [code]so_files: Array[/code] (array of the shared/static objects which contains dictionaries with the following keys: [code]path: String[/code], [code]tags: PackedStringArray[/code], and [code]target_folder: String[/code]).
## If [param embed] is [code]true[/code], PCK content is appended to the end of [param path] file and return [Dictionary] additionally include following keys: [code]embedded_start: int[/code] (embedded PCK offset) and [code]embedded_size: int[/code] (embedded PCK size).
func save_pack(preset: EditorExportPreset, debug: bool, path: String, embed: bool = false) -> Dictionary:
	pass;

## Saves patch PCK archive and returns [Dictionary] with the following keys: [code]result: Error[/code], [code]so_files: Array[/code] (array of the shared/static objects which contains dictionaries with the following keys: [code]path: String[/code], [code]tags: PackedStringArray[/code], and [code]target_folder: String[/code]).
func save_pack_patch(preset: EditorExportPreset, debug: bool, path: String) -> Dictionary:
	pass;

## Saves ZIP archive and returns [Dictionary] with the following keys: [code]result: Error[/code], [code]so_files: Array[/code] (array of the shared/static objects which contains dictionaries with the following keys: [code]path: String[/code], [code]tags: PackedStringArray[/code], and [code]target_folder: String[/code]).
func save_zip(preset: EditorExportPreset, debug: bool, path: String) -> Dictionary:
	pass;

## Saves patch ZIP archive and returns [Dictionary] with the following keys: [code]result: Error[/code], [code]so_files: Array[/code] (array of the shared/static objects which contains dictionaries with the following keys: [code]path: String[/code], [code]tags: PackedStringArray[/code], and [code]target_folder: String[/code]).
func save_zip_patch(preset: EditorExportPreset, debug: bool, path: String) -> Dictionary:
	pass;

## Uploads specified file over SCP protocol to the remote host.
func ssh_push_to_remote(host: String, port: String, scp_args: PackedStringArray, src_file: String, dst_file: String) -> int:
	pass;

## Executes specified command on the remote host via SSH protocol and returns command output in the [param output].
func ssh_run_on_remote(host: String, port: String, ssh_arg: PackedStringArray, cmd_args: String, output: Array = [], port_fwd: int = -1) -> int:
	pass;

## Executes specified command on the remote host via SSH protocol and returns process ID (on the remote host) without waiting for command to finish.
func ssh_run_on_remote_no_wait(host: String, port: String, ssh_args: PackedStringArray, cmd_args: String, port_fwd: int = -1) -> int:
	pass;


