extends Resource
class_name CompositorEffect

## This resource allows for creating a custom rendering effect.
##
## This resource defines a custom rendering effect that can be applied to [Viewport]s through the viewports' [Environment]. You can implement a callback that is called during rendering at a given stage of the rendering pipeline and allows you to insert additional passes. Note that this callback happens on the rendering thread. CompositorEffect is an abstract base class and must be extended to implement specific rendering logic.
##
## @tutorial(The Compositor): https://docs.godotengine.org/en/stable/tutorials/rendering/compositor.html


## The callback is called before our opaque rendering pass, but after depth prepass (if applicable).
## The callback is called after our opaque rendering pass, but before our sky is rendered.
## The callback is called after our sky is rendered, but before our back buffers are created (and if enabled, before subsurface scattering and/or screen space reflections).
## The callback is called before our transparent rendering pass, but after our sky is rendered and we've created our back buffers.
## The callback is called after our transparent rendering pass, but before any built-in post-processing effects and output to our render target.
## Represents the size of the [enum EffectCallbackType] enum.

#enum EffectCallbackType
enum {
    EFFECT_CALLBACK_TYPE_PRE_OPAQUE = 0,
    EFFECT_CALLBACK_TYPE_POST_OPAQUE = 1,
    EFFECT_CALLBACK_TYPE_POST_SKY = 2,
    EFFECT_CALLBACK_TYPE_PRE_TRANSPARENT = 3,
    EFFECT_CALLBACK_TYPE_POST_TRANSPARENT = 4,
    EFFECT_CALLBACK_TYPE_MAX = 5,
}
## If [code]true[/code] and MSAA is enabled, this will trigger a color buffer resolve before the effect is run.
## [b]Note:[/b] In [method _render_callback], to access the resolved buffer use:
## [codeblock]
## var render_scene_buffers = render_data.get_render_scene_buffers()
## var color_buffer = render_scene_buffers.get_texture("render_buffers", "color")
## [/codeblock]
var access_resolved_color: bool:
	get = get_access_resolved_color, set = set_access_resolved_color

## If [code]true[/code] and MSAA is enabled, this will trigger a depth buffer resolve before the effect is run.
## [b]Note:[/b] In [method _render_callback], to access the resolved buffer use:
## [codeblock]
## var render_scene_buffers = render_data.get_render_scene_buffers()
## var depth_buffer = render_scene_buffers.get_texture("render_buffers", "depth")
## [/codeblock]
var access_resolved_depth: bool:
	get = get_access_resolved_depth, set = set_access_resolved_depth

## The type of effect that is implemented, determines at what stage of rendering the callback is called.
var effect_callback_type: int:
	get = get_effect_callback_type, set = set_effect_callback_type

## If [code]true[/code] this rendering effect is applied to any viewport it is added to.
var enabled: bool:
	get = get_enabled, set = set_enabled

## If [code]true[/code] this triggers motion vectors being calculated during the opaque render state.
## [b]Note:[/b] In [method _render_callback], to access the motion vector buffer use:
## [codeblock]
## var render_scene_buffers = render_data.get_render_scene_buffers()
## var motion_buffer = render_scene_buffers.get_velocity_texture()
## [/codeblock]
var needs_motion_vectors: bool:
	get = get_needs_motion_vectors, set = set_needs_motion_vectors

## If [code]true[/code] this triggers normal and roughness data to be output during our depth pre-pass, only applicable for the Forward+ renderer.
## [b]Note:[/b] In [method _render_callback], to access the roughness buffer use:
## [codeblock]
## var render_scene_buffers = render_data.get_render_scene_buffers()
## var roughness_buffer = render_scene_buffers.get_texture("forward_clustered", "normal_roughness")
## [/codeblock]
## The raw normal and roughness buffer is stored in an optimized format, different than the one available in Spatial shaders. When sampling the buffer, a conversion function must be applied. Use this function, copied from [url=https://github.com/godotengine/godot/blob/da5f39889f155658cef7f7ec3cc1abb94e17d815/servers/rendering/renderer_rd/shaders/forward_clustered/scene_forward_clustered_inc.glsl#L334-L341]here[/url]:
## [codeblock]
## vec4 normal_roughness_compatibility(vec4 p_normal_roughness) {
## float roughness = p_normal_roughness.w;
## if (roughness > 0.5) {
## roughness = 1.0 - roughness;
## }
## roughness /= (127.0 / 255.0);
## return vec4(normalize(p_normal_roughness.xyz * 2.0 - 1.0) * 0.5 + 0.5, roughness);
## }
## [/codeblock]
var needs_normal_roughness: bool:
	get = get_needs_normal_roughness, set = set_needs_normal_roughness

## If [code]true[/code] this triggers specular data being rendered to a separate buffer and combined after effects have been applied, only applicable for the Forward+ renderer.
var needs_separate_specular: bool:
	get = get_needs_separate_specular, set = set_needs_separate_specular



## Implement this function with your custom rendering code. [param effect_callback_type] should always match the effect callback type you've specified in [member effect_callback_type]. [param render_data] provides access to the rendering state, it is only valid during rendering and should not be stored.
func _render_callback(effect_callback_type: int, render_data: RenderData) -> void:
	pass;


func get_access_resolved_color() -> bool:
	return access_resolved_color

func set_access_resolved_color(value: bool) -> void:
	access_resolved_color = value

func get_access_resolved_depth() -> bool:
	return access_resolved_depth

func set_access_resolved_depth(value: bool) -> void:
	access_resolved_depth = value

func get_effect_callback_type() -> int:
	return effect_callback_type

func set_effect_callback_type(value: int) -> void:
	effect_callback_type = value

func get_enabled() -> bool:
	return enabled

func set_enabled(value: bool) -> void:
	enabled = value

func get_needs_motion_vectors() -> bool:
	return needs_motion_vectors

func set_needs_motion_vectors(value: bool) -> void:
	needs_motion_vectors = value

func get_needs_normal_roughness() -> bool:
	return needs_normal_roughness

func set_needs_normal_roughness(value: bool) -> void:
	needs_normal_roughness = value

func get_needs_separate_specular() -> bool:
	return needs_separate_specular

func set_needs_separate_specular(value: bool) -> void:
	needs_separate_specular = value

