extends AudioStream
class_name AudioStreamInteractive

## Audio stream that can playback music interactively, combining clips and a transition table.
##
## This is an audio stream that can playback music interactively, combining clips and a transition table. Clips must be added first, and then the transition rules via the [method add_transition]. Additionally, this stream exports a property parameter to control the playback via [AudioStreamPlayer], [AudioStreamPlayer2D], or [AudioStreamPlayer3D].
## The way this is used is by filling a number of clips, then configuring the transition table. From there, clips are selected for playback and the music will smoothly go from the current to the new one while using the corresponding transition rule defined in the transition table.


## Start transition as soon as possible, don't wait for any specific time position.
## Transition when the clip playback position reaches the next beat.
## Transition when the clip playback position reaches the next bar.
## Transition when the current clip finished playing.
## Transition to the same position in the destination clip. This is useful when both clips have exactly the same length and the music should fade between them.
## Transition to the start of the destination clip.
## Do not use fade for the transition. This is useful when transitioning from a clip-end to clip-beginning, and each clip has their begin/end.
## Use a fade-in in the next clip, let the current clip finish.
## Use a fade-out in the current clip, the next clip will start by itself.
## Use a cross-fade between clips.
## Use automatic fade logic depending on the transition from/to. It is recommended to use this by default.
## Disable auto-advance (default).
## Enable auto-advance, a clip must be specified.
## Enable auto-advance, but instead of specifying a clip, the playback will return to hold (see [method add_transition]).
## This constant describes that any clip is valid for a specific transition as either source or destination.
const CLIP_ANY = -1;


#enum TransitionFromTime
enum {
    TRANSITION_FROM_TIME_IMMEDIATE = 0,
    TRANSITION_FROM_TIME_NEXT_BEAT = 1,
    TRANSITION_FROM_TIME_NEXT_BAR = 2,
    TRANSITION_FROM_TIME_END = 3,
}
#enum TransitionToTime
enum {
    TRANSITION_TO_TIME_SAME_POSITION = 0,
    TRANSITION_TO_TIME_START = 1,
}
#enum FadeMode
enum {
    FADE_DISABLED = 0,
    FADE_IN = 1,
    FADE_OUT = 2,
    FADE_CROSS = 3,
    FADE_AUTOMATIC = 4,
}
#enum AutoAdvanceMode
enum {
    AUTO_ADVANCE_DISABLED = 0,
    AUTO_ADVANCE_ENABLED = 1,
    AUTO_ADVANCE_RETURN_TO_HOLD = 2,
}
## Amount of clips contained in this interactive player.
var clip_count: int:
	get = get_clip_count, set = set_clip_count

## Index of the initial clip, which will be played first when this stream is played.
var initial_clip: int:
	get = get_initial_clip, set = set_initial_clip



## Add a transition between two clips. Provide the indices of the source and destination clips, or use the [constant CLIP_ANY] constant to indicate that transition happens to/from any clip to this one.
## * [param from_time] indicates the moment in the current clip the transition will begin after triggered.
## * [param to_time] indicates the time in the next clip that the playback will start from.
## * [param fade_mode] indicates how the fade will happen between clips. If unsure, just use [constant FADE_AUTOMATIC] which uses the most common type of fade for each situation.
## * [param fade_beats] indicates how many beats the fade will take. Using decimals is allowed.
## * [param use_filler_clip] indicates that there will be a filler clip used between the source and destination clips.
## * [param filler_clip] the index of the filler clip.
## * If [param hold_previous] is used, then this clip will be remembered. This can be used together with [constant AUTO_ADVANCE_RETURN_TO_HOLD] to return to this clip after another is done playing.
func add_transition(from_clip: int, to_clip: int, from_time: int, to_time: int, fade_mode: int, fade_beats: float, use_filler_clip: bool = false, filler_clip: int = -1, hold_previous: bool = false) -> void:
	pass;

## Erase a transition by providing [param from_clip] and [param to_clip] clip indices. [constant CLIP_ANY] can be used for either argument or both.
func erase_transition(from_clip: int, to_clip: int) -> void:
	pass;

## Return whether a clip has auto-advance enabled. See [method set_clip_auto_advance].
func get_clip_auto_advance(clip_index: int) -> int:
	pass;

## Return the clip towards which the clip referenced by [param clip_index] will auto-advance to.
func get_clip_auto_advance_next_clip(clip_index: int) -> int:
	pass;

## Return the name of a clip.
func get_clip_name(clip_index: int) -> StringName:
	pass;

## Return the [AudioStream] associated with a clip.
func get_clip_stream(clip_index: int) -> AudioStream:
	pass;

## Return the time (in beats) for a transition (see [method add_transition]).
func get_transition_fade_beats(from_clip: int, to_clip: int) -> float:
	pass;

## Return the mode for a transition (see [method add_transition]).
func get_transition_fade_mode(from_clip: int, to_clip: int) -> int:
	pass;

## Return the filler clip for a transition (see [method add_transition]).
func get_transition_filler_clip(from_clip: int, to_clip: int) -> int:
	pass;

## Return the source time position for a transition (see [method add_transition]).
func get_transition_from_time(from_clip: int, to_clip: int) -> int:
	pass;

## Return the list of transitions (from, to interleaved).
func get_transition_list() -> PackedInt32Array:
	pass;

## Return the destination time position for a transition (see [method add_transition]).
func get_transition_to_time(from_clip: int, to_clip: int) -> int:
	pass;

## Returns [code]true[/code] if a given transition exists (was added via [method add_transition]).
func has_transition(from_clip: int, to_clip: int) -> bool:
	pass;

## Return whether a transition uses the [i]hold previous[/i] functionality (see [method add_transition]).
func is_transition_holding_previous(from_clip: int, to_clip: int) -> bool:
	pass;

## Return whether a transition uses the [i]filler clip[/i] functionality (see [method add_transition]).
func is_transition_using_filler_clip(from_clip: int, to_clip: int) -> bool:
	pass;

## Set whether a clip will auto-advance by changing the auto-advance mode.
func set_clip_auto_advance(clip_index: int, mode: int) -> void:
	pass;

## Set the index of the next clip towards which this clip will auto advance to when finished. If the clip being played loops, then auto-advance will be ignored.
func set_clip_auto_advance_next_clip(clip_index: int, auto_advance_next_clip: int) -> void:
	pass;

## Set the name of the current clip (for easier identification).
func set_clip_name(clip_index: int, name: StringName) -> void:
	pass;

## Set the [AudioStream] associated with the current clip.
func set_clip_stream(clip_index: int, stream: AudioStream) -> void:
	pass;


func get_clip_count() -> int:
	return clip_count

func set_clip_count(value: int) -> void:
	clip_count = value

func get_initial_clip() -> int:
	return initial_clip

func set_initial_clip(value: int) -> void:
	initial_clip = value

