extends AnimationMixer
class_name AnimationPlayer

## A node used for animation playback.
##
## An animation player is used for general-purpose playback of animations. It contains a dictionary of [AnimationLibrary] resources and custom blend times between animation transitions.
## Some methods and properties use a single key to reference an animation directly. These keys are formatted as the key for the library, followed by a forward slash, then the key for the animation within the library, for example [code]"movement/run"[/code]. If the library's key is an empty string (known as the default library), the forward slash is omitted, being the same key used by the library.
## [AnimationPlayer] is better-suited than [Tween] for more complex animations, for example ones with non-trivial timings. It can also be used over [Tween] if the animation track editor is more convenient than doing it in code.
## Updating the target properties of animations occurs at the process frame.
##
## @tutorial(2D Sprite animation): https://docs.godotengine.org/en/stable/tutorials/2d/2d_sprite_animation.html
## @tutorial(Animation documentation index): https://docs.godotengine.org/en/stable/tutorials/animation/index.html
## @tutorial(Third Person Shooter (TPS) Demo): https://godotengine.org/asset-library/asset/2710


## Emitted when a queued animation plays after the previous animation finished. See also [method AnimationPlayer.queue].
## [b]Note:[/b] The signal is not emitted when the animation is changed via [method AnimationPlayer.play] or by an [AnimationTree].
signal animation_changed(old_name: StringName, new_name: StringName)
## Emitted when [member current_animation] changes.
signal current_animation_changed(name: String)

#enum AnimationProcessCallback
enum {
    ANIMATION_PROCESS_PHYSICS = 0,
    ANIMATION_PROCESS_IDLE = 1,
    ANIMATION_PROCESS_MANUAL = 2,
}
#enum AnimationMethodCallMode
enum {
    ANIMATION_METHOD_CALL_DEFERRED = 0,
    ANIMATION_METHOD_CALL_IMMEDIATE = 1,
}
## If playing, the current animation's key, otherwise, the animation last played. When set, this changes the animation, but will not play it unless already playing. See also [member current_animation].
var assigned_animation: String:
	get = get_assigned_animation, set = set_assigned_animation

## The key of the animation to play when the scene loads.
var autoplay: String:
	get = get_autoplay, set = set_autoplay

## The key of the currently playing animation. If no animation is playing, the property's value is an empty string. Changing this value does not restart the animation. See [method play] for more information on playing animations.
## [b]Note:[/b] While this property appears in the Inspector, it's not meant to be edited, and it's not saved in the scene. This property is mainly used to get the currently playing animation, and internally for animation playback tracks. For more information, see [Animation].
var current_animation: String:
	get = get_current_animation, set = set_current_animation

## The length (in seconds) of the currently playing animation.
var current_animation_length: float:
	get = get_current_animation_length

## The position (in seconds) of the currently playing animation.
var current_animation_position: float:
	get = get_current_animation_position

## If [code]true[/code] and the engine is running in Movie Maker mode (see [MovieWriter]), exits the engine with [method SceneTree.quit] as soon as an animation is done playing in this [AnimationPlayer]. A message is printed when the engine quits for this reason.
## [b]Note:[/b] This obeys the same logic as the [signal AnimationMixer.animation_finished] signal, so it will not quit the engine if the animation is set to be looping.
var movie_quit_on_finish: bool:
	get = is_movie_quit_on_finish_enabled, set = set_movie_quit_on_finish_enabled

## If [code]true[/code], performs [method AnimationMixer.capture] before playback automatically. This means just [method play_with_capture] is executed with default arguments instead of [method play].
## [b]Note:[/b] Capture interpolation is only performed if the animation contains a capture track. See also [constant Animation.UPDATE_CAPTURE].
var playback_auto_capture: bool:
	get = is_auto_capture, set = set_auto_capture

## See also [method play_with_capture] and [method AnimationMixer.capture].
## If [member playback_auto_capture_duration] is negative value, the duration is set to the interval between the current position and the first key.
var playback_auto_capture_duration: float:
	get = get_auto_capture_duration, set = set_auto_capture_duration

## The ease type of the capture interpolation. See also [enum Tween.EaseType].
var playback_auto_capture_ease_type: int:
	get = get_auto_capture_ease_type, set = set_auto_capture_ease_type

## The transition type of the capture interpolation. See also [enum Tween.TransitionType].
var playback_auto_capture_transition_type: int:
	get = get_auto_capture_transition_type, set = set_auto_capture_transition_type

## The default time in which to blend animations. Ranges from 0 to 4096 with 0.01 precision.
var playback_default_blend_time: float:
	get = get_default_blend_time, set = set_default_blend_time

## The speed scaling ratio. For example, if this value is [code]1[/code], then the animation plays at normal speed. If it's [code]0.5[/code], then it plays at half speed. If it's [code]2[/code], then it plays at double speed.
## If set to a negative value, the animation is played in reverse. If set to [code]0[/code], the animation will not advance.
var speed_scale: float:
	get = get_speed_scale, set = set_speed_scale



## Returns the key of the animation which is queued to play after the [param animation_from] animation.
func animation_get_next(animation_from: StringName) -> StringName:
	pass;

## Triggers the [param animation_to] animation when the [param animation_from] animation completes.
func animation_set_next(animation_from: StringName, animation_to: StringName) -> void:
	pass;

## Clears all queued, unplayed animations.
func clear_queue() -> void:
	pass;

## Returns the blend time (in seconds) between two animations, referenced by their keys.
func get_blend_time(animation_from: StringName, animation_to: StringName) -> float:
	pass;

## Returns the call mode used for "Call Method" tracks.
func get_method_call_mode() -> int:
	pass;

## Returns the actual playing speed of current animation or [code]0[/code] if not playing. This speed is the [member speed_scale] property multiplied by [code]custom_speed[/code] argument specified when calling the [method play] method.
## Returns a negative value if the current animation is playing backwards.
func get_playing_speed() -> float:
	pass;

## Returns the process notification in which to update animations.
func get_process_callback() -> int:
	pass;

## Returns a list of the animation keys that are currently queued to play.
func get_queue() -> PackedStringArray:
	pass;

## Returns the node which node path references will travel from.
func get_root() -> NodePath:
	pass;

## Returns the end time of the section currently being played.
func get_section_end_time() -> float:
	pass;

## Returns the start time of the section currently being played.
func get_section_start_time() -> float:
	pass;

## Returns [code]true[/code] if an animation is currently playing with a section.
func has_section() -> bool:
	pass;

## Returns [code]true[/code] if an animation is currently playing (even if [member speed_scale] and/or [code]custom_speed[/code] are [code]0[/code]).
func is_playing() -> bool:
	pass;

## Pauses the currently playing animation. The [member current_animation_position] will be kept and calling [method play] or [method play_backwards] without arguments or with the same animation name as [member assigned_animation] will resume the animation.
## See also [method stop].
func pause() -> void:
	pass;

## Plays the animation with key [param name]. Custom blend times and speed can be set.
## The [param from_end] option only affects when switching to a new animation track, or if the same track but at the start or end. It does not affect resuming playback that was paused in the middle of an animation. If [param custom_speed] is negative and [param from_end] is [code]true[/code], the animation will play backwards (which is equivalent to calling [method play_backwards]).
## The [AnimationPlayer] keeps track of its current or last played animation with [member assigned_animation]. If this method is called with that same animation [param name], or with no [param name] parameter, the assigned animation will resume playing if it was paused.
## [b]Note:[/b] The animation will be updated the next time the [AnimationPlayer] is processed. If other variables are updated at the same time this is called, they may be updated too early. To perform the update immediately, call [code]advance(0)[/code].
func play(name: StringName = &"", custom_blend: float = -1, custom_speed: float = 1.0, from_end: bool = false) -> void:
	pass;

## Plays the animation with key [param name] in reverse.
## This method is a shorthand for [method play] with [code]custom_speed = -1.0[/code] and [code]from_end = true[/code], so see its description for more information.
func play_backwards(name: StringName = &"", custom_blend: float = -1) -> void:
	pass;

## Plays the animation with key [param name] and the section starting from [param start_time] and ending on [param end_time]. See also [method play].
## Setting [param start_time] to a value outside the range of the animation means the start of the animation will be used instead, and setting [param end_time] to a value outside the range of the animation means the end of the animation will be used instead. [param start_time] cannot be equal to [param end_time].
func play_section(name: StringName = &"", start_time: float = -1, end_time: float = -1, custom_blend: float = -1, custom_speed: float = 1.0, from_end: bool = false) -> void:
	pass;

## Plays the animation with key [param name] and the section starting from [param start_time] and ending on [param end_time] in reverse.
## This method is a shorthand for [method play_section] with [code]custom_speed = -1.0[/code] and [code]from_end = true[/code], see its description for more information.
func play_section_backwards(name: StringName = &"", start_time: float = -1, end_time: float = -1, custom_blend: float = -1) -> void:
	pass;

## Plays the animation with key [param name] and the section starting from [param start_marker] and ending on [param end_marker].
## If the start marker is empty, the section starts from the beginning of the animation. If the end marker is empty, the section ends on the end of the animation. See also [method play].
func play_section_with_markers(name: StringName = &"", start_marker: StringName = &"", end_marker: StringName = &"", custom_blend: float = -1, custom_speed: float = 1.0, from_end: bool = false) -> void:
	pass;

## Plays the animation with key [param name] and the section starting from [param start_marker] and ending on [param end_marker] in reverse.
## This method is a shorthand for [method play_section_with_markers] with [code]custom_speed = -1.0[/code] and [code]from_end = true[/code], see its description for more information.
func play_section_with_markers_backwards(name: StringName = &"", start_marker: StringName = &"", end_marker: StringName = &"", custom_blend: float = -1) -> void:
	pass;

## See also [method AnimationMixer.capture].
## You can use this method to use more detailed options for capture than those performed by [member playback_auto_capture]. When [member playback_auto_capture] is [code]false[/code], this method is almost the same as the following:
## [codeblock]
## capture(name, duration, trans_type, ease_type)
## play(name, custom_blend, custom_speed, from_end)
## [/codeblock]
## If [param name] is blank, it specifies [member assigned_animation].
## If [param duration] is a negative value, the duration is set to the interval between the current position and the first key, when [param from_end] is [code]true[/code], uses the interval between the current position and the last key instead.
## [b]Note:[/b] The [param duration] takes [member speed_scale] into account, but [param custom_speed] does not, because the capture cache is interpolated with the blend result and the result may contain multiple animations.
func play_with_capture(name: StringName = &"", duration: float = -1.0, custom_blend: float = -1, custom_speed: float = 1.0, from_end: bool = false, trans_type: int = 0, ease_type: int = 0) -> void:
	pass;

## Queues an animation for playback once the current animation and all previously queued animations are done.
## [b]Note:[/b] If a looped animation is currently playing, the queued animation will never play unless the looped animation is stopped somehow.
func queue(name: StringName) -> void:
	pass;

## Resets the current section. Does nothing if a section has not been set.
func reset_section() -> void:
	pass;

## Seeks the animation to the [param seconds] point in time (in seconds). If [param update] is [code]true[/code], the animation updates too, otherwise it updates at process time. Events between the current frame and [param seconds] are skipped.
## If [param update_only] is [code]true[/code], the method / audio / animation playback tracks will not be processed.
## [b]Note:[/b] Seeking to the end of the animation doesn't emit [signal AnimationMixer.animation_finished]. If you want to skip animation and emit the signal, use [method AnimationMixer.advance].
func seek(seconds: float, update: bool = false, update_only: bool = false) -> void:
	pass;

## Specifies a blend time (in seconds) between two animations, referenced by their keys.
func set_blend_time(animation_from: StringName, animation_to: StringName, sec: float) -> void:
	pass;

## Sets the call mode used for "Call Method" tracks.
func set_method_call_mode(mode: int) -> void:
	pass;

## Sets the process notification in which to update animations.
func set_process_callback(mode: int) -> void:
	pass;

## Sets the node which node path references will travel from.
func set_root(path: NodePath) -> void:
	pass;

## Changes the start and end times of the section being played. The current playback position will be clamped within the new section. See also [method play_section].
func set_section(start_time: float = -1, end_time: float = -1) -> void:
	pass;

## Changes the start and end markers of the section being played. The current playback position will be clamped within the new section. See also [method play_section_with_markers].
## If the argument is empty, the section uses the beginning or end of the animation. If both are empty, it means that the section is not set.
func set_section_with_markers(start_marker: StringName = &"", end_marker: StringName = &"") -> void:
	pass;

## Stops the currently playing animation. The animation position is reset to [code]0[/code] and the [code]custom_speed[/code] is reset to [code]1.0[/code]. See also [method pause].
## If [param keep_state] is [code]true[/code], the animation state is not updated visually.
## [b]Note:[/b] The method / audio / animation playback tracks will not be processed by this method.
func stop(keep_state: bool = false) -> void:
	pass;


func get_assigned_animation() -> String:
	return assigned_animation

func set_assigned_animation(value: String) -> void:
	assigned_animation = value

func get_autoplay() -> String:
	return autoplay

func set_autoplay(value: String) -> void:
	autoplay = value

func get_current_animation() -> String:
	return current_animation

func set_current_animation(value: String) -> void:
	current_animation = value

func get_current_animation_length() -> float:
	return current_animation_length

func get_current_animation_position() -> float:
	return current_animation_position

func is_movie_quit_on_finish_enabled() -> bool:
	return movie_quit_on_finish

func set_movie_quit_on_finish_enabled(value: bool) -> void:
	movie_quit_on_finish = value

func is_auto_capture() -> bool:
	return playback_auto_capture

func set_auto_capture(value: bool) -> void:
	playback_auto_capture = value

func get_auto_capture_duration() -> float:
	return playback_auto_capture_duration

func set_auto_capture_duration(value: float) -> void:
	playback_auto_capture_duration = value

func get_auto_capture_ease_type() -> int:
	return playback_auto_capture_ease_type

func set_auto_capture_ease_type(value: int) -> void:
	playback_auto_capture_ease_type = value

func get_auto_capture_transition_type() -> int:
	return playback_auto_capture_transition_type

func set_auto_capture_transition_type(value: int) -> void:
	playback_auto_capture_transition_type = value

func get_default_blend_time() -> float:
	return playback_default_blend_time

func set_default_blend_time(value: float) -> void:
	playback_default_blend_time = value

func get_speed_scale() -> float:
	return speed_scale

func set_speed_scale(value: float) -> void:
	speed_scale = value

