extends AnimationRootNode
class_name AnimationNodeStateMachine

## A state machine with multiple [AnimationRootNode]s, used by [AnimationTree].
##
## Contains multiple [AnimationRootNode]s representing animation states, connected in a graph. State transitions can be configured to happen automatically or via code, using a shortest-path algorithm. Retrieve the [AnimationNodeStateMachinePlayback] object from the [AnimationTree] node to control it programmatically.
## [codeblocks]
## [gdscript]
## var state_machine = $AnimationTree.get("parameters/playback")
## state_machine.travel("some_state")
## [/gdscript]
## [csharp]
## var stateMachine = GetNode<AnimationTree>("AnimationTree").Get("parameters/playback") as AnimationNodeStateMachinePlayback;
## stateMachine.Travel("some_state");
## [/csharp]
## [/codeblocks]
##
## @tutorial(Using AnimationTree): https://docs.godotengine.org/en/stable/tutorials/animation/animation_tree.html


## Seeking to the beginning is treated as playing from the start state. Transition to the end state is treated as exiting the state machine.
## Seeking to the beginning is treated as seeking to the beginning of the animation in the current state. Transition to the end state, or the absence of transitions in each state, is treated as exiting the state machine.
## This is a grouped state machine that can be controlled from a parent state machine. It does not work independently. There must be a state machine with [member state_machine_type] of [constant STATE_MACHINE_TYPE_ROOT] or [constant STATE_MACHINE_TYPE_NESTED] in the parent or ancestor.

#enum StateMachineType
enum {
    STATE_MACHINE_TYPE_ROOT = 0,
    STATE_MACHINE_TYPE_NESTED = 1,
    STATE_MACHINE_TYPE_GROUPED = 2,
}
## If [code]true[/code], allows teleport to the self state with [method AnimationNodeStateMachinePlayback.travel]. When the reset option is enabled in [method AnimationNodeStateMachinePlayback.travel], the animation is restarted. If [code]false[/code], nothing happens on the teleportation to the self state.
var allow_transition_to_self: bool:
	get = is_allow_transition_to_self, set = set_allow_transition_to_self

## If [code]true[/code], treat the cross-fade to the start and end nodes as a blend with the RESET animation.
## In most cases, when additional cross-fades are performed in the parent [AnimationNode] of the state machine, setting this property to [code]false[/code] and matching the cross-fade time of the parent [AnimationNode] and the state machine's start node and end node gives good results.
var reset_ends: bool:
	get = are_ends_reset, set = set_reset_ends

## This property can define the process of transitions for different use cases. See also [enum AnimationNodeStateMachine.StateMachineType].
var state_machine_type: int:
	get = get_state_machine_type, set = set_state_machine_type



## Adds a new animation node to the graph. The [param position] is used for display in the editor.
func add_node(name: StringName, node: AnimationNode, position: Vector2 = Vector2(0, 0)) -> void:
	pass;

## Adds a transition between the given animation nodes.
func add_transition(from: StringName, to: StringName, transition: AnimationNodeStateMachineTransition) -> void:
	pass;

## Returns the draw offset of the graph. Used for display in the editor.
func get_graph_offset() -> Vector2:
	pass;

## Returns the animation node with the given name.
func get_node(name: StringName) -> AnimationNode:
	pass;

## Returns a list containing the names of all animation nodes in this state machine.
func get_node_list() -> Array[StringName]:
	pass;

## Returns the given animation node's name.
func get_node_name(node: AnimationNode) -> StringName:
	pass;

## Returns the given animation node's coordinates. Used for display in the editor.
func get_node_position(name: StringName) -> Vector2:
	pass;

## Returns the given transition.
func get_transition(idx: int) -> AnimationNodeStateMachineTransition:
	pass;

## Returns the number of connections in the graph.
func get_transition_count() -> int:
	pass;

## Returns the given transition's start node.
func get_transition_from(idx: int) -> StringName:
	pass;

## Returns the given transition's end node.
func get_transition_to(idx: int) -> StringName:
	pass;

## Returns [code]true[/code] if the graph contains the given animation node.
func has_node(name: StringName) -> bool:
	pass;

## Returns [code]true[/code] if there is a transition between the given animation nodes.
func has_transition(from: StringName, to: StringName) -> bool:
	pass;

## Deletes the given animation node from the graph.
func remove_node(name: StringName) -> void:
	pass;

## Deletes the transition between the two specified animation nodes.
func remove_transition(from: StringName, to: StringName) -> void:
	pass;

## Deletes the given transition by index.
func remove_transition_by_index(idx: int) -> void:
	pass;

## Renames the given animation node.
func rename_node(name: StringName, new_name: StringName) -> void:
	pass;

## Replaces the given animation node with a new animation node.
func replace_node(name: StringName, node: AnimationNode) -> void:
	pass;

## Sets the draw offset of the graph. Used for display in the editor.
func set_graph_offset(offset: Vector2) -> void:
	pass;

## Sets the animation node's coordinates. Used for display in the editor.
func set_node_position(name: StringName, position: Vector2) -> void:
	pass;


func is_allow_transition_to_self() -> bool:
	return allow_transition_to_self

func set_allow_transition_to_self(value: bool) -> void:
	allow_transition_to_self = value

func are_ends_reset() -> bool:
	return reset_ends

func set_reset_ends(value: bool) -> void:
	reset_ends = value

func get_state_machine_type() -> int:
	return state_machine_type

func set_state_machine_type(value: int) -> void:
	state_machine_type = value

