extends AnimationNodeSync
class_name AnimationNodeOneShot

## Plays an animation once in an [AnimationNodeBlendTree].
##
## A resource to add to an [AnimationNodeBlendTree]. This animation node will execute a sub-animation and return once it finishes. Blend times for fading in and out can be customized, as well as filters.
## After setting the request and changing the animation playback, the one-shot node automatically clears the request on the next process frame by setting its [code]request[/code] value to [constant ONE_SHOT_REQUEST_NONE].
## [codeblocks]
## [gdscript]
## # Play child animation connected to "shot" port.
## animation_tree.set("parameters/OneShot/request", AnimationNodeOneShot.ONE_SHOT_REQUEST_FIRE)
## # Alternative syntax (same result as above).
## animation_tree["parameters/OneShot/request"] = AnimationNodeOneShot.ONE_SHOT_REQUEST_FIRE
## # Abort child animation connected to "shot" port.
## animation_tree.set("parameters/OneShot/request", AnimationNodeOneShot.ONE_SHOT_REQUEST_ABORT)
## # Alternative syntax (same result as above).
## animation_tree["parameters/OneShot/request"] = AnimationNodeOneShot.ONE_SHOT_REQUEST_ABORT
## # Abort child animation with fading out connected to "shot" port.
## animation_tree.set("parameters/OneShot/request", AnimationNodeOneShot.ONE_SHOT_REQUEST_FADE_OUT)
## # Alternative syntax (same result as above).
## animation_tree["parameters/OneShot/request"] = AnimationNodeOneShot.ONE_SHOT_REQUEST_FADE_OUT
## # Get current state (read-only).
## animation_tree.get("parameters/OneShot/active")
## # Alternative syntax (same result as above).
## animation_tree["parameters/OneShot/active"]
## # Get current internal state (read-only).
## animation_tree.get("parameters/OneShot/internal_active")
## # Alternative syntax (same result as above).
## animation_tree["parameters/OneShot/internal_active"]
## [/gdscript]
## [csharp]
## // Play child animation connected to "shot" port.
## animationTree.Set("parameters/OneShot/request", (int)AnimationNodeOneShot.OneShotRequest.Fire);
## // Abort child animation connected to "shot" port.
## animationTree.Set("parameters/OneShot/request", (int)AnimationNodeOneShot.OneShotRequest.Abort);
## // Abort child animation with fading out connected to "shot" port.
## animationTree.Set("parameters/OneShot/request", (int)AnimationNodeOneShot.OneShotRequest.FadeOut);
## // Get current state (read-only).
## animationTree.Get("parameters/OneShot/active");
## // Get current internal state (read-only).
## animationTree.Get("parameters/OneShot/internal_active");
## [/csharp]
## [/codeblocks]
##
## @tutorial(Using AnimationTree): https://docs.godotengine.org/en/stable/tutorials/animation/animation_tree.html
## @tutorial(Third Person Shooter (TPS) Demo): https://godotengine.org/asset-library/asset/2710


## The default state of the request. Nothing is done.
## The request to play the animation connected to "shot" port.
## The request to stop the animation connected to "shot" port.
## The request to fade out the animation connected to "shot" port.
## Blends two animations. See also [AnimationNodeBlend2].
## Blends two animations additively. See also [AnimationNodeAdd2].

#enum OneShotRequest
enum {
    ONE_SHOT_REQUEST_NONE = 0,
    ONE_SHOT_REQUEST_FIRE = 1,
    ONE_SHOT_REQUEST_ABORT = 2,
    ONE_SHOT_REQUEST_FADE_OUT = 3,
}
#enum MixMode
enum {
    MIX_MODE_BLEND = 0,
    MIX_MODE_ADD = 1,
}
## If [code]true[/code], the sub-animation will restart automatically after finishing.
## In other words, to start auto restarting, the animation must be played once with the [constant ONE_SHOT_REQUEST_FIRE] request. The [constant ONE_SHOT_REQUEST_ABORT] request stops the auto restarting, but it does not disable the [member autorestart] itself. So, the [constant ONE_SHOT_REQUEST_FIRE] request will start auto restarting again.
var autorestart: bool:
	get = has_autorestart, set = set_autorestart

## The delay after which the automatic restart is triggered, in seconds.
var autorestart_delay: float:
	get = get_autorestart_delay, set = set_autorestart_delay

## If [member autorestart] is [code]true[/code], a random additional delay (in seconds) between 0 and this value will be added to [member autorestart_delay].
var autorestart_random_delay: float:
	get = get_autorestart_random_delay, set = set_autorestart_random_delay

## If [code]true[/code], breaks the loop at the end of the loop cycle for transition, even if the animation is looping.
var break_loop_at_end: bool:
	get = is_loop_broken_at_end, set = set_break_loop_at_end

## Determines how cross-fading between animations is eased. If empty, the transition will be linear. Should be a unit [Curve].
var fadein_curve: Curve:
	get = get_fadein_curve, set = set_fadein_curve

## The fade-in duration. For example, setting this to [code]1.0[/code] for a 5 second length animation will produce a cross-fade that starts at 0 second and ends at 1 second during the animation.
## [b]Note:[/b] [AnimationNodeOneShot] transitions the current state after the fading has finished.
var fadein_time: float:
	get = get_fadein_time, set = set_fadein_time

## Determines how cross-fading between animations is eased. If empty, the transition will be linear. Should be a unit [Curve].
var fadeout_curve: Curve:
	get = get_fadeout_curve, set = set_fadeout_curve

## The fade-out duration. For example, setting this to [code]1.0[/code] for a 5 second length animation will produce a cross-fade that starts at 4 second and ends at 5 second during the animation.
## [b]Note:[/b] [AnimationNodeOneShot] transitions the current state after the fading has finished.
var fadeout_time: float:
	get = get_fadeout_time, set = set_fadeout_time

## The blend type.
var mix_mode: int:
	get = get_mix_mode, set = set_mix_mode




func has_autorestart() -> bool:
	return autorestart

func set_autorestart(value: bool) -> void:
	autorestart = value

func get_autorestart_delay() -> float:
	return autorestart_delay

func set_autorestart_delay(value: float) -> void:
	autorestart_delay = value

func get_autorestart_random_delay() -> float:
	return autorestart_random_delay

func set_autorestart_random_delay(value: float) -> void:
	autorestart_random_delay = value

func is_loop_broken_at_end() -> bool:
	return break_loop_at_end

func set_break_loop_at_end(value: bool) -> void:
	break_loop_at_end = value

func get_fadein_curve() -> Curve:
	return fadein_curve

func set_fadein_curve(value: Curve) -> void:
	fadein_curve = value

func get_fadein_time() -> float:
	return fadein_time

func set_fadein_time(value: float) -> void:
	fadein_time = value

func get_fadeout_curve() -> Curve:
	return fadeout_curve

func set_fadeout_curve(value: Curve) -> void:
	fadeout_curve = value

func get_fadeout_time() -> float:
	return fadeout_time

func set_fadeout_time(value: float) -> void:
	fadeout_time = value

func get_mix_mode() -> int:
	return mix_mode

func set_mix_mode(value: int) -> void:
	mix_mode = value

