extends SpriteBase3D
class_name AnimatedSprite3D

## 2D sprite node in 3D world, that can use multiple 2D textures for animation.
##
## [AnimatedSprite3D] is similar to the [Sprite3D] node, except it carries multiple textures as animation [member sprite_frames]. Animations are created using a [SpriteFrames] resource, which allows you to import image files (or a folder containing said files) to provide the animation frames for the sprite. The [SpriteFrames] resource can be configured in the editor via the SpriteFrames bottom panel.
##
## @tutorial(2D Sprite animation (also applies to 3D)): https://docs.godotengine.org/en/stable/tutorials/2d/2d_sprite_animation.html


## Emitted when [member animation] changes.
signal animation_changed
## Emitted when the animation reaches the end, or the start if it is played in reverse. When the animation finishes, it pauses the playback.
## [b]Note:[/b] This signal is not emitted if an animation is looping.
signal animation_finished
## Emitted when the animation loops.
signal animation_looped
## Emitted when [member frame] changes.
signal frame_changed
## Emitted when [member sprite_frames] changes.
signal sprite_frames_changed

## The current animation from the [member sprite_frames] resource. If this value is changed, the [member frame] counter and the [member frame_progress] are reset.
var animation: StringName:
	get = get_animation, set = set_animation

## The key of the animation to play when the scene loads.
var autoplay: String:
	get = get_autoplay, set = set_autoplay

## The displayed animation frame's index. Setting this property also resets [member frame_progress]. If this is not desired, use [method set_frame_and_progress].
var frame: int:
	get = get_frame, set = set_frame

## The progress value between [code]0.0[/code] and [code]1.0[/code] until the current frame transitions to the next frame. If the animation is playing backwards, the value transitions from [code]1.0[/code] to [code]0.0[/code].
var frame_progress: float:
	get = get_frame_progress, set = set_frame_progress

## The speed scaling ratio. For example, if this value is [code]1[/code], then the animation plays at normal speed. If it's [code]0.5[/code], then it plays at half speed. If it's [code]2[/code], then it plays at double speed.
## If set to a negative value, the animation is played in reverse. If set to [code]0[/code], the animation will not advance.
var speed_scale: float:
	get = get_speed_scale, set = set_speed_scale

## The [SpriteFrames] resource containing the animation(s). Allows you the option to load, edit, clear, make unique and save the states of the [SpriteFrames] resource.
var sprite_frames: SpriteFrames:
	get = get_sprite_frames, set = set_sprite_frames



## Returns the actual playing speed of current animation or [code]0[/code] if not playing. This speed is the [member speed_scale] property multiplied by [code]custom_speed[/code] argument specified when calling the [method play] method.
## Returns a negative value if the current animation is playing backwards.
func get_playing_speed() -> float:
	pass;

## Returns [code]true[/code] if an animation is currently playing (even if [member speed_scale] and/or [code]custom_speed[/code] are [code]0[/code]).
func is_playing() -> bool:
	pass;

## Pauses the currently playing animation. The [member frame] and [member frame_progress] will be kept and calling [method play] or [method play_backwards] without arguments will resume the animation from the current playback position.
## See also [method stop].
func pause() -> void:
	pass;

## Plays the animation with key [param name]. If [param custom_speed] is negative and [param from_end] is [code]true[/code], the animation will play backwards (which is equivalent to calling [method play_backwards]).
## If this method is called with that same animation [param name], or with no [param name] parameter, the assigned animation will resume playing if it was paused.
func play(name: StringName = &"", custom_speed: float = 1.0, from_end: bool = false) -> void:
	pass;

## Plays the animation with key [param name] in reverse.
## This method is a shorthand for [method play] with [code]custom_speed = -1.0[/code] and [code]from_end = true[/code], so see its description for more information.
func play_backwards(name: StringName = &"") -> void:
	pass;

## Sets [member frame] and [member frame_progress] to the given values. Unlike setting [member frame], this method does not reset the [member frame_progress] to [code]0.0[/code] implicitly.
## [b]Example:[/b] Change the animation while keeping the same [member frame] and [member frame_progress]:
## [codeblocks]
## [gdscript]
## var current_frame = animated_sprite.get_frame()
## var current_progress = animated_sprite.get_frame_progress()
## animated_sprite.play("walk_another_skin")
## animated_sprite.set_frame_and_progress(current_frame, current_progress)
## [/gdscript]
## [/codeblocks]
func set_frame_and_progress(frame: int, progress: float) -> void:
	pass;

## Stops the currently playing animation. The animation position is reset to [code]0[/code] and the [code]custom_speed[/code] is reset to [code]1.0[/code]. See also [method pause].
func stop() -> void:
	pass;


func get_animation() -> StringName:
	return animation

func set_animation(value: StringName) -> void:
	animation = value

func get_autoplay() -> String:
	return autoplay

func set_autoplay(value: String) -> void:
	autoplay = value

func get_frame() -> int:
	return frame

func set_frame(value: int) -> void:
	frame = value

func get_frame_progress() -> float:
	return frame_progress

func set_frame_progress(value: float) -> void:
	frame_progress = value

func get_speed_scale() -> float:
	return speed_scale

func set_speed_scale(value: float) -> void:
	speed_scale = value

func get_sprite_frames() -> SpriteFrames:
	return sprite_frames

func set_sprite_frames(value: SpriteFrames) -> void:
	sprite_frames = value

