extends Node
class_name Timer

## A countdown timer.
##
## The [Timer] node is a countdown timer and is the simplest way to handle time-based logic in the engine. When a timer reaches the end of its [member wait_time], it will emit the [signal timeout] signal.
## After a timer enters the tree, it can be manually started with [method start]. A timer node is also started automatically if [member autostart] is [code]true[/code].
## Without requiring much code, a timer node can be added and configured in the editor. The [signal timeout] signal it emits can also be connected through the Node dock in the editor:
## [codeblock]
## func _on_timer_timeout():
## print("Time to attack!")
## [/codeblock]
## [b]Note:[/b] To create a one-shot timer without instantiating a node, use [method SceneTree.create_timer].
## [b]Note:[/b] Timers are affected by [member Engine.time_scale]. The higher the time scale, the sooner timers will end. How often a timer processes may depend on the framerate or [member Engine.physics_ticks_per_second].
##
## @tutorial(2D Dodge The Creeps Demo): https://godotengine.org/asset-library/asset/2712


## Emitted when the timer reaches the end.
signal timeout
## Update the timer every physics process frame (see [constant Node.NOTIFICATION_INTERNAL_PHYSICS_PROCESS]).
## Update the timer every process (rendered) frame (see [constant Node.NOTIFICATION_INTERNAL_PROCESS]).

#enum TimerProcessCallback
enum {
    TIMER_PROCESS_PHYSICS = 0,
    TIMER_PROCESS_IDLE = 1,
}
## If [code]true[/code], the timer will start immediately when it enters the scene tree.
## [b]Note:[/b] After the timer enters the tree, this property is automatically set to [code]false[/code].
## [b]Note:[/b] This property does nothing when the timer is running in the editor.
var autostart: bool:
	get = has_autostart, set = set_autostart

## If [code]true[/code], the timer will ignore [member Engine.time_scale] and update with the real, elapsed time.
var ignore_time_scale: bool:
	get = is_ignoring_time_scale, set = set_ignore_time_scale

## If [code]true[/code], the timer will stop after reaching the end. Otherwise, as by default, the timer will automatically restart.
var one_shot: bool:
	get = is_one_shot, set = set_one_shot

## If [code]true[/code], the timer is paused. A paused timer does not process until this property is set back to [code]false[/code], even when [method start] is called.
var paused: bool:
	get = is_paused, set = set_paused

## Specifies when the timer is updated during the main loop (see [enum TimerProcessCallback]).
var process_callback: int:
	get = get_timer_process_callback, set = set_timer_process_callback

## The timer's remaining time in seconds. This is always [code]0[/code] if the timer is stopped.
## [b]Note:[/b] This property is read-only and cannot be modified. It is based on [member wait_time].
var time_left: float:
	get = get_time_left

## The time required for the timer to end, in seconds. This property can also be set every time [method start] is called.
## [b]Note:[/b] Timers can only process once per physics or process frame (depending on the [member process_callback]). An unstable framerate may cause the timer to end inconsistently, which is especially noticeable if the wait time is lower than roughly [code]0.05[/code] seconds. For very short timers, it is recommended to write your own code instead of using a [Timer] node. Timers are also affected by [member Engine.time_scale].
var wait_time: float:
	get = get_wait_time, set = set_wait_time



## Returns [code]true[/code] if the timer is stopped or has not started.
func is_stopped() -> bool:
	pass;

## Starts the timer, or resets the timer if it was started already. Fails if the timer is not inside the tree. If [param time_sec] is greater than [code]0[/code], this value is used for the [member wait_time].
## [b]Note:[/b] This method does not resume a paused timer. See [member paused].
func start(time_sec: float = -1) -> void:
	pass;

## Stops the timer.
func stop() -> void:
	pass;


func has_autostart() -> bool:
	return autostart

func set_autostart(value: bool) -> void:
	autostart = value

func is_ignoring_time_scale() -> bool:
	return ignore_time_scale

func set_ignore_time_scale(value: bool) -> void:
	ignore_time_scale = value

func is_one_shot() -> bool:
	return one_shot

func set_one_shot(value: bool) -> void:
	one_shot = value

func is_paused() -> bool:
	return paused

func set_paused(value: bool) -> void:
	paused = value

func get_timer_process_callback() -> int:
	return process_callback

func set_timer_process_callback(value: int) -> void:
	process_callback = value

func get_time_left() -> float:
	return time_left

func get_wait_time() -> float:
	return wait_time

func set_wait_time(value: float) -> void:
	wait_time = value

