extends Node2D
class_name TileMapLayer

## Node for 2D tile-based maps.
##
## Node for 2D tile-based maps. A [TileMapLayer] uses a [TileSet] which contain a list of tiles which are used to create grid-based maps. Unlike the [TileMap] node, which is deprecated, [TileMapLayer] has only one layer of tiles. You can use several [TileMapLayer] to achieve the same result as a [TileMap] node.
## For performance reasons, all TileMap updates are batched at the end of a frame. Notably, this means that scene tiles from a [TileSetScenesCollectionSource] may be initialized after their parent. This is only queued when inside the scene tree.
## To force an update earlier on, call [method update_internals].
## [b]Note:[/b] For performance and compatibility reasons, the coordinates serialized by [TileMapLayer] are limited to 16-bit signed integers, i.e. the range for X and Y coordinates is from [code]-32768[/code] to [code]32767[/code]. When saving tile data, tiles outside this range are wrapped.
##
## @tutorial(Using Tilemaps): https://docs.godotengine.org/en/stable/tutorials/2d/using_tilemaps.html
## @tutorial(2D Platformer Demo): https://godotengine.org/asset-library/asset/2727
## @tutorial(2D Isometric Demo): https://godotengine.org/asset-library/asset/2718
## @tutorial(2D Hexagonal Demo): https://godotengine.org/asset-library/asset/2717
## @tutorial(2D Grid-based Navigation with AStarGrid2D Demo): https://godotengine.org/asset-library/asset/2723
## @tutorial(2D Role Playing Game (RPG) Demo): https://godotengine.org/asset-library/asset/2729
## @tutorial(2D Kinematic Character Demo): https://godotengine.org/asset-library/asset/2719
## @tutorial(2D Dynamic TileMap Layers Demo): https://godotengine.org/asset-library/asset/2713


## Emitted when this [TileMapLayer]'s properties changes. This includes modified cells, properties, or changes made to its assigned [TileSet].
## [b]Note:[/b] This signal may be emitted very often when batch-modifying a [TileMapLayer]. Avoid executing complex processing in a connected function, and consider delaying it to the end of the frame instead (i.e. calling [method Object.call_deferred]).
signal changed
## Hide the collisions or navigation debug shapes in the editor, and use the debug settings to determine their visibility in game (i.e. [member SceneTree.debug_collisions_hint] or [member SceneTree.debug_navigation_hint]).
## Always hide the collisions or navigation debug shapes.
## Always show the collisions or navigation debug shapes.

#enum DebugVisibilityMode
enum {
    DEBUG_VISIBILITY_MODE_DEFAULT = 0,
    DEBUG_VISIBILITY_MODE_FORCE_HIDE = 2,
    DEBUG_VISIBILITY_MODE_FORCE_SHOW = 1,
}
## Enable or disable collisions.
var collision_enabled: bool:
	get = is_collision_enabled, set = set_collision_enabled

## Show or hide the [TileMapLayer]'s collision shapes. If set to [constant DEBUG_VISIBILITY_MODE_DEFAULT], this depends on the show collision debug settings.
var collision_visibility_mode: int:
	get = get_collision_visibility_mode, set = set_collision_visibility_mode

## If [code]false[/code], disables this [TileMapLayer] completely (rendering, collision, navigation, scene tiles, etc.)
var enabled: bool:
	get = is_enabled, set = set_enabled

## If [code]true[/code], navigation regions are enabled.
var navigation_enabled: bool:
	get = is_navigation_enabled, set = set_navigation_enabled

## Show or hide the [TileMapLayer]'s navigation meshes. If set to [constant DEBUG_VISIBILITY_MODE_DEFAULT], this depends on the show navigation debug settings.
var navigation_visibility_mode: int:
	get = get_navigation_visibility_mode, set = set_navigation_visibility_mode

## Enable or disable light occlusion.
var occlusion_enabled: bool:
	get = is_occlusion_enabled, set = set_occlusion_enabled

## The [TileMapLayer]'s quadrant size. A quadrant is a group of tiles to be drawn together on a single canvas item, for optimization purposes. [member rendering_quadrant_size] defines the length of a square's side, in the map's coordinate system, that forms the quadrant. Thus, the default quadrant size groups together [code]16 * 16 = 256[/code] tiles.
## The quadrant size does not apply on a Y-sorted [TileMapLayer], as tiles are grouped by Y position instead in that case.
## [b]Note:[/b] As quadrants are created according to the map's coordinate system, the quadrant's "square shape" might not look like square in the [TileMapLayer]'s local coordinate system.
var rendering_quadrant_size: int:
	get = get_rendering_quadrant_size, set = set_rendering_quadrant_size

## The raw tile map data as a byte array.
var tile_map_data: PackedByteArray:
	get = get_tile_map_data_as_array, set = set_tile_map_data_from_array

## The [TileSet] used by this layer. The textures, collisions, and additional behavior of all available tiles are stored here.
var tile_set: TileSet:
	get = get_tile_set, set = set_tile_set

## If [code]true[/code], this [TileMapLayer] collision shapes will be instantiated as kinematic bodies. This can be needed for moving [TileMapLayer] nodes (i.e. moving platforms).
var use_kinematic_bodies: bool:
	get = is_using_kinematic_bodies, set = set_use_kinematic_bodies

## If [member CanvasItem.y_sort_enabled] is enabled, setting this to [code]true[/code] will reverse the order the tiles are drawn on the X-axis.
var x_draw_order_reversed: bool:
	get = is_x_draw_order_reversed, set = set_x_draw_order_reversed

## This Y-sort origin value is added to each tile's Y-sort origin value. This allows, for example, to fake a different height level. This can be useful for top-down view games.
var y_sort_origin: int:
	get = get_y_sort_origin, set = set_y_sort_origin



## Called with a [TileData] object about to be used internally by the [TileMapLayer], allowing its modification at runtime.
## This method is only called if [method _use_tile_data_runtime_update] is implemented and returns [code]true[/code] for the given tile [param coords].
## [b]Warning:[/b] The [param tile_data] object's sub-resources are the same as the one in the TileSet. Modifying them might impact the whole TileSet. Instead, make sure to duplicate those resources.
## [b]Note:[/b] If the properties of [param tile_data] object should change over time, use [method notify_runtime_tile_data_update] to notify the [TileMapLayer] it needs an update.
func _tile_data_runtime_update(coords: Vector2i, tile_data: TileData) -> void:
	pass;

## Called when this [TileMapLayer]'s cells need an internal update. This update may be caused from individual cells being modified or by a change in the [member tile_set] (causing all cells to be queued for an update). The first call to this function is always for initializing all the [TileMapLayer]'s cells. [param coords] contains the coordinates of all modified cells, roughly in the order they were modified. [param forced_cleanup] is [code]true[/code] when the [TileMapLayer]'s internals should be fully cleaned up. This is the case when:
## - The layer is disabled;
## - The layer is not visible;
## - [member tile_set] is set to [code]null[/code];
## - The node is removed from the tree;
## - The node is freed.
## Note that any internal update happening while one of these conditions is verified is considered to be a "cleanup". See also [method update_internals].
## [b]Warning:[/b] Implementing this method may degrade the [TileMapLayer]'s performance.
func _update_cells(coords: Array[Vector2i], forced_cleanup: bool) -> void:
	pass;

## Should return [code]true[/code] if the tile at coordinates [param coords] requires a runtime update.
## [b]Warning:[/b] Make sure this function only returns [code]true[/code] when needed. Any tile processed at runtime without a need for it will imply a significant performance penalty.
## [b]Note:[/b] If the result of this function should change, use [method notify_runtime_tile_data_update] to notify the [TileMapLayer] it needs an update.
func _use_tile_data_runtime_update(coords: Vector2i) -> bool:
	pass;

## Clears all cells.
func clear() -> void:
	pass;

## Erases the cell at coordinates [param coords].
func erase_cell(coords: Vector2i) -> void:
	pass;

## Clears cells containing tiles that do not exist in the [member tile_set].
func fix_invalid_tiles() -> void:
	pass;

## Returns the tile alternative ID of the cell at coordinates [param coords].
func get_cell_alternative_tile(coords: Vector2i) -> int:
	pass;

## Returns the tile atlas coordinates ID of the cell at coordinates [param coords]. Returns [code]Vector2i(-1, -1)[/code] if the cell does not exist.
func get_cell_atlas_coords(coords: Vector2i) -> Vector2i:
	pass;

## Returns the tile source ID of the cell at coordinates [param coords]. Returns [code]-1[/code] if the cell does not exist.
func get_cell_source_id(coords: Vector2i) -> int:
	pass;

## Returns the [TileData] object associated with the given cell, or [code]null[/code] if the cell does not exist or is not a [TileSetAtlasSource].
## [codeblock]
## func get_clicked_tile_power():
## var clicked_cell = tile_map_layer.local_to_map(tile_map_layer.get_local_mouse_position())
## var data = tile_map_layer.get_cell_tile_data(clicked_cell)
## if data:
## return data.get_custom_data("power")
## else:
## return 0
## [/codeblock]
func get_cell_tile_data(coords: Vector2i) -> TileData:
	pass;

## Returns the coordinates of the tile for given physics body [RID]. Such an [RID] can be retrieved from [method KinematicCollision2D.get_collider_rid], when colliding with a tile.
func get_coords_for_body_rid(body: RID) -> Vector2i:
	pass;

## Returns the [RID] of the [NavigationServer2D] navigation used by this [TileMapLayer].
## By default this returns the default [World2D] navigation map, unless a custom map was provided using [method set_navigation_map].
func get_navigation_map() -> RID:
	pass;

## Returns the neighboring cell to the one at coordinates [param coords], identified by the [param neighbor] direction. This method takes into account the different layouts a TileMap can take.
func get_neighbor_cell(coords: Vector2i, neighbor: int) -> Vector2i:
	pass;

## Creates and returns a new [TileMapPattern] from the given array of cells. See also [method set_pattern].
func get_pattern(coords_array: Array[Vector2i]) -> TileMapPattern:
	pass;

## Returns the list of all neighboring cells to the one at [param coords]. Any neighboring cell is one that is touching edges, so for a square cell 4 cells would be returned, for a hexagon 6 cells are returned.
func get_surrounding_cells(coords: Vector2i) -> Array[Vector2i]:
	pass;

## Returns a [Vector2i] array with the positions of all cells containing a tile. A cell is considered empty if its source identifier equals [code]-1[/code], its atlas coordinate identifier is [code]Vector2(-1, -1)[/code] and its alternative identifier is [code]-1[/code].
func get_used_cells() -> Array[Vector2i]:
	pass;

## Returns a [Vector2i] array with the positions of all cells containing a tile. Tiles may be filtered according to their source ([param source_id]), their atlas coordinates ([param atlas_coords]), or alternative id ([param alternative_tile]).
## If a parameter has its value set to the default one, this parameter is not used to filter a cell. Thus, if all parameters have their respective default values, this method returns the same result as [method get_used_cells].
## A cell is considered empty if its source identifier equals [code]-1[/code], its atlas coordinate identifier is [code]Vector2(-1, -1)[/code] and its alternative identifier is [code]-1[/code].
func get_used_cells_by_id(source_id: int = -1, atlas_coords: Vector2i = Vector2i(-1, -1), alternative_tile: int = -1) -> Array[Vector2i]:
	pass;

## Returns a rectangle enclosing the used (non-empty) tiles of the map.
func get_used_rect() -> Rect2i:
	pass;

## Returns whether the provided [param body] [RID] belongs to one of this [TileMapLayer]'s cells.
func has_body_rid(body: RID) -> bool:
	pass;

## Returns [code]true[/code] if the cell at coordinates [param coords] is flipped horizontally. The result is valid only for atlas sources.
func is_cell_flipped_h(coords: Vector2i) -> bool:
	pass;

## Returns [code]true[/code] if the cell at coordinates [param coords] is flipped vertically. The result is valid only for atlas sources.
func is_cell_flipped_v(coords: Vector2i) -> bool:
	pass;

## Returns [code]true[/code] if the cell at coordinates [param coords] is transposed. The result is valid only for atlas sources.
func is_cell_transposed(coords: Vector2i) -> bool:
	pass;

## Returns the map coordinates of the cell containing the given [param local_position]. If [param local_position] is in global coordinates, consider using [method Node2D.to_local] before passing it to this method. See also [method map_to_local].
func local_to_map(local_position: Vector2) -> Vector2i:
	pass;

## Returns for the given coordinates [param coords_in_pattern] in a [TileMapPattern] the corresponding cell coordinates if the pattern was pasted at the [param position_in_tilemap] coordinates (see [method set_pattern]). This mapping is required as in half-offset tile shapes, the mapping might not work by calculating [code]position_in_tile_map + coords_in_pattern[/code].
func map_pattern(position_in_tilemap: Vector2i, coords_in_pattern: Vector2i, pattern: TileMapPattern) -> Vector2i:
	pass;

## Returns the centered position of a cell in the [TileMapLayer]'s local coordinate space. To convert the returned value into global coordinates, use [method Node2D.to_global]. See also [method local_to_map].
## [b]Note:[/b] This may not correspond to the visual position of the tile, i.e. it ignores the [member TileData.texture_origin] property of individual tiles.
func map_to_local(map_position: Vector2i) -> Vector2:
	pass;

## Notifies the [TileMapLayer] node that calls to [method _use_tile_data_runtime_update] or [method _tile_data_runtime_update] will lead to different results. This will thus trigger a [TileMapLayer] update.
## [b]Warning:[/b] Updating the [TileMapLayer] is computationally expensive and may impact performance. Try to limit the number of calls to this function to avoid unnecessary update.
## [b]Note:[/b] This does not trigger a direct update of the [TileMapLayer], the update will be done at the end of the frame as usual (unless you call [method update_internals]).
func notify_runtime_tile_data_update() -> void:
	pass;

## Sets the tile identifiers for the cell at coordinates [param coords]. Each tile of the [TileSet] is identified using three parts:
## - The source identifier [param source_id] identifies a [TileSetSource] identifier. See [method TileSet.set_source_id],
## - The atlas coordinate identifier [param atlas_coords] identifies a tile coordinates in the atlas (if the source is a [TileSetAtlasSource]). For [TileSetScenesCollectionSource] it should always be [code]Vector2i(0, 0)[/code],
## - The alternative tile identifier [param alternative_tile] identifies a tile alternative in the atlas (if the source is a [TileSetAtlasSource]), and the scene for a [TileSetScenesCollectionSource].
## If [param source_id] is set to [code]-1[/code], [param atlas_coords] to [code]Vector2i(-1, -1)[/code], or [param alternative_tile] to [code]-1[/code], the cell will be erased. An erased cell gets [b]all[/b] its identifiers automatically set to their respective invalid values, namely [code]-1[/code], [code]Vector2i(-1, -1)[/code] and [code]-1[/code].
func set_cell(coords: Vector2i, source_id: int = -1, atlas_coords: Vector2i = Vector2i(-1, -1), alternative_tile: int = 0) -> void:
	pass;

## Update all the cells in the [param cells] coordinates array so that they use the given [param terrain] for the given [param terrain_set]. If an updated cell has the same terrain as one of its neighboring cells, this function tries to join the two. This function might update neighboring tiles if needed to create correct terrain transitions.
## If [param ignore_empty_terrains] is [code]true[/code], empty terrains will be ignored when trying to find the best fitting tile for the given terrain constraints.
## [b]Note:[/b] To work correctly, this method requires the [TileMapLayer]'s TileSet to have terrains set up with all required terrain combinations. Otherwise, it may produce unexpected results.
func set_cells_terrain_connect(cells: Array[Vector2i], terrain_set: int, terrain: int, ignore_empty_terrains: bool = true) -> void:
	pass;

## Update all the cells in the [param path] coordinates array so that they use the given [param terrain] for the given [param terrain_set]. The function will also connect two successive cell in the path with the same terrain. This function might update neighboring tiles if needed to create correct terrain transitions.
## If [param ignore_empty_terrains] is [code]true[/code], empty terrains will be ignored when trying to find the best fitting tile for the given terrain constraints.
## [b]Note:[/b] To work correctly, this method requires the [TileMapLayer]'s TileSet to have terrains set up with all required terrain combinations. Otherwise, it may produce unexpected results.
func set_cells_terrain_path(path: Array[Vector2i], terrain_set: int, terrain: int, ignore_empty_terrains: bool = true) -> void:
	pass;

## Sets a custom [param map] as a [NavigationServer2D] navigation map. If not set, uses the default [World2D] navigation map instead.
func set_navigation_map(map: RID) -> void:
	pass;

## Pastes the [TileMapPattern] at the given [param position] in the tile map. See also [method get_pattern].
func set_pattern(position: Vector2i, pattern: TileMapPattern) -> void:
	pass;

## Triggers a direct update of the [TileMapLayer]. Usually, calling this function is not needed, as [TileMapLayer] node updates automatically when one of its properties or cells is modified.
## However, for performance reasons, those updates are batched and delayed to the end of the frame. Calling this function will force the [TileMapLayer] to update right away instead.
## [b]Warning:[/b] Updating the [TileMapLayer] is computationally expensive and may impact performance. Try to limit the number of updates and how many tiles they impact.
func update_internals() -> void:
	pass;


func is_collision_enabled() -> bool:
	return collision_enabled

func set_collision_enabled(value: bool) -> void:
	collision_enabled = value

func get_collision_visibility_mode() -> int:
	return collision_visibility_mode

func set_collision_visibility_mode(value: int) -> void:
	collision_visibility_mode = value

func is_enabled() -> bool:
	return enabled

func set_enabled(value: bool) -> void:
	enabled = value

func is_navigation_enabled() -> bool:
	return navigation_enabled

func set_navigation_enabled(value: bool) -> void:
	navigation_enabled = value

func get_navigation_visibility_mode() -> int:
	return navigation_visibility_mode

func set_navigation_visibility_mode(value: int) -> void:
	navigation_visibility_mode = value

func is_occlusion_enabled() -> bool:
	return occlusion_enabled

func set_occlusion_enabled(value: bool) -> void:
	occlusion_enabled = value

func get_rendering_quadrant_size() -> int:
	return rendering_quadrant_size

func set_rendering_quadrant_size(value: int) -> void:
	rendering_quadrant_size = value

func get_tile_map_data_as_array() -> PackedByteArray:
	return tile_map_data

func set_tile_map_data_from_array(value: PackedByteArray) -> void:
	tile_map_data = value

func get_tile_set() -> TileSet:
	return tile_set

func set_tile_set(value: TileSet) -> void:
	tile_set = value

func is_using_kinematic_bodies() -> bool:
	return use_kinematic_bodies

func set_use_kinematic_bodies(value: bool) -> void:
	use_kinematic_bodies = value

func is_x_draw_order_reversed() -> bool:
	return x_draw_order_reversed

func set_x_draw_order_reversed(value: bool) -> void:
	x_draw_order_reversed = value

func get_y_sort_origin() -> int:
	return y_sort_origin

func set_y_sort_origin(value: int) -> void:
	y_sort_origin = value

