extends Container
class_name TabContainer

## A container that creates a tab for each child control, displaying only the active tab's control.
##
## Arranges child controls into a tabbed view, creating a tab for each one. The active tab's corresponding control is made visible, while all other child controls are hidden. Ignores non-control children.
## [b]Note:[/b] The drawing of the clickable tabs is handled by this node; [TabBar] is not needed.
##
## @tutorial(Using Containers): https://docs.godotengine.org/en/stable/tutorials/ui/gui_containers.html


## Emitted when the active tab is rearranged via mouse drag. See [member drag_to_rearrange_enabled].
signal active_tab_rearranged(idx_to: int)
## Emitted when the [TabContainer]'s [Popup] button is clicked. See [method set_popup] for details.
signal pre_popup_pressed
## Emitted when the user clicks on the button icon on this tab.
signal tab_button_pressed(tab: int)
## Emitted when switching to another tab.
signal tab_changed(tab: int)
## Emitted when a tab is clicked, even if it is the current tab.
signal tab_clicked(tab: int)
## Emitted when a tab is hovered by the mouse.
signal tab_hovered(tab: int)
## Emitted when a tab is selected via click, directional input, or script, even if it is the current tab.
signal tab_selected(tab: int)
## Places the tab bar at the top.
## Places the tab bar at the bottom. The tab bar's [StyleBox] will be flipped vertically.
## Represents the size of the [enum TabPosition] enum.

#enum TabPosition
enum {
    POSITION_TOP = 0,
    POSITION_BOTTOM = 1,
    POSITION_MAX = 2,
}
## If [code]true[/code], all tabs are drawn in front of the panel. If [code]false[/code], inactive tabs are drawn behind the panel.
var all_tabs_in_front: bool:
	get = is_all_tabs_in_front, set = set_all_tabs_in_front

## If [code]true[/code], tabs overflowing this node's width will be hidden, displaying two navigation buttons instead. Otherwise, this node's minimum size is updated so that all tabs are visible.
var clip_tabs: bool:
	get = get_clip_tabs, set = set_clip_tabs

## The current tab index. When set, this index's [Control] node's [code]visible[/code] property is set to [code]true[/code] and all others are set to [code]false[/code].
## A value of [code]-1[/code] means that no tab is selected.
var current_tab: int:
	get = get_current_tab, set = set_current_tab

## If [code]true[/code], all tabs can be deselected so that no tab is selected. Click on the [member current_tab] to deselect it.
## Only the tab header will be shown if no tabs are selected.
var deselect_enabled: bool:
	get = get_deselect_enabled, set = set_deselect_enabled

## If [code]true[/code], tabs can be rearranged with mouse drag.
var drag_to_rearrange_enabled: bool:
	get = get_drag_to_rearrange_enabled, set = set_drag_to_rearrange_enabled

## Sets the position at which tabs will be placed. See [enum TabBar.AlignmentMode] for details.
var tab_alignment: int:
	get = get_tab_alignment, set = set_tab_alignment

## The focus access mode for the internal [TabBar] node.
var tab_focus_mode: int:
	get = get_tab_focus_mode, set = set_tab_focus_mode

## Sets the position of the tab bar. See [enum TabPosition] for details.
var tabs_position: int:
	get = get_tabs_position, set = set_tabs_position

## [TabContainer]s with the same rearrange group ID will allow dragging the tabs between them. Enable drag with [member drag_to_rearrange_enabled].
## Setting this to [code]-1[/code] will disable rearranging between [TabContainer]s.
var tabs_rearrange_group: int:
	get = get_tabs_rearrange_group, set = set_tabs_rearrange_group

## If [code]true[/code], tabs are visible. If [code]false[/code], tabs' content and titles are hidden.
var tabs_visible: bool:
	get = are_tabs_visible, set = set_tabs_visible

## If [code]true[/code], child [Control] nodes that are hidden have their minimum size take into account in the total, instead of only the currently visible one.
var use_hidden_tabs_for_min_size: bool:
	get = get_use_hidden_tabs_for_min_size, set = set_use_hidden_tabs_for_min_size



## Returns the child [Control] node located at the active tab index.
func get_current_tab_control() -> Control:
	pass;

## Returns the [Popup] node instance if one has been set already with [method set_popup].
## [b]Warning:[/b] This is a required internal node, removing and freeing it may cause a crash. If you wish to hide it or any of its children, use their [member Window.visible] property.
func get_popup() -> Popup:
	pass;

## Returns the previously active tab index.
func get_previous_tab() -> int:
	pass;

## Returns the [TabBar] contained in this container.
## [b]Warning:[/b] This is a required internal node, removing and freeing it or editing its tabs may cause a crash. If you wish to edit the tabs, use the methods provided in [TabContainer].
func get_tab_bar() -> TabBar:
	pass;

## Returns the button icon from the tab at index [param tab_idx].
func get_tab_button_icon(tab_idx: int) -> Texture2D:
	pass;

## Returns the [Control] node from the tab at index [param tab_idx].
func get_tab_control(tab_idx: int) -> Control:
	pass;

## Returns the number of tabs.
func get_tab_count() -> int:
	pass;

## Returns the [Texture2D] for the tab at index [param tab_idx] or [code]null[/code] if the tab has no [Texture2D].
func get_tab_icon(tab_idx: int) -> Texture2D:
	pass;

## Returns the maximum allowed width of the icon for the tab at index [param tab_idx].
func get_tab_icon_max_width(tab_idx: int) -> int:
	pass;

## Returns the index of the tab at local coordinates [param point]. Returns [code]-1[/code] if the point is outside the control boundaries or if there's no tab at the queried position.
func get_tab_idx_at_point(point: Vector2) -> int:
	pass;

## Returns the index of the tab tied to the given [param control]. The control must be a child of the [TabContainer].
func get_tab_idx_from_control(control: Control) -> int:
	pass;

## Returns the metadata value set to the tab at index [param tab_idx] using [method set_tab_metadata]. If no metadata was previously set, returns [code]null[/code] by default.
func get_tab_metadata(tab_idx: int) -> Variant:
	pass;

## Returns the title of the tab at index [param tab_idx]. Tab titles default to the name of the indexed child node, but this can be overridden with [method set_tab_title].
func get_tab_title(tab_idx: int) -> String:
	pass;

## Returns the tooltip text of the tab at index [param tab_idx].
func get_tab_tooltip(tab_idx: int) -> String:
	pass;

## Returns [code]true[/code] if the tab at index [param tab_idx] is disabled.
func is_tab_disabled(tab_idx: int) -> bool:
	pass;

## Returns [code]true[/code] if the tab at index [param tab_idx] is hidden.
func is_tab_hidden(tab_idx: int) -> bool:
	pass;

## Selects the first available tab with greater index than the currently selected. Returns [code]true[/code] if tab selection changed.
func select_next_available() -> bool:
	pass;

## Selects the first available tab with lower index than the currently selected. Returns [code]true[/code] if tab selection changed.
func select_previous_available() -> bool:
	pass;

## If set on a [Popup] node instance, a popup menu icon appears in the top-right corner of the [TabContainer] (setting it to [code]null[/code] will make it go away). Clicking it will expand the [Popup] node.
func set_popup(popup: Node) -> void:
	pass;

## Sets the button icon from the tab at index [param tab_idx].
func set_tab_button_icon(tab_idx: int, icon: Texture2D) -> void:
	pass;

## If [param disabled] is [code]true[/code], disables the tab at index [param tab_idx], making it non-interactable.
func set_tab_disabled(tab_idx: int, disabled: bool) -> void:
	pass;

## If [param hidden] is [code]true[/code], hides the tab at index [param tab_idx], making it disappear from the tab area.
func set_tab_hidden(tab_idx: int, hidden: bool) -> void:
	pass;

## Sets an icon for the tab at index [param tab_idx].
func set_tab_icon(tab_idx: int, icon: Texture2D) -> void:
	pass;

## Sets the maximum allowed width of the icon for the tab at index [param tab_idx]. This limit is applied on top of the default size of the icon and on top of [theme_item icon_max_width]. The height is adjusted according to the icon's ratio.
func set_tab_icon_max_width(tab_idx: int, width: int) -> void:
	pass;

## Sets the metadata value for the tab at index [param tab_idx], which can be retrieved later using [method get_tab_metadata].
func set_tab_metadata(tab_idx: int, metadata: Variant) -> void:
	pass;

## Sets a custom title for the tab at index [param tab_idx] (tab titles default to the name of the indexed child node). Set it back to the child's name to make the tab default to it again.
func set_tab_title(tab_idx: int, title: String) -> void:
	pass;

## Sets a custom tooltip text for tab at index [param tab_idx].
## [b]Note:[/b] By default, if the [param tooltip] is empty and the tab text is truncated (not all characters fit into the tab), the title will be displayed as a tooltip. To hide the tooltip, assign [code]" "[/code] as the [param tooltip] text.
func set_tab_tooltip(tab_idx: int, tooltip: String) -> void:
	pass;


func is_all_tabs_in_front() -> bool:
	return all_tabs_in_front

func set_all_tabs_in_front(value: bool) -> void:
	all_tabs_in_front = value

func get_clip_tabs() -> bool:
	return clip_tabs

func set_clip_tabs(value: bool) -> void:
	clip_tabs = value

func get_current_tab() -> int:
	return current_tab

func set_current_tab(value: int) -> void:
	current_tab = value

func get_deselect_enabled() -> bool:
	return deselect_enabled

func set_deselect_enabled(value: bool) -> void:
	deselect_enabled = value

func get_drag_to_rearrange_enabled() -> bool:
	return drag_to_rearrange_enabled

func set_drag_to_rearrange_enabled(value: bool) -> void:
	drag_to_rearrange_enabled = value

func get_tab_alignment() -> int:
	return tab_alignment

func set_tab_alignment(value: int) -> void:
	tab_alignment = value

func get_tab_focus_mode() -> int:
	return tab_focus_mode

func set_tab_focus_mode(value: int) -> void:
	tab_focus_mode = value

func get_tabs_position() -> int:
	return tabs_position

func set_tabs_position(value: int) -> void:
	tabs_position = value

func get_tabs_rearrange_group() -> int:
	return tabs_rearrange_group

func set_tabs_rearrange_group(value: int) -> void:
	tabs_rearrange_group = value

func are_tabs_visible() -> bool:
	return tabs_visible

func set_tabs_visible(value: bool) -> void:
	tabs_visible = value

func get_use_hidden_tabs_for_min_size() -> bool:
	return use_hidden_tabs_for_min_size

func set_use_hidden_tabs_for_min_size(value: bool) -> void:
	use_hidden_tabs_for_min_size = value

