extends Control
class_name Range

## Abstract base class for controls that represent a number within a range.
##
## Range is an abstract base class for controls that represent a number within a range, using a configured [member step] and [member page] size. See e.g. [ScrollBar] and [Slider] for examples of higher-level nodes using Range.


## Emitted when [member min_value], [member max_value], [member page], or [member step] change.
signal changed
## Emitted when [member value] changes. When used on a [Slider], this is called continuously while dragging (potentially every frame). If you are performing an expensive operation in a function connected to [signal value_changed], consider using a [i]debouncing[/i] [Timer] to call the function less often.
## [b]Note:[/b] Unlike signals such as [signal LineEdit.text_changed], [signal value_changed] is also emitted when [param value] is set directly via code.
signal value_changed(value: float)

## If [code]true[/code], [member value] may be greater than [member max_value].
var allow_greater: bool:
	get = is_greater_allowed, set = set_allow_greater

## If [code]true[/code], [member value] may be less than [member min_value].
var allow_lesser: bool:
	get = is_lesser_allowed, set = set_allow_lesser

## If [code]true[/code], and [member min_value] is greater than 0, [member value] will be represented exponentially rather than linearly.
var exp_edit: bool:
	get = is_ratio_exp, set = set_exp_ratio

## Maximum value. Range is clamped if [member value] is greater than [member max_value].
var max_value: float:
	get = get_max, set = set_max

## Minimum value. Range is clamped if [member value] is less than [member min_value].
var min_value: float:
	get = get_min, set = set_min

## Page size. Used mainly for [ScrollBar]. A [ScrollBar]'s grabber length is the [ScrollBar]'s size multiplied by [member page] over the difference between [member min_value] and [member max_value].
var page: float:
	get = get_page, set = set_page

## The value mapped between 0 and 1.
var ratio: float:
	get = get_as_ratio, set = set_as_ratio

## If [code]true[/code], [member value] will always be rounded to the nearest integer.
var rounded: bool:
	get = is_using_rounded_values, set = set_use_rounded_values

var size_flags_vertical: int:
	get = get_v_size_flags, set = set_v_size_flags

## If greater than 0, [member value] will always be rounded to a multiple of this property's value. If [member rounded] is also [code]true[/code], [member value] will first be rounded to a multiple of this property's value, then rounded to the nearest integer.
var step: float:
	get = get_step, set = set_step

## Range's current value. Changing this property (even via code) will trigger [signal value_changed] signal. Use [method set_value_no_signal] if you want to avoid it.
var value: float:
	get = get_value, set = set_value



## Called when the [Range]'s value is changed (following the same conditions as [signal value_changed]).
func _value_changed(new_value: float) -> void:
	pass;

## Sets the [Range]'s current value to the specified [param value], without emitting the [signal value_changed] signal.
func set_value_no_signal(value: float) -> void:
	pass;

## Binds two [Range]s together along with any ranges previously grouped with either of them. When any of range's member variables change, it will share the new value with all other ranges in its group.
func share(with: Node) -> void:
	pass;

## Stops the [Range] from sharing its member variables with any other.
func unshare() -> void:
	pass;


func is_greater_allowed() -> bool:
	return allow_greater

func set_allow_greater(value: bool) -> void:
	allow_greater = value

func is_lesser_allowed() -> bool:
	return allow_lesser

func set_allow_lesser(value: bool) -> void:
	allow_lesser = value

func is_ratio_exp() -> bool:
	return exp_edit

func set_exp_ratio(value: bool) -> void:
	exp_edit = value

func get_max() -> float:
	return max_value

func set_max(value: float) -> void:
	max_value = value

func get_min() -> float:
	return min_value

func set_min(value: float) -> void:
	min_value = value

func get_page() -> float:
	return page

func set_page(value: float) -> void:
	page = value

func get_as_ratio() -> float:
	return ratio

func set_as_ratio(value: float) -> void:
	ratio = value

func is_using_rounded_values() -> bool:
	return rounded

func set_use_rounded_values(value: bool) -> void:
	rounded = value

func get_v_size_flags() -> int:
	return size_flags_vertical

func set_v_size_flags(value: int) -> void:
	size_flags_vertical = value

func get_step() -> float:
	return step

func set_step(value: float) -> void:
	step = value

func get_value() -> float:
	return value

func set_value(value: float) -> void:
	value = value

