extends Node
class_name NavigationAgent2D

## A 2D agent used to pathfind to a position while avoiding obstacles.
##
## A 2D agent used to pathfind to a position while avoiding static and dynamic obstacles. The calculation can be used by the parent node to dynamically move it along the path. Requires navigation data to work correctly.
## Dynamic obstacles are avoided using RVO collision avoidance. Avoidance is computed before physics, so the pathfinding information can be used safely in the physics step.
## [b]Note:[/b] After setting the [member target_position] property, the [method get_next_path_position] method must be used once every physics frame to update the internal path logic of the navigation agent. The vector position it returns should be used as the next movement position for the agent's parent node.
##
## @tutorial(Using NavigationAgents): https://docs.godotengine.org/en/stable/tutorials/navigation/navigation_using_navigationagents.html


## Signals that the agent reached a navigation link. Emitted when the agent moves within [member path_desired_distance] of the next position of the path when that position is a navigation link.
## The details dictionary may contain the following keys depending on the value of [member path_metadata_flags]:
## - [code]position[/code]: The start position of the link that was reached.
## - [code]type[/code]: Always [constant NavigationPathQueryResult2D.PATH_SEGMENT_TYPE_LINK].
## - [code]rid[/code]: The [RID] of the link.
## - [code]owner[/code]: The object which manages the link (usually [NavigationLink2D]).
## - [code]link_entry_position[/code]: If [code]owner[/code] is available and the owner is a [NavigationLink2D], it will contain the global position of the link's point the agent is entering.
## - [code]link_exit_position[/code]: If [code]owner[/code] is available and the owner is a [NavigationLink2D], it will contain the global position of the link's point which the agent is exiting.
signal link_reached(details: Dictionary)
## Signals that the agent's navigation has finished. If the target is reachable, navigation ends when the target is reached. If the target is unreachable, navigation ends when the last waypoint of the path is reached. This signal is emitted only once per loaded path.
## This signal will be emitted just after [signal target_reached] when the target is reachable.
signal navigation_finished
## Emitted when the agent had to update the loaded path:
## - because path was previously empty.
## - because navigation map has changed.
## - because agent pushed further away from the current path segment than the [member path_max_distance].
signal path_changed
## Signals that the agent reached the target, i.e. the agent moved within [member target_desired_distance] of the [member target_position]. This signal is emitted only once per loaded path.
## This signal will be emitted just before [signal navigation_finished] when the target is reachable.
## It may not always be possible to reach the target but it should always be possible to reach the final position. See [method get_final_position].
signal target_reached
## Notifies when the collision avoidance velocity is calculated. Emitted every update as long as [member avoidance_enabled] is [code]true[/code] and the agent has a navigation map.
signal velocity_computed(safe_velocity: Vector2)
## Signals that the agent reached a waypoint. Emitted when the agent moves within [member path_desired_distance] of the next position of the path.
## The details dictionary may contain the following keys depending on the value of [member path_metadata_flags]:
## - [code]position[/code]: The position of the waypoint that was reached.
## - [code]type[/code]: The type of navigation primitive (region or link) that contains this waypoint.
## - [code]rid[/code]: The [RID] of the containing navigation primitive (region or link).
## - [code]owner[/code]: The object which manages the containing navigation primitive (region or link).
signal waypoint_reached(details: Dictionary)

## If [code]true[/code] the agent is registered for an RVO avoidance callback on the [NavigationServer2D]. When [member velocity] is used and the processing is completed a [code]safe_velocity[/code] Vector2 is received with a signal connection to [signal velocity_computed]. Avoidance processing with many registered agents has a significant performance cost and should only be enabled on agents that currently require it.
var avoidance_enabled: bool:
	get = get_avoidance_enabled, set = set_avoidance_enabled

## A bitfield determining the avoidance layers for this NavigationAgent. Other agents with a matching bit on the [member avoidance_mask] will avoid this agent.
var avoidance_layers: int:
	get = get_avoidance_layers, set = set_avoidance_layers

## A bitfield determining what other avoidance agents and obstacles this NavigationAgent will avoid when a bit matches at least one of their [member avoidance_layers].
var avoidance_mask: int:
	get = get_avoidance_mask, set = set_avoidance_mask

## The agent does not adjust the velocity for other agents that would match the [member avoidance_mask] but have a lower [member avoidance_priority]. This in turn makes the other agents with lower priority adjust their velocities even more to avoid collision with this agent.
var avoidance_priority: float:
	get = get_avoidance_priority, set = set_avoidance_priority

## If [code]true[/code] shows debug visuals for this agent.
var debug_enabled: bool:
	get = get_debug_enabled, set = set_debug_enabled

## If [member debug_use_custom] is [code]true[/code] uses this color for this agent instead of global color.
var debug_path_custom_color: Color:
	get = get_debug_path_custom_color, set = set_debug_path_custom_color

## If [member debug_use_custom] is [code]true[/code] uses this line width for rendering paths for this agent instead of global line width.
var debug_path_custom_line_width: float:
	get = get_debug_path_custom_line_width, set = set_debug_path_custom_line_width

## If [member debug_use_custom] is [code]true[/code] uses this rasterized point size for rendering path points for this agent instead of global point size.
var debug_path_custom_point_size: float:
	get = get_debug_path_custom_point_size, set = set_debug_path_custom_point_size

## If [code]true[/code] uses the defined [member debug_path_custom_color] for this agent instead of global color.
var debug_use_custom: bool:
	get = get_debug_use_custom, set = set_debug_use_custom

## The maximum number of neighbors for the agent to consider.
var max_neighbors: int:
	get = get_max_neighbors, set = set_max_neighbors

## The maximum speed that an agent can move.
var max_speed: float:
	get = get_max_speed, set = set_max_speed

## A bitfield determining which navigation layers of navigation regions this agent will use to calculate a path. Changing it during runtime will clear the current navigation path and generate a new one, according to the new navigation layers.
var navigation_layers: int:
	get = get_navigation_layers, set = set_navigation_layers

## The distance to search for other agents.
var neighbor_distance: float:
	get = get_neighbor_distance, set = set_neighbor_distance

## The distance threshold before a path point is considered to be reached. This allows agents to not have to hit a path point on the path exactly, but only to reach its general area. If this value is set too high, the NavigationAgent will skip points on the path, which can lead to it leaving the navigation mesh. If this value is set too low, the NavigationAgent will be stuck in a repath loop because it will constantly overshoot the distance to the next point on each physics frame update.
var path_desired_distance: float:
	get = get_path_desired_distance, set = set_path_desired_distance

## The maximum distance the agent is allowed away from the ideal path to the final position. This can happen due to trying to avoid collisions. When the maximum distance is exceeded, it recalculates the ideal path.
var path_max_distance: float:
	get = get_path_max_distance, set = set_path_max_distance

## Additional information to return with the navigation path.
var path_metadata_flags: int:
	get = get_path_metadata_flags, set = set_path_metadata_flags

## The path postprocessing applied to the raw path corridor found by the [member pathfinding_algorithm].
var path_postprocessing: int:
	get = get_path_postprocessing, set = set_path_postprocessing

## The pathfinding algorithm used in the path query.
var pathfinding_algorithm: int:
	get = get_pathfinding_algorithm, set = set_pathfinding_algorithm

## The radius of the avoidance agent. This is the "body" of the avoidance agent and not the avoidance maneuver starting radius (which is controlled by [member neighbor_distance]).
## Does not affect normal pathfinding. To change an actor's pathfinding radius bake [NavigationMesh] resources with a different [member NavigationMesh.agent_radius] property and use different navigation maps for each actor size.
var radius: float:
	get = get_radius, set = set_radius

## The path simplification amount in worlds units.
var simplify_epsilon: float:
	get = get_simplify_epsilon, set = set_simplify_epsilon

## If [code]true[/code] a simplified version of the path will be returned with less critical path points removed. The simplification amount is controlled by [member simplify_epsilon]. The simplification uses a variant of Ramer-Douglas-Peucker algorithm for curve point decimation.
## Path simplification can be helpful to mitigate various path following issues that can arise with certain agent types and script behaviors. E.g. "steering" agents or avoidance in "open fields".
var simplify_path: bool:
	get = get_simplify_path, set = set_simplify_path

## The distance threshold before the target is considered to be reached. On reaching the target, [signal target_reached] is emitted and navigation ends (see [method is_navigation_finished] and [signal navigation_finished]).
## You can make navigation end early by setting this property to a value greater than [member path_desired_distance] (navigation will end before reaching the last waypoint).
## You can also make navigation end closer to the target than each individual path position by setting this property to a value lower than [member path_desired_distance] (navigation won't immediately end when reaching the last waypoint). However, if the value set is too low, the agent will be stuck in a repath loop because it will constantly overshoot the distance to the target on each physics frame update.
var target_desired_distance: float:
	get = get_target_desired_distance, set = set_target_desired_distance

## If set, a new navigation path from the current agent position to the [member target_position] is requested from the NavigationServer.
var target_position: Vector2:
	get = get_target_position, set = set_target_position

## The minimal amount of time for which this agent's velocities, that are computed with the collision avoidance algorithm, are safe with respect to other agents. The larger the number, the sooner the agent will respond to other agents, but less freedom in choosing its velocities. A too high value will slow down agents movement considerably. Must be positive.
var time_horizon_agents: float:
	get = get_time_horizon_agents, set = set_time_horizon_agents

## The minimal amount of time for which this agent's velocities, that are computed with the collision avoidance algorithm, are safe with respect to static avoidance obstacles. The larger the number, the sooner the agent will respond to static avoidance obstacles, but less freedom in choosing its velocities. A too high value will slow down agents movement considerably. Must be positive.
var time_horizon_obstacles: float:
	get = get_time_horizon_obstacles, set = set_time_horizon_obstacles

## Sets the new wanted velocity for the agent. The avoidance simulation will try to fulfill this velocity if possible but will modify it to avoid collision with other agents and obstacles. When an agent is teleported to a new position, use [method set_velocity_forced] as well to reset the internal simulation velocity.
var velocity: Vector2:
	get = get_velocity, set = set_velocity



## Returns the distance to the target position, using the agent's global position. The user must set [member target_position] in order for this to be accurate.
func distance_to_target() -> float:
	pass;

## Returns whether or not the specified layer of the [member avoidance_layers] bitmask is enabled, given a [param layer_number] between 1 and 32.
func get_avoidance_layer_value(layer_number: int) -> bool:
	pass;

## Returns whether or not the specified mask of the [member avoidance_mask] bitmask is enabled, given a [param mask_number] between 1 and 32.
func get_avoidance_mask_value(mask_number: int) -> bool:
	pass;

## Returns this agent's current path from start to finish in global coordinates. The path only updates when the target position is changed or the agent requires a repath. The path array is not intended to be used in direct path movement as the agent has its own internal path logic that would get corrupted by changing the path array manually. Use the intended [method get_next_path_position] once every physics frame to receive the next path point for the agents movement as this function also updates the internal path logic.
func get_current_navigation_path() -> PackedVector2Array:
	pass;

## Returns which index the agent is currently on in the navigation path's [PackedVector2Array].
func get_current_navigation_path_index() -> int:
	pass;

## Returns the path query result for the path the agent is currently following.
func get_current_navigation_result() -> NavigationPathQueryResult2D:
	pass;

## Returns the reachable final position of the current navigation path in global coordinates. This position can change if the agent needs to update the navigation path which makes the agent emit the [signal path_changed] signal.
func get_final_position() -> Vector2:
	pass;

## Returns whether or not the specified layer of the [member navigation_layers] bitmask is enabled, given a [param layer_number] between 1 and 32.
func get_navigation_layer_value(layer_number: int) -> bool:
	pass;

## Returns the [RID] of the navigation map for this NavigationAgent node. This function returns always the map set on the NavigationAgent node and not the map of the abstract agent on the NavigationServer. If the agent map is changed directly with the NavigationServer API the NavigationAgent node will not be aware of the map change. Use [method set_navigation_map] to change the navigation map for the NavigationAgent and also update the agent on the NavigationServer.
func get_navigation_map() -> RID:
	pass;

## Returns the next position in global coordinates that can be moved to, making sure that there are no static objects in the way. If the agent does not have a navigation path, it will return the position of the agent's parent. The use of this function once every physics frame is required to update the internal path logic of the NavigationAgent.
func get_next_path_position() -> Vector2:
	pass;

## Returns the [RID] of this agent on the [NavigationServer2D].
func get_rid() -> RID:
	pass;

## Returns [code]true[/code] if the agent's navigation has finished. If the target is reachable, navigation ends when the target is reached. If the target is unreachable, navigation ends when the last waypoint of the path is reached.
## [b]Note:[/b] While [code]true[/code] prefer to stop calling update functions like [method get_next_path_position]. This avoids jittering the standing agent due to calling repeated path updates.
func is_navigation_finished() -> bool:
	pass;

## Returns [code]true[/code] if [method get_final_position] is within [member target_desired_distance] of the [member target_position].
func is_target_reachable() -> bool:
	pass;

## Returns [code]true[/code] if the agent reached the target, i.e. the agent moved within [member target_desired_distance] of the [member target_position]. It may not always be possible to reach the target but it should always be possible to reach the final position. See [method get_final_position].
func is_target_reached() -> bool:
	pass;

## Based on [param value], enables or disables the specified layer in the [member avoidance_layers] bitmask, given a [param layer_number] between 1 and 32.
func set_avoidance_layer_value(layer_number: int, value: bool) -> void:
	pass;

## Based on [param value], enables or disables the specified mask in the [member avoidance_mask] bitmask, given a [param mask_number] between 1 and 32.
func set_avoidance_mask_value(mask_number: int, value: bool) -> void:
	pass;

## Based on [param value], enables or disables the specified layer in the [member navigation_layers] bitmask, given a [param layer_number] between 1 and 32.
func set_navigation_layer_value(layer_number: int, value: bool) -> void:
	pass;

## Sets the [RID] of the navigation map this NavigationAgent node should use and also updates the [code]agent[/code] on the NavigationServer.
func set_navigation_map(navigation_map: RID) -> void:
	pass;

## Replaces the internal velocity in the collision avoidance simulation with [param velocity]. When an agent is teleported to a new position this function should be used in the same frame. If called frequently this function can get agents stuck.
func set_velocity_forced(velocity: Vector2) -> void:
	pass;


func get_avoidance_enabled() -> bool:
	return avoidance_enabled

func set_avoidance_enabled(value: bool) -> void:
	avoidance_enabled = value

func get_avoidance_layers() -> int:
	return avoidance_layers

func set_avoidance_layers(value: int) -> void:
	avoidance_layers = value

func get_avoidance_mask() -> int:
	return avoidance_mask

func set_avoidance_mask(value: int) -> void:
	avoidance_mask = value

func get_avoidance_priority() -> float:
	return avoidance_priority

func set_avoidance_priority(value: float) -> void:
	avoidance_priority = value

func get_debug_enabled() -> bool:
	return debug_enabled

func set_debug_enabled(value: bool) -> void:
	debug_enabled = value

func get_debug_path_custom_color() -> Color:
	return debug_path_custom_color

func set_debug_path_custom_color(value: Color) -> void:
	debug_path_custom_color = value

func get_debug_path_custom_line_width() -> float:
	return debug_path_custom_line_width

func set_debug_path_custom_line_width(value: float) -> void:
	debug_path_custom_line_width = value

func get_debug_path_custom_point_size() -> float:
	return debug_path_custom_point_size

func set_debug_path_custom_point_size(value: float) -> void:
	debug_path_custom_point_size = value

func get_debug_use_custom() -> bool:
	return debug_use_custom

func set_debug_use_custom(value: bool) -> void:
	debug_use_custom = value

func get_max_neighbors() -> int:
	return max_neighbors

func set_max_neighbors(value: int) -> void:
	max_neighbors = value

func get_max_speed() -> float:
	return max_speed

func set_max_speed(value: float) -> void:
	max_speed = value

func get_navigation_layers() -> int:
	return navigation_layers

func set_navigation_layers(value: int) -> void:
	navigation_layers = value

func get_neighbor_distance() -> float:
	return neighbor_distance

func set_neighbor_distance(value: float) -> void:
	neighbor_distance = value

func get_path_desired_distance() -> float:
	return path_desired_distance

func set_path_desired_distance(value: float) -> void:
	path_desired_distance = value

func get_path_max_distance() -> float:
	return path_max_distance

func set_path_max_distance(value: float) -> void:
	path_max_distance = value

func get_path_metadata_flags() -> int:
	return path_metadata_flags

func set_path_metadata_flags(value: int) -> void:
	path_metadata_flags = value

func get_path_postprocessing() -> int:
	return path_postprocessing

func set_path_postprocessing(value: int) -> void:
	path_postprocessing = value

func get_pathfinding_algorithm() -> int:
	return pathfinding_algorithm

func set_pathfinding_algorithm(value: int) -> void:
	pathfinding_algorithm = value

func get_radius() -> float:
	return radius

func set_radius(value: float) -> void:
	radius = value

func get_simplify_epsilon() -> float:
	return simplify_epsilon

func set_simplify_epsilon(value: float) -> void:
	simplify_epsilon = value

func get_simplify_path() -> bool:
	return simplify_path

func set_simplify_path(value: bool) -> void:
	simplify_path = value

func get_target_desired_distance() -> float:
	return target_desired_distance

func set_target_desired_distance(value: float) -> void:
	target_desired_distance = value

func get_target_position() -> Vector2:
	return target_position

func set_target_position(value: Vector2) -> void:
	target_position = value

func get_time_horizon_agents() -> float:
	return time_horizon_agents

func set_time_horizon_agents(value: float) -> void:
	time_horizon_agents = value

func get_time_horizon_obstacles() -> float:
	return time_horizon_obstacles

func set_time_horizon_obstacles(value: float) -> void:
	time_horizon_obstacles = value

func get_velocity() -> Vector2:
	return velocity

func set_velocity(value: Vector2) -> void:
	velocity = value

