extends Font
class_name FontVariation

## A variation of a font with additional settings.
##
## Provides OpenType variations, simulated bold / slant, and additional font settings like OpenType features and extra spacing.
## To use simulated bold font variant:
## [codeblocks]
## [gdscript]
## var fv = FontVariation.new()
## fv.base_font = load("res://BarlowCondensed-Regular.ttf")
## fv.variation_embolden = 1.2
## $Label.add_theme_font_override("font", fv)
## $Label.add_theme_font_size_override("font_size", 64)
## [/gdscript]
## [csharp]
## var fv = new FontVariation();
## fv.SetBaseFont(ResourceLoader.Load<FontFile>("res://BarlowCondensed-Regular.ttf"));
## fv.SetVariationEmbolden(1.2);
## GetNode("Label").AddThemeFontOverride("font", fv);
## GetNode("Label").AddThemeFontSizeOverride("font_size", 64);
## [/csharp]
## [/codeblocks]
## To set the coordinate of multiple variation axes:
## [codeblock]
## var fv = FontVariation.new();
## var ts = TextServerManager.get_primary_interface()
## fv.base_font = load("res://BarlowCondensed-Regular.ttf")
## fv.variation_opentype = { ts.name_to_tag("wght"): 900, ts.name_to_tag("custom_hght"): 900 }
## [/codeblock]



## Base font used to create a variation. If not set, default [Theme] font is used.
var base_font: Font:
	get = get_base_font, set = set_base_font

## Extra baseline offset (as a fraction of font height).
var baseline_offset: float:
	get = get_baseline_offset, set = set_baseline_offset

## A set of OpenType feature tags. More info: [url=https://docs.microsoft.com/en-us/typography/opentype/spec/featuretags]OpenType feature tags[/url].
var opentype_features: Dictionary:
	get = get_opentype_features, set = set_opentype_features

## Extra spacing at the bottom of the line in pixels.
var spacing_bottom: int:
	get = get_spacing, set = set_spacing

## Extra spacing between graphical glyphs.
var spacing_glyph: int:
	get = get_spacing, set = set_spacing

## Extra width of the space glyphs.
var spacing_space: int:
	get = get_spacing, set = set_spacing

## Extra spacing at the top of the line in pixels.
var spacing_top: int:
	get = get_spacing, set = set_spacing

## If is not equal to zero, emboldens the font outlines. Negative values reduce the outline thickness.
## [b]Note:[/b] Emboldened fonts might have self-intersecting outlines, which will prevent MSDF fonts and [TextMesh] from working correctly.
var variation_embolden: float:
	get = get_variation_embolden, set = set_variation_embolden

## Active face index in the TrueType / OpenType collection file.
var variation_face_index: int:
	get = get_variation_face_index, set = set_variation_face_index

## Font OpenType variation coordinates. More info: [url=https://docs.microsoft.com/en-us/typography/opentype/spec/dvaraxisreg]OpenType variation tags[/url].
## [b]Note:[/b] This [Dictionary] uses OpenType tags as keys. Variation axes can be identified both by tags ([int], e.g. [code]0x77678674[/code]) and names ([String], e.g. [code]wght[/code]). Some axes might be accessible by multiple names. For example, [code]wght[/code] refers to the same axis as [code]weight[/code]. Tags on the other hand are unique. To convert between names and tags, use [method TextServer.name_to_tag] and [method TextServer.tag_to_name].
## [b]Note:[/b] To get available variation axes of a font, use [method Font.get_supported_variation_list].
var variation_opentype: Dictionary:
	get = get_variation_opentype, set = set_variation_opentype

## 2D transform, applied to the font outlines, can be used for slanting, flipping and rotating glyphs.
## For example, to simulate italic typeface by slanting, apply the following transform [code]Transform2D(1.0, slant, 0.0, 1.0, 0.0, 0.0)[/code].
var variation_transform: Transform2D:
	get = get_variation_transform, set = set_variation_transform



## Sets the spacing for [param spacing] (see [enum TextServer.SpacingType]) to [param value] in pixels (not relative to the font size).
func set_spacing(spacing: int, value: int) -> void:
	pass;


func get_base_font() -> Font:
	return base_font

func set_base_font(value: Font) -> void:
	base_font = value

func get_baseline_offset() -> float:
	return baseline_offset

func set_baseline_offset(value: float) -> void:
	baseline_offset = value

func get_opentype_features() -> Dictionary:
	return opentype_features

func set_opentype_features(value: Dictionary) -> void:
	opentype_features = value

func get_spacing() -> int:
	return spacing_bottom

func set_spacing(value: int) -> void:
	spacing_bottom = value

func get_spacing() -> int:
	return spacing_glyph

func set_spacing(value: int) -> void:
	spacing_glyph = value

func get_spacing() -> int:
	return spacing_space

func set_spacing(value: int) -> void:
	spacing_space = value

func get_spacing() -> int:
	return spacing_top

func set_spacing(value: int) -> void:
	spacing_top = value

func get_variation_embolden() -> float:
	return variation_embolden

func set_variation_embolden(value: float) -> void:
	variation_embolden = value

func get_variation_face_index() -> int:
	return variation_face_index

func set_variation_face_index(value: int) -> void:
	variation_face_index = value

func get_variation_opentype() -> Dictionary:
	return variation_opentype

func set_variation_opentype(value: Dictionary) -> void:
	variation_opentype = value

func get_variation_transform() -> Transform2D:
	return variation_transform

func set_variation_transform(value: Transform2D) -> void:
	variation_transform = value

