extends Object
class_name Engine

## Provides access to engine properties.
##
## The [Engine] singleton allows you to query and modify the project's run-time parameters, such as frames per second, time scale, and others. It also stores information about the current build of Godot, such as the current version.



## The maximum number of frames that can be rendered every second (FPS). A value of [code]0[/code] means the framerate is uncapped.
## Limiting the FPS can be useful to reduce the host machine's power consumption, which reduces heat, noise emissions, and improves battery life.
## If [member ProjectSettings.display/window/vsync/vsync_mode] is [b]Enabled[/b] or [b]Adaptive[/b], the setting takes precedence and the max FPS number cannot exceed the monitor's refresh rate.
## If [member ProjectSettings.display/window/vsync/vsync_mode] is [b]Enabled[/b], on monitors with variable refresh rate enabled (G-Sync/FreeSync), using an FPS limit a few frames lower than the monitor's refresh rate will [url=https://blurbusters.com/howto-low-lag-vsync-on/]reduce input lag while avoiding tearing[/url].
## See also [member physics_ticks_per_second] and [member ProjectSettings.application/run/max_fps].
## [b]Note:[/b] The actual number of frames per second may still be below this value if the CPU or GPU cannot keep up with the project's logic and rendering.
## [b]Note:[/b] If [member ProjectSettings.display/window/vsync/vsync_mode] is [b]Disabled[/b], limiting the FPS to a high value that can be consistently reached on the system can reduce input lag compared to an uncapped framerate. Since this works by ensuring the GPU load is lower than 100%, this latency reduction is only effective in GPU-bottlenecked scenarios, not CPU-bottlenecked scenarios.
var max_fps: int:
	get = get_max_fps, set = set_max_fps

## The maximum number of physics steps that can be simulated each rendered frame.
## [b]Note:[/b] The default value is tuned to prevent expensive physics simulations from triggering even more expensive simulations indefinitely. However, the game will appear to slow down if the rendering FPS is less than [code]1 / max_physics_steps_per_frame[/code] of [member physics_ticks_per_second]. This occurs even if [code]delta[/code] is consistently used in physics calculations. To avoid this, increase [member max_physics_steps_per_frame] if you have increased [member physics_ticks_per_second] significantly above its default value.
var max_physics_steps_per_frame: int:
	get = get_max_physics_steps_per_frame, set = set_max_physics_steps_per_frame

## How much physics ticks are synchronized with real time. If [code]0[/code] or less, the ticks are fully synchronized. Higher values cause the in-game clock to deviate more from the real clock, but they smooth out framerate jitters.
## [b]Note:[/b] The default value of [code]0.5[/code] should be good enough for most cases; values above [code]2[/code] could cause the game to react to dropped frames with a noticeable delay and are not recommended.
## [b]Note:[/b] When using a custom physics interpolation solution, or within a network game, it's recommended to disable the physics jitter fix by setting this property to [code]0[/code].
var physics_jitter_fix: float:
	get = get_physics_jitter_fix, set = set_physics_jitter_fix

## The number of fixed iterations per second. This controls how often physics simulation and [method Node._physics_process] methods are run. This value should generally always be set to [code]60[/code] or above, as Godot doesn't interpolate the physics step. As a result, values lower than [code]60[/code] will look stuttery. This value can be increased to make input more reactive or work around collision tunneling issues, but keep in mind doing so will increase CPU usage. See also [member max_fps] and [member ProjectSettings.physics/common/physics_ticks_per_second].
## [b]Note:[/b] Only [member max_physics_steps_per_frame] physics ticks may be simulated per rendered frame at most. If more physics ticks have to be simulated per rendered frame to keep up with rendering, the project will appear to slow down (even if [code]delta[/code] is used consistently in physics calculations). Therefore, it is recommended to also increase [member max_physics_steps_per_frame] if increasing [member physics_ticks_per_second] significantly above its default value.
var physics_ticks_per_second: int:
	get = get_physics_ticks_per_second, set = set_physics_ticks_per_second

## If [code]false[/code], stops printing error and warning messages to the console and editor Output log. This can be used to hide error and warning messages during unit test suite runs. This property is equivalent to the [member ProjectSettings.application/run/disable_stderr] project setting.
## [b]Note:[/b] This property does not impact the editor's Errors tab when running a project from the editor.
## [b]Warning:[/b] If set to [code]false[/code] anywhere in the project, important error messages may be hidden even if they are emitted from other scripts. In a [code]@tool[/code] script, this will also impact the editor itself. Do [i]not[/i] report bugs before ensuring error messages are enabled (as they are by default).
var print_error_messages: bool:
	get = is_printing_error_messages, set = set_print_error_messages

## If [code]false[/code], stops printing messages (for example using [method @GlobalScope.print]) to the console, log files, and editor Output log. This property is equivalent to the [member ProjectSettings.application/run/disable_stdout] project setting.
## [b]Note:[/b] This does not stop printing errors or warnings produced by scripts to the console or log files, for more details see [member print_error_messages].
var print_to_stdout: bool:
	get = is_printing_to_stdout, set = set_print_to_stdout

## The speed multiplier at which the in-game clock updates, compared to real time. For example, if set to [code]2.0[/code] the game runs twice as fast, and if set to [code]0.5[/code] the game runs half as fast.
## This value affects [Timer], [SceneTreeTimer], and all other simulations that make use of [code]delta[/code] time (such as [method Node._process] and [method Node._physics_process]).
## [b]Note:[/b] It's recommended to keep this property above [code]0.0[/code], as the game may behave unexpectedly otherwise.
## [b]Note:[/b] This does not affect audio playback speed. Use [member AudioServer.playback_speed_scale] to adjust audio playback speed independently of [member Engine.time_scale].
## [b]Note:[/b] This does not automatically adjust [member physics_ticks_per_second]. With values above [code]1.0[/code] physics simulation may become less precise, as each physics tick will stretch over a larger period of engine time. If you're modifying [member Engine.time_scale] to speed up simulation by a large factor, consider also increasing [member physics_ticks_per_second] to make the simulation more reliable.
var time_scale: float:
	get = get_time_scale, set = set_time_scale



## Returns the name of the CPU architecture the Godot binary was built for. Possible return values include [code]"x86_64"[/code], [code]"x86_32"[/code], [code]"arm64"[/code], [code]"arm32"[/code], [code]"rv64"[/code], [code]"riscv"[/code], [code]"ppc64"[/code], [code]"ppc"[/code], [code]"wasm64"[/code], and [code]"wasm32"[/code].
## To detect whether the current build is 64-bit, or the type of architecture, don't use the architecture name. Instead, use [method OS.has_feature] to check for the [code]"64"[/code] feature tag, or tags such as [code]"x86"[/code] or [code]"arm"[/code]. See the [url=$DOCS_URL/tutorials/export/feature_tags.html]Feature Tags[/url] documentation for more details.
## [b]Note:[/b] This method does [i]not[/i] return the name of the system's CPU architecture (like [method OS.get_processor_name]). For example, when running an [code]x86_32[/code] Godot binary on an [code]x86_64[/code] system, the returned value will still be [code]"x86_32"[/code].
func get_architecture_name() -> String:
	pass;

## Returns the engine author information as a [Dictionary], where each entry is an [Array] of strings with the names of notable contributors to the Godot Engine: [code]lead_developers[/code], [code]founders[/code], [code]project_managers[/code], and [code]developers[/code].
func get_author_info() -> Dictionary:
	pass;

## Returns an [Array] of dictionaries with copyright information for every component of Godot's source code.
## Every [Dictionary] contains a [code]name[/code] identifier, and a [code]parts[/code] array of dictionaries. It describes the component in detail with the following entries:
## - [code]files[/code] - [Array] of file paths from the source code affected by this component;
## - [code]copyright[/code] - [Array] of owners of this component;
## - [code]license[/code] - The license applied to this component (such as "[url=https://en.wikipedia.org/wiki/MIT_License#Ambiguity_and_variants]Expat[/url]" or "[url=https://creativecommons.org/licenses/by/4.0/]CC-BY-4.0[/url]").
func get_copyright_info() -> Array[Dictionary]:
	pass;

## Returns a [Dictionary] of categorized donor names. Each entry is an [Array] of strings:
## {[code]platinum_sponsors[/code], [code]gold_sponsors[/code], [code]silver_sponsors[/code], [code]bronze_sponsors[/code], [code]mini_sponsors[/code], [code]gold_donors[/code], [code]silver_donors[/code], [code]bronze_donors[/code]}
func get_donor_info() -> Dictionary:
	pass;

## Returns the total number of frames drawn since the engine started.
## [b]Note:[/b] On headless platforms, or if rendering is disabled with [code]--disable-render-loop[/code] via command line, this method always returns [code]0[/code]. See also [method get_process_frames].
func get_frames_drawn() -> int:
	pass;

## Returns the average frames rendered every second (FPS), also known as the framerate.
func get_frames_per_second() -> float:
	pass;

## Returns a [Dictionary] of licenses used by Godot and included third party components. Each entry is a license name (such as "[url=https://en.wikipedia.org/wiki/MIT_License#Ambiguity_and_variants]Expat[/url]") and its associated text.
func get_license_info() -> Dictionary:
	pass;

## Returns the full Godot license text.
func get_license_text() -> String:
	pass;

## Returns the instance of the [MainLoop]. This is usually the main [SceneTree] and is the same as [method Node.get_tree].
## [b]Note:[/b] The type instantiated as the main loop can changed with [member ProjectSettings.application/run/main_loop_type].
func get_main_loop() -> MainLoop:
	pass;

## Returns the total number of frames passed since the engine started. This number is increased every [b]physics frame[/b]. See also [method get_process_frames].
## This method can be used to run expensive logic less often without relying on a [Timer]:
## [codeblocks]
## [gdscript]
## func _physics_process(_delta):
## if Engine.get_physics_frames() % 2 == 0:
## pass # Run expensive logic only once every 2 physics frames here.
## [/gdscript]
## [csharp]
## public override void _PhysicsProcess(double delta)
## {
## base._PhysicsProcess(delta);
## if (Engine.GetPhysicsFrames() % 2 == 0)
## {
## // Run expensive logic only once every 2 physics frames here.
## }
## }
## [/csharp]
## [/codeblocks]
func get_physics_frames() -> int:
	pass;

## Returns the fraction through the current physics tick we are at the time of rendering the frame. This can be used to implement fixed timestep interpolation.
func get_physics_interpolation_fraction() -> float:
	pass;

## Returns the total number of frames passed since the engine started. This number is increased every [b]process frame[/b], regardless of whether the render loop is enabled. See also [method get_frames_drawn] and [method get_physics_frames].
## This method can be used to run expensive logic less often without relying on a [Timer]:
## [codeblocks]
## [gdscript]
## func _process(_delta):
## if Engine.get_process_frames() % 5 == 0:
## pass # Run expensive logic only once every 5 process (render) frames here.
## [/gdscript]
## [csharp]
## public override void _Process(double delta)
## {
## base._Process(delta);
## if (Engine.GetProcessFrames() % 5 == 0)
## {
## // Run expensive logic only once every 5 process (render) frames here.
## }
## }
## [/csharp]
## [/codeblocks]
func get_process_frames() -> int:
	pass;

## Returns an instance of a [ScriptLanguage] with the given [param index].
func get_script_language(index: int) -> ScriptLanguage:
	pass;

## Returns the number of available script languages. Use with [method get_script_language].
func get_script_language_count() -> int:
	pass;

## Returns the global singleton with the given [param name], or [code]null[/code] if it does not exist. Often used for plugins. See also [method has_singleton] and [method get_singleton_list].
## [b]Note:[/b] Global singletons are not the same as autoloaded nodes, which are configurable in the project settings.
func get_singleton(name: StringName) -> Object:
	pass;

## Returns a list of names of all available global singletons. See also [method get_singleton].
func get_singleton_list() -> PackedStringArray:
	pass;

## Returns the current engine version information as a [Dictionary] containing the following entries:
## - [code]major[/code] - Major version number as an int;
## - [code]minor[/code] - Minor version number as an int;
## - [code]patch[/code] - Patch version number as an int;
## - [code]hex[/code] - Full version encoded as a hexadecimal int with one byte (2 hex digits) per number (see example below);
## - [code]status[/code] - Status (such as "beta", "rc1", "rc2", "stable", etc.) as a String;
## - [code]build[/code] - Build name (e.g. "custom_build") as a String;
## - [code]hash[/code] - Full Git commit hash as a String;
## - [code]timestamp[/code] - Holds the Git commit date UNIX timestamp in seconds as an int, or [code]0[/code] if unavailable;
## - [code]string[/code] - [code]major[/code], [code]minor[/code], [code]patch[/code], [code]status[/code], and [code]build[/code] in a single String.
## The [code]hex[/code] value is encoded as follows, from left to right: one byte for the major, one byte for the minor, one byte for the patch version. For example, "3.1.12" would be [code]0x03010C[/code].
## [b]Note:[/b] The [code]hex[/code] value is still an [int] internally, and printing it will give you its decimal representation, which is not particularly meaningful. Use hexadecimal literals for quick version comparisons from code:
## [codeblocks]
## [gdscript]
## if Engine.get_version_info().hex >= 0x040100:
## pass # Do things specific to version 4.1 or later.
## else:
## pass # Do things specific to versions before 4.1.
## [/gdscript]
## [csharp]
## if ((int)Engine.GetVersionInfo()["hex"] >= 0x040100)
## {
## // Do things specific to version 4.1 or later.
## }
## else
## {
## // Do things specific to versions before 4.1.
## }
## [/csharp]
## [/codeblocks]
func get_version_info() -> Dictionary:
	pass;

## Returns the path to the [MovieWriter]'s output file, or an empty string if the engine wasn't started in Movie Maker mode. The default path can be changed in [member ProjectSettings.editor/movie_writer/movie_file].
func get_write_movie_path() -> String:
	pass;

## Returns [code]true[/code] if a singleton with the given [param name] exists in the global scope. See also [method get_singleton].
## [codeblocks]
## [gdscript]
## print(Engine.has_singleton("OS"))          # Prints true
## print(Engine.has_singleton("Engine"))      # Prints true
## print(Engine.has_singleton("AudioServer")) # Prints true
## print(Engine.has_singleton("Unknown"))     # Prints false
## [/gdscript]
## [csharp]
## GD.Print(Engine.HasSingleton("OS"));          // Prints True
## GD.Print(Engine.HasSingleton("Engine"));      // Prints True
## GD.Print(Engine.HasSingleton("AudioServer")); // Prints True
## GD.Print(Engine.HasSingleton("Unknown"));     // Prints False
## [/csharp]
## [/codeblocks]
## [b]Note:[/b] Global singletons are not the same as autoloaded nodes, which are configurable in the project settings.
func has_singleton(name: StringName) -> bool:
	pass;

## Returns [code]true[/code] if the script is currently running inside the editor, otherwise returns [code]false[/code]. This is useful for [code]@tool[/code] scripts to conditionally draw editor helpers, or prevent accidentally running "game" code that would affect the scene state while in the editor:
## [codeblocks]
## [gdscript]
## if Engine.is_editor_hint():
## draw_gizmos()
## else:
## simulate_physics()
## [/gdscript]
## [csharp]
## if (Engine.IsEditorHint())
## DrawGizmos();
## else
## SimulatePhysics();
## [/csharp]
## [/codeblocks]
## See [url=$DOCS_URL/tutorials/plugins/running_code_in_the_editor.html]Running code in the editor[/url] in the documentation for more information.
## [b]Note:[/b] To detect whether the script is running on an editor [i]build[/i] (such as when pressing [kbd]F5[/kbd]), use [method OS.has_feature] with the [code]"editor"[/code] argument instead. [code]OS.has_feature("editor")[/code] evaluate to [code]true[/code] both when the script is running in the editor and when running the project from the editor, but returns [code]false[/code] when run from an exported project.
func is_editor_hint() -> bool:
	pass;

## Returns [code]true[/code] if the engine is running embedded in the editor. This is useful to prevent attempting to update window mode or window flags that are not supported when running the project embedded in the editor.
func is_embedded_in_editor() -> bool:
	pass;

## Returns [code]true[/code] if the engine is inside the fixed physics process step of the main loop.
## [codeblock]
## func _enter_tree():
## # Depending on when the node is added to the tree,
## # prints either "true" or "false".
## print(Engine.is_in_physics_frame())
## func _process(delta):
## print(Engine.is_in_physics_frame()) # Prints false
## func _physics_process(delta):
## print(Engine.is_in_physics_frame()) # Prints true
## [/codeblock]
func is_in_physics_frame() -> bool:
	pass;

## Registers a [ScriptLanguage] instance to be available with [code]ScriptServer[/code].
## Returns:
## - [constant OK] on success;
## - [constant ERR_UNAVAILABLE] if [code]ScriptServer[/code] has reached the limit and cannot register any new language;
## - [constant ERR_ALREADY_EXISTS] if [code]ScriptServer[/code] already contains a language with similar extension/name/type.
func register_script_language(language: ScriptLanguage) -> int:
	pass;

## Registers the given [Object] [param instance] as a singleton, available globally under [param name]. Useful for plugins.
func register_singleton(name: StringName, instance: Object) -> void:
	pass;

## Unregisters the [ScriptLanguage] instance from [code]ScriptServer[/code].
## Returns:
## - [constant OK] on success;
## - [constant ERR_DOES_NOT_EXIST] if the language is not registered in [code]ScriptServer[/code].
func unregister_script_language(language: ScriptLanguage) -> int:
	pass;

## Removes the singleton registered under [param name]. The singleton object is [i]not[/i] freed. Only works with user-defined singletons registered with [method register_singleton].
func unregister_singleton(name: StringName) -> void:
	pass;


func get_max_fps() -> int:
	return max_fps

func set_max_fps(value: int) -> void:
	max_fps = value

func get_max_physics_steps_per_frame() -> int:
	return max_physics_steps_per_frame

func set_max_physics_steps_per_frame(value: int) -> void:
	max_physics_steps_per_frame = value

func get_physics_jitter_fix() -> float:
	return physics_jitter_fix

func set_physics_jitter_fix(value: float) -> void:
	physics_jitter_fix = value

func get_physics_ticks_per_second() -> int:
	return physics_ticks_per_second

func set_physics_ticks_per_second(value: int) -> void:
	physics_ticks_per_second = value

func is_printing_error_messages() -> bool:
	return print_error_messages

func set_print_error_messages(value: bool) -> void:
	print_error_messages = value

func is_printing_to_stdout() -> bool:
	return print_to_stdout

func set_print_to_stdout(value: bool) -> void:
	print_to_stdout = value

func get_time_scale() -> float:
	return time_scale

func set_time_scale(value: float) -> void:
	time_scale = value

