extends Node
class_name Viewport

## Abstract base class for viewports. Encapsulates drawing and interaction with a game world.
##
## A [Viewport] creates a different view into the screen, or a sub-view inside another viewport. Child 2D nodes will display on it, and child Camera3D 3D nodes will render on it too.
## Optionally, a viewport can have its own 2D or 3D world, so it doesn't share what it draws with other viewports.
## Viewports can also choose to be audio listeners, so they generate positional audio depending on a 2D or 3D camera child of it.
## Also, viewports can be assigned to different screens in case the devices have multiple screens.
## Finally, viewports can also behave as render targets, in which case they will not be visible unless the associated texture is used to draw.
##
## @tutorial(Using Viewports): https://docs.godotengine.org/en/stable/tutorials/rendering/viewports.html
## @tutorial(Viewport and canvas transforms): https://docs.godotengine.org/en/stable/tutorials/2d/2d_transforms.html
## @tutorial(GUI in 3D Viewport Demo): https://godotengine.org/asset-library/asset/2807
## @tutorial(3D in 2D Viewport Demo): https://godotengine.org/asset-library/asset/2804
## @tutorial(2D in 3D Viewport Demo): https://godotengine.org/asset-library/asset/2803
## @tutorial(Screen Capture Demo): https://godotengine.org/asset-library/asset/2808
## @tutorial(Dynamic Split Screen Demo): https://godotengine.org/asset-library/asset/2806
## @tutorial(3D Resolution Scaling Demo): https://godotengine.org/asset-library/asset/2805


## Emitted when a Control node grabs keyboard focus.
## [b]Note:[/b] A Control node losing focus doesn't cause this signal to be emitted.
signal gui_focus_changed(node: Control)
## Emitted when the size of the viewport is changed, whether by resizing of window, or some other means.
signal size_changed
## This quadrant will not be used.
## This quadrant will only be used by one shadow map.
## This quadrant will be split in 4 and used by up to 4 shadow maps.
## This quadrant will be split 16 ways and used by up to 16 shadow maps.
## This quadrant will be split 64 ways and used by up to 64 shadow maps.
## This quadrant will be split 256 ways and used by up to 256 shadow maps. Unless the [member positional_shadow_atlas_size] is very high, the shadows in this quadrant will be very low resolution.
## This quadrant will be split 1024 ways and used by up to 1024 shadow maps. Unless the [member positional_shadow_atlas_size] is very high, the shadows in this quadrant will be very low resolution.
## Represents the size of the [enum PositionalShadowAtlasQuadrantSubdiv] enum.
## Use bilinear scaling for the viewport's 3D buffer. The amount of scaling can be set using [member scaling_3d_scale]. Values less than [code]1.0[/code] will result in undersampling while values greater than [code]1.0[/code] will result in supersampling. A value of [code]1.0[/code] disables scaling.
## Use AMD FidelityFX Super Resolution 1.0 upscaling for the viewport's 3D buffer. The amount of scaling can be set using [member scaling_3d_scale]. Values less than [code]1.0[/code] will be result in the viewport being upscaled using FSR. Values greater than [code]1.0[/code] are not supported and bilinear downsampling will be used instead. A value of [code]1.0[/code] disables scaling.
## Use AMD FidelityFX Super Resolution 2.2 upscaling for the viewport's 3D buffer. The amount of scaling can be set using [member Viewport.scaling_3d_scale]. Values less than [code]1.0[/code] will be result in the viewport being upscaled using FSR2. Values greater than [code]1.0[/code] are not supported and bilinear downsampling will be used instead. A value of [code]1.0[/code] will use FSR2 at native resolution as a TAA solution.
## Use the [url=https://developer.apple.com/documentation/metalfx/mtlfxspatialscaler#overview]MetalFX spatial upscaler[/url] for the viewport's 3D buffer.
## The amount of scaling can be set using [member scaling_3d_scale].
## Values less than [code]1.0[/code] will be result in the viewport being upscaled using MetalFX. Values greater than [code]1.0[/code] are not supported and bilinear downsampling will be used instead. A value of [code]1.0[/code] disables scaling.
## More information: [url=https://developer.apple.com/documentation/metalfx]MetalFX[/url].
## [b]Note:[/b] Only supported when the Metal rendering driver is in use, which limits this scaling mode to macOS and iOS.
## Use the [url=https://developer.apple.com/documentation/metalfx/mtlfxtemporalscaler#overview]MetalFX temporal upscaler[/url] for the viewport's 3D buffer.
## The amount of scaling can be set using [member scaling_3d_scale]. To determine the minimum input scale, use the [method RenderingDevice.limit_get] method with [constant RenderingDevice.LIMIT_METALFX_TEMPORAL_SCALER_MIN_SCALE].
## Values less than [code]1.0[/code] will be result in the viewport being upscaled using MetalFX. Values greater than [code]1.0[/code] are not supported and bilinear downsampling will be used instead. A value of [code]1.0[/code] will use MetalFX at native resolution as a TAA solution.
## More information: [url=https://developer.apple.com/documentation/metalfx]MetalFX[/url].
## [b]Note:[/b] Only supported when the Metal rendering driver is in use, which limits this scaling mode to macOS and iOS.
## Represents the size of the [enum Scaling3DMode] enum.
## Multisample antialiasing mode disabled. This is the default value, and is also the fastest setting.
## Use 2× Multisample Antialiasing. This has a moderate performance cost. It helps reduce aliasing noticeably, but 4× MSAA still looks substantially better.
## Use 4× Multisample Antialiasing. This has a significant performance cost, and is generally a good compromise between performance and quality.
## Use 8× Multisample Antialiasing. This has a very high performance cost. The difference between 4× and 8× MSAA may not always be visible in real gameplay conditions. Likely unsupported on low-end and older hardware.
## Represents the size of the [enum MSAA] enum.
## Anisotropic filtering is disabled.
## Use 2× anisotropic filtering.
## Use 4× anisotropic filtering. This is the default value.
## Use 8× anisotropic filtering.
## Use 16× anisotropic filtering.
## Represents the size of the [enum AnisotropicFiltering] enum.
## Do not perform any antialiasing in the full screen post-process.
## Use fast approximate antialiasing. FXAA is a popular screen-space antialiasing method, which is fast but will make the image look blurry, especially at lower resolutions. It can still work relatively well at large resolutions such as 1440p and 4K.
## Represents the size of the [enum ScreenSpaceAA] enum.
## Amount of objects in frame.
## Amount of vertices in frame.
## Amount of draw calls in frame.
## Represents the size of the [enum RenderInfo] enum.
## Visible render pass (excluding shadows).
## Shadow render pass. Objects will be rendered several times depending on the number of amounts of lights with shadows and the number of directional shadow splits.
## Canvas item rendering. This includes all 2D rendering.
## Represents the size of the [enum RenderInfoType] enum.
## Objects are displayed normally.
## Objects are displayed without light information.
## Objects are displayed without textures and only with lighting information.
## Objects are displayed semi-transparent with additive blending so you can see where they are drawing over top of one another. A higher overdraw means you are wasting performance on drawing pixels that are being hidden behind others.
## Objects are displayed as wireframe models.
## [b]Note:[/b] [method RenderingServer.set_debug_generate_wireframes] must be called before loading any meshes for wireframes to be visible when using the Compatibility renderer.
## Objects are displayed without lighting information and their textures replaced by normal mapping.
## Objects are displayed with only the albedo value from [VoxelGI]s.
## Objects are displayed with only the lighting value from [VoxelGI]s.
## Objects are displayed with only the emission color from [VoxelGI]s.
## Draws the shadow atlas that stores shadows from [OmniLight3D]s and [SpotLight3D]s in the upper left quadrant of the [Viewport].
## Draws the shadow atlas that stores shadows from [DirectionalLight3D]s in the upper left quadrant of the [Viewport].
## Draws the scene luminance buffer (if available) in the upper left quadrant of the [Viewport].
## Draws the screen-space ambient occlusion texture instead of the scene so that you can clearly see how it is affecting objects. In order for this display mode to work, you must have [member Environment.ssao_enabled] set in your [WorldEnvironment].
## Draws the screen-space indirect lighting texture instead of the scene so that you can clearly see how it is affecting objects. In order for this display mode to work, you must have [member Environment.ssil_enabled] set in your [WorldEnvironment].
## Colors each PSSM split for the [DirectionalLight3D]s in the scene a different color so you can see where the splits are. In order, they will be colored red, green, blue, and yellow.
## Draws the decal atlas used by [Decal]s and light projector textures in the upper left quadrant of the [Viewport].
## Draws the cascades used to render signed distance field global illumination (SDFGI).
## Does nothing if the current environment's [member Environment.sdfgi_enabled] is [code]false[/code] or SDFGI is not supported on the platform.
## Draws the probes used for signed distance field global illumination (SDFGI).
## Does nothing if the current environment's [member Environment.sdfgi_enabled] is [code]false[/code] or SDFGI is not supported on the platform.
## Draws the buffer used for global illumination (GI).
## Draws all of the objects at their highest polycount, without low level of detail (LOD).
## Draws the cluster used by [OmniLight3D] nodes to optimize light rendering.
## Draws the cluster used by [SpotLight3D] nodes to optimize light rendering.
## Draws the cluster used by [Decal] nodes to optimize decal rendering.
## Draws the cluster used by [ReflectionProbe] nodes to optimize decal rendering.
## Draws the buffer used for occlusion culling.
## Draws vector lines over the viewport to indicate the movement of pixels between frames.
## Draws the internal resolution buffer of the scene before post-processing is applied.
## The texture filter reads from the nearest pixel only. This makes the texture look pixelated from up close, and grainy from a distance (due to mipmaps not being sampled).
## The texture filter blends between the nearest 4 pixels. This makes the texture look smooth from up close, and grainy from a distance (due to mipmaps not being sampled).
## The texture filter blends between the nearest 4 pixels and between the nearest 2 mipmaps (or uses the nearest mipmap if [member ProjectSettings.rendering/textures/default_filters/use_nearest_mipmap_filter] is [code]true[/code]). This makes the texture look smooth from up close, and smooth from a distance.
## Use this for non-pixel art textures that may be viewed at a low scale (e.g. due to [Camera2D] zoom or sprite scaling), as mipmaps are important to smooth out pixels that are smaller than on-screen pixels.
## The texture filter reads from the nearest pixel and blends between the nearest 2 mipmaps (or uses the nearest mipmap if [member ProjectSettings.rendering/textures/default_filters/use_nearest_mipmap_filter] is [code]true[/code]). This makes the texture look pixelated from up close, and smooth from a distance.
## Use this for non-pixel art textures that may be viewed at a low scale (e.g. due to [Camera2D] zoom or sprite scaling), as mipmaps are important to smooth out pixels that are smaller than on-screen pixels.
## Represents the size of the [enum DefaultCanvasItemTextureFilter] enum.
## Disables textures repeating. Instead, when reading UVs outside the 0-1 range, the value will be clamped to the edge of the texture, resulting in a stretched out look at the borders of the texture.
## Enables the texture to repeat when UV coordinates are outside the 0-1 range. If using one of the linear filtering modes, this can result in artifacts at the edges of a texture when the sampler filters across the edges of the texture.
## Flip the texture when repeating so that the edge lines up instead of abruptly changing.
## Represents the size of the [enum DefaultCanvasItemTextureRepeat] enum.
## The signed distance field only covers the viewport's own rectangle.
## The signed distance field is expanded to cover 20% of the viewport's size around the borders.
## The signed distance field is expanded to cover 50% of the viewport's size around the borders.
## The signed distance field is expanded to cover 100% (double) of the viewport's size around the borders.
## Represents the size of the [enum SDFOversize] enum.
## The signed distance field is rendered at full resolution.
## The signed distance field is rendered at half the resolution of this viewport.
## The signed distance field is rendered at a quarter the resolution of this viewport.
## Represents the size of the [enum SDFScale] enum.
## Variable Rate Shading is disabled.
## Variable Rate Shading uses a texture. Note, for stereoscopic use a texture atlas with a texture for each view.
## Variable Rate Shading's texture is supplied by the primary [XRInterface].
## Represents the size of the [enum VRSMode] enum.
## The input texture for variable rate shading will not be processed.
## The input texture for variable rate shading will be processed once.
## The input texture for variable rate shading will be processed each frame.
## Represents the size of the [enum VRSUpdateMode] enum.

#enum PositionalShadowAtlasQuadrantSubdiv
enum {
    SHADOW_ATLAS_QUADRANT_SUBDIV_DISABLED = 0,
    SHADOW_ATLAS_QUADRANT_SUBDIV_1 = 1,
    SHADOW_ATLAS_QUADRANT_SUBDIV_4 = 2,
    SHADOW_ATLAS_QUADRANT_SUBDIV_16 = 3,
    SHADOW_ATLAS_QUADRANT_SUBDIV_64 = 4,
    SHADOW_ATLAS_QUADRANT_SUBDIV_256 = 5,
    SHADOW_ATLAS_QUADRANT_SUBDIV_1024 = 6,
    SHADOW_ATLAS_QUADRANT_SUBDIV_MAX = 7,
}
#enum Scaling3DMode
enum {
    SCALING_3D_MODE_BILINEAR = 0,
    SCALING_3D_MODE_FSR = 1,
    SCALING_3D_MODE_FSR2 = 2,
    SCALING_3D_MODE_METALFX_SPATIAL = 3,
    SCALING_3D_MODE_METALFX_TEMPORAL = 4,
    SCALING_3D_MODE_MAX = 5,
}
#enum MSAA
enum {
    MSAA_DISABLED = 0,
    MSAA_2X = 1,
    MSAA_4X = 2,
    MSAA_8X = 3,
    MSAA_MAX = 4,
}
#enum AnisotropicFiltering
enum {
    ANISOTROPY_DISABLED = 0,
    ANISOTROPY_2X = 1,
    ANISOTROPY_4X = 2,
    ANISOTROPY_8X = 3,
    ANISOTROPY_16X = 4,
    ANISOTROPY_MAX = 5,
}
#enum ScreenSpaceAA
enum {
    SCREEN_SPACE_AA_DISABLED = 0,
    SCREEN_SPACE_AA_FXAA = 1,
    SCREEN_SPACE_AA_MAX = 2,
}
#enum RenderInfo
enum {
    RENDER_INFO_OBJECTS_IN_FRAME = 0,
    RENDER_INFO_PRIMITIVES_IN_FRAME = 1,
    RENDER_INFO_DRAW_CALLS_IN_FRAME = 2,
    RENDER_INFO_MAX = 3,
}
#enum RenderInfoType
enum {
    RENDER_INFO_TYPE_VISIBLE = 0,
    RENDER_INFO_TYPE_SHADOW = 1,
    RENDER_INFO_TYPE_CANVAS = 2,
    RENDER_INFO_TYPE_MAX = 3,
}
#enum DebugDraw
enum {
    DEBUG_DRAW_DISABLED = 0,
    DEBUG_DRAW_UNSHADED = 1,
    DEBUG_DRAW_LIGHTING = 2,
    DEBUG_DRAW_OVERDRAW = 3,
    DEBUG_DRAW_WIREFRAME = 4,
    DEBUG_DRAW_NORMAL_BUFFER = 5,
    DEBUG_DRAW_VOXEL_GI_ALBEDO = 6,
    DEBUG_DRAW_VOXEL_GI_LIGHTING = 7,
    DEBUG_DRAW_VOXEL_GI_EMISSION = 8,
    DEBUG_DRAW_SHADOW_ATLAS = 9,
    DEBUG_DRAW_DIRECTIONAL_SHADOW_ATLAS = 10,
    DEBUG_DRAW_SCENE_LUMINANCE = 11,
    DEBUG_DRAW_SSAO = 12,
    DEBUG_DRAW_SSIL = 13,
    DEBUG_DRAW_PSSM_SPLITS = 14,
    DEBUG_DRAW_DECAL_ATLAS = 15,
    DEBUG_DRAW_SDFGI = 16,
    DEBUG_DRAW_SDFGI_PROBES = 17,
    DEBUG_DRAW_GI_BUFFER = 18,
    DEBUG_DRAW_DISABLE_LOD = 19,
    DEBUG_DRAW_CLUSTER_OMNI_LIGHTS = 20,
    DEBUG_DRAW_CLUSTER_SPOT_LIGHTS = 21,
    DEBUG_DRAW_CLUSTER_DECALS = 22,
    DEBUG_DRAW_CLUSTER_REFLECTION_PROBES = 23,
    DEBUG_DRAW_OCCLUDERS = 24,
    DEBUG_DRAW_MOTION_VECTORS = 25,
    DEBUG_DRAW_INTERNAL_BUFFER = 26,
}
#enum DefaultCanvasItemTextureFilter
enum {
    DEFAULT_CANVAS_ITEM_TEXTURE_FILTER_NEAREST = 0,
    DEFAULT_CANVAS_ITEM_TEXTURE_FILTER_LINEAR = 1,
    DEFAULT_CANVAS_ITEM_TEXTURE_FILTER_LINEAR_WITH_MIPMAPS = 2,
    DEFAULT_CANVAS_ITEM_TEXTURE_FILTER_NEAREST_WITH_MIPMAPS = 3,
    DEFAULT_CANVAS_ITEM_TEXTURE_FILTER_MAX = 4,
}
#enum DefaultCanvasItemTextureRepeat
enum {
    DEFAULT_CANVAS_ITEM_TEXTURE_REPEAT_DISABLED = 0,
    DEFAULT_CANVAS_ITEM_TEXTURE_REPEAT_ENABLED = 1,
    DEFAULT_CANVAS_ITEM_TEXTURE_REPEAT_MIRROR = 2,
    DEFAULT_CANVAS_ITEM_TEXTURE_REPEAT_MAX = 3,
}
#enum SDFOversize
enum {
    SDF_OVERSIZE_100_PERCENT = 0,
    SDF_OVERSIZE_120_PERCENT = 1,
    SDF_OVERSIZE_150_PERCENT = 2,
    SDF_OVERSIZE_200_PERCENT = 3,
    SDF_OVERSIZE_MAX = 4,
}
#enum SDFScale
enum {
    SDF_SCALE_100_PERCENT = 0,
    SDF_SCALE_50_PERCENT = 1,
    SDF_SCALE_25_PERCENT = 2,
    SDF_SCALE_MAX = 3,
}
#enum VRSMode
enum {
    VRS_DISABLED = 0,
    VRS_TEXTURE = 1,
    VRS_XR = 2,
    VRS_MAX = 3,
}
#enum VRSUpdateMode
enum {
    VRS_UPDATE_DISABLED = 0,
    VRS_UPDATE_ONCE = 1,
    VRS_UPDATE_ALWAYS = 2,
    VRS_UPDATE_MAX = 3,
}
## Sets the maximum number of samples to take when using anisotropic filtering on textures (as a power of two). A higher sample count will result in sharper textures at oblique angles, but is more expensive to compute. A value of [code]0[/code] forcibly disables anisotropic filtering, even on materials where it is enabled.
## The anisotropic filtering level also affects decals and light projectors if they are configured to use anisotropic filtering. See [member ProjectSettings.rendering/textures/decals/filter] and [member ProjectSettings.rendering/textures/light_projectors/filter].
## [b]Note:[/b] In 3D, for this setting to have an effect, set [member BaseMaterial3D.texture_filter] to [constant BaseMaterial3D.TEXTURE_FILTER_LINEAR_WITH_MIPMAPS_ANISOTROPIC] or [constant BaseMaterial3D.TEXTURE_FILTER_NEAREST_WITH_MIPMAPS_ANISOTROPIC] on materials.
## [b]Note:[/b] In 2D, for this setting to have an effect, set [member CanvasItem.texture_filter] to [constant CanvasItem.TEXTURE_FILTER_LINEAR_WITH_MIPMAPS_ANISOTROPIC] or [constant CanvasItem.TEXTURE_FILTER_NEAREST_WITH_MIPMAPS_ANISOTROPIC] on the [CanvasItem] node displaying the texture (or in [CanvasTexture]). However, anisotropic filtering is rarely useful in 2D, so only enable it for textures in 2D if it makes a meaningful visual difference.
var anisotropic_filtering_level: int:
	get = get_anisotropic_filtering_level, set = set_anisotropic_filtering_level

## If [code]true[/code], the viewport will process 2D audio streams.
var audio_listener_enable_2d: bool:
	get = is_audio_listener_2d, set = set_as_audio_listener_2d

## If [code]true[/code], the viewport will process 3D audio streams.
var audio_listener_enable_3d: bool:
	get = is_audio_listener_3d, set = set_as_audio_listener_3d

## The rendering layers in which this [Viewport] renders [CanvasItem] nodes.
var canvas_cull_mask: int:
	get = get_canvas_cull_mask, set = set_canvas_cull_mask

## Sets the default filter mode used by [CanvasItem]s in this Viewport. See [enum DefaultCanvasItemTextureFilter] for options.
var canvas_item_default_texture_filter: int:
	get = get_default_canvas_item_texture_filter, set = set_default_canvas_item_texture_filter

## Sets the default repeat mode used by [CanvasItem]s in this Viewport. See [enum DefaultCanvasItemTextureRepeat] for options.
var canvas_item_default_texture_repeat: int:
	get = get_default_canvas_item_texture_repeat, set = set_default_canvas_item_texture_repeat

## The canvas transform of the viewport, useful for changing the on-screen positions of all child [CanvasItem]s. This is relative to the global canvas transform of the viewport.
var canvas_transform: Transform2D:
	get = get_canvas_transform, set = set_canvas_transform

## The overlay mode for test rendered geometry in debug purposes.
var debug_draw: int:
	get = get_debug_draw, set = set_debug_draw

## Disable 3D rendering (but keep 2D rendering).
var disable_3d: bool:
	get = is_3d_disabled, set = set_disable_3d

## Determines how sharp the upscaled image will be when using the FSR upscaling mode. Sharpness halves with every whole number. Values go from 0.0 (sharpest) to 2.0. Values above 2.0 won't make a visible difference.
## To control this property on the root viewport, set the [member ProjectSettings.rendering/scaling_3d/fsr_sharpness] project setting.
var fsr_sharpness: float:
	get = get_fsr_sharpness, set = set_fsr_sharpness

## The global canvas transform of the viewport. The canvas transform is relative to this.
var global_canvas_transform: Transform2D:
	get = get_global_canvas_transform, set = set_global_canvas_transform

## If [code]true[/code], the viewport will not receive input events.
var gui_disable_input: bool:
	get = is_input_disabled, set = set_disable_input

## If [code]true[/code], sub-windows (popups and dialogs) will be embedded inside application window as control-like nodes. If [code]false[/code], they will appear as separate windows handled by the operating system.
var gui_embed_subwindows: bool:
	get = is_embedding_subwindows, set = set_embedding_subwindows

## If [code]true[/code], the GUI controls on the viewport will lay pixel perfectly.
var gui_snap_controls_to_pixels: bool:
	get = is_snap_controls_to_pixels_enabled, set = set_snap_controls_to_pixels

## If [code]true[/code], this viewport will mark incoming input events as handled by itself. If [code]false[/code], this is instead done by the first parent viewport that is set to handle input locally.
## A [SubViewportContainer] will automatically set this property to [code]false[/code] for the [Viewport] contained inside of it.
## See also [method set_input_as_handled] and [method is_input_handled].
var handle_input_locally: bool:
	get = is_handling_input_locally, set = set_handle_input_locally

## The automatic LOD bias to use for meshes rendered within the [Viewport] (this is analogous to [member ReflectionProbe.mesh_lod_threshold]). Higher values will use less detailed versions of meshes that have LOD variations generated. If set to [code]0.0[/code], automatic LOD is disabled. Increase [member mesh_lod_threshold] to improve performance at the cost of geometry detail.
## To control this property on the root viewport, set the [member ProjectSettings.rendering/mesh_lod/lod_change/threshold_pixels] project setting.
## [b]Note:[/b] [member mesh_lod_threshold] does not affect [GeometryInstance3D] visibility ranges (also known as "manual" LOD or hierarchical LOD).
var mesh_lod_threshold: float:
	get = get_mesh_lod_threshold, set = set_mesh_lod_threshold

## The multisample antialiasing mode for 2D/Canvas rendering. A higher number results in smoother edges at the cost of significantly worse performance. A value of [constant Viewport.MSAA_2X] or [constant Viewport.MSAA_4X] is best unless targeting very high-end systems. This has no effect on shader-induced aliasing or texture aliasing.
## See also [member ProjectSettings.rendering/anti_aliasing/quality/msaa_2d] and [method RenderingServer.viewport_set_msaa_2d].
var msaa_2d: int:
	get = get_msaa_2d, set = set_msaa_2d

## The multisample antialiasing mode for 3D rendering. A higher number results in smoother edges at the cost of significantly worse performance. A value of [constant Viewport.MSAA_2X] or [constant Viewport.MSAA_4X] is best unless targeting very high-end systems. See also bilinear scaling 3D [member scaling_3d_mode] for supersampling, which provides higher quality but is much more expensive. This has no effect on shader-induced aliasing or texture aliasing.
## See also [member ProjectSettings.rendering/anti_aliasing/quality/msaa_3d] and [method RenderingServer.viewport_set_msaa_3d].
var msaa_3d: int:
	get = get_msaa_3d, set = set_msaa_3d

## If [code]true[/code], the viewport will use a unique copy of the [World3D] defined in [member world_3d].
var own_world_3d: bool:
	get = is_using_own_world_3d, set = set_use_own_world_3d

var physics_interpolation_mode: int:
	get = get_physics_interpolation_mode, set = set_physics_interpolation_mode

## If [code]true[/code], the objects rendered by viewport become subjects of mouse picking process.
## [b]Note:[/b] The number of simultaneously pickable objects is limited to 64 and they are selected in a non-deterministic order, which can be different in each picking process.
var physics_object_picking: bool:
	get = get_physics_object_picking, set = set_physics_object_picking

## If [code]true[/code], the input_event signal will only be sent to one physics object in the mouse picking process. If you want to get the top object only, you must also enable [member physics_object_picking_sort].
## If [code]false[/code], an input_event signal will be sent to all physics objects in the mouse picking process.
## This applies to 2D CanvasItem object picking only.
var physics_object_picking_first_only: bool:
	get = get_physics_object_picking_first_only, set = set_physics_object_picking_first_only

## If [code]true[/code], objects receive mouse picking events sorted primarily by their [member CanvasItem.z_index] and secondarily by their position in the scene tree. If [code]false[/code], the order is undetermined.
## [b]Note:[/b] This setting is disabled by default because of its potential expensive computational cost.
## [b]Note:[/b] Sorting happens after selecting the pickable objects. Because of the limitation of 64 simultaneously pickable objects, it is not guaranteed that the object with the highest [member CanvasItem.z_index] receives the picking event.
var physics_object_picking_sort: bool:
	get = get_physics_object_picking_sort, set = set_physics_object_picking_sort

## Use 16 bits for the omni/spot shadow depth map. Enabling this results in shadows having less precision and may result in shadow acne, but can lead to performance improvements on some devices.
var positional_shadow_atlas_16_bits: bool:
	get = get_positional_shadow_atlas_16_bits, set = set_positional_shadow_atlas_16_bits

## The subdivision amount of the first quadrant on the shadow atlas.
var positional_shadow_atlas_quad_0: int:
	get = get_positional_shadow_atlas_quadrant_subdiv, set = set_positional_shadow_atlas_quadrant_subdiv

## The subdivision amount of the second quadrant on the shadow atlas.
var positional_shadow_atlas_quad_1: int:
	get = get_positional_shadow_atlas_quadrant_subdiv, set = set_positional_shadow_atlas_quadrant_subdiv

## The subdivision amount of the third quadrant on the shadow atlas.
var positional_shadow_atlas_quad_2: int:
	get = get_positional_shadow_atlas_quadrant_subdiv, set = set_positional_shadow_atlas_quadrant_subdiv

## The subdivision amount of the fourth quadrant on the shadow atlas.
var positional_shadow_atlas_quad_3: int:
	get = get_positional_shadow_atlas_quadrant_subdiv, set = set_positional_shadow_atlas_quadrant_subdiv

## The shadow atlas' resolution (used for omni and spot lights). The value is rounded up to the nearest power of 2.
## [b]Note:[/b] If this is set to [code]0[/code], no positional shadows will be visible at all. This can improve performance significantly on low-end systems by reducing both the CPU and GPU load (as fewer draw calls are needed to draw the scene without shadows).
var positional_shadow_atlas_size: int:
	get = get_positional_shadow_atlas_size, set = set_positional_shadow_atlas_size

## Sets scaling 3D mode. Bilinear scaling renders at different resolution to either undersample or supersample the viewport. FidelityFX Super Resolution 1.0, abbreviated to FSR, is an upscaling technology that produces high quality images at fast framerates by using a spatially aware upscaling algorithm. FSR is slightly more expensive than bilinear, but it produces significantly higher image quality. FSR should be used where possible.
## To control this property on the root viewport, set the [member ProjectSettings.rendering/scaling_3d/mode] project setting.
var scaling_3d_mode: int:
	get = get_scaling_3d_mode, set = set_scaling_3d_mode

## Scales the 3D render buffer based on the viewport size uses an image filter specified in [member ProjectSettings.rendering/scaling_3d/mode] to scale the output image to the full viewport size. Values lower than [code]1.0[/code] can be used to speed up 3D rendering at the cost of quality (undersampling). Values greater than [code]1.0[/code] are only valid for bilinear mode and can be used to improve 3D rendering quality at a high performance cost (supersampling). See also [member ProjectSettings.rendering/anti_aliasing/quality/msaa_3d] for multi-sample antialiasing, which is significantly cheaper but only smooths the edges of polygons.
## When using FSR upscaling, AMD recommends exposing the following values as preset options to users "Ultra Quality: 0.77", "Quality: 0.67", "Balanced: 0.59", "Performance: 0.5" instead of exposing the entire scale.
## To control this property on the root viewport, set the [member ProjectSettings.rendering/scaling_3d/scale] project setting.
var scaling_3d_scale: float:
	get = get_scaling_3d_scale, set = set_scaling_3d_scale

## Sets the screen-space antialiasing method used. Screen-space antialiasing works by selectively blurring edges in a post-process shader. It differs from MSAA which takes multiple coverage samples while rendering objects. Screen-space AA methods are typically faster than MSAA and will smooth out specular aliasing, but tend to make scenes appear blurry.
## See also [member ProjectSettings.rendering/anti_aliasing/quality/screen_space_aa] and [method RenderingServer.viewport_set_screen_space_aa].
var screen_space_aa: int:
	get = get_screen_space_aa, set = set_screen_space_aa

## Controls how much of the original viewport's size should be covered by the 2D signed distance field. This SDF can be sampled in [CanvasItem] shaders and is also used for [GPUParticles2D] collision. Higher values allow portions of occluders located outside the viewport to still be taken into account in the generated signed distance field, at the cost of performance. If you notice particles falling through [LightOccluder2D]s as the occluders leave the viewport, increase this setting.
## The percentage is added on each axis and on both sides. For example, with the default [constant SDF_OVERSIZE_120_PERCENT], the signed distance field will cover 20% of the viewport's size outside the viewport on each side (top, right, bottom, left).
var sdf_oversize: int:
	get = get_sdf_oversize, set = set_sdf_oversize

## The resolution scale to use for the 2D signed distance field. Higher values lead to a more precise and more stable signed distance field as the camera moves, at the cost of performance.
var sdf_scale: int:
	get = get_sdf_scale, set = set_sdf_scale

## If [code]true[/code], [CanvasItem] nodes will internally snap to full pixels. Their position can still be sub-pixel, but the decimals will not have effect. This can lead to a crisper appearance at the cost of less smooth movement, especially when [Camera2D] smoothing is enabled.
var snap_2d_transforms_to_pixel: bool:
	get = is_snap_2d_transforms_to_pixel_enabled, set = set_snap_2d_transforms_to_pixel

## If [code]true[/code], vertices of [CanvasItem] nodes will snap to full pixels. Only affects the final vertex positions, not the transforms. This can lead to a crisper appearance at the cost of less smooth movement, especially when [Camera2D] smoothing is enabled.
var snap_2d_vertices_to_pixel: bool:
	get = is_snap_2d_vertices_to_pixel_enabled, set = set_snap_2d_vertices_to_pixel

## Affects the final texture sharpness by reading from a lower or higher mipmap (also called "texture LOD bias"). Negative values make mipmapped textures sharper but grainier when viewed at a distance, while positive values make mipmapped textures blurrier (even when up close).
## Enabling temporal antialiasing ([member use_taa]) will automatically apply a [code]-0.5[/code] offset to this value, while enabling FXAA ([member screen_space_aa]) will automatically apply a [code]-0.25[/code] offset to this value. If both TAA and FXAA are enabled at the same time, an offset of [code]-0.75[/code] is applied to this value.
## [b]Note:[/b] If [member scaling_3d_scale] is lower than [code]1.0[/code] (exclusive), [member texture_mipmap_bias] is used to adjust the automatic mipmap bias which is calculated internally based on the scale factor. The formula for this is [code]log2(scaling_3d_scale) + mipmap_bias[/code].
## To control this property on the root viewport, set the [member ProjectSettings.rendering/textures/default_filters/texture_mipmap_bias] project setting.
var texture_mipmap_bias: float:
	get = get_texture_mipmap_bias, set = set_texture_mipmap_bias

## If [code]true[/code], the viewport should render its background as transparent.
var transparent_bg: bool:
	get = has_transparent_background, set = set_transparent_background

## If [code]true[/code], uses a fast post-processing filter to make banding significantly less visible in 3D. 2D rendering is [i]not[/i] affected by debanding unless the [member Environment.background_mode] is [constant Environment.BG_CANVAS].
## In some cases, debanding may introduce a slightly noticeable dithering pattern. It's recommended to enable debanding only when actually needed since the dithering pattern will make lossless-compressed screenshots larger.
## See also [member ProjectSettings.rendering/anti_aliasing/quality/use_debanding] and [method RenderingServer.viewport_set_use_debanding].
var use_debanding: bool:
	get = is_using_debanding, set = set_use_debanding

## If [code]true[/code], 2D rendering will use an high dynamic range (HDR) format framebuffer matching the bit depth of the 3D framebuffer. When using the Forward+ renderer this will be an [code]RGBA16[/code] framebuffer, while when using the Mobile renderer it will be an [code]RGB10_A2[/code] framebuffer. Additionally, 2D rendering will take place in linear color space and will be converted to sRGB space immediately before blitting to the screen (if the Viewport is attached to the screen). Practically speaking, this means that the end result of the Viewport will not be clamped into the [code]0-1[/code] range and can be used in 3D rendering without color space adjustments. This allows 2D rendering to take advantage of effects requiring high dynamic range (e.g. 2D glow) as well as substantially improves the appearance of effects requiring highly detailed gradients.
## [b]Note:[/b] This setting will have no effect when using the Compatibility renderer, which always renders in low dynamic range for performance reasons.
var use_hdr_2d: bool:
	get = is_using_hdr_2d, set = set_use_hdr_2d

## If [code]true[/code], [OccluderInstance3D] nodes will be usable for occlusion culling in 3D for this viewport. For the root viewport, [member ProjectSettings.rendering/occlusion_culling/use_occlusion_culling] must be set to [code]true[/code] instead.
## [b]Note:[/b] Enabling occlusion culling has a cost on the CPU. Only enable occlusion culling if you actually plan to use it, and think whether your scene can actually benefit from occlusion culling. Large, open scenes with few or no objects blocking the view will generally not benefit much from occlusion culling. Large open scenes generally benefit more from mesh LOD and visibility ranges ([member GeometryInstance3D.visibility_range_begin] and [member GeometryInstance3D.visibility_range_end]) compared to occlusion culling.
## [b]Note:[/b] Due to memory constraints, occlusion culling is not supported by default in Web export templates. It can be enabled by compiling custom Web export templates with [code]module_raycast_enabled=yes[/code].
var use_occlusion_culling: bool:
	get = is_using_occlusion_culling, set = set_use_occlusion_culling

## Enables temporal antialiasing for this viewport. TAA works by jittering the camera and accumulating the images of the last rendered frames, motion vector rendering is used to account for camera and object motion.
## [b]Note:[/b] The implementation is not complete yet, some visual instances such as particles and skinned meshes may show artifacts.
## See also [member ProjectSettings.rendering/anti_aliasing/quality/use_taa] and [method RenderingServer.viewport_set_use_taa].
var use_taa: bool:
	get = is_using_taa, set = set_use_taa

## If [code]true[/code], the viewport will use the primary XR interface to render XR output. When applicable this can result in a stereoscopic image and the resulting render being output to a headset.
var use_xr: bool:
	get = is_using_xr, set = set_use_xr

## The Variable Rate Shading (VRS) mode that is used for this viewport. Note, if hardware does not support VRS this property is ignored.
var vrs_mode: int:
	get = get_vrs_mode, set = set_vrs_mode

## Texture to use when [member vrs_mode] is set to [constant Viewport.VRS_TEXTURE].
## The texture [i]must[/i] use a lossless compression format so that colors can be matched precisely. The following VRS densities are mapped to various colors, with brighter colors representing a lower level of shading precision:
## [codeblock lang=text]
## - 1×1 = rgb(0, 0, 0)     - #000000
## - 1×2 = rgb(0, 85, 0)    - #005500
## - 2×1 = rgb(85, 0, 0)    - #550000
## - 2×2 = rgb(85, 85, 0)   - #555500
## - 2×4 = rgb(85, 170, 0)  - #55aa00
## - 4×2 = rgb(170, 85, 0)  - #aa5500
## - 4×4 = rgb(170, 170, 0) - #aaaa00
## - 4×8 = rgb(170, 255, 0) - #aaff00 - Not supported on most hardware
## - 8×4 = rgb(255, 170, 0) - #ffaa00 - Not supported on most hardware
## - 8×8 = rgb(255, 255, 0) - #ffff00 - Not supported on most hardware
## [/codeblock]
var vrs_texture: Texture2D:
	get = get_vrs_texture, set = set_vrs_texture

## Sets the update mode for Variable Rate Shading (VRS) for the viewport. VRS requires the input texture to be converted to the format usable by the VRS method supported by the hardware. The update mode defines how often this happens. If the GPU does not support VRS, or VRS is not enabled, this property is ignored.
var vrs_update_mode: int:
	get = get_vrs_update_mode, set = set_vrs_update_mode

## The custom [World2D] which can be used as 2D environment source.
var world_2d: World2D:
	get = get_world_2d, set = set_world_2d

## The custom [World3D] which can be used as 3D environment source.
var world_3d: World3D:
	get = get_world_3d, set = set_world_3d



## Returns the first valid [World2D] for this viewport, searching the [member world_2d] property of itself and any Viewport ancestor.
func find_world_2d() -> World2D:
	pass;

## Returns the first valid [World3D] for this viewport, searching the [member world_3d] property of itself and any Viewport ancestor.
func find_world_3d() -> World3D:
	pass;

## Returns the currently active 2D audio listener. Returns [code]null[/code] if there are no active 2D audio listeners, in which case the active 2D camera will be treated as listener.
func get_audio_listener_2d() -> AudioListener2D:
	pass;

## Returns the currently active 3D audio listener. Returns [code]null[/code] if there are no active 3D audio listeners, in which case the active 3D camera will be treated as listener.
func get_audio_listener_3d() -> AudioListener3D:
	pass;

## Returns the currently active 2D camera. Returns [code]null[/code] if there are no active cameras.
func get_camera_2d() -> Camera2D:
	pass;

## Returns the currently active 3D camera.
func get_camera_3d() -> Camera3D:
	pass;

## Returns an individual bit on the rendering layer mask.
func get_canvas_cull_mask_bit(layer: int) -> bool:
	pass;

## Returns a list of the visible embedded [Window]s inside the viewport.
## [b]Note:[/b] [Window]s inside other viewports will not be listed.
func get_embedded_subwindows() -> Array[Window]:
	pass;

## Returns the transform from the viewport's coordinate system to the embedder's coordinate system.
func get_final_transform() -> Transform2D:
	pass;

## Returns the mouse's position in this [Viewport] using the coordinate system of this [Viewport].
func get_mouse_position() -> Vector2:
	pass;

## Returns the positional shadow atlas quadrant subdivision of the specified quadrant.
func get_positional_shadow_atlas_quadrant_subdiv(quadrant: int) -> int:
	pass;

## Returns rendering statistics of the given type. See [enum RenderInfoType] and [enum RenderInfo] for options.
func get_render_info(type: int, info: int) -> int:
	pass;

## Returns the transform from the Viewport's coordinates to the screen coordinates of the containing window manager window.
func get_screen_transform() -> Transform2D:
	pass;

## Returns the automatically computed 2D stretch transform, taking the [Viewport]'s stretch settings into account. The final value is multiplied by [member Window.content_scale_factor], but only for the root viewport. If this method is called on a [SubViewport] (e.g., in a scene tree with [SubViewportContainer] and [SubViewport]), the scale factor of the root window will not be applied. Using [method Transform2D.get_scale] on the returned value, this can be used to compensate for scaling when zooming a [Camera2D] node, or to scale down a [TextureRect] to be pixel-perfect regardless of the automatically computed scale factor.
## [b]Note:[/b] Due to how pixel scaling works, the returned transform's X and Y scale may differ slightly, even when [member Window.content_scale_aspect] is set to a mode that preserves the pixels' aspect ratio. If [member Window.content_scale_aspect] is [constant Window.CONTENT_SCALE_ASPECT_IGNORE], the X and Y scale may differ [i]significantly[/i].
func get_stretch_transform() -> Transform2D:
	pass;

## Returns the viewport's texture.
## [b]Note:[/b] When trying to store the current texture (e.g. in a file), it might be completely black or outdated if used too early, especially when used in e.g. [method Node._ready]. To make sure the texture you get is correct, you can await [signal RenderingServer.frame_post_draw] signal.
## [codeblock]
## func _ready():
## await RenderingServer.frame_post_draw
## $Viewport.get_texture().get_image().save_png("user://Screenshot.png")
## [/codeblock]
## [b]Note:[/b] When [member use_hdr_2d] is [code]true[/code] the returned texture will be an HDR image encoded in linear space.
func get_texture() -> ViewportTexture:
	pass;

## Returns the viewport's RID from the [RenderingServer].
func get_viewport_rid() -> RID:
	pass;

## Returns the visible rectangle in global screen coordinates.
func get_visible_rect() -> Rect2:
	pass;

## Cancels the drag operation that was previously started through [method Control._get_drag_data] or forced with [method Control.force_drag].
func gui_cancel_drag() -> void:
	pass;

## Returns the drag data from the GUI, that was previously returned by [method Control._get_drag_data].
func gui_get_drag_data() -> Variant:
	pass;

## Returns the currently focused [Control] within this viewport. If no [Control] is focused, returns [code]null[/code].
func gui_get_focus_owner() -> Control:
	pass;

## Returns the [Control] that the mouse is currently hovering over in this viewport. If no [Control] has the cursor, returns [code]null[/code].
## Typically the leaf [Control] node or deepest level of the subtree which claims hover. This is very useful when used together with [method Node.is_ancestor_of] to find if the mouse is within a control tree.
func gui_get_hovered_control() -> Control:
	pass;

## Returns [code]true[/code] if the drag operation is successful.
func gui_is_drag_successful() -> bool:
	pass;

## Returns [code]true[/code] if a drag operation is currently ongoing and where the drop action could happen in this viewport.
## Alternative to [constant Node.NOTIFICATION_DRAG_BEGIN] and [constant Node.NOTIFICATION_DRAG_END] when you prefer polling the value.
func gui_is_dragging() -> bool:
	pass;

## Removes the focus from the currently focused [Control] within this viewport. If no [Control] has the focus, does nothing.
func gui_release_focus() -> void:
	pass;

## Returns whether the current [InputEvent] has been handled. Input events are not handled until [method set_input_as_handled] has been called during the lifetime of an [InputEvent].
## This is usually done as part of input handling methods like [method Node._input], [method Control._gui_input] or others, as well as in corresponding signal handlers.
## If [member handle_input_locally] is set to [code]false[/code], this method will try finding the first parent viewport that is set to handle input locally, and return its value for [method is_input_handled] instead.
func is_input_handled() -> bool:
	pass;

## Inform the Viewport that the mouse has entered its area. Use this function before sending an [InputEventMouseButton] or [InputEventMouseMotion] to the [Viewport] with [method Viewport.push_input]. See also [method notify_mouse_exited].
## [b]Note:[/b] In most cases, it is not necessary to call this function because [SubViewport] nodes that are children of [SubViewportContainer] are notified automatically. This is only necessary when interacting with viewports in non-default ways, for example as textures in [TextureRect] or with an [Area3D] that forwards input events.
func notify_mouse_entered() -> void:
	pass;

## Inform the Viewport that the mouse has left its area. Use this function when the node that displays the viewport notices the mouse has left the area of the displayed viewport. See also [method notify_mouse_entered].
## [b]Note:[/b] In most cases, it is not necessary to call this function because [SubViewport] nodes that are children of [SubViewportContainer] are notified automatically. This is only necessary when interacting with viewports in non-default ways, for example as textures in [TextureRect] or with an [Area3D] that forwards input events.
func notify_mouse_exited() -> void:
	pass;

## Triggers the given [param event] in this [Viewport]. This can be used to pass an [InputEvent] between viewports, or to locally apply inputs that were sent over the network or saved to a file.
## If [param in_local_coords] is [code]false[/code], the event's position is in the embedder's coordinates and will be converted to viewport coordinates. If [param in_local_coords] is [code]true[/code], the event's position is in viewport coordinates.
## While this method serves a similar purpose as [method Input.parse_input_event], it does not remap the specified [param event] based on project settings like [member ProjectSettings.input_devices/pointing/emulate_touch_from_mouse].
## Calling this method will propagate calls to child nodes for following methods in the given order:
## - [method Node._input]
## - [method Control._gui_input] for [Control] nodes
## - [method Node._shortcut_input]
## - [method Node._unhandled_key_input]
## - [method Node._unhandled_input]
## If an earlier method marks the input as handled via [method set_input_as_handled], any later method in this list will not be called.
## If none of the methods handle the event and [member physics_object_picking] is [code]true[/code], the event is used for physics object picking.
func push_input(event: InputEvent, in_local_coords: bool = false) -> void:
	pass;

## Helper method which calls the [code]set_text()[/code] method on the currently focused [Control], provided that it is defined (e.g. if the focused Control is [Button] or [LineEdit]).
func push_text_input(text: String) -> void:
	pass;

## Triggers the given [param event] in this [Viewport]. This can be used to pass an [InputEvent] between viewports, or to locally apply inputs that were sent over the network or saved to a file.
## If [param in_local_coords] is [code]false[/code], the event's position is in the embedder's coordinates and will be converted to viewport coordinates. If [param in_local_coords] is [code]true[/code], the event's position is in viewport coordinates.
## Calling this method will propagate calls to child nodes for following methods in the given order:
## - [method Node._shortcut_input]
## - [method Node._unhandled_key_input]
## - [method Node._unhandled_input]
## If an earlier method marks the input as handled via [method set_input_as_handled], any later method in this list will not be called.
## If none of the methods handle the event and [member physics_object_picking] is [code]true[/code], the event is used for physics object picking.
## [b]Note:[/b] This method doesn't propagate input events to embedded [Window]s or [SubViewport]s.
func push_unhandled_input(event: InputEvent, in_local_coords: bool = false) -> void:
	pass;

## Set/clear individual bits on the rendering layer mask. This simplifies editing this [Viewport]'s layers.
func set_canvas_cull_mask_bit(layer: int, enable: bool) -> void:
	pass;

## Stops the input from propagating further down the [SceneTree].
## [b]Note:[/b] This does not affect the methods in [Input], only the way events are propagated.
func set_input_as_handled() -> void:
	pass;

## Sets the number of subdivisions to use in the specified quadrant. A higher number of subdivisions allows you to have more shadows in the scene at once, but reduces the quality of the shadows. A good practice is to have quadrants with a varying number of subdivisions and to have as few subdivisions as possible.
func set_positional_shadow_atlas_quadrant_subdiv(quadrant: int, subdiv: int) -> void:
	pass;

## Force instantly updating the display based on the current mouse cursor position. This includes updating the mouse cursor shape and sending necessary [signal Control.mouse_entered], [signal CollisionObject2D.mouse_entered], [signal CollisionObject3D.mouse_entered] and [signal Window.mouse_entered] signals and their respective [code]mouse_exited[/code] counterparts.
func update_mouse_cursor_state() -> void:
	pass;

## Moves the mouse pointer to the specified position in this [Viewport] using the coordinate system of this [Viewport].
## [b]Note:[/b] [method warp_mouse] is only supported on Windows, macOS and Linux. It has no effect on Android, iOS and Web.
func warp_mouse(position: Vector2) -> void:
	pass;


func get_anisotropic_filtering_level() -> int:
	return anisotropic_filtering_level

func set_anisotropic_filtering_level(value: int) -> void:
	anisotropic_filtering_level = value

func is_audio_listener_2d() -> bool:
	return audio_listener_enable_2d

func set_as_audio_listener_2d(value: bool) -> void:
	audio_listener_enable_2d = value

func is_audio_listener_3d() -> bool:
	return audio_listener_enable_3d

func set_as_audio_listener_3d(value: bool) -> void:
	audio_listener_enable_3d = value

func get_canvas_cull_mask() -> int:
	return canvas_cull_mask

func set_canvas_cull_mask(value: int) -> void:
	canvas_cull_mask = value

func get_default_canvas_item_texture_filter() -> int:
	return canvas_item_default_texture_filter

func set_default_canvas_item_texture_filter(value: int) -> void:
	canvas_item_default_texture_filter = value

func get_default_canvas_item_texture_repeat() -> int:
	return canvas_item_default_texture_repeat

func set_default_canvas_item_texture_repeat(value: int) -> void:
	canvas_item_default_texture_repeat = value

func get_canvas_transform() -> Transform2D:
	return canvas_transform

func set_canvas_transform(value: Transform2D) -> void:
	canvas_transform = value

func get_debug_draw() -> int:
	return debug_draw

func set_debug_draw(value: int) -> void:
	debug_draw = value

func is_3d_disabled() -> bool:
	return disable_3d

func set_disable_3d(value: bool) -> void:
	disable_3d = value

func get_fsr_sharpness() -> float:
	return fsr_sharpness

func set_fsr_sharpness(value: float) -> void:
	fsr_sharpness = value

func get_global_canvas_transform() -> Transform2D:
	return global_canvas_transform

func set_global_canvas_transform(value: Transform2D) -> void:
	global_canvas_transform = value

func is_input_disabled() -> bool:
	return gui_disable_input

func set_disable_input(value: bool) -> void:
	gui_disable_input = value

func is_embedding_subwindows() -> bool:
	return gui_embed_subwindows

func set_embedding_subwindows(value: bool) -> void:
	gui_embed_subwindows = value

func is_snap_controls_to_pixels_enabled() -> bool:
	return gui_snap_controls_to_pixels

func set_snap_controls_to_pixels(value: bool) -> void:
	gui_snap_controls_to_pixels = value

func is_handling_input_locally() -> bool:
	return handle_input_locally

func set_handle_input_locally(value: bool) -> void:
	handle_input_locally = value

func get_mesh_lod_threshold() -> float:
	return mesh_lod_threshold

func set_mesh_lod_threshold(value: float) -> void:
	mesh_lod_threshold = value

func get_msaa_2d() -> int:
	return msaa_2d

func set_msaa_2d(value: int) -> void:
	msaa_2d = value

func get_msaa_3d() -> int:
	return msaa_3d

func set_msaa_3d(value: int) -> void:
	msaa_3d = value

func is_using_own_world_3d() -> bool:
	return own_world_3d

func set_use_own_world_3d(value: bool) -> void:
	own_world_3d = value

func get_physics_interpolation_mode() -> int:
	return physics_interpolation_mode

func set_physics_interpolation_mode(value: int) -> void:
	physics_interpolation_mode = value

func get_physics_object_picking() -> bool:
	return physics_object_picking

func set_physics_object_picking(value: bool) -> void:
	physics_object_picking = value

func get_physics_object_picking_first_only() -> bool:
	return physics_object_picking_first_only

func set_physics_object_picking_first_only(value: bool) -> void:
	physics_object_picking_first_only = value

func get_physics_object_picking_sort() -> bool:
	return physics_object_picking_sort

func set_physics_object_picking_sort(value: bool) -> void:
	physics_object_picking_sort = value

func get_positional_shadow_atlas_16_bits() -> bool:
	return positional_shadow_atlas_16_bits

func set_positional_shadow_atlas_16_bits(value: bool) -> void:
	positional_shadow_atlas_16_bits = value

func get_positional_shadow_atlas_quadrant_subdiv() -> int:
	return positional_shadow_atlas_quad_0

func set_positional_shadow_atlas_quadrant_subdiv(value: int) -> void:
	positional_shadow_atlas_quad_0 = value

func get_positional_shadow_atlas_quadrant_subdiv() -> int:
	return positional_shadow_atlas_quad_1

func set_positional_shadow_atlas_quadrant_subdiv(value: int) -> void:
	positional_shadow_atlas_quad_1 = value

func get_positional_shadow_atlas_quadrant_subdiv() -> int:
	return positional_shadow_atlas_quad_2

func set_positional_shadow_atlas_quadrant_subdiv(value: int) -> void:
	positional_shadow_atlas_quad_2 = value

func get_positional_shadow_atlas_quadrant_subdiv() -> int:
	return positional_shadow_atlas_quad_3

func set_positional_shadow_atlas_quadrant_subdiv(value: int) -> void:
	positional_shadow_atlas_quad_3 = value

func get_positional_shadow_atlas_size() -> int:
	return positional_shadow_atlas_size

func set_positional_shadow_atlas_size(value: int) -> void:
	positional_shadow_atlas_size = value

func get_scaling_3d_mode() -> int:
	return scaling_3d_mode

func set_scaling_3d_mode(value: int) -> void:
	scaling_3d_mode = value

func get_scaling_3d_scale() -> float:
	return scaling_3d_scale

func set_scaling_3d_scale(value: float) -> void:
	scaling_3d_scale = value

func get_screen_space_aa() -> int:
	return screen_space_aa

func set_screen_space_aa(value: int) -> void:
	screen_space_aa = value

func get_sdf_oversize() -> int:
	return sdf_oversize

func set_sdf_oversize(value: int) -> void:
	sdf_oversize = value

func get_sdf_scale() -> int:
	return sdf_scale

func set_sdf_scale(value: int) -> void:
	sdf_scale = value

func is_snap_2d_transforms_to_pixel_enabled() -> bool:
	return snap_2d_transforms_to_pixel

func set_snap_2d_transforms_to_pixel(value: bool) -> void:
	snap_2d_transforms_to_pixel = value

func is_snap_2d_vertices_to_pixel_enabled() -> bool:
	return snap_2d_vertices_to_pixel

func set_snap_2d_vertices_to_pixel(value: bool) -> void:
	snap_2d_vertices_to_pixel = value

func get_texture_mipmap_bias() -> float:
	return texture_mipmap_bias

func set_texture_mipmap_bias(value: float) -> void:
	texture_mipmap_bias = value

func has_transparent_background() -> bool:
	return transparent_bg

func set_transparent_background(value: bool) -> void:
	transparent_bg = value

func is_using_debanding() -> bool:
	return use_debanding

func set_use_debanding(value: bool) -> void:
	use_debanding = value

func is_using_hdr_2d() -> bool:
	return use_hdr_2d

func set_use_hdr_2d(value: bool) -> void:
	use_hdr_2d = value

func is_using_occlusion_culling() -> bool:
	return use_occlusion_culling

func set_use_occlusion_culling(value: bool) -> void:
	use_occlusion_culling = value

func is_using_taa() -> bool:
	return use_taa

func set_use_taa(value: bool) -> void:
	use_taa = value

func is_using_xr() -> bool:
	return use_xr

func set_use_xr(value: bool) -> void:
	use_xr = value

func get_vrs_mode() -> int:
	return vrs_mode

func set_vrs_mode(value: int) -> void:
	vrs_mode = value

func get_vrs_texture() -> Texture2D:
	return vrs_texture

func set_vrs_texture(value: Texture2D) -> void:
	vrs_texture = value

func get_vrs_update_mode() -> int:
	return vrs_update_mode

func set_vrs_update_mode(value: int) -> void:
	vrs_update_mode = value

func get_world_2d() -> World2D:
	return world_2d

func set_world_2d(value: World2D) -> void:
	world_2d = value

func get_world_3d() -> World3D:
	return world_3d

func set_world_3d(value: World3D) -> void:
	world_3d = value

