extends Viewport
class_name SubViewport

## An interface to a game world that doesn't create a window or draw to the screen directly.
##
## [SubViewport] Isolates a rectangular region of a scene to be displayed independently. This can be used, for example, to display UI in 3D space.
## [b]Note:[/b] [SubViewport] is a [Viewport] that isn't a [Window], i.e. it doesn't draw anything by itself. To display anything, [SubViewport] must have a non-zero size and be either put inside a [SubViewportContainer] or assigned to a [ViewportTexture].
## [b]Note:[/b] [InputEvent]s are not passed to a standalone [SubViewport] by default. To ensure [InputEvent] propagation, a [SubViewport] can be placed inside of a [SubViewportContainer].
##
## @tutorial(Using Viewports): https://docs.godotengine.org/en/stable/tutorials/rendering/viewports.html
## @tutorial(Viewport and canvas transforms): https://docs.godotengine.org/en/stable/tutorials/2d/2d_transforms.html
## @tutorial(GUI in 3D Viewport Demo): https://godotengine.org/asset-library/asset/2807
## @tutorial(3D in 2D Viewport Demo): https://godotengine.org/asset-library/asset/2804
## @tutorial(2D in 3D Viewport Demo): https://godotengine.org/asset-library/asset/2803
## @tutorial(Screen Capture Demo): https://godotengine.org/asset-library/asset/2808
## @tutorial(Dynamic Split Screen Demo): https://godotengine.org/asset-library/asset/2806
## @tutorial(3D Resolution Scaling Demo): https://godotengine.org/asset-library/asset/2805


## Always clear the render target before drawing.
## Never clear the render target.
## Clear the render target on the next frame, then switch to [constant CLEAR_MODE_NEVER].
## Do not update the render target.
## Update the render target once, then switch to [constant UPDATE_DISABLED].
## Update the render target only when it is visible. This is the default value.
## Update the render target only when its parent is visible.
## Always update the render target.

#enum ClearMode
enum {
    CLEAR_MODE_ALWAYS = 0,
    CLEAR_MODE_NEVER = 1,
    CLEAR_MODE_ONCE = 2,
}
#enum UpdateMode
enum {
    UPDATE_DISABLED = 0,
    UPDATE_ONCE = 1,
    UPDATE_WHEN_VISIBLE = 2,
    UPDATE_WHEN_PARENT_VISIBLE = 3,
    UPDATE_ALWAYS = 4,
}
## The clear mode when the sub-viewport is used as a render target.
## [b]Note:[/b] This property is intended for 2D usage.
var render_target_clear_mode: int:
	get = get_clear_mode, set = set_clear_mode

## The update mode when the sub-viewport is used as a render target.
var render_target_update_mode: int:
	get = get_update_mode, set = set_update_mode

## The width and height of the sub-viewport. Must be set to a value greater than or equal to 2 pixels on both dimensions. Otherwise, nothing will be displayed.
## [b]Note:[/b] If the parent node is a [SubViewportContainer] and its [member SubViewportContainer.stretch] is [code]true[/code], the viewport size cannot be changed manually.
var size: Vector2i:
	get = get_size, set = set_size

## The 2D size override of the sub-viewport. If either the width or height is [code]0[/code], the override is disabled.
var size_2d_override: Vector2i:
	get = get_size_2d_override, set = set_size_2d_override

## If [code]true[/code], the 2D size override affects stretch as well.
var size_2d_override_stretch: bool:
	get = is_size_2d_override_stretch_enabled, set = set_size_2d_override_stretch




func get_clear_mode() -> int:
	return render_target_clear_mode

func set_clear_mode(value: int) -> void:
	render_target_clear_mode = value

func get_update_mode() -> int:
	return render_target_update_mode

func set_update_mode(value: int) -> void:
	render_target_update_mode = value

func get_size() -> Vector2i:
	return size

func set_size(value: Vector2i) -> void:
	size = value

func get_size_2d_override() -> Vector2i:
	return size_2d_override

func set_size_2d_override(value: Vector2i) -> void:
	size_2d_override = value

func is_size_2d_override_stretch_enabled() -> bool:
	return size_2d_override_stretch

func set_size_2d_override_stretch(value: bool) -> void:
	size_2d_override_stretch = value

